(**
   Implements popup menues.

  TODO
  * Add support for check menu and radio menu

  * Cleanup code and share more code with VOWindow
**)

MODULE VOMenu;

(*
    Implements menues.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)


IMPORT D   := VODisplay,
       E   := VOEvent,
       F   := VOFrame,
       G   := VOGUIObject,
       K   := VOKeyHandler,
       O   := VOObject,
       T   := VOText,
       U   := VOUtil,
       V   := VOVecImage;

CONST
  selectedMsg * = 0; (* a menuitem has been selected *)
  closeMsg    * = 1; (* a toplevel menu has been closed *)

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 menuFrame*,            (* the frame to use for the menu *)
                 stripFrame* : LONGINT; (* the frame to use for the menustrip *)
               END;

  (* Menustrip stuff *)

  StripEntry     = POINTER TO StripEntryDesc;
  MenuStrip*     = POINTER TO MenuStripDesc;

  (* menu stuff *)

  Menu*         = POINTER TO MenuDesc;

  MenuEntry     = POINTER TO MenuEntryDesc;
  MenuEntryDesc = RECORD (G.GadgetDesc)
                    w1,w2  : LONGINT;
                  END;

  (* Menustrip stuff *)

  StripEntryDesc = RECORD
                    next : StripEntry;
                    text : T.Text;
                    menu : Menu;
                  END;


  MenuStripDesc* = RECORD(G.GroupDesc)
                     prefs      : Prefs;

                     frame      : F.Frame;
                     menuList,
                     lastMenu,
                     current    : StripEntry;
                     selected   : BOOLEAN;
                     selId      : LONGINT;
                   END;

  (* menu stuff *)

  MenuItem      = POINTER TO MenuItemDesc;
  MenuItemDesc  = RECORD (MenuEntryDesc)
                    name,
                    shortcut : G.Object;
                  END;

  MenuTitle     = POINTER TO MenuTitleDesc;
  MenuTitleDesc = RECORD (MenuEntryDesc)
                    name     : G.Object;
                  END;

  Separator     = POINTER TO SeparatorDesc;
  SeparatorDesc = RECORD (MenuEntryDesc)
                  END;

  SubMenu       = POINTER TO SubMenuDesc;
  SubMenuDesc   = RECORD(MenuEntryDesc)
                    name    : G.Object;
                    arrow   : V.VecImage;
                    subMenu : Menu;
                    subOpen : BOOLEAN;
                  END;

  MenuDesc*     = RECORD (D.WindowDesc)
                    prefs            : Prefs;
                    backgroundObject : G.Background;
                    parentMenu       : SubMenu;

                    child            : Menu;

                    frame            : F.Frame;   (* frames menu *)

                    selected         : MenuEntry; (* the currently selected entry *)

                    count            : LONGINT;   (* number of menuentries *)
                    list,last        : MenuEntry; (* menuentry-list *)

                    (* Pulldown stuff *)
                    reference        : G.Object;
                    strip            : MenuStrip;
                    offset           : LONGINT;

                    popup            : BOOLEAN;   (* Top menu of a popup menu *)
                    pullDown         : BOOLEAN;   (* This menu is the top of a pullDown *)
                  END;

  SelectedMsg*     = POINTER TO SelectedMsgDesc;
  SelectedMsgDesc* = RECORD (O.MessageDesc)
                       id* : LONGINT;
                     END;

  CloseMsg*     = POINTER TO CloseMsgDesc;
  CloseMsgDesc* = RECORD (O.MessageDesc)
                  END;

VAR
  prefs* : Prefs;

  (**
    Initalisation of preferences.
  **)

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.menuFrame:=F.double3DOut;
    p.stripFrame:=F.double3DOut;
  END Init;

  PROCEDURE (p : Prefs) SetPrefs(m : Menu);

  BEGIN
    m.prefs:=p;   (* We set the prefs *)

    IF p.background#NIL THEN
      m.backgroundObject:=p.background.Copy();
      m.backgroundObject.source:=NIL;
    END;
  END SetPrefs;

  (* ---------- Menu stuff ------------------*)

  PROCEDURE InitMenu*(d : D.Display):Menu;

  VAR
    m : Menu;

  BEGIN
    NEW(m);
    m.Init;

    m.SetDisplay(d);

    RETURN m;
  END InitMenu;

  PROCEDURE (m : Menu) Init*;

  BEGIN
    m.Init^;

    prefs.SetPrefs(m);

    m.CreateAllways(TRUE);
    m.Borderless(TRUE);

    m.count:=0;
    m.list:=NIL;
    m.last:=NIL;
    m.selected:=NIL;
    m.parentMenu:=NIL;
    m.child:=NIL;
    m.reference:=NIL;
    m.strip:=NIL;
    m.popup:=TRUE;
    m.pullDown:=FALSE;

    NEW(m.frame);
    m.frame.Init;
    m.frame.SetFlags({G.horizontalFlex,G.verticalFlex});
  END Init;

  (**
    This makes the menu a pulldown relative to the given object.

    NOTE
    The pulldown flag gets cleared after closing the menu, you must explizitely
    set it before calling Menu.Open.
  **)

  PROCEDURE (m : Menu) SetPullDown(strip : MenuStrip; reference : G.Object; offset : LONGINT);

  BEGIN
    m.pullDown:=TRUE;
    m.reference:=reference;
    m.strip:=strip;
    m.offset:=offset;
  END SetPullDown;

  (* ---------- Menustrip stuff ------------------*)

  PROCEDURE (m : MenuStrip) Init*;

  BEGIN
    m.Init^;

    m.SetFlags({G.horizontalFlex});

    m.prefs:=prefs;
    m.SetBackgroundObject(m.prefs.background);

    NEW(m.frame);
    m.frame.Init;
    m.frame.SetFlags({G.horizontalFlex,G.verticalFlex});

    m.current:=NIL;
  END Init;

  (**
    Adds an entry to the menustrip and bind that entry to a menu.
  **)

  PROCEDURE(m : MenuStrip) AddTextStrip*(title : ARRAY OF CHAR; menu : Menu);

  VAR
    entry : StripEntry;
    help  : U.Text;

  BEGIN
    NEW(entry);
    NEW(entry.text); (* Allocate a VOText.Text object for displaying text *)
    entry.text.Init; (* We must call Text.Init asfor all objects *)
    entry.text.SetFlags({G.horizontalFlex,G.verticalFlex}); (* Our text should be resizeable in all directions *)
    entry.text.SetBackgroundObject(menu.backgroundObject);
    entry.text.SetDefault(T.centered,{},D.normalFont); (* We want the text of tex textimage centered in its bounds *)
    help:=U.EscapeString(title);    (* We escape the string because it could contain escape-sequences.
                                       This is because Oberon-2 does not specify an escape-character for strings *)
    entry.text.SetText(help^);            (* Set the escaped text to the Text-object *)

    entry.menu:=menu;

    IF m.menuList=NIL THEN
      m.menuList:=entry;
    ELSE
      m.lastMenu.next:=entry;
    END;
    m.lastMenu:=entry;
  END AddTextStrip;

  PROCEDURE (m : MenuStrip) CalcSize*(display : D.Display);

  VAR
    entry : StripEntry;

  BEGIN
    m.width:=0;
    m.height:=0;

    m.frame.SetInternalFrame(m.prefs.stripFrame);
    m.frame.CalcSize(display);

    entry:=m.menuList;
    WHILE entry#NIL DO

      entry.text.CalcSize(display);
      entry.text.Resize(entry.text.width+2*display.spaceWidth,-1);
      m.height:=G.MaxLong(m.height,entry.text.oHeight);
      INC(m.width,entry.text.oWidth);

      entry:=entry.next;
    END;

    INC(m.height,m.frame.topBorder+display.spaceHeight+m.frame.bottomBorder);

    m.CalcSize^(display);
  END CalcSize;

  PROCEDURE (m : MenuStrip) GetFocus*(event : E.Event):G.Object;

  VAR
    entry : StripEntry;

  BEGIN
    IF ~m.visible OR m.disabled THEN
      RETURN NIL;
    END;

    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseDown) & m.PointIsIn(event.x,event.y) & (event.qualifier={})
      & ((event.button=E.button1) OR (event.button=E.button3)) THEN
        entry:=m.menuList;
        WHILE (entry#NIL) & ~(entry.text.PointIsIn(event.x,event.y)) DO
          entry:=entry.next;
        END;

        IF entry#NIL THEN
          m.selected:=FALSE;

          m.current:=entry;
          entry.menu.SetPullDown(m,entry.text,-m.frame.leftBorder);
          entry.menu.Open();
          RETURN m;
        END;
      END;
    ELSE
    END;

    RETURN NIL;
  END GetFocus;

  PROCEDURE (m : MenuStrip) HandleEvent*(event : E.Event):BOOLEAN;

  VAR
    entry    : StripEntry;
    selected : SelectedMsg;

  BEGIN
    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseUp)
      &  ((event.button=E.button1) OR (event.button=E.button3)) THEN
        m.current:=NIL;

        IF m.selected THEN
          NEW(selected);
          selected.id:=m.selId;
          m.Send(selected,selectedMsg);
        END;

        RETURN TRUE;
      END;
    | event : E.MotionEvent DO
      IF  (m.current=NIL) OR ~m.current.menu.open THEN
        entry:=m.menuList;
        WHILE (entry#NIL) & ~entry.text.PointIsIn(event.x,event.y) DO
          entry:=entry.next;
        END;
        IF (entry#NIL) & ((entry#m.current) OR ~m.current.menu.maped) THEN
          m.current:=entry;
          entry.menu.SetPullDown(m,entry.text,-m.frame.leftBorder);
          entry.menu.Open;
        END;
      END;
    ELSE
    END;

    RETURN FALSE;
  END HandleEvent;

  PROCEDURE (m : MenuStrip) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  VAR
    entry : StripEntry;
    pos   : LONGINT;

  BEGIN
    m.Draw^(x,y,draw);

    m.DrawBackground(m.x,m.y,m.width,m.height);

    m.frame.Resize(m.width,m.height);
    m.frame.Draw(m.x,m.y,draw);

    pos:=m.x+m.frame.leftBorder;
    entry:=m.menuList;
    WHILE entry#NIL DO
      entry.text.Resize(-1,m.height-m.frame.topBorder-m.frame.bottomBorder);
      entry.text.Draw(pos,m.y + (m.height-entry.text.oHeight) DIV 2,draw);
      INC(pos,entry.text.width);

      entry:=entry.next;
    END;
  END Draw;

  PROCEDURE (m : MenuStrip) Refresh*(x,y,w,h : LONGINT);

  VAR
    entry : StripEntry;

  BEGIN
    IF m.visible & m.Intersect(x,y,w,h) THEN
      m.RestrictToBounds(x,y,w,h);
      m.DrawBackground(x,y,w,h);
      entry:=m.menuList;
      WHILE entry#NIL DO
        entry.text.Refresh(x,y,w,h);
        entry:=entry.next;
      END;
      m.frame.Refresh(x,y,w,h);
    END;
  END Refresh;

  PROCEDURE (m : MenuStrip) Hide*;

  VAR
    entry : StripEntry;

  BEGIN
    IF m.visible THEN
      entry:=m.menuList;
      WHILE entry#NIL DO
        entry.text.Hide;
        entry:=entry.next;
      END;
      m.frame.Hide;
    END;
    m.Hide^;
  END Hide;





  (* ---------- Menu stuff ------------------*)

  PROCEDURE (e : MenuEntry) ResizeMenu(w1,w2, height : LONGINT);

  BEGIN
    e.w1:=w1;
    e.w2:=w2;
(*    e.Resize(w1+w2,height);*)
    e.width:=w1+w2;
    e.height:=height;
  END ResizeMenu;

  PROCEDURE (e : MenuItem) Init*;

  BEGIN
    e.Init^;

    INCL(e.flags,G.handleSC);
    INCL(e.flags,G.scAlways);
  END Init;

  (* MenuItem *)

  PROCEDURE (i : MenuItem) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  BEGIN
    i.Draw^(x,y,draw);

    IF D.selected IN draw.mode THEN
      draw.PushForeground(D.fillColor);
      draw.FillRectangle(i.x,i.y,i.width,i.height);
      draw.PopForeground;
    ELSE
      i.DrawBackground(i.x,i.y,i.width,i.height);
    END;


    i.name.Draw(i.x+i.display.spaceWidth DIV 2,
                i.y+(i.height-i.name.height) DIV 2,draw);
    IF i.shortcut#NIL THEN
      i.shortcut.Draw(i.x+i.w1+2*i.display.spaceWidth,
                      i.y+(i.height-i.shortcut.height) DIV 2,draw);
    END;
  END Draw;

  PROCEDURE (i : MenuItem) HandleShortcutEvent*(id,state : LONGINT);

  VAR
    menu     : Menu;
    help     : D.Window;
    selected : SelectedMsg;

  BEGIN
    IF state=G.pressed THEN
      (* Nothing to do *)
    ELSE
      IF state=G.released THEN
        help:=i.GetMenuObject();
        menu:=help(Menu);
        WHILE menu.parentMenu#NIL DO
          help:=menu.parentMenu.GetMenuObject();
          menu:=help(Menu);
        END;
        NEW(selected);
        selected.id:=i.id;
        IF menu.pullDown THEN (* send msg after menu has been closed! *)
          menu.strip.Send(selected,selectedMsg);
        ELSE
          menu.Send(selected,selectedMsg);
        END;
      ELSE
        (* Nothing to do *)
      END;
    END;
  END HandleShortcutEvent;

  (* MenuTitle *)

  PROCEDURE (i : MenuTitle) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  BEGIN
    i.Draw^(x,y,draw);

    i.DrawBackground(i.x,i.y,i.width,i.height);

    draw.mode:={};
    i.name.Draw(i.x+(i.width-i.name.width) DIV 2,i.y,draw);
    draw.PushForeground(D.shadowColor);
    draw.DrawLine(i.x,i.y+i.name.height,i.x+i.width-1,i.y+i.name.height);
    draw.PopForeground;
    draw.PushForeground(D.shineColor);
    draw.DrawLine(i.x,i.y+i.name.height+1,i.x+i.width-1,i.y+i.name.height+1);
    draw.PopForeground;
  END Draw;

  PROCEDURE (i : MenuTitle) ResizeMenu(w1,w2, height : LONGINT);

  BEGIN
    i.ResizeMenu^(w1,w2,height);
    i.name.Resize(w1+w2,height-3);
  END ResizeMenu;

  (* Separator *)

  PROCEDURE (s : Separator) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  BEGIN
    s.Draw^(x,y,draw);

    s.DrawBackground(s.x,s.y,s.width,s.height);

    draw.PushForeground(D.shadowColor);
    draw.DrawLine(s.x,s.y+s.display.spaceHeight DIV 2,
                  s.x+s.width-1,s.y+s.display.spaceHeight DIV 2);
    draw.PopForeground;
    draw.PushForeground(D.shineColor);
    draw.DrawLine(s.x,s.y+1+s.display.spaceHeight DIV 2,
                  s.x+s.width-1,s.y+1+s.display.spaceHeight DIV 2);
    draw.PopForeground;
  END Draw;

  (* SubMenuI *)

  PROCEDURE (s : SubMenu) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  VAR
    help : D.Window;

  BEGIN
    s.Draw^(x,y,draw);

    s.DrawBackground(s.x,s.y,s.width,s.height);

    IF (D.selected IN draw.mode) & ~s.subOpen THEN
      s.subMenu.Open;
      help:=s.GetMenuObject();
      help(Menu).child:=s.subMenu;
      s.subOpen:=TRUE;
    ELSIF s.subOpen THEN
      s.subMenu.Close;
    END;
    s.arrow.Draw(s.x+s.w1+s.w2-s.arrow.width-s.display.spaceWidth DIV 2,
                 s.y+s.display.spaceHeight DIV 2,draw);
    draw.mode:={};
    s.name.Draw(s.x+s.display.spaceWidth DIV 2,s.y+s.display.spaceHeight DIV 2,draw);
  END Draw;

  (**
     Initializes an empty menu.
  **)

  PROCEDURE (m : Menu) Add(entry : MenuEntry);

  BEGIN
    IF m.list=NIL THEN
      m.list:=entry;
    ELSE
      m.last.next:=entry;
    END;
    m.last:=entry;
    entry.SetMenuObject(m);
    entry.display:=m.display;
    entry.SetBackgroundObject(m.backgroundObject);

    INC(m.count);
  END Add;

  PROCEDURE (m : Menu) AddItem*(name, shortcut : G.Object; id : LONGINT);

  VAR
    item : MenuItem;

  BEGIN
    NEW(item);
    item.Init;

    item.name:=name;
    item.shortcut:=shortcut;
    item.SetId(id);

    item.name.SetFlags({G.horizontalFlex,G.verticalFlex});
    item.name.SetBackgroundObject(m.backgroundObject);
    item.name.CalcSize(m.display);
    item.w1:=item.name.width+m.display.spaceWidth DIV 2;
    item.height:=item.name.height+m.display.spaceHeight;

    IF item.shortcut#NIL THEN
      item.shortcut.SetFlags({G.horizontalFlex,G.verticalFlex});
      item.shortcut.SetBackgroundObject(m.backgroundObject);
      item.shortcut.CalcSize(m.display);
      item.w2:=item.shortcut.width+2*m.display.spaceWidth+m.display.spaceWidth DIV 2;
      item.height:=G.MaxLong(item.height,item.name.height+m.display.spaceHeight);
    ELSE
      item.w2:=m.display.spaceWidth DIV 2;
    END;

    m.Add(item);
  END AddItem;

  PROCEDURE (m : Menu) AddTextItem*(label, shortcut : ARRAY OF CHAR; id : LONGINT);

  VAR
    labelText,
    scText    : T.Text;
    help      : U.Text;

  BEGIN
    NEW(labelText);
    labelText.Init;
    labelText.SetFlags({G.horizontalFlex,G.verticalFlex});
    labelText.SetDefault(T.leftAlligned,{},D.normalFont);
    help:=U.EscapeString(label);
    labelText.SetText(help^);

    IF shortcut#"" THEN
      NEW(scText);
      scText.Init;
      scText.SetFlags({G.horizontalFlex,G.verticalFlex});
      scText.SetDefault(T.leftAlligned,{},D.normalFont);
      help:=U.EscapeString(shortcut);
      scText.SetText(help^);
    ELSE
      scText:=NIL;
    END;

    m.AddItem(labelText,scText,id);
  END AddTextItem;

  PROCEDURE (m : Menu) AddTextItemSC*(label, shortcut : ARRAY OF CHAR;
                                      id : LONGINT; qualifier : SET; char : CHAR;
                                      keyHandler : K.KeyHandler);

  BEGIN
    m.AddTextItem(label,shortcut,id);
    IF keyHandler#NIL THEN
      keyHandler.AddShortcutObject(m.last,qualifier,char,id,K.none); (* m.last = hack! *)
    END;
  END AddTextItemSC;

  PROCEDURE (m : Menu) AddSeparator*;

  VAR
    s : Separator;

  BEGIN
    NEW(s);
    s.w1:=2;
    s.height:=2+m.display.spaceHeight;

    m.Add(s);
  END AddSeparator;

  PROCEDURE (m : Menu) AddTitle*(name : G.Object);

  VAR
    item : MenuTitle;

  BEGIN
    NEW(item);
    item.name:=name;
    item.name.SetFlags({G.horizontalFlex,G.verticalFlex});
    item.name.SetBackgroundObject(m.backgroundObject);
    item.name.CalcSize(m.display);
    item.w1:=item.name.width+m.display.spaceWidth;
    item.height:=item.name.height+m.display.spaceHeight;
    INC(item.height,3);
    m.Add(item);
  END AddTitle;

  PROCEDURE (m : Menu) AddTextTitle*(label : ARRAY OF CHAR);

  VAR
    labelText : T.Text;
    help      : U.Text;

  BEGIN
    NEW(labelText);
    labelText.Init;
    labelText.SetFlags({G.horizontalFlex,G.verticalFlex});
    labelText.SetDefault(T.centered,{},D.normalFont);
    help:=U.EscapeString(label);
    labelText.SetText(help^);

    m.AddTitle(labelText);
  END AddTextTitle;

  PROCEDURE (m : Menu) AddSubMenu*(name : G.Object; menu : Menu);

  VAR
    sub : SubMenu;

  BEGIN
    m.popup:=FALSE;
    NEW(sub);
    sub.subMenu:=menu;
    sub.subMenu.parentMenu:=sub;
    sub.name:=name;
    sub.name.SetFlags({G.horizontalFlex,G.verticalFlex});
    sub.name.SetBackgroundObject(m.backgroundObject);
    sub.name.CalcSize(m.display);
    sub.w1:=sub.name.width+m.display.spaceWidth;
    NEW(sub.arrow);
    sub.arrow.Init;
    sub.arrow.SetFlags({G.horizontalFlex,G.verticalFlex});
    sub.arrow.SetBackgroundObject(m.backgroundObject);
    sub.arrow.Set(V.arrowRight);
    sub.arrow.CalcSize(m.display);
    sub.w2:=2*m.display.spaceWidth+sub.arrow.width;
    sub.height:=G.MaxLong(sub.name.height,sub.arrow.height)+m.display.spaceHeight;

    m.Add(sub);

  END AddSubMenu;

  PROCEDURE (m : Menu) AddTextSubMenu*(label : ARRAY OF CHAR; menu : Menu);

  VAR
    labelText : T.Text;
    help      : U.Text;

  BEGIN
    NEW(labelText);
    labelText.Init;
    labelText.SetFlags({G.horizontalFlex,G.verticalFlex});
    labelText.SetDefault(T.centered,{},D.normalFont);
    help:=U.EscapeString(label);
    labelText.SetText(help^);
    m.AddSubMenu(labelText,menu);
  END AddTextSubMenu;

  PROCEDURE (m : Menu) CalcSize;

  VAR
    entry   : MenuEntry;
    w1,w2,h : LONGINT;

  BEGIN
    m.frame.SetInternalFrame(m.prefs.menuFrame);
    m.frame.CalcSize(m.display);

    w1:=0;
    w2:=0;
    h:=0;

    entry:=m.list;
    WHILE entry#NIL DO
      INC(h,entry.height);
      w1:=G.MaxLong(w1,entry.w1);
      w2:=G.MaxLong(w2,entry.w2);
      IF entry.next#NIL THEN
        entry:=entry.next(MenuEntry);
      ELSE
        entry:=NIL;
      END;
    END;

    entry:=m.list;
    WHILE entry#NIL DO
      entry.ResizeMenu(w1,w2,entry.height);
      IF entry.next#NIL THEN
        entry:=entry.next(MenuEntry);
      ELSE
        entry:=NIL;
      END;
    END;

    m.SetSize(m.frame.leftBorder+w1+w2+m.frame.rightBorder,
              m.frame.topBorder+h+m.frame.bottomBorder);

    m.frame.Resize(m.width,m.height);
  END CalcSize;

  PROCEDURE (m : Menu) Send*(message : O.Message; type : LONGINT);

  VAR
    menu : Menu;
    help : D.Window;

  BEGIN
    menu:=m;
    WHILE (menu.parentMenu#NIL) & (menu.parentMenu.GetMenuObject()#NIL) DO
      help:=menu.parentMenu.GetMenuObject();
      menu:=help(Menu);
    END;

    IF menu.strip#NIL THEN
      menu.strip.Send(message,type);
    ELSIF menu#m THEN
      menu.Send(message,type);
    ELSE
      menu.Send^(message,type);
    END;
  END Send;

  PROCEDURE (m : Menu) CursorIsIn():BOOLEAN;

  VAR
    rx,ry,
    wx,wy  : LONGINT;

  BEGIN
    m.GetMousePos(rx,ry,wx,wy);
    IF (rx>=m.x) & (rx<=m.x+m.width-1) & (ry>=m.y) & (ry<=m.y+m.height-1) THEN
      RETURN TRUE;
    ELSE
      RETURN FALSE;
    END;
  END CursorIsIn;

  PROCEDURE (m : Menu) GetSelected():MenuEntry;

  VAR
    entry  : MenuEntry;

    rx,ry,
    wx,wy  : LONGINT;

  BEGIN
    m.GetMousePos(rx,ry,wx,wy);
    entry:=m.list;
    WHILE entry#NIL DO
      IF (rx>=m.x) & (rx<=m.x+m.width-1) & (wy>=entry.y) & (wy<=entry.y+entry.height-1) THEN
        RETURN entry;
      END;
      IF entry.next#NIL THEN
        entry:=entry.next(MenuEntry);
      ELSE
        entry:=NIL;
      END;
    END;
    RETURN entry;
  END GetSelected;

  PROCEDURE (m : Menu) PreInit*;

  VAR
    rx,ry,
    cx,cy,
    x,y    : LONGINT;
    help   : D.Window;

  BEGIN
    m.CalcSize;

    m.display.currentWin.GetMousePos(rx,ry,cx,cy);

    IF m.selected=NIL THEN
      m.selected:=m.list;
    END;
    IF m.pullDown & (m.reference#NIL) THEN
      m.display.currentWin.GetXY(x,y);
      INC(x,m.reference.x+m.offset);
      INC(y,m.strip.y+m.strip.height);
      m.Grab(TRUE);
    ELSE
      IF m.parentMenu#NIL THEN (* submenu *)
        help:=m.parentMenu.GetMenuObject();
        x:=help.x+help.width;
        y:=help.y+m.parentMenu.y-help(Menu).list.y;
      ELSE (* main-menu *)
        IF m.popup THEN
          x:=rx+m.display.spaceWidth;
          y:=ry+m.display.spaceHeight;
        ELSE
          x:=rx-m.selected.x-m.selected.width DIV 2;
          y:=ry-m.selected.y-m.selected.height DIV 2;
        END;
        IF ~m.pullDown THEN
          m.Grab(TRUE);
        END;
      END;
    END;

    x:=G.RoundRange(x,0,m.display.scrWidth-1-m.width);
    y:=G.RoundRange(y,0,m.display.scrHeight-1-m.height);

    m.SetPos(x,y);

    m.PreInit^;
  END PreInit;

  PROCEDURE (m : Menu) Close*;

  VAR
    close : CloseMsg;
    help  : D.Window;

  BEGIN
    m.Grab(FALSE);

    IF m.child#NIL THEN
      m.child.Close;
      m.child:=NIL;
    END;
    IF m.parentMenu#NIL THEN
      m.parentMenu.subOpen:=FALSE;
      help:=m.parentMenu.GetMenuObject();
      help(Menu).child:=NIL;
    ELSE
      NEW(close);
      m.Send(close,closeMsg);
    END;

    m.pullDown:=FALSE;
    m.reference:=NIL;

    m.Close^;
  END Close;

  PROCEDURE ( m : Menu) CloseAll;

  VAR
    current : Menu;
    help    : D.Window;

  BEGIN
    m.pullDown:=FALSE;
    m.reference:=NIL;
    current:=m;
    WHILE current#NIL DO
      current.Close;
      IF current.parentMenu#NIL THEN
        help:=current.parentMenu.GetMenuObject();
        current:=help(Menu);
      ELSE
        current:=NIL;
      END;
    END;
  END CloseAll;

  PROCEDURE (m : Menu) Draw*;

  VAR
    entry : MenuEntry;
    x,y   : LONGINT;

  BEGIN
    m.frame.Draw(0,0,m.draw);

    x:=m.frame.topBorder;
    y:=m.frame.leftBorder;

    m.selected:=m.GetSelected();
    entry:=m.list;
    WHILE entry#NIL DO

      IF entry=m.selected THEN
        m.draw.mode:={D.selected};
      END;

      entry.Draw(x,y,m.draw);
      m.draw.mode:={};

      INC(y,entry.height);

      IF entry.next#NIL THEN
        entry:=entry.next(MenuEntry);
      ELSE
        entry:=NIL;
      END;
    END;
  END Draw;

  PROCEDURE (m : Menu) Hide*;

  VAR
    entry : MenuEntry;

  BEGIN
    m.frame.Hide;

    entry:=m.list;
    WHILE entry#NIL DO
      entry.Hide;
      IF entry.next#NIL THEN
        entry:=entry.next(MenuEntry);
      ELSE
        entry:=NIL;
      END;
    END;
  END Hide;

  PROCEDURE (m : Menu) Hidden*;

  BEGIN
    m.CloseAll;
  END Hidden;

  (* ---------------------------------- *)

  PROCEDURE (m : Menu) HandleEvent*(event : E.Event):BOOLEAN;

  VAR
    newSelected : MenuEntry;
    selected    : SelectedMsg;
    menu        : Menu;
    rx,ry,wx,wy : LONGINT;

  BEGIN
    IF m.HandleEvent^(event) THEN
      RETURN TRUE;
    END;

    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseUp)
      &  ((event.button=E.button3) OR (event.button=E.button1)) THEN
        menu:=m;
        WHILE (menu#NIL) & ~(menu.GetSelected()#NIL) & (menu.selected#NIL) DO
          IF menu.selected IS SubMenu THEN
            menu:=menu.selected(SubMenu).subMenu;
          ELSE
            menu:=NIL;
          END;
        END;

        IF menu#NIL THEN
          menu.selected:=menu.GetSelected();
          IF (menu.selected#NIL) & (menu.selected IS MenuItem) THEN
            IF m.pullDown THEN (* send msg after menu has been closed! *)
              m.strip.selected:=TRUE;
              m.strip.selId:=menu.selected(MenuItem).id;
            ELSE
              NEW(selected);
              selected.id:=menu.selected(MenuItem).id;
              m.Send(selected,selectedMsg);
            END;
          END;
        END;
        IF m.pullDown THEN (* The strip must get its mouse up *)
          m.display.PutBackEvent(event,m.strip.draw.vWindow);
        END;
        m.CloseAll;
      END;

    | event : E.MotionEvent DO
      menu:=m;
      WHILE (menu#NIL) & (menu.selected#NIL) & ~menu.CursorIsIn() DO
        IF (menu.selected#NIL) & (menu.selected IS SubMenu) THEN
          menu:=menu.selected(SubMenu).subMenu;
        ELSE
          menu:=NIL;
        END;
      END;

      IF menu#NIL THEN

        IF menu.CursorIsIn() THEN
          newSelected:=menu.GetSelected();
          IF newSelected#menu.selected THEN
            IF menu.selected#NIL THEN
              menu.selected.Redraw;
            END;
            IF newSelected#NIL THEN
              menu.draw.mode:={D.selected};
              newSelected.Redraw;
              menu.draw.mode:={};
            END;
            menu.selected:=newSelected;
          END;
        ELSIF m.pullDown THEN
          m.reference.draw.vWindow.GetMousePos(rx,ry,wx,wy);
          IF m.strip.MouseIsIn() &
          ((wx<m.reference.x) OR (wx>m.reference.x+m.reference.width-1)) THEN
            m.CloseAll;
          END;
        END;

      ELSE

        menu:=m;
        WHILE (menu#NIL) & (menu.selected#NIL) & (menu.selected IS SubMenu) DO
          menu:=menu.selected(SubMenu).subMenu;
        END;
        IF (menu#NIL) & (menu.selected#NIL) THEN
          menu.selected.Redraw;
          menu.selected:=NIL;
        END;

        IF m.pullDown THEN
          m.reference.draw.vWindow.GetMousePos(rx,ry,wx,wy);
          IF m.strip.MouseIsIn() &
          ((wx<m.reference.x) OR (wx>m.reference.x+m.reference.width-1)) THEN
            m.CloseAll;
          END;
        END;
      END;

    ELSE
    END;
    RETURN TRUE;
  END HandleEvent;

BEGIN
  NEW(prefs);
  prefs.Init;
END VOMenu.