(**
   A groupclass that places nicely formated textlabels (or any other
   object) to the left of a vertical list of objects.
**)

MODULE VOLabel;

(*
    A layouting class that gives you labels.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D   := VODisplay,
       G   := VOGUIObject,
       T   := VOText,
       U   := VOUtil;

CONST
  left*  = 0;
  right* = 1;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button row is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 labelPos*    : LONGINT;
                 labelAlign*  : LONGINT;
                 objectAlign* : LONGINT;
               END;


  Text       = POINTER TO TextDesc;
  TextDesc   = RECORD
                 next     : Text;
                 text     : G.Object;
               END;

  Label*     = POINTER TO LabelDesc;
  LabelDesc* = RECORD(G.GroupDesc)
                 prefs      : Prefs;
                 lSize,
                 rSize      : LONGINT;
                 labelList,
                 lLast      : Text;
                 labelFlex  : BOOLEAN;
               END;

VAR
  prefs* : Prefs;


  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.labelPos:=left;
    p.labelAlign:=right;
    p.objectAlign:=left;
  END Init;

  PROCEDURE (p : Prefs) SetPrefs(l : Label);

  BEGIN
    l.prefs:=p;   (* We set the prefs *)

    IF p.background#NIL THEN
      l.SetBackgroundObject(p.background.Copy());
      l.backgroundObject.source:=l;
    END;
  END SetPrefs;

  PROCEDURE (l : Label) Init*;

  BEGIN
    l.Init^;

    prefs.SetPrefs(l);

    l.labelList:=NIL;
    l.labelFlex:=FALSE;
  END Init;

  (**
    Add a label entry to label group. A label entry consists of two objects. One
    object is the label for the other.

  **)

  PROCEDURE (l : Label) AddLabel*(text, object : G.Object);

  VAR
    label : Text;

  BEGIN
    IF l.list=NIL THEN
      l.list:=object;
    ELSE
      l.last.next:=object;
    END;
    l.last:=object;

    NEW(label);
    label.text:=text;

    IF l.labelList=NIL THEN
      l.labelList:=label;
    ELSE
      l.lLast.next:=label;
    END;
    l.lLast:=label;

    text.parent:=l;
    object.parent:=l;

    INC(l.count);

    object.SetLabelObject(text);
  END AddLabel;

  (**
    Add a textual label entry. The label object will generate a text object out of
    the given string and will then assign it as label object to the other object.
  **)

  PROCEDURE (l : Label) AddTextLabel*(string : ARRAY OF CHAR; object : G.Object);

  VAR
    text : T.Text;
    help : U.Text;

  BEGIN
    NEW(text);
    text.Init;
    help:=U.EscapeString(string);
    text.SetText(help^);
    l.CopyBackground(text);
    l.AddLabel(text,object);
  END AddTextLabel;

  (**
    Tell, if the label object should stretch the label or the object when beeing resized
    horizontally.
  **)

  PROCEDURE ( l : Label) Set*(labelFlex : BOOLEAN);

  BEGIN
    l.labelFlex:=labelFlex;
  END Set;

  PROCEDURE (l : Label) CalcSize*(display : D.Display);

  VAR
    object    : G.Object;
    label     : Text;
    rMinWidth : LONGINT;

  BEGIN
    l.height:=0;
    l.minHeight:=0;
    l.lSize:=0;
    l.rSize:=0;
    rMinWidth:=0;

    object:=l.list;
    label:=l.labelList;
    WHILE (label#NIL) & (object#NIL) DO

      object.CalcSize(display);
      label.text.CalcSize(display);

      l.lSize:=G.MaxLong(label.text.width,l.lSize);

      IF object.oHeight<label.text.oHeight THEN
        INC(l.height,label.text.oHeight);
        INC(l.minHeight,label.text.oMinHeight);
      ELSE
        INC(l.height,object.oHeight);
        INC(l.minHeight,object.oMinHeight);
      END;

      l.rSize:=G.MaxLong(l.rSize,object.oWidth);
      rMinWidth:=G.MaxLong(rMinWidth,object.oMinWidth);

      label:=label.next;
      object:=object.next;
    END;

    IF l.count>1 THEN
      INC(l.height,(l.count-1)*display.spaceHeight);
      INC(l.minHeight,(l.count-1)*display.spaceHeight);
    END;

    l.width:=l.lSize+display.spaceWidth+l.rSize;
    l.minWidth:=l.lSize+display.spaceWidth+rMinWidth;

    l.CalcSize^(display);

    IF l.labelFlex THEN
      l.lSize:=l.width-l.rSize-l.display.spaceWidth;
    ELSE
      l.rSize:=l.width-l.lSize-l.display.spaceWidth;
    END;
  END CalcSize;

  PROCEDURE (l : Label) Resize*(width,height : LONGINT);

  BEGIN
    l.Resize^(width,height);

    IF l.labelFlex THEN
      l.lSize:=l.width-l.rSize-l.display.spaceWidth;
    ELSE
      l.rSize:=l.width-l.lSize-l.display.spaceWidth;
    END;
  END Resize;

  PROCEDURE (l : Label) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  VAR
    object    : G.Object;
    label     : Text;
    pos,
    curHeight,
    old,
    count     : LONGINT;

  BEGIN
    l.Draw^(x,y,draw);

    l.DrawBackground(l.x,l.y,l.width,l.height);

    curHeight:=0;
    label:=l.labelList;
    object:=l.list;
    WHILE (label#NIL) & (object#NIL) DO
      IF object.oHeight<label.text.oHeight THEN
        object.Resize(-1,label.text.oHeight);
        INC(curHeight,label.text.oHeight);
      ELSIF object.oHeight>label.text.oHeight THEN
        label.text.Resize(-1,object.oHeight);
        INC(curHeight,object.oHeight);
      ELSE
        INC(curHeight,object.oHeight);
      END;
      label.text.Resize(l.lSize,-1);
      object.Resize(l.rSize,-1);
      label:=label.next;
      object:=object.next;
    END;

    IF l.count>1 THEN
      INC(curHeight,(l.count-1)*l.display.spaceHeight);
    END;

    IF curHeight<l.height THEN
      LOOP
        count:=0;
        object:=l.list;
        WHILE object#NIL DO
          IF object.CanResize(l.height>curHeight,FALSE) THEN
            INC(count);
          END;
          object:=object.next;
        END;

        IF count=0 THEN
          EXIT;
        END;

        object:=l.list;
        WHILE object#NIL DO
          IF object.CanResize(l.height>curHeight,FALSE) THEN
            old:=object.oHeight;
            object.Resize(-1,object.oHeight+U.UpDiv(l.height-curHeight,count));
            INC(curHeight,object.oHeight-old);
            DEC(count);
          END;
          object:=object.next;
        END;

        IF curHeight=l.height THEN
          EXIT;
        END;
      END;
    END;

    (* Draw all objects *)

    object:=l.list;
    label:=l.labelList;
    pos:=l.y;
    WHILE (label#NIL) & (object#NIL) DO

      IF l.prefs.labelPos=right THEN
        IF l.prefs.objectAlign=left THEN
          object.Draw(l.x,pos,draw);
        ELSE
          object.Draw(l.x+l.rSize-object.oWidth,pos,draw);
        END;
        IF l.prefs.labelAlign=right THEN
          label.text.Draw(l.x+l.rSize+ draw.display.spaceWidth+l.lSize-label.text.oWidth,
                          pos+object.oHeight-label.text.oHeight,draw);
        ELSE
          label.text.Draw(l.x+l.rSize+ draw.display.spaceWidth,
                          pos+object.oHeight-label.text.oHeight,draw);
        END;
      ELSE
        IF l.prefs.objectAlign=left THEN
          object.Draw(l.x+l.lSize+draw.display.spaceWidth,pos,draw);
        ELSE
          object.Draw(l.x+l.lSize+draw.display.spaceWidth+l.rSize-object.oWidth,pos,draw);
        END;
        IF l.prefs.labelAlign=right THEN
          label.text.Draw(l.x+l.lSize-label.text.oWidth,pos+object.oHeight-label.text.oHeight,draw);
        ELSE
          label.text.Draw(l.x,pos+object.oHeight-label.text.oHeight,draw);
        END;
      END;

      IF object.oHeight>label.text.height THEN
        INC(pos,object.oHeight);
      ELSE
        INC(pos,label.text.height);
      END;

      INC(pos,l.display.spaceHeight);

      object:=object.next;
      label:=label.next;
    END;
  END Draw;

  PROCEDURE (l : Label) Refresh*(x,y,w,h : LONGINT);

  VAR
    object : G.Object;
    label  : Text;

  BEGIN
    IF l.visible & l.Intersect(x,y,w,h) THEN
      l.RestrictToBounds(x,y,w,h);
      l.DrawBackground(x,y,w,h);
      object:=l.list;
      label:=l.labelList;
      WHILE (label#NIL) & (object#NIL) DO

        object.Refresh(x,y,w,h);
        label.text.Refresh(x,y,w,h);

        object:=object.next;
        label:=label.next;
      END;
    END;
  END Refresh;

  PROCEDURE (l : Label) Hide*;

  VAR
    object : G.Object;
    label  : Text;

  BEGIN
    IF l.visible THEN
      object:=l.list;
      label:=l.labelList;
      WHILE object#NIL DO
        object.Hide;
        object:=object.next;
        label.text.Hide;
        label:=label.next;
      END;
      l.DrawHide;
      l.Hide^;
    END;
  END Hide;

  PROCEDURE (l : Label) GetPosObject*(x,y : LONGINT; type : LONGINT):G.Object;

  VAR
    object,
    return : G.Object;
    label  : Text;

  BEGIN
    IF l.visible THEN
      object:=l.list;
      label:=l.labelList;
      WHILE object#NIL DO
        return:=object.GetPosObject(x,y,type);
        IF return#NIL THEN
          RETURN return;
        END;
        object:=object.next;

        return:=label.text.GetPosObject(x,y,type);
        IF return#NIL THEN
          RETURN return;
        END;
        label:=label.next;
      END;
    END;

    RETURN l.GetPosObject^(x,y,type);
  END GetPosObject;

  PROCEDURE (l : Label) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):G.Object;

  VAR
    object,
    return : G.Object;
    label  : Text;

  BEGIN
    IF l.visible THEN
      object:=l.list;
      label:=l.labelList;
      WHILE object#NIL DO
        return:=object.GetDnDObject(x,y,drag);
        IF return#NIL THEN
          RETURN return;
        END;
        object:=object.next;

        return:=label.text.GetDnDObject(x,y,drag);
        IF return#NIL THEN
          RETURN return;
        END;
        label:=label.next;
      END;
    END;

    RETURN NIL;
  END GetDnDObject;

BEGIN
  NEW(prefs);
  prefs.Init;
END VOLabel.