/*  gtksourceview
*  Copyright (C) 2001
* Mikael Hermansson<mikeh@bahnhof.se>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU Library General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU Library General Public License for more details.
*
*  You should have received a copy of the GNU Library General Public License*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <vdk/gtksourceview.h>

enum
{
  UNDO,
  REDO,
  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };


static gboolean foreach_character(gunichar ch, gpointer data);

static GObjectClass *parent_class = NULL;

static void gtk_source_view_real_undo(GtkSourceView *view);
static void gtk_source_view_real_redo(GtkSourceView *view);

static gint find_correct_bracket(GtkTextBuffer *buf, GtkTextIter *iter);
static void gtk_source_view_init (GtkSourceView *text);
static void gtk_source_view_class_init (GtkSourceViewClass *text);
static gint gtk_source_view_expose (GtkWidget *widget, GdkEventExpose *ev);
static gint gtk_source_view_key_press (GtkWidget *widget, GdkEventKey *ev);
static gint
get_lines (GtkTextView  *text_view,
           gint          first_y,
           gint          last_y,
           GArray       *buffer_coords,
           gint         *countp);

gint
get_lines (GtkTextView  *text_view,
           gint          first_y,
           gint          last_y,
           GArray       *buffer_coords,
           gint         *countp)
{
  GtkTextIter iter;
  gint count;
  gint size;  
  gint num;

  g_array_set_size (buffer_coords, 0);
  
  /* Get iter at first y */
  gtk_text_view_get_line_at_y (text_view, &iter, first_y, NULL);

  /* For each iter, get its location and add it to the arrays.
   * Stop when we pass last_y
   */
  count = 0;
  size = 0;

  num = gtk_text_iter_get_line (&iter);
  while (!gtk_text_iter_is_end (&iter))
    {
      gint y, height;
      
      gtk_text_view_get_line_yrange (text_view, &iter, &y, &height);

      g_array_append_val (buffer_coords, y);
      
      ++count;

      if ((y + height) >= last_y)
        break;
      
      gtk_text_iter_forward_line (&iter);
    }

  *countp = count;
  
  return num;
}

gboolean foreach_character (gunichar ch, gpointer data)
{
  return ch == GPOINTER_TO_INT(data);   
}

gboolean
find_correct_bracket(GtkTextBuffer *buf,GtkTextIter *cur)
{
  gint dummy=0,e=0;
  GtkTextIter iter1=*cur;

  do 
    {
       if(gtk_text_iter_get_char(&iter1) == '{') 
       {  dummy--; e++; }
       else if(gtk_text_iter_get_char(&iter1) == '}')
         {    
            dummy++;
            e--;
         }
      if(!dummy && e > 1)
        return e;

      gtk_text_iter_backward_char(&iter1);
    }
  // while(!gtk_text_iter_is_first(&iter1));
    while(!gtk_text_iter_is_start(&iter1));

  return e >= 1 ? e : 0;
}
gint 
gtk_source_view_key_press(GtkWidget *widget, GdkEventKey *ev)
{
  gunichar prevchar;
  GtkTextIter cur,iter,iter2,lineiter;
  GtkTextMark *mark;
  GtkTextBuffer *buf=gtk_text_view_get_buffer (GTK_TEXT_VIEW(widget));

  mark = gtk_text_buffer_get_mark(buf,"insert");
  gtk_text_buffer_get_iter_at_mark(buf,&cur,mark);
  iter=iter2=lineiter=cur;

  gtk_text_iter_backward_char(&cur);
  prevchar=gtk_text_iter_get_char(&cur);

  /* we set it back to cur pos */
  cur=iter;

  if (!(ev->state & GDK_CONTROL_MASK) && ev->length)
  {
    if (ev->keyval == GDK_Tab ) 
    {
      gtk_text_buffer_insert_interactive_at_cursor (buf, "  ", 2,
                                                    GTK_TEXT_VIEW(widget)->editable);
      gtk_text_view_scroll_mark_onscreen (GTK_TEXT_VIEW(widget),
                                          gtk_text_buffer_get_mark (gtk_text_view_get_buffer (GTK_TEXT_VIEW(widget)),
                                                                    "insert"));

/*   gtk_text_layout_set_cursor_visible (GTK_TEXT_VIEW(widget)->layout, TRUE);*/

      return TRUE;
    }
    else if(ev->keyval == GDK_Return  && 
                GTK_SOURCE_VIEW(widget)->auto_indent && 
                gtk_text_iter_ends_line(&lineiter)               
            )
    {
      gint loffset=find_correct_bracket(buf,&iter);
      if(loffset)
      {
        gtk_text_buffer_begin_user_action(buf);

        gtk_text_buffer_insert(buf,&lineiter,"\n", 1);
        while (loffset--)  
         gtk_text_buffer_insert(buf,&lineiter,"  ", 2);
  
        gtk_text_buffer_end_user_action(buf);

      gtk_text_view_scroll_mark_onscreen (GTK_TEXT_VIEW(widget),
                                          gtk_text_buffer_get_mark (gtk_text_view_get_buffer (GTK_TEXT_VIEW(widget)),
                                                                    "insert"));
    
        return TRUE;
      }
    }
#if 0 
   else if(ev->keyval == '{'  && 
                GTK_SOURCE_VIEW(widget)->auto_indent &&
               find_correct_bracket(buf,&iter)
              )
    {

        gtk_text_buffer_begin_user_action(buf);
        gtk_text_buffer_insert(buf,&cur,"  ", 2);
        gtk_text_buffer_insert(buf,&cur,ev->string, ev->length);
        gtk_text_buffer_end_user_action(buf);

        return 0;
    }
#endif
    else if(ev->keyval == '}' &&
                GTK_SOURCE_VIEW(widget)->auto_indent &&
                 find_correct_bracket(buf,&iter)
            )
    {
      iter=cur;
      gtk_text_iter_backward_chars(&iter,2);
      gtk_text_buffer_begin_user_action(buf);
    
    /* make sure it really is an whitespace we are deleting */
      if(gtk_text_iter_get_char(&iter)==' ')
        gtk_text_buffer_delete(buf,&iter,&cur);

      gtk_text_buffer_insert(buf,&cur,"}", 1);
      gtk_text_buffer_end_user_action(buf);

      return TRUE;
    }
    else if((ev->keyval == ',' || ev->keyval == '=' ) &&
                GTK_SOURCE_VIEW(widget)->auto_indent)
    {

      if(ev->keyval == ',' || 
                prevchar == '!' || prevchar=='=' ||
                prevchar == '<' || prevchar=='>' 
           )
      {
        gtk_text_buffer_begin_user_action(buf);
        gtk_text_buffer_insert(buf,&cur,ev->string, ev->length);
        gtk_text_buffer_insert(buf,&cur," ", 1);
        gtk_text_buffer_end_user_action(buf);

        return TRUE;
      }
      else if(g_unichar_isalpha(prevchar) && ev->keyval == '=')
      {
        gtk_text_buffer_begin_user_action(buf);
        gtk_text_buffer_insert(buf,&cur," ", 1);
        gtk_text_buffer_insert(buf,&cur,ev->string, ev->length);
        gtk_text_buffer_end_user_action(buf);

        return TRUE;
      }
    }
  else if( (ev->keyval == '(' || ev->keyval =='<') && 
          g_unichar_isalpha(prevchar) &&
          GTK_SOURCE_VIEW(widget)->auto_indent)      
    {
        gtk_text_buffer_begin_user_action(buf);
        gtk_text_buffer_insert(buf,&cur," ", 1);
        gtk_text_buffer_insert(buf,&cur,ev->string, ev->length);
        gtk_text_buffer_end_user_action(buf);

        return TRUE;
    }
    else if(g_unichar_isalnum(ev->keyval) &&
                  prevchar == '='  &&
                GTK_SOURCE_VIEW(widget)->auto_indent)
    {
        gtk_text_buffer_begin_user_action(buf);
        gtk_text_buffer_insert(buf,&cur," ", 1);
        gtk_text_buffer_insert(buf,&cur,ev->string, ev->length);
        gtk_text_buffer_end_user_action(buf);

        return TRUE;
    }
  }

  return (* GTK_WIDGET_CLASS (parent_class)->key_press_event) (widget, ev);
  //  (* GTK_WIDGET_CLASS (parent_class)->key_press_event) (widget, ev);
  //  return FALSE;
}

gint 
gtk_source_view_expose(GtkWidget *widget, GdkEventExpose *ev)
{
  GdkWindow *win=0;
  GArray *pixels=0;
  gint pos=0;
  gint y1=0,y2=0;
  gint i = 0;
  gint num,count;
  GtkTextView *tw = GTK_TEXT_VIEW(widget);
  PangoLayout *layout;
  
  win = gtk_text_view_get_window (tw, GTK_TEXT_WINDOW_LEFT);
  if(ev->window != win || !GTK_SOURCE_VIEW(tw)->show_line_numbers)
    return (* GTK_WIDGET_CLASS (parent_class)->expose_event) (widget, ev);

  pixels = g_array_new (FALSE, FALSE, sizeof (gint));

  y1 = ev->area.y;
  y2 = y1 + ev->area.height;
  
  gtk_text_view_window_to_buffer_coords (tw,
                                         GTK_TEXT_WINDOW_LEFT,
                                         0,
                                         y1,
                                         NULL,
                                         &y1);

  gtk_text_view_window_to_buffer_coords (tw,
                                         GTK_TEXT_WINDOW_LEFT,
                                         0,
                                         y2,
                                         NULL,
                                         &y2);

  num = get_lines (tw,
             y1,
             y2,
             pixels,
             &count);

  layout = gtk_widget_create_pango_layout (GTK_WIDGET(tw), "");
  
  while (i < count)
    {
      gchar *str;
      
      gtk_text_view_buffer_to_window_coords (tw,
                                             GTK_TEXT_WINDOW_LEFT,
                                             0,
                                             g_array_index (pixels, gint, i),
                                             NULL,
                                             &pos);


      str = g_strdup_printf ("<b>%4d:</b>", num);

      pango_layout_set_markup (layout, str, -1);


      gdk_draw_layout (win,
                       widget->style->fg_gc [widget->state],
                       /* 2 is just a random padding */
                       10, pos + 2,
                       layout);

      g_free (str);

      i++;
      num++;      
    }

  g_array_free (pixels, TRUE); 
  
  g_object_unref (G_OBJECT (layout));
  

  return TRUE;

}

void
gtk_source_view_init (GtkSourceView *text)
{
  text->show_line_numbers = TRUE;
  text->auto_indent = TRUE;

  gtk_text_view_set_border_window_size (GTK_TEXT_VIEW(text),
                                        GTK_TEXT_WINDOW_LEFT,
                                        50);
}

void 
gtk_source_view_class_init (GtkSourceViewClass *klass)
{
  GObjectClass *object_class;
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);
  GtkBindingSet *binding_set;
 
  object_class = (GObjectClass*) klass;
  parent_class = g_type_class_peek_parent (klass);

  widget_class->expose_event = gtk_source_view_expose;
  widget_class->key_press_event = gtk_source_view_key_press;
  
  klass->undo = gtk_source_view_real_undo;
  klass->redo = gtk_source_view_real_redo;

  signals[UNDO] =
    gtk_signal_new ("undo",
                    GTK_RUN_LAST | GTK_RUN_ACTION,
                    GTK_CLASS_TYPE (object_class),
                    GTK_SIGNAL_OFFSET (GtkSourceViewClass, undo),
                    gtk_marshal_VOID__VOID,
                    GTK_TYPE_NONE, 0);
  signals[REDO] =
    gtk_signal_new ("redo",
                    GTK_RUN_LAST | GTK_RUN_ACTION,
                    GTK_CLASS_TYPE (object_class),
                    GTK_SIGNAL_OFFSET (GtkSourceViewClass, redo),
                    gtk_marshal_VOID__VOID,
                    GTK_TYPE_NONE, 0);


  binding_set = gtk_binding_set_by_class (klass);

  gtk_binding_entry_add_signal (binding_set, GDK_z, GDK_CONTROL_MASK,
                                "undo", 0);

  gtk_binding_entry_add_signal (binding_set, GDK_r, GDK_CONTROL_MASK,
                                "redo", 0);
  /*
    added by mario motta
  */
  gtk_binding_entry_add_signal (binding_set, GDK_Insert, GDK_CONTROL_MASK,
                                "copy_clipboard", 0);

  gtk_binding_entry_add_signal (binding_set, GDK_Insert, GDK_SHIFT_MASK,
                                "paste_clipboard", 0);
}

void 
gtk_source_view_real_undo(GtkSourceView *view)
{
  g_return_if_fail(GTK_IS_SOURCE_VIEW(view));

  gtk_source_buffer_undo(GTK_SOURCE_BUFFER(gtk_text_view_get_buffer(GTK_TEXT_VIEW(view))));
}

void
gtk_source_view_real_redo(GtkSourceView *view)
{
  g_return_if_fail(GTK_IS_SOURCE_VIEW(view));

  gtk_source_buffer_redo(GTK_SOURCE_BUFFER(gtk_text_view_get_buffer(GTK_TEXT_VIEW(view))));
}

void 
gtk_source_view_set_show_line_numbers(GtkSourceView *source, gboolean show)
{
  g_return_if_fail (GTK_IS_SOURCE_VIEW(source));

  source->show_line_numbers = show;

  gtk_text_view_set_border_window_size (GTK_TEXT_VIEW(source),
                                        GTK_TEXT_WINDOW_LEFT,
                                        show ? 50 : 0);

  if(GTK_WIDGET_REALIZED(source))
    gtk_widget_queue_draw(GTK_WIDGET(source));
}

gboolean
gtk_source_view_get_show_line_numbers(GtkSourceView *source)
{
  g_return_val_if_fail (GTK_IS_SOURCE_VIEW(source), FALSE);

  return source->show_line_numbers;
}

void 
gtk_source_view_set_auto_indent(GtkSourceView *source, gboolean set)
{
  g_return_if_fail (GTK_IS_SOURCE_VIEW(source));

  source->auto_indent = set;
}

gboolean
gtk_source_view_get_auto_indent(GtkSourceView *source)
{
  g_return_val_if_fail (GTK_IS_SOURCE_VIEW(source), FALSE);

  return source->auto_indent;
}

GtkWidget *
gtk_source_view_new()
{
  return GTK_WIDGET(g_object_new(gtk_source_view_get_type(),NULL));
}

GtkWidget *
gtk_source_view_new_with_buffer(GtkTextBuffer *buf)
{
  GtkWidget *widget = gtk_source_view_new();

  gtk_text_view_set_buffer(GTK_TEXT_VIEW(widget), buf);  

  return widget;
}

GType
gtk_source_view_get_type (void)
{
  static GType our_type = 0;

  if (our_type == 0)
    {
      static const GTypeInfo our_info =
      {
        sizeof (GtkSourceViewClass),
        (GBaseInitFunc) NULL,
        (GBaseFinalizeFunc) NULL,
        (GClassInitFunc) gtk_source_view_class_init,
        NULL,           /* class_finalize */
        NULL,           /* class_data */
        sizeof (GtkSourceView),
        0,              /* n_preallocs */
        (GInstanceInitFunc) gtk_source_view_init
      };

      our_type = g_type_register_static (GTK_TYPE_TEXT_VIEW,
                                         "GtkSourceView",
                                         &our_info,
                                         0);
    }

  return our_type;
}
