/*
 * GLX Hardware Device Driver for SiS 6326
 * Copyright (C) 2000 Jim Duchek
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *  Based on mach64tex.c
 *    Jim Duchek <jim@linuxpimps.com>
 */

#include <stdlib.h>
#include "sis6326glx.h"
#include "glx_symbols.h"


/*
 * sis6326DestroyTexObj
 * Free all memory associated with a texture and NULL any pointers
 * to it.
 */
static void sis6326DestroyTexObj( sis6326ContextPtr ctx, sis6326TextureObjectPtr t )
{
	sis6326TextureObjectPtr	p, prev;
	int			i;

	hwMsg( 10, "sis6326DestroyTexObj( %p )\n", t->tObj );

 	if ( !t ) {
  		return;
  	}

 	if ( t->magic != SIS6326_TEXTURE_OBJECT_MAGIC ) {
 		hwError( "sis6326DestroyTexObj: t->magic != SIS6326_TEXTURE_OBJECT_MAGIC\n" );
		return;
	}

	/* free the texture memory */
	if ( sis6326glx.agpTextures ) {
		mmFreeMem( t->memBlocks[0] );
		mmFreeMem( t->memBlocks[1] );
	} else {
		mmFreeMem( t->memBlock );
	}

 	/* free mesa's link */
	t->tObj->DriverData = NULL;

	/* see if it was the driver's current object */
	if ( sis6326glx.currentTexture == t ) {
		hwMsg( 10, "sis6326DestroyTexObj: destroyed current\n" );
		sis6326glx.currentTexture = NULL;
	}

	/* remove from the driver texobj list */
	p = sis6326glx.textureList;
	prev = NULL;
	while ( p ) {
		if ( p == t ) {
			if ( prev ) {
				prev->next = t->next;
			} else {
				sis6326glx.textureList = t->next;
			}
    			break;
    		}
		prev = p;
		p = p->next;
	}

	/* clear magic to catch any bad future references */
	t->magic = 0;

	/* free the structure */
	free( t );

	/* dump the heap contents if loglevel is high enough */
	if ( hwGetLogLevel() >= 15 ) {
		mmDumpMemInfo( textureHeap );
	}
}


/*
 * sis6326DestroyOldestTexObj
 * Throw out a texture to try to make room for a new texture
 */
static int sis6326DestroyOldestTexObj( void )
{
	sis6326TextureObjectPtr	t, oldest;
	hwUI32			old;

	hwMsg(10,"  Swapping out texture.\n");

 	/* find the best texture to toss */
	old = 0x7fffffff;
	oldest = NULL;
	for ( t = sis6326glx.textureList; t ; t = t->next ) {
		/* never swap out used textures */
		if ( t == sis6326glx.currentTexture ) {
			continue;
		}
		if ( t->age < old ) {
			old = t->age;
			oldest = t;
		}
	}

	/* if the oldest texture was in use on the previous frame, then
	   we are in a texture thrashing state.  Note that we can't just
	   test for "in THIS frame", because textures from the same working
	   set may be used in different order, and it could register as not
	   thrashing.  The solution is to pick the MOST recently used texture
	   that isn't currently needed for multitexture.  This will allow the
	   other textures to stay resident for the next frame, rather than
	   swapping everything out in order. */

	if ( old >= sis6326glx.swapBuffersCount - 1 ) {
		/* newly created texture objects are always added to the
		   front of the list, so just find the first one that isn't
		   used for multitexture */
	        hwMsg( 10, "sis6326DestroyOldestTexObj: thrashing\n" );
		for ( t = sis6326glx.textureList ; t ; t = t->next ) {
			/* never swap out textures used */
			if ( t == sis6326glx.currentTexture ) {
				continue;
			}
			break;
		}
		oldest = t;

	} else {
		hwMsg( 10, "sis6326DestroyOldestTexObj\n" );
	}

	if ( !oldest ) {
		/* This shouldn't happen unless the 2D resolution is high enough that
		   a single texture can't be allocated in the remaining memory */
		hwError("  No Texture to swap out -> Out of Memory!\n");
		mmDumpMemInfo( textureHeap );
		return -1;
	}

	/* just destroy the texture, because it can always
	   be recreated directly from the mesa texture */

	sis6326DestroyTexObj( sis6326Ctx, oldest );

	return 0;
}


/*=============================================*/


/*
 * sis6326ConvertLocalTexture
 * Converts a block of mesa format texture to the apropriate hardware format.
 * Does a host data blit, only for local card memory textures.
 */
static void sis6326ConvertLocalTexture( hwUI32 *dest, int texelBytes, struct gl_texture_image *image,
				       int x, int y, int width, int height, int pitch )
{
	int i, j;
	hwUI32 *destPtr = dest;
	hwUI8 *src;
	
	if ( texelBytes == 1 && (image->Format == GL_INTENSITY || image->Format == GL_ALPHA ) ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (hwUI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width >> 2  ; j ; j-- ) {
				int	pix;
			
				pix = src[0] | ( src[1] << 8 ) | ( src[2] << 16 ) | ( src[3] << 24 );
				*destPtr++ = pix;
				src += 4;
			}
		}
	} else if (texelBytes == 1 && (image->Format == GL_LUMINANCE) ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (hwUI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width >> 2  ; j ; j-- ) {
				int	pix;
				pix = src[0] | ( src[1] << 8 ) | ( src[2] << 16 ) | ( src[3] << 24 );
				*destPtr++ = pix;
				src += 4;
			}
		}
	} else if ( texelBytes == 2 && image->Format == GL_RGB ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (hwUI8 *)image->Data + ( ( y + i ) * image->Width + x ) * 3;
			for ( j = width >> 1  ; j ; j-- ) {
				int	pix;
				pix = SIS6326PACKCOLOR555(src[0],src[1],src[2], 0) |  
					( SIS6326PACKCOLOR555(src[3],src[4],src[5], 0) << 16 );
				*destPtr++ = pix;
				src += 6;
			}
		}
	} else if ( texelBytes == 2 && image->Format == GL_RGBA ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (hwUI8 *)image->Data + ( ( y + i ) * image->Width + x ) * 4;
			for ( j = width >> 1  ; j ; j-- ) {
				int	pix;
				pix = SIS6326PACKCOLOR4444(src[0],src[1],src[2],src[3]) |  
					( SIS6326PACKCOLOR4444(src[4],src[5],src[6],src[7]) << 16 );
				*destPtr++ = pix;
				src += 8;
			}
		}
	} else {
		hwError( "Unsupported texelBytes %i, image->Format %i\n", texelBytes, image->Format );
	}

}



/*=============================================*/


/*
 * sis6326UploadLocalSubImage
 *
 * Perform an host blit based update of a resident buffer.  This is used for
 * both initial loading of the entire image, and texSubImage updates.
 *
 * This is complicated by a few factors:
 *
 * sis6326ConvertTexture() must upload at least 32 bits for each row, making
 * problems for 16 bit 1 texel wide and 8 bit 3 texel or less wide images.
 *
 * The host blit hardware has a minimum pitch of 64 texels, causing problems
 * for all images less than that wide.
 *
 * Very large images may not be able to be be converted into a single dma
 * buffer, forcing a subdivision into multiple parts.
 */
static void sis6326UploadLocalSubImage( sis6326TextureObjectPtr t, int level,
				       int x, int y, int width, int height )
{
	int		x2;
	int		dwords;
	struct gl_texture_image *image;
	int		texelBytes, texelsPerDword;
	int		pitch, ofs;
	int		blitX, blitY, blitWidth, blitHeight;
	int		imageWidth, imageHeight;

	image = t->tObj->Image[level];
	
	
	if ( !image ) {
		hwError( "sis6326UploadLocalSubImage: NULL image\n" );
		return;
	}

	texelBytes = t->texelBytes;
	texelsPerDword = 4 / texelBytes;  // Better be even!

	imageWidth = image->Width;
	imageHeight = image->Height;

	if ( imageWidth < texelsPerDword ) {
		hwMsg(1, "PROBLEM!\n"); // FIXME, dipshit
	}

	/* pad the size out to dwords.  The image is a pointer to the entire image,
	   so we can safely reference outside the x,y,width,height bounds if we need to */
	x2 = x + width;
	x2 = ( x2 + ( texelsPerDword - 1 ) ) & ~(texelsPerDword-1);

	x = ( x + ( texelsPerDword - 1 ) ) & ~(texelsPerDword-1);
	width = x2 - x;

	/* the texture walker and the blitter look identical */
	blitX = x;
	blitY = y;
	blitWidth = width;
	blitHeight = height;
	pitch = imageWidth >> 3;

	ofs = t->memBlock->ofs + t->offsets[level];

	/* bump the performance counter */
	sis6326glx.c_textureSwaps += ( dwords << 2 );

	/* make sure we overflow here instead of in convertTexture */

	hwMsg( 1, "sis6326UploadLocalSubImage: %i,%i of %i,%i at %i,%i\n",
		   width, height, image->Width, image->Height, x, y );
	hwMsg( 1, "                blit size: %i,%i at %i,%i\n",
		   blitWidth, blitHeight, blitX, blitY );

	sis6326ConvertLocalTexture( (hwUI32 *)(sis6326glx.linearBase + ofs), texelBytes, image, x, y, width, height, imageWidth );
}


/*
 * sis6326UploadAGPSubImage
 *
 * Perform an agp memory blit update of a resident buffer.  This is used for
 * both initial loading of the entire image, and texSubImage updates.
 *
 * This is much simpler than the host blit based update, but we have to
 * upload the entire image due to the double texture buffers.  Otherwise,
 * any updates to the other buffer could be lost when we upload this round
 * of updates.
 *
 * FIXME: Save updated region, so we can get away with only updating the
 * union of the previous and current regions?
 */
static void sis6326UploadAGPSubImage( sis6326TextureObjectPtr t, int level,
				     int x, int y, int width, int height )
{
	int		dwords;
	struct gl_texture_image *image;
	int		texelBytes, texelsPerDword;
	hwUI32		*dest;

	image = t->tObj->Image[0];
	if ( !image ) {
		hwError( "sis6326UploadAGPSubImage: NULL image\n" );
		return;
	}

	/* hasAlpha is needed for programming SCALE_3D_CNTL */
	if ( ( image->Format == GL_RGBA ) || ( image->Format == GL_ALPHA ) || ( image->Format == GL_LUMINANCE_ALPHA ) ) {
		t->hasAlpha = 1;
	} else {
		t->hasAlpha = 0;
	}

	/* textureFormat is used for programming DP_PIX_WIDTH */
	/* FIXME: 32 bit and 8 bit intensity */
	if ( t->hasAlpha ) {
		t->textureFormat = 15;	// 4444
	} else {
		t->textureFormat = 4;	// 565
	}

	texelBytes = t->texelBytes;
	texelsPerDword = 2;

	dwords =  width * height * texelBytes;

	/* bump the performance counter */
	sis6326glx.c_textureSwaps += ( dwords << 2 );

	/* find the proper destination offset */
	dest = (hwUI32 *)(sis6326glx.agpMemory + t->memBlock->ofs);

	hwMsg( 10, "sis6326UploadAGPSubImage: %i,%i of %i,%i at %i,%i\n",
	       image->Width, image->Height, image->Width, image->Height, 0, 0 );

	/* write directly to agp memory the properly converted texels from the mesa buffer */
//	sis6326ConvertAGPTexture( dest, texelBytes, image, 0, 0, image->Width, image->Height, image->Width );

	/* the normal state registers must be reprogrammed before drawing */
}


/*=============================================*/


static int Log2( unsigned a )
{
	unsigned	i;

	for ( i = 0 ; i < 32 ; i++ ) {
		if ( ( 1<<i ) >= a ) {
			return i;
		}
	}
	return 31;
}

/*
 * sis6326CreateTexObj
 * Allocate space for and load the mesa images into the texture memory block.
 * This will happen before drawing with a new texture, or drawing with a
 * texture after it was swapped out or teximaged again.
 */
void sis6326CreateTexObj( sis6326ContextPtr ctx, struct gl_texture_object *tObj )
{
	sis6326TextureObjectPtr	t;
	int			ofs, size;
	PMemBlock		mem[2];
	int 			lastlev;
	int i;
	struct gl_texture_image *image;

	hwMsg( 10,"sis6326CreateTexObj( %p )\n", tObj );

	image = tObj->Image[ 0 ];
	if ( !image ) {
		return;
	}
	
	t = malloc( sizeof( *t ) );
	if ( !t ) {
		FatalError( "sis6326CreateTexObj: Failed to malloc textureObject\n" );
	}
	
	memset( t, 0, sizeof( *t ) );

	switch(image->Format) {
	case GL_LUMINANCE:
		t->texelBytes = 1;
		t->textureFormat = S_TSET1_FORMAT_LUM | S_TSET1_LUM_L8;
		hwMsg(0, "Got a LUM\n");
		t->hasAlpha = 0;
		break;
	case GL_RGB:
		t->texelBytes = 2;
		t->textureFormat = S_TSET1_FORMAT_16 | S_TSET1_16_RGB555;
		t->hasAlpha = 0;
		break;		
	case GL_RGBA:
		t->texelBytes = 2;
		t->textureFormat = S_TSET1_FORMAT_16 | S_TSET1_16_ARGB4444;
		t->hasAlpha = 1;
		break;
	case GL_ALPHA:
		t->texelBytes = 1;
		t->textureFormat = S_TSET1_FORMAT_LUM | S_TSET1_LUM_AL88;
		t->hasAlpha = 1;
		hwMsg(0, "Cannot Got handle GL_ALPHA correctly!!!!\n"); // FIXME, crackhead..
		break;
	case GL_LUMINANCE_ALPHA:
		t->texelBytes = 2;
		t->textureFormat = S_TSET1_FORMAT_LUM | S_TSET1_LUM_AL88;
		t->hasAlpha = 1;
		hwMsg(0, "LumAlpha\n");
		break;
	case GL_INTENSITY:
		t->texelBytes = 1;
		t->textureFormat = S_TSET1_FORMAT_LUM | S_TSET1_LUM_L8;
		hwMsg(0, "Cannot handle Got intensity!\n");
		break;
	case GL_COLOR_INDEX:
		t->texelBytes = 1;
		t->textureFormat = S_TSET1_FORMAT_LUM | S_TSET1_LUM_L8;
		hwMsg(0, "No paletted textures, results may be crappy.\n");
		break;
	default:
		hwMsg(0, "Got Odd image type %08x.\n", image->Format);
		break;
	}

	lastlev = 9;
	
	ofs = 0;
	
	for (i=0; i<=lastlev; i++) {
		int levWidth, levHeight;
		
		t->offsets[i] = ofs;
		
		image = tObj->Image[i];
		
		if (!image) {
			t->offsets[i] = -1;
			lastlev = i - 1;
			hwMsg(10, "Images missing after %d\n", lastlev);
			break;
		}
		
		levWidth = image->Width;
		levHeight = image->Height;
		
		size = levWidth * levHeight * t->texelBytes;
		size = ( size + 31 ) & ~31;	/* 32 byte aligned */
		ofs += size;
	}
	
	t->totalSize = ofs;

        while ( ( mem[0] = mmAllocMem( textureHeap, t->totalSize, 6, 0 ) ) == 0 ) {
        	if ( sis6326DestroyOldestTexObj() ) {
        		/* can't hold this texture at all */
			hwMsg( 10, "sis6326CreateTexObj: Couldn't allocate buffer\n" );
        		free( t );
        		return;
        	}
        }


	/* fill in our texture object */
	t->magic = SIS6326_TEXTURE_OBJECT_MAGIC;
	t->tObj = tObj;
	t->ctx = ctx;
	t->next = sis6326glx.textureList;
	sis6326glx.textureList = t;



	/* if we're using agp memory, allocate the second buffer */
	if ( sis6326glx.agpTextures ) {
		while ( ( mem[1] = mmAllocMem( textureHeap, ofs, 6, 0 ) ) == 0 ) {
			if ( sis6326DestroyOldestTexObj() ) {
				/* can't hold this texture at all */
				hwMsg( 10, "sis6326CreateTexObj: Couldn't allocate buffer\n" );
				free( t );
				return;
			}
		}
	}

	/* dump the heap contents if loglevel is high enough */
	if ( hwGetLogLevel() >= 15 ) {
		mmDumpMemInfo( textureHeap );
	}

	/* init agp texture double buffer */
	t->activeMemBlock = 0;

	if ( sis6326glx.agpTextures ) {
		t->memBlocks[0] = mem[0];
		t->memBlocks[1] = mem[1];
	}
	t->memBlock = mem[0];

	/* base image */
	image = tObj->Image[ 0 ];


	t->widthLog2 = Log2( image->Width );
	t->heightLog2 = Log2( image->Height );
	t->maxLog2 = t->widthLog2 > t->heightLog2 ? t->widthLog2 : t->heightLog2;


  	tObj->DriverData = t;

	for (i=0; i<=lastlev; i++) {
		image = tObj->Image[i];
		/* load the texels */
		if ( sis6326glx.agpTextures ) {
	//		sis6326UploadAGPSubImage( t, 0, 0, 0, image->Width, image->Height );
		} else {
			sis6326UploadLocalSubImage( t, i, 0, 0, image->Width, image->Height );
		}
	}
}


#if 0	// FIXME: also fix in mga
/*
 * mgaDestroyContextTextures
 */
void mgaDestroyContextTextures( mgaContextPtr ctx ) {
	hwMsg( 1, "mgaDestroyContextTextures( %p )\n", ctx );
	/* FIXME: do we actually need to do anything here?  Shouldn't MESA
	   call the normal delete function before freeing textures? */
#if 0
	while ( ctx->textureList )
		mgaDestroyTexObj( ctx, ctx->textureList );
#endif
}
#endif


/*
============================================================================

Driver functions called directly from mesa

============================================================================
*/


/*
 * sis6326TexImage
 */
void sis6326TexImage( GLcontext *ctx, GLenum target,
		     struct gl_texture_object *tObj, GLint level,
		     GLint internalFormat,
		     const struct gl_texture_image *image )
{
	sis6326TextureObject_t *t;

	hwMsg( 10,"sis6326TexImage( %p, level %i )\n", tObj, level );
	

  	/* free the driver texture if it exists */
	t = (sis6326TextureObjectPtr) tObj->DriverData;
	if ( t ) {
 	 	sis6326DestroyTexObj( sis6326Ctx, t );
	}

	/* create it */
	sis6326CreateTexObj( sis6326Ctx, tObj );

	/* make the texture current */
	sis6326glx.currentTexture = t;
	sis6326glx.texChange = 1;
}

/*
 * sis6326TexSubImage
 */
void sis6326TexSubImage( GLcontext *ctx, GLenum target,
			struct gl_texture_object *tObj, GLint level,
			GLint xoffset, GLint yoffset,
			GLsizei width, GLsizei height,
			GLint internalFormat,
			const struct gl_texture_image *image )
{
	sis6326TextureObject_t *t;

	hwMsg( 10, "sis6326TexSubImage() size: %d,%d of %d,%d; level %d\n",
		   width, height, image->Width,image->Height, level );
		   
	/* immediately upload it if it is resident */
	t = (sis6326TextureObject_t *) tObj->DriverData;
	if ( t ) {
		/* local and agp textures require different paths */
		if ( sis6326glx.agpTextures ) {
			/* swap to using the other buffer */
			t->activeMemBlock ^= 1;
			t->memBlock = t->memBlocks[ t->activeMemBlock ];

			sis6326UploadAGPSubImage( t, level, xoffset, yoffset, width, height );
		} else {
			sis6326UploadLocalSubImage( t, level, xoffset, yoffset, width, height );
		}
	}

	/* make the texture current */
	sis6326glx.currentTexture = t;
}

/*
 * sis6326DeleteTexture
 */
void sis6326DeleteTexture( GLcontext *ctx, struct gl_texture_object *tObj )
{
	hwMsg( 10, "sis6326DeleteTexture( %p )\n", tObj );

	/* delete our driver data */
	if ( tObj->DriverData ) {
		sis6326DestroyTexObj( sis6326Ctx, (sis6326TextureObject_t *)(tObj->DriverData) );
	}
}

/*
 * sis6326IsTextureResident
 */
GLboolean sis6326IsTextureResident( GLcontext *ctx, struct gl_texture_object *tObj )
{
	GLboolean	is;

   	is = (tObj->DriverData != NULL);

	hwMsg( 10, "sis6326IsTextureResident( %p ) == %i\n", tObj, is );

	return is;
}


/*
 * Local Variables:
 * mode: c
 * tab-width: 8
 * c-basic-offset: 8
 * End:
 */
