/*
 * GLX Hardware Device Driver for SiS 6326
 * Copyright (C) 2000 Jim Duchek
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *  Some bits based on mach64swap.c
 *  Jim Duchek <jim@linuxpimps.com>
 */

#include <stdlib.h>

#include "context.h"
#include "depth.h"
#include "macros.h"
#include "texstate.h"
#include "triangle.h"
#include "vb.h"
#include "types.h"

#include "xsmesaP.h"
#include "glx_log.h"
#include "mesaglx/context.h"
#include "mesaglx/matrix.h"
#include "mesaglx/types.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "servermd.h" /* PixmapBytePad */
#include "scrnintstr.h"
#include "regionstr.h"
#include "windowstr.h"
#undef GC

#include "glx_symbols.h"

#include "sis6326glx.h"

/*
 * sis6326BackToFront
 * Blit the visible rectangles from the back buffer to the screen
 */
void sis6326BackToFront( DrawablePtr drawable, sis6326BufferPtr buf ) {
	RegionPtr	prgnClip;
	BoxPtr		pbox;
	int 		i, nbox;
	int	j;
	short 	blah[200];
	int 		xorg, yorg, pitch;
	int		screenFormat;

	prgnClip = &((WindowPtr)drawable)->clipList;
	pbox = REGION_RECTS( prgnClip );
	nbox = REGION_NUM_RECTS( prgnClip );
	
	if( !nbox ) {
		/* window is completely covered */
		return;
	}

	xorg = drawable->x;
	yorg = drawable->y;
	pitch = buf->pitch;

	WAITFIFOEMPTY((3 + (nbox * 4)));

	OUTREG(SIS6326_BITBLT_FG_COLOR, (0xCC << 24));
	OUTREG(SIS6326_BITBLT_BG_COLOR, (0xCC << 24));
	OUTREG(SIS6326_BITBLT_DST_SRC_PITCH, (((sis6326glx.displayWidth * sis6326glx.bytesPerPixel) << 16) | (sis6326DB->pitch * sis6326glx.bytesPerPixel)));


	/* xy bitblt each visible rectangle */
	for ( i = nbox ; i > 0 ; i-- ) {
		int	x = pbox->x1 - xorg;
		int	y = pbox->y1 - yorg;
		int	w = pbox->x2 - pbox->x1;
		int	h = pbox->y2 - pbox->y1;
		OUTREG(SIS6326_BITBLT_SRC_ADDR, (sis6326DB->backBufferBlock->ofs + ((y * sis6326DB->pitch + x) * sis6326glx.bytesPerPixel)));
		OUTREG(SIS6326_BITBLT_DST_ADDR, ((pbox->y1 * sis6326glx.displayWidth + pbox->x1) * sis6326glx.bytesPerPixel));
		OUTREG(SIS6326_BITBLT_HEIGHT_WIDTH, (((h-1) << 16) | ((w) * sis6326glx.bytesPerPixel)));
		OUTREGW(SIS6326_BITBLT_CMD, ((0x00 << 8) | (0x32)));

		pbox++;
	}
}


/*
 * ClearBox
 * Add hardware commands to draw a filled box for the
 * debugging display.
 */
static void ClearBox( int x, int y, int w, int h, int r, int g, int b )
{
#if 0
    	DMALOCALS;
	int	color;

    	DMAGETPTR( 32 );

	DMAOUTREG( SIS6326REG_DST_OFF_PITCH, ( (sis6326DB->pitch/8) << 22 ) | (sis6326DB->backBufferBlock->ofs>>3) ) ;

	DMAOUTREG( SIS6326REG_Z_CNTL, 0 );			/* no z buffering */
	DMAOUTREG( SIS6326REG_SCALE_3D_CNTL, 0 );
	DMAOUTREG( SIS6326REG_CLR_CMP_CNTL, 0 );			/* disable color compare */

	DMAOUTREG( SIS6326REG_DP_SRC,
		   DP_bkgd_src_foreground | DP_frgd_src_foreground | DP_mono_src_1 );

	color = sis6326PackColor( r, g, b, 0 );

	DMAOUTREG( SIS6326REG_DP_FRGD_CLR, color );
	DMAOUTREG( SIS6326REG_DP_WRITE_MASK, 0xffffffff );	/* write to all */
	DMAOUTREG( SIS6326REG_DP_MIX, 0x00070003 );		/* bkground leave alone */
	DMAOUTREG( SIS6326REG_DP_SRC,
		   DP_bkgd_src_foreground | DP_frgd_src_foreground | DP_mono_src_1 );
	DMAOUTREG( SIS6326REG_GUI_TRAJ_CNTL, 3 );		/* left to right, top to bottom */
	DMAOUTREG( SIS6326REG_DST_X_Y, (y<<16)|x );
	DMAOUTREG( SIS6326REG_DST_WIDTH_HEIGHT, (h<<16)|w );

	/* put it back where X expects it, until we get full dma going */
	DMAOUTREG( SIS6326REG_DST_OFF_PITCH, ( (sis6326glx.displayWidth/8) << 22 ) | 0 );

    	DMAADVANCE();
    	
#endif
}


/*
 * performanceBoxes
 * Draw some small boxesin the corner of the buffer
 * based on some performance information
 */
void sis6326PerformanceBoxes( int is_direct )
{
#if 0
	int		w, t;

	if ( !sis6326glx.boxes || !sis6326DB ) {
		return;
	}

	/* draw a box to show we are active, so if it is't seen
	   it means that it is completely software based rendering  */
	/* draw a purple box if we are direct rendering */
	if ( is_direct ) {			/* purple = direct (client dma) rendering */
		ClearBox( 4, 4, 8, 8, 255, 0, 255 );
	} else if ( sis6326glx.dmaDriver ) {	/* white = server dma rendering */
		ClearBox( 4, 4, 8, 8, 255, 255, 255 );
	} else {				/* grey = server PDMA */
		ClearBox( 4, 4, 8, 8, 128, 128, 128 );
	}

	/* draw a red box if we had to wait for drawing to complete
	   (software render or texture swap) */
	if ( sis6326glx.c_drawWaits ) {
		ClearBox( 16, 4, 8, 8, 255, 0, 0 );
		sis6326glx.c_drawWaits = 0;
	}

	/* draw a yellow box if textures were swapped */
	if ( sis6326glx.c_textureSwaps ) {
		ClearBox( 40, 4, 8, 8, 255, 255, 0 );
	}

	/* draw a green box if we had to wait for dma to complete (full
	   utilization) on the previous frame */
	if ( !sis6326glx.hardwareWentIdle ) {
		ClearBox( 64, 4, 8, 8, 0, 255, 0 );
	}
	sis6326glx.hardwareWentIdle = 0;

	/* show buffer utilization */
	if ( sis6326glx.c_dmaFlush > 1 ) {
		/* draw a solid bar if we flushed more than one buffer */
		ClearBox( 4, 16, 252, 4, 255, 32, 32 );
	} else {
		/* draw bars to represent the utilization of primary buffer */
		ClearBox( 4, 16, 252, 4, 32, 32, 32 );
		t = sis6326glx.dma_buffer->maxBufferDwords;
		w = 252 * sis6326glx.dma_buffer->bufferDwords / t;
		if ( w < 1 ) {
			w = 1;
		}
		ClearBox( 4, 16, w, 4, 196, 128, 128 );
	}
	sis6326glx.c_dmaFlush = 0;
#endif
}


/*
 * Copy the back buffer to the front buffer.
 */
void sis6326GLXSwapBuffers( XSMesaBuffer b ) {

	/* make sure mesa flushes any open vertex buffers */
	if ( sis6326Ctx && sis6326Ctx->gl_ctx ) {
		FLUSH_VB( sis6326Ctx->gl_ctx, "swapbuffers" );
	}
	sis6326GLXSwapBuffersWithoutFlush( b );
}

/*
 * this needs to be separate so the auto-front-buffer-swap can
 * be done inside renderFinish without recursion caused by the flush
 */
void sis6326GLXSwapBuffersWithoutFlush( XSMesaBuffer b ) {
	sis6326BufferPtr	buf;

	sis6326glx.swapBuffersCount++;

	ValidateGC( b->frontbuffer, b->cleargc );

	if ( !b->backimage ) {
		return;	/* when is this ever hit? */
	}

	buf = (sis6326BufferPtr)b->backimage->devPriv;
	if ( !VALID_SIS6326_BUFFER( buf ) ) {
		hwError( "BackToFront(): invalid back buffer\n" );
		return;
	}

	if ( !__glx_is_server ) {
		/* direct rendering case */
		/* send a message to the server to flush and swap */
		sis6326DirectClientSwapBuffers( b );
	} else {
		/* we are the X server */
		if ( buf->backBufferBlock ) {
			/* diagnostic drawing tools */
			sis6326PerformanceBoxes( 0 );

			/* hardware accelerated back to front blit */
			sis6326BackToFront( (DrawablePtr)b->frontbuffer, buf );
//			usleep(250000);
		} else {
			/* software blit ... */
			(*b->cleargc->ops->PutImage)( (DrawablePtr)b->frontbuffer,
						      b->cleargc,
						      b->frontbuffer->depth,
						      0, 0,
						      buf->pitch,
						      buf->height,
						      0, ZPixmap, buf->backBuffer );
		}
	}

	/* report performance counters */
	hwMsg( 10, "swapBuffers: c_triangles:%i  c_setup:%i c_textures:%i c_numCmds:%i\n",
	       sis6326glx.c_triangles, sis6326glx.c_setupPointers, sis6326glx.c_textureSwaps, sis6326glx.c_numCmds);

	sis6326glx.c_triangles = 0;
	sis6326glx.c_setupPointers = 0;
	sis6326glx.c_textureSwaps = 0;
	sis6326glx.c_drawWaits = 0;
	sis6326glx.c_clears = 0;
	sis6326glx.c_numCmds = 0;

	hwMsg( 10, "---------------------------------------------------------\n" );
}


/*
 * Local Variables:
 * mode: c
 * tab-width: 8
 * c-basic-offset: 8
 * End:
 */
