/* $Id: mach64dma.h,v 1.21 2001/01/23 17:58:13 svartalf Exp $ */

/*
 * GLX Hardware Device Driver for ATI Rage Pro
 * Copyright (C) 1999 Gareth Hughes
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on MGA driver: mgadma.h ???
 *
 *    Gareth Hughes <gareth@precisioninsight.com>
 */

#ifndef __MACH64_DMA_H__
#define __MACH64_DMA_H__

/*
 * The Rage Pro DMA stream operates in 4K chunks of dword pairs, in the
 * following format:
 *
 * ADDR (in MM offset format)
 * DATA
 * ADDR (in MM offset format)
 * DATA
 * ADDR (in MM offset format)
 * DATA
 * ....
 *
 * The data is accessed through descriptor tables, with each descriptor
 * entry consisting of 4 dwords:
 *
 * BM_FRAME_BUF_OFFSET	- Frame buffer offset for data transfer
 * BM_SYSTEM_MEM_ADDR	- Physical system memory address for data transfer
 * BM_COMMAND		- Count of bytes to transfer (4K max) + flags
 * RESERVED		- Always set to 0
 *
 * For more info, see the mach64 Programmer's Guide, section 8.10
 */

/*
 * Not sure how this compares with the G200, but the Rage Pro has two
 * banks of registers, with bank 0 at (aperture base + memmap offset - 1KB)
 * and bank 1 at (aperture base + memmap offset - 2KB).  But, to send them
 * via DMA, we need to encode them as memory map select rather than physical
 * offsets.
 */
#define DWMREG0		0x0400
#define DWMREG0_END	0x07ff
#define DWMREG1		0x0000
#define DWMREG1_END	0x03ff

#define ISREG0(r)	( ( (r) >= DWMREG0 ) && ( (r) <= DWMREG0_END ) )
#define ADRINDEX0(r)	( ((r) - DWMREG0) >> 2 )
#define ADRINDEX1(r)	( ( ((r) - DWMREG1) >> 2 ) | 0x0100 )
#define ADRINDEX(r)	( ISREG0(r) ? ADRINDEX0(r) : ADRINDEX1(r) )

#define MMREG0		0x0000
#define MMREG0_END	0x00ff

#define ISMMREG0(r)	( ( (r) >= MMREG0 ) && ( (r) <= MMREG0_END ) )
#define MMSELECT0(r)	( ((r)<<2) + DWMREG0 )
#define MMSELECT1(r)	( ( (((r) & 0xff)<<2) + DWMREG1 ) )
#define MMSELECT(r)	( ISMMREG0(r) ? MMSELECT0(r) : MMSELECT1(r) )

#define DMA_FRAME_BUF_OFFSET	0
#define DMA_SYS_MEM_ADDR	1
#define DMA_COMMAND		2
#define DMA_RESERVED		3

/* the main initialization of the entire mach64 hardware driver */
GLboolean mach64InitGLX( void );

/* a flush command will guarantee that all data added to the dma buffer
is on its way to the card, and will eventually complete with no more
intervention.  If running with pseudo dma, this will be the same as a finish
call, but if async dma is active then the card will be executing the commands
while the cpu is doing other work. */
void mach64DmaFlush( void );

/* the overflow function is called when a block can't be allocated
in the current dma buffer.  It flushes the current buffer and
records some information */
void mach64DmaOverflow( int newDwords );

/* a finish command will guarantee that all dma commands have actually
been consumed by the card, and that the engine is completely idle.  It
is safe to do software rendering after this returns */
void mach64DmaFinish( void );

/* reset the engine when dma transfers go wrong */
void mach64EngineReset( void );
void mach64DumpEngineState( void );

/* for routines that need to add a variable stream of register writes, use
the DMAGETPTR() / DMAOUTREG() / DMAADVANCE() interface.  Call
mach64DmaGetPtr() with a length in dwords that is guaranteed to be greater
than what you will be writing.  Using a large value, like 100, does not
waste anything.  Program all the registers you need with DMAOUTREG(),
When you have finished,call DMAADVANCE(), which will add any necessary padding
and commit the data to the dma buffer

DMALOCALS must be included at the top of all functions that use these
macros to declare temporary variables.

*/

typedef struct _dma_buffer
{
    hwUI32 	*virtualBuffer;

    hwUI32	bufferDwords;		/* amount currently allocated */
    hwUI32	overflowBufferDwords;
    hwUI32	maxBufferDwords;
} mach64Dma_buffer;

/* cardHeap is the 8 / 16 / 32 megs of memory on the video card */
extern	memHeap_t	*cardHeap;

/* These are only required by mach64direct.c:
 */
extern mach64Dma_buffer	*dmaBuffers[2];

extern int mach64WaitForDmaCompletion( void );
extern void mach64DmaResetBuffer( void );

#define DMALOCALS		int outcount; hwUI32 *dma_ptr

#define DMAGETPTR( length )						\
    if ( (mach64glx.dma_buffer->overflowBufferDwords - mach64glx.dma_buffer->bufferDwords) < length ) \
	mach64DmaOverflow( length );					\
    dma_ptr = (mach64glx.dma_buffer->virtualBuffer + mach64glx.dma_buffer->bufferDwords); \
    outcount = 0;


#ifdef __PPC__

// hardware registers and dma buffers need to be in little endian format 
unsigned _SWAP(unsigned a);
#define SWAP(a) _SWAP(a)

#else	// __PPC__
#define SWAP(a) a
#endif  // __PPC__ 


/* standard register writes: */
#define DMAOUTREG( reg, val )						\
    do {								\
	dma_ptr[outcount++] = SWAP( ADRINDEX( reg ) );			\
	dma_ptr[outcount++] = SWAP( val );				\
    } while ( 0 )

#ifdef __PPC__
#define DMAOUTFLOAT( reg, val )						\
    do {								\
    	int	temp;							\
	dma_ptr[outcount++] = SWAP( ADRINDEX( reg ) );			\
	*(float *)&temp = val;						\
	dma_ptr[outcount++] = SWAP( temp );				\
    } while ( 0 )
#else
#define DMAOUTFLOAT( reg, val )						\
    do {								\
	dma_ptr[outcount++] = ADRINDEX( reg );				\
	*(float *)&dma_ptr[outcount++] = val;				\
    } while ( 0 )
#endif /* __PPC__ */


/* host data block writes: */
#define DMAOUTHOSTDATA( val )						\
    do {								\
	if ( (outcount & 15) == 0 ) {					\
	    dma_ptr[outcount++] = SWAP( ADRINDEX( MACH64_HOST_DATA0 ) | (14<<16) ); \
	}								\
	dma_ptr[outcount++] = SWAP( val );				\
    } while ( 0 )

#define DMAHOSTDATAEND()						\
    do {								\
	int	dwords = (outcount & 15);				\
	if ( dwords != 0 ) {						\
	    dma_ptr[outcount-dwords] = SWAP( ADRINDEX( MACH64_HOST_DATA0 ) | ((dwords-2)<<16) ); \
	}								\
    } while ( 0 )


/* commit data to buffer: */
#define DMAADVANCE()	mach64glx.dma_buffer->bufferDwords += outcount;


#endif

/*
 * Local Variables:
 * mode: c
 * tab-width: 8
 * c-basic-offset: 8
 * End:
 */
