/* varray.c,v 1.2 1998/10/10 11:24:22 ripperda Exp */

/*
 * Mesa 3-D graphics library
 * Version:  3.0
 * Copyright (C) 1995-1998  Brian Paul
 * Copyright (C) 1998 Terence Ripperda (ripperda@sgi.com)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * BRIAN PAUL, TERENCE RIPPERDA, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY 
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT 
 * OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR  
 * THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */


/*
 * varray.c,v
 * Revision 1.2  1998/10/10 11:24:22  ripperda
 * updates regarding adding of api table, changing all function names
 * from direct OpenGL calls to __glx_ prefaced.
 *
 * Revision 1.1  1998/10/05 04:26:35  ripperda
 * needed files for vertex arrays (misc is for error handling)
 *
 * Revision 3.6  1998/08/21 01:50:53  brianp
 * added support for texture coordinate set interleave factor
 *
 * Revision 3.5  1998/06/07 22:18:52  brianp
 * implemented GL_EXT_multitexture extension
 *
 * Revision 3.4  1998/03/27 04:26:44  brianp
 * fixed G++ warnings
 *
 * Revision 3.3  1998/02/20 04:53:07  brianp
 * implemented GL_SGIS_multitexture
 *
 * Revision 3.2  1998/02/01 20:05:10  brianp
 * added __glx_DrawRangeElements()
 *
 * Revision 3.1  1998/02/01 16:37:19  brianp
 * added GL_EXT_rescale_normal extension
 *
 * Revision 3.0  1998/01/31 21:06:45  brianp
 * initial rev
 *
 */



/*
 * NOTE:  At this time, only three vertex array configurations are optimized:
 *  1.  glVertex3fv(), zero stride
 *  2.  glNormal3fv() with glVertex3fv(), zero stride
 *  3.  glNormal3fv() with glVertex4fv(), zero stride
 *
 * More optimized array configurations can be added.
 */

#include <assert.h>
#include "glxlib.h"
#include "misc.h"
#include "render.h"
#include "glx_varray.h"



void __glx_client_state( GLenum cap, GLboolean state )
{
   assert(GLCurrent);
   switch (cap) {
      case GL_VERTEX_ARRAY:
         GLCurrent->Array.VertexEnabled = state;
         if (state) GLCurrent->enabled_varrays |= VARRAY_VERTS;
         else GLCurrent->enabled_varrays &= ~VARRAY_VERTS;
         break;
      case GL_NORMAL_ARRAY:
         GLCurrent->Array.NormalEnabled = state;
         if (state) GLCurrent->enabled_varrays |= VARRAY_NORMS;
         else GLCurrent->enabled_varrays &= ~VARRAY_NORMS;
         break;
      case GL_COLOR_ARRAY:
         GLCurrent->Array.ColorEnabled = state;
         if (state) GLCurrent->enabled_varrays |= VARRAY_COLORS;
         else GLCurrent->enabled_varrays &= ~VARRAY_COLORS;
         break;
      case GL_INDEX_ARRAY:
         GLCurrent->Array.IndexEnabled = state;
         if (state) GLCurrent->enabled_varrays |= VARRAY_INDEX;
         else GLCurrent->enabled_varrays &= ~VARRAY_INDEX;
         break;
      case GL_TEXTURE_COORD_ARRAY:
         GLCurrent->Array.TexCoordEnabled[0] = state;
         if (state) GLCurrent->enabled_varrays |= VARRAY_TEX;
         else GLCurrent->enabled_varrays &= ~VARRAY_TEX;
         break;
      case GL_EDGE_FLAG_ARRAY:
         GLCurrent->Array.EdgeFlagEnabled = state;
         if (state) GLCurrent->enabled_varrays |= VARRAY_EDGE;
         else GLCurrent->enabled_varrays &= ~VARRAY_EDGE;
         break;
      default:
         __glx_error( GL_INVALID_ENUM, "glEnable/DisableClientState" );
   }
}



void __glx_EnableClientState( GLenum cap )
{
   __glx_client_state( cap, GL_TRUE );
}



void __glx_DisableClientState( GLenum cap )
{
   __glx_client_state( cap, GL_FALSE );
}


void __glx_VertexPointer( GLint size, GLenum type, GLsizei stride,
                       const GLvoid *ptr )
{
   struct gl_array_attrib *a_ptr = NULL;

   assert(GLCurrent);
   a_ptr = &GLCurrent->Array;
   if (size<2 || size>4) {
      __glx_error( GL_INVALID_VALUE, "__glx_VertexPointer(size)" );
      return;
   }
   if (stride<0) {
      __glx_error( GL_INVALID_VALUE, "__glx_VertexPointer(stride)" );
      return;
   }
   switch (type) {
      case GL_SHORT:
         a_ptr->VertexStrideB = stride ? stride : size*sizeof(GLshort);
         switch (size) {
            case 2:  
               a_ptr->VertexFuncPtr = (void (*)(void *)) glVertex2sv; 
               break;
            case 3:  
               a_ptr->VertexFuncPtr = (void (*)(void *)) glVertex3sv; 
               break;
            case 4:  
               a_ptr->VertexFuncPtr = (void (*)(void *)) glVertex4sv; 
               break;
         }
         break;
      case GL_INT:
         a_ptr->VertexStrideB = stride ? stride : size*sizeof(GLint);
         switch (size) {
            case 2:  
               a_ptr->VertexFuncPtr = (void (*)(void *)) glVertex2iv; 
               break;
            case 3:  
               a_ptr->VertexFuncPtr = (void (*)(void *)) glVertex3iv; 
               break;
            case 4:  
               a_ptr->VertexFuncPtr = (void (*)(void *)) glVertex4iv; 
               break;
         }
         break;
      case GL_FLOAT:
         a_ptr->VertexStrideB = stride ? stride : size*sizeof(GLfloat);
         switch (size) {
            case 2:  
               a_ptr->VertexFuncPtr = (void (*)(void *)) glVertex2fv; 
               break;
            case 3:  
               a_ptr->VertexFuncPtr = (void (*)(void *)) glVertex3fv; 
               break;
            case 4:  
               a_ptr->VertexFuncPtr = (void (*)(void *)) glVertex4fv; 
               break;
         }
         break;
      case GL_DOUBLE:
         a_ptr->VertexStrideB = stride ? stride : size*sizeof(GLdouble);
         switch (size) {
            case 2:  
               a_ptr->VertexFuncPtr = (void (*)(void *)) glVertex2dv; 
               break;
            case 3:  
               a_ptr->VertexFuncPtr = (void (*)(void *)) glVertex3dv; 
               break;
            case 4:  
               a_ptr->VertexFuncPtr = (void (*)(void *)) glVertex4dv; 
               break;
         }
         break;
      default:
         __glx_error( GL_INVALID_ENUM, "__glx_VertexPointer(type)" );
         return;
   }

   a_ptr->VertexSize = size;
   a_ptr->VertexType = type;
   a_ptr->VertexStride = stride;
   a_ptr->VertexPtr = (void *) ptr;
}




void __glx_NormalPointer( GLenum type, GLsizei stride, const GLvoid *ptr )
{
   struct gl_array_attrib *a_ptr = NULL;

   assert(GLCurrent);
   a_ptr = &GLCurrent->Array;
   if (stride<0) {
      __glx_error( GL_INVALID_VALUE, "__glx_NormalPointer(stride)" );
      return;
   }
   switch (type) {
      case GL_BYTE:
         a_ptr->NormalStrideB = stride ? stride : 3*sizeof(GLbyte);
         a_ptr->NormalFuncPtr = (void (*) (void *)) glNormal3bv;
         break;
      case GL_SHORT:
         a_ptr->NormalStrideB = stride ? stride : 3*sizeof(GLshort);
         a_ptr->NormalFuncPtr = (void (*) (void *)) glNormal3sv;
         break;
      case GL_INT:
         a_ptr->NormalStrideB = stride ? stride : 3*sizeof(GLint);
         a_ptr->NormalFuncPtr = (void (*) (void *)) glNormal3iv;
         break;
      case GL_FLOAT:
         a_ptr->NormalStrideB = stride ? stride : 3*sizeof(GLfloat);
         a_ptr->NormalFuncPtr = (void (*) (void *)) glNormal3fv;
         break;
      case GL_DOUBLE:
         a_ptr->NormalStrideB = stride ? stride : 3*sizeof(GLdouble);
         a_ptr->NormalFuncPtr = (void (*) (void *)) glNormal3dv;
         break;
      default:
         __glx_error( GL_INVALID_ENUM, "__glx_NormalPointer(type)" );
         return;
   }
   a_ptr->NormalType = type;
   a_ptr->NormalStride = stride;
   a_ptr->NormalPtr = (void *) ptr;
}



void __glx_ColorPointer( GLint size, GLenum type, GLsizei stride,
                      const GLvoid *ptr )
{
   struct gl_array_attrib *a_ptr = NULL;

   assert(GLCurrent);
   a_ptr = &GLCurrent->Array;
   if (size<3 || size>4) {
      __glx_error( GL_INVALID_VALUE, "__glx_ColorPointer(size)" );
      return;
   }
   if (stride<0) {
      __glx_error( GL_INVALID_VALUE, "__glx_ColorPointer(stride)" );
      return;
   }
   switch (type) {
      case GL_BYTE:
         a_ptr->ColorStrideB = stride ? stride : size*sizeof(GLbyte);
         if (size == 3) a_ptr->ColorFuncPtr = (void (*)(void *)) glColor3bv; 
         else           a_ptr->ColorFuncPtr = (void (*)(void *)) glColor4bv;  
         break;
      case GL_UNSIGNED_BYTE:
         a_ptr->ColorStrideB = stride ? stride : size*sizeof(GLubyte);
         if (size == 3) a_ptr->ColorFuncPtr = (void (*)(void *)) glColor3ubv; 
         else           a_ptr->ColorFuncPtr = (void (*)(void *)) glColor4ubv;  
         break;
      case GL_SHORT:
         a_ptr->ColorStrideB = stride ? stride : size*sizeof(GLshort);
         if (size == 3) a_ptr->ColorFuncPtr = (void (*)(void *)) glColor3sv; 
         else           a_ptr->ColorFuncPtr = (void (*)(void *)) glColor4sv;  
         break;
      case GL_UNSIGNED_SHORT:
         a_ptr->ColorStrideB = stride ? stride : size*sizeof(GLushort);
         if (size == 3) a_ptr->ColorFuncPtr = (void (*)(void *)) glColor3usv; 
         else           a_ptr->ColorFuncPtr = (void (*)(void *)) glColor4usv;  
         break;
      case GL_INT:
         a_ptr->ColorStrideB = stride ? stride : size*sizeof(GLint);
         if (size == 3) a_ptr->ColorFuncPtr = (void (*)(void *)) glColor3iv; 
         else           a_ptr->ColorFuncPtr = (void (*)(void *)) glColor4iv;  
         break;
      case GL_UNSIGNED_INT:
         a_ptr->ColorStrideB = stride ? stride : size*sizeof(GLuint);
         if (size == 3) a_ptr->ColorFuncPtr = (void (*)(void *)) glColor3uiv; 
         else           a_ptr->ColorFuncPtr = (void (*)(void *)) glColor4uiv;  
         break;
      case GL_FLOAT:
         a_ptr->ColorStrideB = stride ? stride : size*sizeof(GLfloat);
         if (size == 3) a_ptr->ColorFuncPtr = (void (*)(void *)) glColor3fv; 
         else           a_ptr->ColorFuncPtr = (void (*)(void *)) glColor4fv;  
         break;
      case GL_DOUBLE:
         a_ptr->ColorStrideB = stride ? stride : size*sizeof(GLdouble);
         if (size == 3) a_ptr->ColorFuncPtr = (void (*)(void *)) glColor3dv; 
         else           a_ptr->ColorFuncPtr = (void (*)(void *)) glColor4dv;  
         break;
      default:
         __glx_error( GL_INVALID_ENUM, "__glx_ColorPointer(type)" );
         return;
   }
   a_ptr->ColorSize = size;
   a_ptr->ColorType = type;
   a_ptr->ColorStride = stride;
   a_ptr->ColorPtr = (void *) ptr;
}



void __glx_IndexPointer( GLenum type, GLsizei stride, const GLvoid *ptr )
{
   struct gl_array_attrib *a_ptr = NULL;

   assert(GLCurrent);
   a_ptr = &GLCurrent->Array;
   if (stride<0) {
      __glx_error( GL_INVALID_VALUE, "__glx_IndexPointer(stride)" );
      return;
   }
   switch (type) {
      case GL_UNSIGNED_BYTE:
         a_ptr->IndexStrideB = stride ? stride : sizeof(GLbyte);
         a_ptr->IndexFuncPtr = (void (*)(void *)) glIndexubv;
         break;
      case GL_SHORT:
         a_ptr->IndexStrideB = stride ? stride : sizeof(GLshort);
         a_ptr->IndexFuncPtr = (void (*)(void *)) glIndexsv;
         break;
      case GL_INT:
         a_ptr->IndexStrideB = stride ? stride : sizeof(GLint);
         a_ptr->IndexFuncPtr = (void (*)(void *)) glIndexiv;
         break;
      case GL_FLOAT:
         a_ptr->IndexStrideB = stride ? stride : sizeof(GLfloat);
         a_ptr->IndexFuncPtr = (void (*)(void *)) glIndexfv;
         break;
      case GL_DOUBLE:
         a_ptr->IndexStrideB = stride ? stride : sizeof(GLdouble);
         a_ptr->IndexFuncPtr = (void (*)(void *)) glIndexdv;
         break;
      default:
         __glx_error( GL_INVALID_ENUM, "__glx_IndexPointer(type)" );
         return;
   }
   a_ptr->IndexType = type;
   a_ptr->IndexStride = stride;
   a_ptr->IndexPtr = (void *) ptr;
}



void __glx_TexCoordPointer( GLint size, GLenum type, GLsizei stride,
                         const GLvoid *ptr )
{
   GLuint texSet = 0;
   struct gl_array_attrib *a_ptr = NULL;

   assert(GLCurrent);
   a_ptr = &GLCurrent->Array;

   if (size<1 || size>4) {
      __glx_error( GL_INVALID_VALUE, "__glx_TexCoordPointer(size)" );
      return;
   }
   if (stride<0) {
      __glx_error( GL_INVALID_VALUE, "__glx_TexCoordPointer(stride)" );
      return;
   }
   switch (type) {
      case GL_SHORT:
         a_ptr->TexCoordStrideB[texSet] = stride ? stride : 
            size * sizeof(GLshort);
         switch (size) {
            case 1:  
               a_ptr->TexCoordFuncPtr = (void (*)(void *)) glTexCoord1sv; 
               break;
            case 2:  
               a_ptr->TexCoordFuncPtr = (void (*)(void *)) glTexCoord2sv; 
               break;
            case 3:  
               a_ptr->TexCoordFuncPtr = (void (*)(void *)) glTexCoord3sv; 
               break;
            case 4:  
               a_ptr->TexCoordFuncPtr = (void (*)(void *)) glTexCoord4sv; 
               break;
         }
         break;
      case GL_INT:
         a_ptr->TexCoordStrideB[texSet] = stride ? stride : 
            size * sizeof(GLint);
         switch (size) {
            case 1:  
               a_ptr->TexCoordFuncPtr = (void (*)(void *)) glTexCoord1iv; 
               break;
            case 2:  
               a_ptr->TexCoordFuncPtr = (void (*)(void *)) glTexCoord2iv; 
               break;
            case 3:  
               a_ptr->TexCoordFuncPtr = (void (*)(void *)) glTexCoord3iv; 
               break;
            case 4:  
               a_ptr->TexCoordFuncPtr = (void (*)(void *)) glTexCoord4iv; 
               break;
         }
         break;
      case GL_FLOAT:
         a_ptr->TexCoordStrideB[texSet] = stride ? stride : 
            size * sizeof(GLfloat);
         switch (size) {
            case 1:  
               a_ptr->TexCoordFuncPtr = (void (*)(void *)) glTexCoord1fv; 
               break;
            case 2:  
               a_ptr->TexCoordFuncPtr = (void (*)(void *)) glTexCoord2fv; 
               break;
            case 3:  
               a_ptr->TexCoordFuncPtr = (void (*)(void *)) glTexCoord3fv; 
               break;
            case 4:  
               a_ptr->TexCoordFuncPtr = (void (*)(void *)) glTexCoord4fv; 
               break;
         }
         break;
      case GL_DOUBLE:
         a_ptr->TexCoordStrideB[texSet] = stride ? stride : 
            size * sizeof(GLdouble);
         switch (size) {
            case 1:  
               a_ptr->TexCoordFuncPtr = (void (*)(void *)) glTexCoord1dv; 
               break;
            case 2:  
               a_ptr->TexCoordFuncPtr = (void (*)(void *)) glTexCoord2dv; 
               break;
            case 3:  
               a_ptr->TexCoordFuncPtr = (void (*)(void *)) glTexCoord3dv; 
               break;
            case 4:  
               a_ptr->TexCoordFuncPtr = (void (*)(void *)) glTexCoord4dv; 
               break;
         }
         break;
      default:
         __glx_error( GL_INVALID_ENUM, "__glx_TexCoordPointer(type)" );
         return;
   }

   a_ptr->TexCoordSize[texSet] = size;
   a_ptr->TexCoordType[texSet] = type;
   a_ptr->TexCoordStride[texSet] = stride;
   a_ptr->TexCoordPtr[texSet] = (void *) ptr;
}


#if 0
/* GL_SGIS_multitexture */
void __glx_MultiTexCoordPointer( GLenum target, GLint size, GLenum type, 
                              GLsizei stride, const GLvoid *ptr )
{
   GLuint texSet;
   assert(GLCurrent);
   if (target < GL_TEXTURE0_SGIS || target > GL_TEXTURE1_SGIS) {
      __glx_error( GL_INVALID_ENUM, "glMultiTexCoord(target)");
      return;
   }
   if (stride<0) {
      __glx_error( GL_INVALID_VALUE, "glMultiTexCoordPointer(stride)" );
      return;
   }
   if (size<1 || size>4) {
      __glx_error( GL_INVALID_VALUE, "glMultiTexCoordPointer(size)" );
      return;
   }
   texSet = target - GL_TEXTURE0_SGIS;
   switch (type) {
      case GL_SHORT:
         GLCurrent->Array.TexCoordStrideB[texSet] = stride ? stride : 
            size * sizeof(GLshort);
         break;
      case GL_INT:
         GLCurrent->Array.TexCoordStrideB[texSet] = stride ? stride : 
            size * sizeof(GLint);
         break;
      case GL_FLOAT:
         GLCurrent->Array.TexCoordStrideB[texSet] = stride ? stride : 
            size * sizeof(GLfloat);
         break;
      case GL_DOUBLE:
         GLCurrent->Array.TexCoordStrideB[texSet] = stride ? stride : 
            size * sizeof(GLdouble);
         break;
      default:
         __glx_error( GL_INVALID_ENUM, "glMultiTexCoordPointer(type)" );
         return;
   }
   GLCurrent->Array.TexCoordSize[texSet] = size;
   GLCurrent->Array.TexCoordType[texSet] = type;
   GLCurrent->Array.TexCoordStride[texSet] = stride;
   GLCurrent->Array.TexCoordPtr[texSet] = (void *) ptr;
}
#endif

#if 0
void __glx_InterleavedTextureCoordSets( GLint factor )
{
   GLint maxSets;
   assert(GLCurrent);
   maxSets  = __glx_max_texture_coord_sets(GLCurrent);
   if (factor < 1 || factor > maxSets) {
      __glx_error( GL_INVALID_VALUE, "glInterleavedTextureCoordSets" );
      return;
   }
   GLCurrent->Array.TexCoordInterleaveFactor = factor;
}
#endif


void __glx_EdgeFlagPointer( GLsizei stride, const GLboolean *ptr )
{
   assert(GLCurrent);
   if (stride<0) {
      __glx_error( GL_INVALID_VALUE, "glEdgeFlagPointer(stride)" );
      return;
   }
   GLCurrent->Array.EdgeFlagStride = stride;
   GLCurrent->Array.EdgeFlagStrideB = stride ? stride : sizeof(GLboolean);
   GLCurrent->Array.EdgeFlagFuncPtr = (void (*)(void *)) glEdgeFlagv;
   GLCurrent->Array.EdgeFlagPtr = (GLboolean *) ptr;
}



/*
 * Execute
 */
void __glx_ArrayElement( GLint i )
{
   GLuint texSet = 0;
   struct gl_array_attrib *a_ptr = NULL;

   assert(GLCurrent);
   a_ptr = &GLCurrent->Array;

   if (a_ptr->NormalEnabled) {
      (a_ptr->NormalFuncPtr) ( (void *)
          ( (unsigned char *) a_ptr->NormalPtr + i * a_ptr->NormalStrideB) );
   }

   if (a_ptr->ColorEnabled) {
      (a_ptr->ColorFuncPtr) ( (void *) 
         ( (unsigned char *) a_ptr->ColorPtr + i * a_ptr->ColorStrideB) );
   }

   if (a_ptr->IndexEnabled) {
      (a_ptr->IndexFuncPtr) ( (void *) 
         ( (unsigned char *) a_ptr->IndexPtr + i * a_ptr->IndexStrideB) );
   }

#if 0
   for (texSet=0; texSet<MAX_TEX_SETS; texSet++) {
       if (a_ptr->TexCoordEnabled[texSet]) {
          GLbyte *p = (GLbyte*) a_ptr->TexCoordPtr[texSet]
                      + i * a_ptr->TexCoordStrideB[texSet];
          switch (a_ptr->TexCoordType[texSet]) {
             case GL_SHORT:
                switch (a_ptr->TexCoordSize[texSet]) {
                }
                break;
             case GL_INT:
                switch (a_ptr->TexCoordSize[texSet]) {
                }
                break;
             case GL_FLOAT:
                switch (a_ptr->TexCoordSize[texSet]) {
                }
                break;
             case GL_DOUBLE:
                switch (a_ptr->TexCoordSize[texSet]) {
                }
                break;
             default:
                __glx_problem("Bad texcoord type in __glx_ArrayElement");
                return;
          }
       }
   }
#else
   if (a_ptr->TexCoordEnabled[texSet]) {
       (a_ptr->TexCoordFuncPtr) ( (void *) 
          ( (unsigned char *) a_ptr->TexCoordPtr[texSet] + 
              i * a_ptr->TexCoordStrideB[texSet]) );
   }
#endif

   if (a_ptr->EdgeFlagEnabled) {
      (a_ptr->EdgeFlagFuncPtr) ( (void *) 
         ( (unsigned char *) a_ptr->EdgeFlagPtr + i * a_ptr->EdgeFlagStrideB) );
   }

   if (a_ptr->VertexEnabled) {
      (a_ptr->VertexFuncPtr) ( (void *) 
         ( (unsigned char *) a_ptr->VertexPtr + i * a_ptr->VertexStrideB) );
   }

}



/*
 * Execute
 */
void __glx_DrawArrays( GLenum mode, GLint first, GLsizei count )
{
   GLint i;

   if (count<0) {
      __glx_error( GL_INVALID_VALUE, "glDrawArrays(count)" );
      return;
   }

   switch (mode) {
      case GL_POINTS:
      case GL_LINES:
      case GL_LINE_STRIP:
      case GL_LINE_LOOP:
      case GL_TRIANGLES:
      case GL_TRIANGLE_STRIP:
      case GL_TRIANGLE_FAN:
      case GL_QUADS:
      case GL_QUAD_STRIP:
      case GL_POLYGON:
         /* OK */
         break;
      default:
         __glx_error( GL_INVALID_ENUM, "glDrawArrays(mode)" );
         return;
   }

   __glx_Begin(mode);
      for (i = 0; i < count; i++) {
         __glx_ArrayElement( first + i );
      }
   __glx_End();

}






/*
 * Execute only
 */
void __glx_DrawElements( GLenum mode, GLsizei count,
                      GLenum type, const GLvoid *indices )
{

   if (count<0) {
      __glx_error( GL_INVALID_VALUE, "glDrawElements(count)" );
      return;
   }

   switch (mode) {
      case GL_POINTS:
      case GL_LINES:
      case GL_LINE_STRIP:
      case GL_LINE_LOOP:
      case GL_TRIANGLES:
      case GL_TRIANGLE_STRIP:
      case GL_TRIANGLE_FAN:
      case GL_QUADS:
      case GL_QUAD_STRIP:
      case GL_POLYGON:
         /* OK */
         break;
      default:
         __glx_error( GL_INVALID_ENUM, "glDrawArrays(mode)" );
         return;
   }

   switch (type) {
      case GL_UNSIGNED_BYTE:
         {
            GLubyte *ub_indices = (GLubyte *) indices;
            GLint i;
            __glx_Begin( mode );
            for (i=0;i<count;i++) {
               __glx_ArrayElement( (GLint) ub_indices[i] );
            }
            __glx_End();
         }
         break;
      case GL_UNSIGNED_SHORT:
         {
            GLushort *us_indices = (GLushort *) indices;
            GLint i;
            __glx_Begin( mode );
            for (i=0;i<count;i++) {
               __glx_ArrayElement( (GLint) us_indices[i] );
            }
            __glx_End( );
         }
         break;
      case GL_UNSIGNED_INT:
         {
            GLuint *ui_indices = (GLuint *) indices;
            GLint i;
            __glx_Begin( mode );
            for (i=0;i<count;i++) {
               __glx_ArrayElement( (GLint) ui_indices[i] );
            }
            __glx_End( );
         }
         break;
      default:
         __glx_error(GL_INVALID_ENUM, "glDrawElements(type)" );
         return;
   }
}



void __glx_InterleavedArrays( GLenum format, GLsizei stride,
                           const GLvoid *pointer )
{
   GLboolean tflag, cflag, nflag;  /* enable/disable flags */
   GLint tcomps, ccomps, vcomps;   /* components per texcoord, color, vertex */
   GLenum ctype = 0;                   /* color type */
   GLint coffset=0, noffset=0, voffset=0; /* color, normal, vertex offsets */
   GLint defstride;                /* default stride */
   GLint c, f;
   GLint coordSetSave;

   f = sizeof(GLfloat);
   c = f * ((4*sizeof(GLubyte) + (f-1)) / f);

   if (stride<0) {
      __glx_error( GL_INVALID_VALUE, "glInterleavedArrays(stride)" );
      return;
   }

   switch (format) {
      case GL_V2F:
         tflag = GL_FALSE;  cflag = GL_FALSE;  nflag = GL_FALSE;
         tcomps = 0;  ccomps = 0;  vcomps = 2;
         voffset = 0;
         defstride = 2*f;
         break;
      case GL_V3F:
         tflag = GL_FALSE;  cflag = GL_FALSE;  nflag = GL_FALSE;
         tcomps = 0;  ccomps = 0;  vcomps = 3;
         voffset = 0;
         defstride = 3*f;
         break;
      case GL_C4UB_V2F:
         tflag = GL_FALSE;  cflag = GL_TRUE;  nflag = GL_FALSE;
         tcomps = 0;  ccomps = 4;  vcomps = 2;
         ctype = GL_UNSIGNED_BYTE;
         coffset = 0;
         voffset = c;
         defstride = c + 2*f;
         break;
      case GL_C4UB_V3F:
         tflag = GL_FALSE;  cflag = GL_TRUE;  nflag = GL_FALSE;
         tcomps = 0;  ccomps = 4;  vcomps = 3;
         ctype = GL_UNSIGNED_BYTE;
         coffset = 0;
         voffset = c;
         defstride = c + 3*f;
         break;
      case GL_C3F_V3F:
         tflag = GL_FALSE;  cflag = GL_TRUE;  nflag = GL_FALSE;
         tcomps = 0;  ccomps = 3;  vcomps = 3;
         ctype = GL_FLOAT;
         coffset = 0;
         voffset = 3*f;
         defstride = 6*f;
         break;
      case GL_N3F_V3F:
         tflag = GL_FALSE;  cflag = GL_FALSE;  nflag = GL_TRUE;
         tcomps = 0;  ccomps = 0;  vcomps = 3;
         noffset = 0;
         voffset = 3*f;
         defstride = 6*f;
         break;
      case GL_C4F_N3F_V3F:
         tflag = GL_FALSE;  cflag = GL_TRUE;  nflag = GL_TRUE;
         tcomps = 0;  ccomps = 4;  vcomps = 3;
         ctype = GL_FLOAT;
         coffset = 0;
         noffset = 4*f;
         voffset = 7*f;
         defstride = 10*f;
         break;
      case GL_T2F_V3F:
         tflag = GL_TRUE;  cflag = GL_FALSE;  nflag = GL_FALSE;
         tcomps = 2;  ccomps = 0;  vcomps = 3;
         voffset = 2*f;
         defstride = 5*f;
         break;
      case GL_T4F_V4F:
         tflag = GL_TRUE;  cflag = GL_FALSE;  nflag = GL_FALSE;
         tcomps = 4;  ccomps = 0;  vcomps = 4;
         voffset = 4*f;
         defstride = 8*f;
         break;
      case GL_T2F_C4UB_V3F:
         tflag = GL_TRUE;  cflag = GL_TRUE;  nflag = GL_FALSE;
         tcomps = 2;  ccomps = 4;  vcomps = 3;
         ctype = GL_UNSIGNED_BYTE;
         coffset = 2*f;
         voffset = c+2*f;
         defstride = c+5*f;
         break;
      case GL_T2F_C3F_V3F:
         tflag = GL_TRUE;  cflag = GL_TRUE;  nflag = GL_FALSE;
         tcomps = 2;  ccomps = 3;  vcomps = 3;
         ctype = GL_FLOAT;
         coffset = 2*f;
         voffset = 5*f;
         defstride = 8*f;
         break;
      case GL_T2F_N3F_V3F:
         tflag = GL_TRUE;  cflag = GL_FALSE;  nflag = GL_TRUE;
         tcomps = 2;  ccomps = 0;  vcomps = 3;
         noffset = 2*f;
         voffset = 5*f;
         defstride = 8*f;
         break;
      case GL_T2F_C4F_N3F_V3F:
         tflag = GL_TRUE;  cflag = GL_TRUE;  nflag = GL_TRUE;
         tcomps = 2;  ccomps = 4;  vcomps = 3;
         ctype = GL_FLOAT;
         coffset = 2*f;
         noffset = 6*f;
         voffset = 9*f;
         defstride = 12*f;
         break;
      case GL_T4F_C4F_N3F_V4F:
         tflag = GL_TRUE;  cflag = GL_TRUE;  nflag = GL_TRUE;
         tcomps = 4;  ccomps = 4;  vcomps = 4;
         ctype = GL_FLOAT;
         coffset = 4*f;
         noffset = 8*f;
         voffset = 11*f;
         defstride = 15*f;
         break;
      default:
         __glx_error( GL_INVALID_ENUM, "glInterleavedArrays(format)" );
         return;
   }

   if (stride==0) {
      stride = defstride;
   }

   __glx_DisableClientState( GL_EDGE_FLAG_ARRAY );
   __glx_DisableClientState( GL_INDEX_ARRAY );

   coordSetSave = GLCurrent->TexCoordSet;
   if (tflag) {
      GLint i;
      for (i = 0; i < GLCurrent->Array.TexCoordInterleaveFactor; i++) {
#if 0
         __glx_SelectTextureCoordSet( GL_TEXTURE0_EXT + i );
#endif
         __glx_EnableClientState( GL_TEXTURE_COORD_ARRAY );
         __glx_TexCoordPointer( tcomps, GL_FLOAT, stride,
                             (GLubyte *) pointer + i * coffset );
      }
      for (i = 0; i < MAX_TEX_COORD_SETS; i++) {
#if 0
         __glx_SelectTextureCoordSet( GL_TEXTURE0_EXT + i );
#endif
         __glx_DisableClientState( GL_TEXTURE_COORD_ARRAY );
      }
   }
   else {
      GLint i;
      for (i = 0; i < MAX_TEX_COORD_SETS; i++) {
#if 0
         __glx_SelectTextureCoordSet( GL_TEXTURE0_EXT + i );
#endif
         __glx_DisableClientState( GL_TEXTURE_COORD_ARRAY );
      }
   }
#if 0
   /* Restore texture coordinate set index */
   __glx_SelectTextureCoordSet( coordSetSave );
#endif

   if (cflag) {
      __glx_EnableClientState( GL_COLOR_ARRAY );
      __glx_ColorPointer( ccomps, ctype, stride, (GLubyte*) pointer + coffset );
   }
   else {
      __glx_DisableClientState( GL_COLOR_ARRAY );
   }

   if (nflag) {
      __glx_EnableClientState( GL_NORMAL_ARRAY );
      __glx_NormalPointer( GL_FLOAT, stride, (GLubyte*) pointer + noffset );
   }
   else {
      __glx_DisableClientState( GL_NORMAL_ARRAY );
   }

   __glx_EnableClientState( GL_VERTEX_ARRAY );
   __glx_VertexPointer( vcomps, GL_FLOAT, stride, (GLubyte *) pointer + voffset );
}



void __glx_DrawRangeElements( GLenum mode, GLuint start,
                           GLuint end, GLsizei count, GLenum type,
                           const GLvoid *indices )
{
   /* XXX TODO optimize this function someday- it would be worthwhile */
   if (end < start) {
      __glx_error(GL_INVALID_VALUE, "glDrawRangeElements( end < start )");
      return;
   }
   (void) start;
   (void) end;
   __glx_DrawElements( mode, count, type, indices );
}


