# Copyright (C) 2007  Lars Wirzenius <liw@iki.fi>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.


"""Unit tests for the Unperish plugin for Subversion operations."""


import os
import shutil
import tempfile
import unittest

import unperishlib


class SvnTestBase(unittest.TestCase):

    def create_file(self, path, contents):
        f = file(path, "w")
        f.write(contents)
        f.close()

    def create_svn_stuff(self):
        """Create svn repository and a dummy project."""
        root = tempfile.mkdtemp()
        repo = os.path.join(root, "repo")
        unperishlib.run(["svnadmin", "create", repo])
        proj = os.path.join(root, "proj")
        os.mkdir(proj)
        self.create_file(os.path.join(proj, "foo"), "foo\n")
        unperishlib.run(["svn", "-q", "import", "-m", "initial_import",
                         proj, "file://" + repo])
        shutil.rmtree(proj)
        unperishlib.run(["svn", "checkout", "file://" + repo, proj])
        return root, repo, proj


class IsSvnBranchTests(SvnTestBase):

    def setUp(self):
        pm = unperishlib.PluginManager()
        pm.set_plugin_directories(["plugins"])
        pm.load_plugins()
        self.oper = pm.get_operation("is-svn-branch")
        self.root, self.repo, self.proj = self.create_svn_stuff()
        self.options = unperishlib.FakeOptions(is_svn_branch=False)

    def tearDown(self):
        shutil.rmtree(self.root)

    def testNoticesNonSvnDirectory(self):
        self.options.directory = self.root
        self.failUnlessEqual(self.oper.do_it(self.options), None)
        self.failUnlessEqual(self.options.is_svn_branch, False)

    def testAcceptsSvnDirectory(self):
        self.options.directory = self.proj
        self.failUnlessEqual(self.oper.do_it(self.options), None)
        self.failUnlessEqual(self.options.is_svn_branch, True)


class BuildTarballFromSvnTests(SvnTestBase):

    def setUp(self):
        self.app = unperishlib.Application(plugins=["plugins"])
        self.pm = self.app.get_plugin_manager()
        self.root, self.repo, self.proj = self.create_svn_stuff()

    def tearDown(self):
        shutil.rmtree(self.root)

    def testRemovesDirectoryForBasename(self):
        self.pm.load_plugins()
        oper = self.pm.get_operation("build-tarball-from-svn")
        self.failUnlessEqual(oper.tarball_basename("foo/bar"), "bar")

    def testRemovesTarGzSuffixForBasename(self):
        self.pm.load_plugins()
        oper = self.pm.get_operation("build-tarball-from-svn")
        self.failUnlessEqual(oper.tarball_basename("foo/bar.tar.gz"), "bar")

    def testRemovesTgzSuffixForBasename(self):
        self.pm.load_plugins()
        oper = self.pm.get_operation("build-tarball-from-svn")
        self.failUnlessEqual(oper.tarball_basename("foo/bar.tar.gz"), "bar")

    def testRemovesTarBz2SuffixForBasename(self):
        self.pm.load_plugins()
        oper = self.pm.get_operation("build-tarball-from-svn")
        self.failUnlessEqual(oper.tarball_basename("foo/bar.tar.bz2"), "bar")

    def testCreatesTarball(self):
        build_area = tempfile.mkdtemp()

        options, opers = self.app.parse_args(["--project-name=pink",
                                              "--project-version=4.2", 
                                              "--directory=" + self.proj,
                                              "--build-area=" + build_area])
        self.failUnlessEqual(options.upstream_tarball, None)
        
        oper = self.pm.get_operation("build-tarball-from-svn")
        oper.do_it(options)
        
        tarball_base = "pink-4.2.tar.gz"
        tarball_path = os.path.join(build_area, tarball_base)

        self.failUnless(os.path.exists(tarball_path))
        self.failUnlessEqual(options.upstream_tarball, tarball_path)

        shutil.rmtree(build_area)
