# Copyright (C) 2007  Lars Wirzenius <liw@iki.fi>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.


"""Unperish plugin for Subversion operations."""


import os
import shutil
import tempfile

import unperishlib


class IsSvnBranch(unperishlib.Operation):

    """Verify that the directory is a Subversion branch.
    
    This operation exists just to verify that the directory we operate
    on is a Subversion branch. This way other operations in this plugin
    do not need to embed the check themselves, but can just require the
    is_svn_branch option this operation provides.
    
    """

    name = "is-svn-branch"
    
    provided_options = ["is_svn_branch"]
    
    def add_options(self, parser):
        parser.add_option("--is-svn-branch", action="store_true",
                          help="Assume the source directory is a Subversion "
                               "branch.")

    def do_it(self, options):
        if os.path.isdir(os.path.join(options.directory, ".svn")):
            options.is_svn_branch = True


class BuildTarballFromSvn(unperishlib.Operation):

    """Create a release tarball from a Subversion branch.
    
    The entire contents of the Subversion branch are exported into a .tar.gz
    file in the build area. The full path to the upstream tarball
    (not just basename) is stored into the upstream_tarball option.
    
    """
    
    name = "build-tarball-from-svn"

    required_options = ["is_svn_branch", "build_area_exists", "project_name", 
                        "project_version"]

    provided_options = ["upstream_tarball"]

    def tarball_basename(self, upstream_tarball):
        """Return basename of upstream tarball.
        
        The basename excludes any directory parts, and any .tar.gz or
        .tar.bz2 suffix.

        """
        basename = os.path.basename(upstream_tarball)
        for suffix in [".tar.gz", ".tgz", ".tar.bz2"]:
            if basename.endswith(suffix):
                basename = basename[:-len(suffix)]
                break
        return basename

    def do_it(self, options):
        if options.upstream_tarball is None:
            base = ("%s-%s.tar.gz" % 
                        (options.project_name, options.project_version))
            options.upstream_tarball = os.path.join(options.build_area, base)
        if not os.path.exists(options.upstream_tarball):
            basename = self.tarball_basename(options.upstream_tarball)
            temp = tempfile.mkdtemp()
            temp2 = os.path.join(temp, basename)
            unperishlib.run(["svn", "--quiet", "export", ".", temp2],
                            cwd=options.directory)
            unperishlib.run(["tar", "-C", temp, 
                             "-czf", options.upstream_tarball, 
                             basename])
            shutil.rmtree(temp)
