/*
 * Copyright (c) 2003-2005 Erez Zadok
 * Copyright (c) 2003-2005 Charles P. Wright
 * Copyright (c) 2003-2005 Mohammad Nayyer Zubair
 * Copyright (c) 2003-2005 Puja Gupta
 * Copyright (c) 2003-2005 Harikesavan Krishnan
 * Copyright (c) 2003-2005 Stony Brook University
 * Copyright (c) 2003-2005 The Research Foundation of State University of New York
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package.
 *
 * This Copyright notice must be kept intact and distributed with all sources.
 */
/*
 *  $Id: subr.c,v 1.98 2005/02/08 15:17:38 cwright Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#include "fist.h"
#include "unionfs.h"

/* lookup files/directories from index 'bstart' onwards, does not looksup bstart */
STATIC int
unionfs_partial_lookup(dentry_t *dentry)
{
    int err = 0;
    int dentry_count = 0;
    int bstart, bindex;
    const char *name;
    unsigned int namelen;
    dentry_t *hidden_dentry;
    dentry_t *hidden_dir_dentry;
    int positive;

    print_entry_location();

    fist_print_dentry("IN: unionfs_partial_lookup: ", dentry);
    lock_dpd(dentry);
    bstart = dbstart(dentry);

    name = dentry->d_name.name;
    namelen = dentry->d_name.len;

    positive = dentry->d_inode ? 1 :  0;

    for (bindex = bstart + 1; bindex < sbmax(dentry->d_sb); bindex++) {
	/* Check if there is already something there. */
	hidden_dentry = dtohd_index(dentry, bindex);
	if (hidden_dentry) {
	    continue;
	}

	/* Find out hidden parent dentry */
	hidden_dir_dentry = dtohd_index(dentry->d_parent, bindex);

	/* continue if hidden parent doesn't exist */
	if (!hidden_dir_dentry) {
	    continue;
	}

	/* lookup underlying file system */
	hidden_dentry = lookup_one_len(name, hidden_dir_dentry, namelen);
	if (IS_ERR(hidden_dentry)) {
	    printk("ERR from hidden_dentry!!!\n");
	    err = PTR_ERR(hidden_dentry);
	    goto out;
	}

	if (hidden_dentry->d_inode) {
	    /* this is not a negative dentry, so init dtohd with
	     * this new dentry.  Also, change bend.
	     */
	    set_dtohd_index(dentry, bindex, hidden_dentry);
	    set_dbend(dentry, bindex);
	    dentry_count++;
	} else {
	    /* dput the unnecessary negative dentry */
	    dput(hidden_dentry);
	}
    } // end for

    if (dentry_count) {
	if (positive) {
    		/* need to connect the new inodes to the pre-existing
     		 * unionfs inode, unionfs_reinterpose
      		 */
		unionfs_reinterpose(dentry);
	} else {
		int bstart;
		int bend;

		/* Somehow the file was created for us, so we need to interpose it with a new inode. */

		bstart = dbstart(dentry);
		bend = dbend(dentry);

        	for (bindex = bstart; bindex <= bend; bindex++) {
               		if (dtohd_index(dentry, bindex)) {
				if (!dtohd_index(dentry, bindex)->d_inode) {
					dput(dtohd_index(dentry, bindex));
					set_dtohd_index(dentry, bindex, NULL);
				} else {
					set_dbstart(dentry, bindex);
                        		break;
				}
			}
		}

		unionfs_interpose(dentry, dentry->d_sb, 0);
	}
    }

 out:
    unlock_dpd(dentry);
    fist_print_dentry("OUT: unionfs_partial_lookup: ", dentry);
    print_exit_status(err);
    return err;
}


/* Pass an unionfs dentry and an index.  It will try to create a whiteout for the filename
 * in dentry, and will try in branch 'index'.  On error, it will proceed to a branch
 * to the left
 */
int create_whiteout(dentry_t *dentry, int start)
{
    int bstart, bend, bindex;
    dentry_t *hidden_dir_dentry;
    dentry_t *hidden_dentry;
    dentry_t *hidden_wh_dentry;
    char *name = NULL;
    int err = -EUNIONFS_NO_WHITEOUT;

    print_entry_location();

    PASSERT(dentry);
    PASSERT(dentry->d_parent);

    fist_print_dentry("IN create_whiteout", dentry);
    bstart = dbstart(dentry);
    bend = dbend(dentry);

    /* create dentry's whiteout equivalent */
    name = KMALLOC(sizeof(char) * (dentry->d_name.len + 5), GFP_UNIONFS);
    if (!name) {
        err = -ENOMEM;
        goto out;
    }
    strcpy(name, ".wh.");
    strncat(name, dentry->d_name.name, dentry->d_name.len);
    name[dentry->d_name.len + 4] = '\0';

    /* start from 'start', create foo then rename it to .wh.foo, on error go to
       the next left branch, if hidden_dentry is NULL do recursive directory creation */
    for (bindex = start; bindex >= 0; bindex--) {
	hidden_dentry = dtohd_index(dentry, bindex);

	if (!hidden_dentry) {
	    /* if hidden dentry is not present, create the entire
	     * hidden dentry directory structure and go ahead.
	     * Since we want to just create whiteout, we only want the parent
	     * dentry, and hence get rid of this dentry.
	     */
	    hidden_dentry = unionfs_create_dirs(dentry->d_inode, dentry, bindex);
	    if (!hidden_dentry || IS_ERR(hidden_dentry)) {
		fist_dprint(8, "hidden dentry NULL for bindex = %d\n", bindex);
		continue;
	    }
	}
	hidden_wh_dentry = lookup_one_len(name, hidden_dentry->d_parent, dentry->d_name.len + 4);

	if (!hidden_wh_dentry || IS_ERR(hidden_wh_dentry)) {
	    continue;
	}
	ASSERT(!hidden_wh_dentry->d_inode);

	hidden_dir_dentry = lock_parent(hidden_wh_dentry);
    	if (!(err = is_robranch_super(dentry->d_sb, bindex))) {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	    err = vfs_create(hidden_dir_dentry->d_inode, hidden_wh_dentry, ~current->fs->umask & S_IRWXUGO);
#else
	    err = vfs_create(hidden_dir_dentry->d_inode, hidden_wh_dentry, ~current->fs->umask & S_IRWXUGO,NULL);

#endif
	}
	unlock_dir(hidden_dir_dentry);
	dput(hidden_wh_dentry);


	if (!err) {
	    break;
        } else {
            if (IS_SET(dentry->d_sb, GLOBAL_ERR_PASSUP)) {
                goto out;
            }
            /* break out of for loop if error returned was NOT -EROFS */
            if (!IS_COPYUP_ERR(err)) {
                break;
            }
        }
    } // end for

    fist_print_dentry("OUT create_whiteout", dentry);
out:
    if (name) {
       KFREE(name);
    }
    print_exit_status(err);
    return err;
}

/* Pass an unionfs dentry, name of file, and an index.  It will try to create a whiteout
 * for the filename passed in the parent directory passed.  It will start trying
 * in branch start.  On error, it will proceed to a branch to the left
 */
int create_whiteout_parent(dentry_t *parent_dentry, const char *filename, int start)
{
    int bindex;
    int old_bstart, old_bend;
    dentry_t *hidden_dir_dentry;
    dentry_t *hidden_grand_parent_dentry;
    dentry_t *hidden_parent_dentry;
    dentry_t *hidden_wh_dentry;
    char *name = NULL;
    int err = -EUNIONFS_NO_WHITEOUT;

    print_entry_location();

    PASSERT(parent_dentry);
    old_bstart = dbstart(parent_dentry);
    old_bend = dbend(parent_dentry);

    fist_print_dentry("IN create_whiteout_parent", parent_dentry);

    /* create dentry's whiteout equivalent */
    name = KMALLOC(sizeof(char) * (strlen(filename) + 5), GFP_UNIONFS);
    if (!name) {
        err = -ENOMEM;
        goto out;
    }
    strcpy(name, ".wh.");
    strncat(name, filename, strlen(filename));
    name[strlen(filename) + 4] = '\0';

    for (bindex = start; bindex >= 0; bindex--) {

        hidden_parent_dentry = dtohd_index(parent_dentry, bindex);

	if (!hidden_parent_dentry) {
            /* create the recursive directory structure and return the
             * negative dentry for the parent where we want to create whiteout
             */
            ASSERT(parent_dentry->d_inode != NULL);
	    hidden_parent_dentry = unionfs_create_dirs(parent_dentry->d_inode, parent_dentry, bindex);
	    if (!hidden_parent_dentry || IS_ERR(hidden_parent_dentry)) {
		fist_dprint(8, "hidden dentry NULL for bindex = %d\n", bindex);
		continue;
	    } else {
	        /* create directory of the hidden parent, if it is negative.
                 * This is where whiteout is created
                 */
                hidden_grand_parent_dentry = lock_parent(hidden_parent_dentry);
                PASSERT(hidden_grand_parent_dentry->d_inode);

                /* We shouldn't create things in a read-only branch. */
                if (!(err = is_robranch_super(parent_dentry->d_sb, bindex))) {
                    err = vfs_mkdir(hidden_grand_parent_dentry->d_inode, hidden_parent_dentry, S_IRWXU);
                }

                unlock_dir(hidden_grand_parent_dentry);
                if (err || !hidden_parent_dentry->d_inode) {
		    dput(hidden_parent_dentry);
                    /* if error, we need to revert back the old bstart and bend for this
                     * parent and return back
                     */
                    set_dbstart(parent_dentry, old_bstart);
                    set_dbend(parent_dentry, old_bend);
                    set_dtohd_index(parent_dentry, bindex, NULL);

                    if (IS_SET(parent_dentry->d_sb, GLOBAL_ERR_PASSUP)) {
                        goto out;
                    }
                    /* break out of for loop if error returned was NOT -EROFS */
                    if (!IS_COPYUP_ERR(err)) {
                        break;
                    }

                    continue;
                }
                itohi_index(parent_dentry->d_inode, bindex) = igrab(hidden_parent_dentry->d_inode);
            }
        }

        /* lookup for the whiteout dentry that we want to create */
	hidden_wh_dentry = lookup_one_len(name, hidden_parent_dentry, strlen(filename) + 4);
	if (!hidden_wh_dentry || IS_ERR(hidden_wh_dentry)) {
	    continue;
	}
	ASSERT(!hidden_wh_dentry->d_inode);

        /* hidden_dir_dentry and hidden_parent_dentry are going to be same only */
	hidden_dir_dentry = lock_parent(hidden_wh_dentry);

        /* We shouldn't create things in a read-only branch. */
        if (!(err = is_robranch_super(parent_dentry->d_sb, bindex))) {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
            err = vfs_create(hidden_dir_dentry->d_inode, hidden_wh_dentry, ~current->fs->umask & S_IRWXUGO);
#else
			err = vfs_create(hidden_dir_dentry->d_inode, hidden_wh_dentry, ~current->fs->umask & S_IRWXUGO,NULL);
#endif
	}

        unlock_dir(hidden_dir_dentry);
	dput(hidden_wh_dentry);

	if (!err) {
	    break;
        } else {
            if (IS_SET(parent_dentry->d_sb, GLOBAL_ERR_PASSUP)) {
                goto out;
            }
            /* break out of for loop if error returned was NOT -EROFS */
            if (!IS_COPYUP_ERR(err)) {
                break;
            }
        }
    } // end for

out:
    fist_print_dentry("OUT create_whiteout", parent_dentry);
    if (name) {
       KFREE(name);
    }
    print_exit_status(err);
    return err;
}

/* This is a helper function for rename, which ends up with hosed over dentries
 * when it needs to revert. */
int unionfs_refresh_hidden_dentry(struct dentry *dentry, int bindex) {
    struct dentry *hidden_dentry;
    struct dentry *hidden_parent;
    int err = 0;

    print_entry(" bindex = %d", bindex);

    hidden_parent = dtohd_index(dentry->d_parent, bindex);

    PASSERT(hidden_parent);
    PASSERT(hidden_parent->d_inode);
    ASSERT(S_ISDIR(hidden_parent->d_inode->i_mode));

    hidden_dentry = lookup_one_len(dentry->d_name.name, hidden_parent, dentry->d_name.len);
    if (IS_ERR(hidden_dentry)) {
	err = PTR_ERR(hidden_dentry);
	goto out;
    }

    if (dtohd_index(dentry, bindex)) {
    	dput(dtohd_index(dentry, bindex));
    }
    if(itohi_index(dentry->d_inode, bindex)) {
    	iput(itohi_index(dentry->d_inode, bindex));
	itohi_index(dentry->d_inode, bindex) =  NULL;
    }
    if (!hidden_dentry->d_inode) {
	dput(hidden_dentry);
    	set_dtohd_index(dentry, bindex, NULL);
    } else {
    	set_dtohd_index(dentry, bindex, hidden_dentry);
    	itohi_index(dentry->d_inode, bindex) = igrab(hidden_dentry->d_inode);
    }

out:
    print_exit_status(err);
    return err;
}

/*
 * vim:shiftwidth=4
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
