/* XML lowlevel I/O.

   Copyright (C) 2000 Daiki Ueno <ueno@unixuser.org>

   Author: Daiki Ueno <ueno@unixuser.org>
   Created: 2000-05-16

   This file is part of UltraPoint.

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2, or (at your option)  
   any later version.                                                   

   This program is distributed in the hope that it will be useful,      
   but WITHOUT ANY WARRANTY; without even the implied warranty of       
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the        
   GNU General Public License for more details.                         

   You should have received a copy of the GNU General Public License    
   along with GNU Emacs; see the file COPYING.  If not, write to the    
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,         
   Boston, MA 02111-1307, USA.                                          

*/

#include "xml.h"

static gchar *upt_ns_url = NULL;
static xmlNsPtr upt_ns = NULL;

typedef UptXMLTerm *(*UptXMLTermFunc)(xmlDocPtr doc, xmlNodePtr node);

static GHashTable *term_hash_table = NULL;

static UptXMLSlide *parse_slide (xmlDocPtr doc, xmlNodePtr node);
static UptXMLSlideShow *parse_ss_info (xmlDocPtr doc, xmlNodePtr node);
static gchar *parse_ss_info_1 (xmlDocPtr doc, xmlNodePtr node,
			       const gchar *name);
static gint list_type_of_node (xmlNodePtr node, gint last_type);
static UptXMLTerm *term_from_xml_node (xmlDocPtr doc, xmlNodePtr node,
				       gint list_type);
static GNode *gnode_from_xml_node (xmlDocPtr doc, xmlNodePtr node,
				   gint list_type);
static UptXMLTerm *term_parse_para (xmlDocPtr doc, xmlNodePtr node);
static UptXMLTerm *term_parse_imagedata (xmlDocPtr doc, xmlNodePtr node);
static gint validate_doc (xmlDocPtr doc);

static gchar *
parse_ss_info_1 (doc, node, name)
     xmlDocPtr doc;
     xmlNodePtr node;
     const gchar *name;
{
  gchar *value = NULL;

  for (node = node->childs; !value && node; node = node->next)
    {
      if (!node->childs)
	continue;
      if (!g_strcasecmp (name, node->name))
	value = xmlNodeListGetString (doc, node->childs, 1);
      else
	value = parse_ss_info_1 (doc, node, name);
    }

  return value;
}     

static UptXMLSlideShow *
parse_ss_info (doc, node)
     xmlDocPtr doc;
     xmlNodePtr node;
{
  UptXMLSlideShow *slideshow;

  slideshow = g_new0 (UptXMLSlideShow, 1);
  g_assert (slideshow != NULL);
  slideshow->free = g_free;

  slideshow->title =
    parse_ss_info_1 (doc, node, "title");
  slideshow->author =
    parse_ss_info_1 (doc, node, "author");
  slideshow->affiliation =
    parse_ss_info_1 (doc, node, "affiliation");
  slideshow->email =
    parse_ss_info_1 (doc, node, "email");

  return slideshow;
}

static void
slide_free (slide)
     UptXMLSlide *slide;
{
  if (slide->body)
    g_free (slide->body);
  g_free (slide);
}

static gint
list_type_of_node (node, last_type)
     xmlNodePtr node;
     gint last_type;
{
  gchar *prop;

  if (!g_strncasecmp ("listitem", node->name, 8))
    return last_type;
  if (!g_strncasecmp ("itemizedlist", node->name, 35))
    return LIST_ITEMIZE;

  if (g_strncasecmp ("orderedlist", node->name, 34))
    return LIST_DEFAULT;

  prop = xmlGetProp (node, "numeration");
  if (!prop)
    return LIST_ORDER_DEFAULT;
  if (!g_strncasecmp ("arabic", prop, 6))
    return LIST_ORDER_ARABIC;
  if (!g_strncasecmp ("upperalpha", prop, 10))
    return LIST_ORDER_UPPERALPHA;
  if (!g_strncasecmp ("loweralpha", prop, 10))
    return LIST_ORDER_LOWERALPHA;

  return LIST_DEFAULT;
}

static UptXMLTerm *
term_from_xml_node (doc, node, list_type)
     xmlDocPtr doc;
     xmlNodePtr node;
     gint list_type;
{
  UptXMLTermFunc term_func;
  UptXMLTerm *term;
  const gchar *name = node->name;

  term_func = g_hash_table_lookup (term_hash_table, name);
  if (!term_func)
    return NULL;

  term = (*term_func) (doc, node);
  if (!term)
    return NULL;

  term->list_type = list_type;
  return term;
}

static UptXMLTerm *
term_parse_para (doc, node)
     xmlDocPtr doc;
     xmlNodePtr node;
{
  UptXMLTerm *term;
  gchar *string;

  string = xmlNodeListGetString (doc, node->childs, 1);
  if (!string)
    return NULL;

  term = g_new0 (UptXMLTerm, 1);
  term->data.text = string;

  return term;
}

static UptXMLTerm *
term_parse_imagedata (doc, node)
     xmlDocPtr doc;
     xmlNodePtr node;
{
  UptXMLTerm *term;
  gchar *prop, *filename;

  filename = xmlGetProp (node, "fileref");
  if (!filename)
    return NULL;

  term = g_new0 (UptXMLTerm, 1);
  term->type = TERM_IMAGE;
  term->data.image.filename = filename;
  term->data.image.width = -1;
  term->data.image.height = -1;
  term->data.image.scale = -1;

  prop = xmlGetProp (node, "width");
  if (prop)
    term->data.image.width = MAX (atoi (prop), -1);
  prop = xmlGetProp (node, "height");
  if (prop)
    term->data.image.height = MAX (atoi (prop), -1);
  prop = xmlGetProp (node, "scale");
  if (prop)
    term->data.image.scale = MAX (g_strtod (prop, NULL), -1);

  return term;
}

static GNode *
gnode_from_xml_node (doc, node, list_type)
     xmlDocPtr doc;
     xmlNodePtr node;
     gint list_type;
{
  UptXMLTerm *term;
  GNode *root, *child = NULL;

  root = g_node_new (NULL);
  for (node = node->childs; node; node = node->next, child = NULL)
    {
      term = term_from_xml_node (doc, node, list_type);
      if (term)
	child = g_node_new (term);
      else if (node->childs)
	child =
	  gnode_from_xml_node (doc, node,
			       list_type_of_node (node, list_type));
      if (child)
	g_node_append (root, child);
    }
  return root;
}

static gboolean
gnode_print_traverse_func (node, data)
     GNode *node;
     gpointer *data;
{
  gint i, depth;
  UptXMLTerm *term = node->data;

  if (term->type != TERM_TEXT)
    return FALSE;

  depth = g_node_depth (node);
  for (i = 0; i<depth; i++)
    putchar (' ');

  printf ("%s\n", term->data.text);
  return FALSE;
}

static UptXMLSlide *
parse_slide (doc, node)
     xmlDocPtr doc;
     xmlNodePtr node;
{
  UptXMLSlide *slide;

  slide = g_new0 (UptXMLSlide, 1);
  g_assert (slide != NULL);

  slide->body = NULL;
  slide->free = slide_free;
  for (node = node->childs; node; node = node->next)
    {
      if (!node->childs && node->ns != upt_ns)
	continue;
      if (!g_strncasecmp (node->name, "head", 4))
	slide->head =
	  xmlNodeListGetString (doc, node->childs, 1);
      else if (!g_strncasecmp (node->name, "body", 4))
	slide->body = gnode_from_xml_node (doc, node, LIST_DEFAULT);
    }
  return slide;
}

static gint
validate_doc (doc)
     xmlDocPtr doc;
{
  xmlDtdPtr subset;
  xmlValidCtxt vctxt;
  gint ret;

  vctxt.userData = stderr;
  vctxt.error = (xmlValidityErrorFunc) fprintf;
  vctxt.warning = (xmlValidityWarningFunc) fprintf;

  subset = xmlParseDTD (NULL, upt_ns_url);
  if (!subset)
    return -1;

  ret = xmlValidateDtd (&vctxt, doc, subset);
  xmlFreeDtd (subset);

  return ret;
}

UptXMLSlideShow *
upt_parse_file (filename, dtd)
     gchar *filename;
     gchar *dtd;
{
  UptXMLSlideShow *slideshow = NULL;
  xmlDocPtr doc;
  xmlNodePtr node;

  doc = xmlParseFile (filename);
  if (!doc)
    return NULL;

  node = doc->root;
  if (!node)
    goto clean;
  
  if (dtd)
    upt_ns_url = dtd;
  upt_ns = xmlSearchNsByHref (doc, node, upt_ns_url);

  if (dtd && validate_doc (doc) < 0)
    goto clean;

  if (g_strncasecmp (node->name, "slideshow", 9))
    goto clean;

  node = doc->root->childs;
  while (!slideshow && node)
    {
      if (!node->childs && node->ns != upt_ns)
	continue;
      if (!g_strncasecmp (node->name, "slideshowinfo", 13))
	slideshow = parse_ss_info (doc, node);
      node = node->next;
    }

  term_hash_table = g_hash_table_new (g_str_hash, g_str_equal);
  g_hash_table_insert (term_hash_table, "para", term_parse_para);
#ifdef HAVE_LIBMAGICK
  g_hash_table_insert (term_hash_table, "imagedata", term_parse_imagedata);
#endif

  while (node)
    {
      if (!node->childs && node->ns != upt_ns)
	continue;
      if (!g_strncasecmp (node->name, "slide", 5))
	{
	  UptXMLSlide *slide;

	  slide = parse_slide (doc, node);
	  slideshow->slides =
	    g_list_append (slideshow->slides, slide);
	}
      node = node->next;
    }
  xmlFreeDoc (doc);
  return slideshow;

 clean:
  xmlFreeDoc (doc);
  return NULL;
}
