/* ==================================================== ======== ======= *
 *
 *  umserver.cpp : Ubit Multi Mouse/Message Server 
 *  (multiple event flows and pointers, device drivers, bi-manual interaction, etc.)
 *
 *  Ubit Project [Elc::2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)umserver.cpp	ubit:03.06.04"
#include <algorithm>
#include <unistd.h>       // darwin
#include <stdio.h>
#include <ctype.h>
#include <sys/stat.h>
#include <X11/Xlib.h>
#include <X11/Xmu/WinUtil.h>
#include <netinet/in.h>
#include <sys/socket.h>
//#include <arpa/inet.h>
#include <ubit/umsproto.hpp>
#include <ubit/umsclient.hpp>
#include "umserver.hpp"
#include "umsflow.hpp"
#include "umssource.hpp"
using namespace std;

int UMServer::xerror;

/* ==================================================== ======== ======= */
// utility classes.

UMSbuttonMapping::UMSbuttonMapping(u_int _source_button, 
				   u_int _source_modifiers,
				   u_int _flow_button,
				   u_int _flow_modifiers) {
  // input from the device
  source_button      = _source_button;
  source_button_mask = Button1Mask << (_source_button-1);
  source_modifiers   = _source_modifiers;     // button_mask not included

  // output to the event flow
  flow_button        = _flow_button;
  flow_button_mask   = Button1Mask << (_flow_button-1);
  flow_modifiers     = _flow_modifiers;       // button_mask not included
}

/* ==================================================== ======== ======= */

UMScalibration::UMScalibration() {
  curpoint = 0;
  x = y = 0;
  width = height = 0;
  message = null;
}

/* ==================================================== ======== ======= */

UMSpos::UMSpos() {
  rx = ry = 0; 
  wx = wy = 0; 
  win = None;
  ptr_in_uwin = false;
  winsock = -1;
}

UMSpos::UMSpos(int _rx, int _ry, int _wx, int _wy, Window _win,
               bool _ptr_in_uwin, int _winsock) {
  rx = _rx, ry = _ry; 
  wx = _wx, wy = _wy; 
  win = _win;
  ptr_in_uwin = _ptr_in_uwin;
  winsock = _winsock;
}

/* ==================================================== ======== ======= */

struct UMSactionKey {
  u_int keycode, modifier;
  void (*press_action)(class UMServer&, XEvent&); 
  void (*release_action)(class UMServer&, XEvent&); 

  UMSactionKey(u_int keycode, u_int modifier,
	       void (*press_action)(class UMServer&, XEvent&),
	       void (*release_action)(class UMServer&, XEvent&));
};


UMSactionKey::UMSactionKey(u_int _keycode, u_int _modifier,
			   void (*_press_action)(class UMServer&, XEvent&),
			   void (*_release_action)(class UMServer&, XEvent&)) {
  keycode  = _keycode;
  modifier = _modifier;
  press_action = _press_action;
  release_action = _release_action;
}

/* ==================================================== ======== ======= */

struct UMScnx {
  int sock;
  char* app_name;

  UMScnx(int sock, const char* app_name);
  ~UMScnx();
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UMServer::UMServer(int ums_port, const char* display_name) :
  sock_server(* new sockaddr_in())
{
  is_init = false;

  // inits the UMS server
  if (ums_port == 0) ums_port = UMSprotocol::UMS_DEFAULT_PORT;
  ums_socket = openServerSocket(ums_port);
  if (ums_socket == -1) {
    cerr << "UMServer: can't open UMS socket on port: " << ums_port << endl;
    return;
  } 

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  // open and init the X server

  if (!initX(display_name)) {
    cerr << "UMServer: can't open X server on display: " << display_name << endl;
    return;
  }
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  // create the native mouse flow 

  // eflows[0] corresponds to the standard X server (its id must be 0)
  eflows.push_back(new UMSmouseFlow(this, /*flow_id*/0, /*no pointer*/false));

  cout << " - UMS port: " << ntohs(sock_server.sin_port) << endl
       << " - X display: " << (display_name ? display_name : "localhost:0") << endl;

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

  for (int l = 0; l < 3; l++)
    for (int c = 0; c < 3; c++)
      neighbors[l][c] = null;

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

  is_init = true;
}

UMServer::~UMServer() {
  u_int k;
  for (k = 0; k < sources.size(); k++) delete sources[k];
  for (k = 0; k < eflows.size(); k++) delete eflows[k];
  for (k = 0; k < cnxs.size(); k++) delete cnxs[k];
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UMSneighbor::UMSneighbor(const char* _host, int _port) {
  host = strdup(_host);
  port = _port;
  client = new UMSclient("UMS");
  cnx_time = 0;
}

bool UMServer::addNeighbor(int l, int c, const char* host, int port) {
  if (!host || !*host) {
    //cerr << "UMServer::addNeighbor: null host name" << endl;
    return false;
  }
  if (l < 0 || l > 2 || c < 0 || c > 2 || (l == 1 && c == 1)) {
    //cerr << "UMServer::addNeighbor: wrong values for l and c" << endl;
    return false;
  }

  if (port == 0) port = UMSprotocol::UMS_DEFAULT_PORT;
  neighbors[l][c] = new UMSneighbor(host, port);
  
  return true;
}

bool UMServer::addNeighbor(const char* pos, const char* host, int port) {
  if (!pos || !host) return false; 
  switch(pos[0]){
  case 'n':
    if (pos[1] == 'w') return addNeighbor(0,0, host);
    else if (pos[1] == 'e') return addNeighbor(0,2, host);
    else return addNeighbor(0,1, host);
    break;
  case 'w':
    return addNeighbor(1,0, host);
    break;
  case 'e':
    return addNeighbor(1,2, host);
    break;
  case 's':
    if (pos[1] == 'w') return addNeighbor(2,0, host);
    else if (pos[1] == 'e') return addNeighbor(2,2, host);
    else return addNeighbor(2,1, host);
    break;
  }
  return false;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

bool UMServer::addEventFlow(UMSeventFlow* fl) {
  if (fl->getID() == 0) {
    cerr << "UMServer::addEventFlow: invalid ID (ID = 0 is reserved for the native X event flow)" << endl;
    return false;
  }

  if (getEventFlow(fl->getID())) {
    cerr << "UMServer::addEventFlow: invalid ID (an event flow with the same ID is already registered)" << endl;
    return false;
  }

  eflows.push_back(fl);
  return true;
}

/* ==================================================== ======== ======= */

bool UMServer::removeEventFlow(UMSeventFlow* fl) {
  if (fl->getID() == 0) {
    cerr << "UMServer::removeEventFlow: invalid ID (ID = 0 is reserved for the native X event flow and can't be removed)" << endl;
    return false;
  }

  unsigned int sz = eflows.size();
  eflows.erase( remove(eflows.begin(), eflows.end(), fl), eflows.end());

  if (sz == eflows.size()) {
    cerr << "UMServer::removeEventFlow: flow not found" << endl;
    return false;
  }
  else return true;
}

/* ==================================================== ======== ======= */

UMSeventFlow* UMServer::getEventFlow(int id) const {
  for (unsigned int k = 0; k < eflows.size(); k++) {
    if (eflows[k]->getID() == id) return eflows[k];
  }
  return null;  // not found
}

UMSmouseFlow* UMServer::getNativeMouseFlow() const {
  return dynamic_cast<UMSmouseFlow*>(eflows[0]);
}

UMSmouseFlow* UMServer::getOrCreateMouseFlow(int id) {
  UMSeventFlow* eflow = getEventFlow(id);
  if (eflow) {
    return dynamic_cast<UMSmouseFlow*>(eflow);
  }
  else {
    UMSmouseFlow* mflow = new UMSmouseFlow(this, id, true);
    if (addEventFlow(mflow)) return mflow;
    else {
      delete mflow;
      return null;
    }
  }
}

/* ==================================================== ======== ======= */

bool UMServer::addEventSource(UMSeventSource* so) {
  for (unsigned int k = 0; k < sources.size(); k++) {
    if (sources[k] == so) {
      cerr << "UMServer::addSource: this source is already registered" << endl;
      return false;
    }
  }
  sources.push_back(so);
  return true;
}

bool UMServer::removeEventSource(UMSeventSource* so) {
  unsigned int sz = sources.size();
  sources.erase( remove(sources.begin(), sources.end(), so), sources.end());

  if (sz == sources.size()) {
    cerr << "UMServer::removeSource: source not found" << endl;
    return false;
  }
  else return true;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// returns UMSrequest status
// !ATT: CloseCnx ferme la connection!

int UMServer::processRequest(int sock_com, UMSrequest& req) {
  // req.data[0] is the octal size (must be * by 8)
  // req.data[1] is the request type
  req.count = 2;     // skip request size and type 

  switch (req.data[1]) {    

  case UMSrequest::MOUSE_CTRL : {
    if (processMouseRequest(req))
      return UMSrequest::Ok;
    else return UMSrequest::Error;
  }

  case UMSrequest::SEND_EVENT : {
    if (processEventRequest(req))
      return UMSrequest::Ok;
    else return UMSrequest::Error;
  }

  //  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

  case UMSrequest::SEND_MESSAGE : {
    if (processMessageRequest(req))
      return UMSrequest::Ok;
    else return UMSrequest::Error;
  }
    /*
  case UMSrequest::BROADCAST_MESSAGE : {
    cerr << "BROADCAST_MESSAGE not yet implemented" << endl;
    return UMSrequest::Error;
  }
    */

  //  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

  case UMSrequest::OPEN_WIN: {
    //uint32_t win = req.readLong();
    long win = req.readLong();
    if (win == None) return UMSrequest::Ok;
    
    bool added = addWin(cnx_wins, win, false, sock_com);
    if (added) {
      //cerr<< "OPEN_WIN " << win << endl;
      return UMSrequest::Ok;
    }
    else {
      cerr << "OPEN_WIN "<< win << " already registered" << endl;
      return UMSrequest::Ok;
    }
  }

  //  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

  case UMSrequest::CLOSE_WIN : {
    //uint32_t win = req.readLong();
    long win = req.readLong();
    if (win == None) return UMSrequest::Ok;

    //cerr << "CLOSE_WIN " << win << endl;
    removeWin(cnx_wins, win);
    return UMSrequest::Ok;
  }

  //  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

  case UMSrequest::OPEN_CNX : {
    bool added = addCnx(sock_com, req.getString());
    if (added) {
      //cerr<< "OPEN_CNX: "<<  cdata.args << " socket: "<< sock_com << endl;
      return UMSrequest::KeepCnx;
    }
    else {
      cerr << "OPEN_CNX: appli already registered" << endl;
      return UMSrequest::Ok;
    }
  }

  //  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

  case UMSrequest::CLOSE_CNX : {
    //cerr << "CLOSE_CNX " << sock_com << endl;
    removeCnx(sock_com);
    return UMSrequest::CloseCnx;
  }
  } //endswitch()
    
  return UMSrequest::Ok;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/* protocol:
 * --------------------------------
* target can be:
* winid   with winid = ascii-name-without-blanks or quoted 'asci name'
or decimal (1234) or hexa number (0x1234a)
*
* #win:/[display]/winid[:evflow]/    (display is opt and ... never used)
* #ptr:/[display]/ptrid[:evflow]/    with ptrid = 0, 1 ... (useless)
*
* pattern:
 * --------------------------------
 * xxx:/display/'name'[id1][:id2]/
 */

struct RequestAction {
  enum {NONE, PTR, WIN, APP} target_type;
  UMSpos pos;
  UMSeventFlow* output_flow;
  char* message;
};

/* ==================================================== ======== ======= */

bool UMServer::parseRequest(UMSrequest& req, RequestAction& action) {
  char* args = req.getString();

  if (!args || !*args) return false;
  bool sharp;
  
  if (args[0] == '#') {
    args++;
    if (!*args) return false;
    sharp = true;
  }
  else sharp = false;

  char *sl1 = null, *sl2 = null;
  
  if (!sharp) sl2 = args-1;
  else {
     sl1 = strchr(args+1, '/');
    if (!sl1 || sl1[-1] != ':') return false;

    sl2 = strchr(sl1+1, '/');
    if (!sl2) return false;
  }
  
  char *q1 = null, *q2 = null;
  if (sl2[1] == '\'') {
    q1 = sl2+1;
    q2 = strchr(q1+1, '\'');
    if (!q2) return false;
  }
  else if (sl2[1] == '"') {
    q1 = sl2+1;
    q2 = strchr(q1+1, '"');
    if (!q2) return false;
  }

  //char* sl3 = q2 ? strchr(q2+1, '/') : strchr(sl2+1, '/');
  //if (!sl3) return false;
  char* sl3 = args + strlen(args);

  char display[200] = "";
  char name[200] = "";
  long id1 = 0, id2 = 0;
  bool id1_def = false, id2_def = false;

  if (sharp) {
    strncpy(display, sl1+1, sl2-sl1-1);
    display[sl2-sl1-1] = 0;
  }
  
  if (q1) {
    strncpy(name, q1+1, q2-q1-1);
    name[q2-q1-1] = 0;

    char* p = strchr(q2+1, ':');
    if (p) {id2 = strtol(p+1, null, 0); id2_def = true;}
  }

  else {
    char* p = strchr(sl2+1, ':');
    if (!p) {
      q1 = sl2; q2 = sl3;
    }
    else {
      q1 = sl2; q2 = p;
      id2 = strtol(p+1, null, 0); id2_def = true;
    }

    if (isdigit(q1[1])) {
      id1 = strtol(q1+1, null, 0); id1_def = true;
    }
    else {
      strncpy(name, q1+1, q2-q1-1);
      name[q2-q1-1] = 0;
    }
  }

  action.target_type = RequestAction::NONE;

  // skip the final 0 of the first string
  //action.message = sl3+2;
  action.message = sl3+1;

  //  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  if (!sharp || (sharp && args[0] == 'w')) {
    Window win = None;

    if (id1_def) win = (Window)id1;
    else if (name[0] != 0) {
      const UMSwin* ww = searchWinName(root_children, name);

      if (!ww || ww->client_win == None) return false;
      // else win = XmuClientWindow(xdisplay, ww->win);
      else win = ww->client_win;
      //cerr << "-- found win: "; showWindowName(win); cerr << endl;
    }
    else return false;

    if (!id2_def) action.output_flow = getNativeMouseFlow();  // default X flow
    else {
      action.output_flow = getEventFlow(id2);
      if (!action.output_flow) return false;
    }

    if (!locateWin(win, action.pos)) return false;
    action.target_type = RequestAction::WIN;
    return true;
  }

  else if (sharp && args[0] == 'p') {
    if (!id1_def) return false;

    // flow must exist and be an mouse flow
    UMSeventFlow* ptr_flow = getEventFlow(id1);
    UMSmouseFlow* mptr_flow = 
      (ptr_flow ? dynamic_cast<UMSmouseFlow*>(ptr_flow) : null);
    if (!mptr_flow) return false;

    if (!id2_def) action.output_flow = mptr_flow;  // same value as id1
    else {
      action.output_flow = getEventFlow(id2);
      if (!action.output_flow) return false;
    }

    bool stat = locatePointer(*mptr_flow, action.pos);
    if (!stat) return false;
    action.target_type = RequestAction::PTR; 
    return true;
  }

  else {
    //cerr << " invalid target type " << endl;
    return false;
  }
}

/* ==================================================== ======== ======= */
// protocol
// SEND_MESSAGE | (s)target_id  (s)any_string

bool UMServer::processMessageRequest(UMSrequest& req) {
  RequestAction action;
  if (!parseRequest(req, action)) return false;
  if (!action.message) return false;

  sendMessage(*action.output_flow, action.pos, action.message);
  return true;
}

/* ==================================================== ======== ======= */
// protocol:
// SEND_EVENT | (s)target_id  (i)ev_type (i)x (i)y l(btn)

bool UMServer::processEventRequest(UMSrequest& req) {
  u_char event_type;
  u_char event_flow; 
  long x, y;
  unsigned long detail;
    
  req.readEvent(event_type, event_flow, x, y, detail);

  RequestAction action;
  if (!parseRequest(req, action)) return false;
  if (!action.message) return false;

  UMSmouseFlow* mf = dynamic_cast<UMSmouseFlow*>(action.output_flow);
  if (!mf) return false;

  UMSpos pos;
  if (!locatePoint(action.pos.win, 
		   action.pos.rx + action.pos.wx + x,
		   action.pos.ry + action.pos.wy + y, 
		   pos))
    return false;
  pos.winsock = action.pos.winsock;

  //fprintf(stderr, "rx %d ry %d / wx %d wy %d \n", 
  //	  action.pos.rx, action.pos.ry, action.pos.wx, action.pos.wy);

  switch (event_type) {
  case ButtonPress:
    sendButtonEvent(*mf, pos, detail/*mouse_btn_mask*/, true);
    return true;

  case ButtonRelease:
    sendButtonEvent(*mf, pos, detail/*mouse_btn_mask*/, false);
    return true;

  default:
    //cerr << "invalid event type: " << event_type << endl;
    return false;
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// protocol:
// MOUSE_CONTROL | (i)mouse_flow (i)ev_type (i)x (i)y (l)aux={btn|abs_coord}

bool UMServer::processMouseRequest(UMSrequest& req) {
  u_char event_type;
  u_char event_flow; 
  long x,y;
  unsigned long detail;

  req.readEvent(event_type, event_flow, x, y, detail);

  UMSeventFlow* ef = getEventFlow(event_flow);
  UMSmouseFlow* mf = (ef ? dynamic_cast<UMSmouseFlow*>(ef) : null);
  if (!mf) return false;

  switch (event_type) {
  case MotionNotify:
    mf->moveMouse(x, y, detail);  // detail = abs_coord
    return true;
  case ButtonPress:
    mf->pressMouse(detail,0);       // detail = mouse_btn_mask
    return true;
  case ButtonRelease:
    mf->releaseMouse(detail,0);     // detail = mouse_btn_mask
    return true;
  default:
    return false;
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

#define MAX_FD(maxfd,fd) {if ((fd) > (maxfd)) (maxfd) = (fd);}

void UMServer::start() {
  if (!isInit()) {
    cerr << "UMServer::start: UMS not initialized; can't start" << endl;
    return;
  }

  while (1) {

    while (XPending(xdisplay)) {
      XEvent e;
      XNextEvent(xdisplay, &e);    // gets the next X event
      processEvents(e);
    }
      
    fd_set read_set;
    int maxfd = 0;
    FD_ZERO(&read_set);

    // test X events
    FD_SET(xconnection, &read_set);
    MAX_FD(maxfd, xconnection);
    
    // test input sources from drivers
    for (unsigned int k = 0; k < sources.size(); k++) {
      FD_SET(sources[k]->filedesc(), &read_set);
      MAX_FD(maxfd, sources[k]->filedesc());
    }

    // test opened connections on client sockets
    for (unsigned int k = 0; k < cnxs.size(); k++) {
      FD_SET(cnxs[k]->sock, &read_set);
      MAX_FD(maxfd, cnxs[k]->sock);
    }

    // test requests on the server socket
    FD_SET(ums_socket, &read_set);
    MAX_FD(maxfd, ums_socket);

    int has_input = select(maxfd+1,
			   &read_set, /*read*/
			   NULL,      /*write*/
			   NULL,      /*except*/
			   NULL);     /*timeout*/

    if (has_input < 0) {
      cerr << "UMServer::mainLoop: error on select" << endl;
      struct stat statbuf;

      // test open connections which have become invalid
      for (unsigned int k = 0; k < cnxs.size(); k++) {
	if (fstat(cnxs[k]->sock, &statbuf) < 0) {
	  removeCnx(cnxs[k]->sock);
	}
      }
      continue;
    }

    // process X events
    /*
    if (FD_ISSET(xconnection, &read_set)) {
      readXEvents();
    }
    */

    // process input sources
    for (unsigned int k = 0; k < sources.size(); k++) {
      if (FD_ISSET(sources[k]->filedesc(), &read_set))
	sources[k]->read();
    }

    // process opened connections on client sockets
    for (unsigned int k = 0; k < cnxs.size(); k++) {
      if (FD_ISSET(cnxs[k]->sock, &read_set))
	readComSocket(cnxs[k]->sock, false);  // not called by serv
    }

    // process requests on server socket
    if (FD_ISSET(ums_socket, &read_set))
      readServerSocket();
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UMScnx::UMScnx(int _sock, const char* _app_name) {
  sock = _sock; 
  app_name = strdup(_app_name);
}

UMScnx::~UMScnx() {
  free(app_name);
  if (sock > 0) {
    //cerr << "closing socket: " << sock << endl;
    ::shutdown(sock, 2);
    ::close(sock);
  }
}

/* ==================================================== ======== ======= */

bool UMServer::addCnx(int sock, const char* app_name) {
  for (unsigned int k = 0; k < cnxs.size(); k++) {
    if (cnxs[k]->sock == sock) return false;
  }
  UMScnx* cc = new UMScnx(sock, app_name);
  cnxs.push_back(cc);
  return true;
}

void UMServer::removeCnx(int sock) {
  for (unsigned int k = 0; k < cnxs.size(); k++) 
    if (cnxs[k]->sock == sock) {
      delete cnxs[k];
      cnxs.erase(cnxs.begin() + k);
      k--;
    }

  for (UMSwinList::iterator k = cnx_wins.begin(); k != cnx_wins.end(); ) {
    if ((*k)->winsock != sock) k++;
    else {
      UMSwinList::iterator k2 = k; k2++;
      delete *k;
      cnx_wins.erase(k);
      k = k2;
    }
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

bool UMServer::initX(const char* display_name) {

  xdisplay = XOpenDisplay(display_name);
  if (!xdisplay) return false;
  
  xconnection   = XConnectionNumber(xdisplay);
  xscreen       = DefaultScreenOfDisplay(xdisplay);
  xrootwin      = RootWindowOfScreen(xscreen);
  screen_width  = WidthOfScreen(xscreen);
  screen_height = HeightOfScreen(xscreen);
  screen_depth  = DefaultDepthOfScreen(xscreen);
  WM_DELETE_WINDOW = XInternAtom(xdisplay, "WM_DELETE_WINDOW", False);
  UBIT_MESSAGE  = XInternAtom(xdisplay, "_UBIT_MESSAGE", False);
  UBIT_WINDOW   = XInternAtom(xdisplay, "_UBIT_WINDOW", False);

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  // we don't want stupid X errors to crash the UMS server
  XSetErrorHandler(quietXErrorHandler);

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  // calibration window

  calibrating = false;

  XSetWindowAttributes wattr;
  unsigned long wattr_mask = 0;

  wattr.background_pixel = WhitePixelOfScreen(xscreen);
  wattr_mask |= CWBackPixel;

  calwin = XCreateWindow(xdisplay, xrootwin,
			 0, 0, 800, 600, 0,
			 screen_depth,
			 InputOutput,  // Class
			 CopyFromParent,
			 wattr_mask, &wattr);

  XSelectInput(xdisplay, calwin, ExposureMask | StructureNotifyMask
               | ButtonPressMask | ButtonReleaseMask
               );

  XStoreName(xdisplay, calwin, "Calibration Window");
  //cant destroy this window (nor the X connection)
  XSetWMProtocols(xdisplay, calwin, &WM_DELETE_WINDOW, 1);

  XGCValues gcval;
  gcval.function   = GXcopy;
  gcval.foreground = BlackPixelOfScreen(xscreen);
  gcval.background = WhitePixelOfScreen(xscreen);

  calgc = XCreateGC(xdisplay, calwin,
		    GCFunction | GCForeground | GCBackground, &gcval);

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  // this will make it possible to detect window creations & deletions,
  // and window (un)mapping, moving and stacking operations
  
  XSelectInput(xdisplay, xrootwin, 
	       SubstructureNotifyMask | ResizeRedirectMask);

  // inits the root children list
  retrieveWindows();

  return true;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

int UMServer::quietXErrorHandler(Display*, XErrorEvent *xer) {
  xerror = xer->error_code;
  return 0;
}

void UMServer::setActionKey(u_int keycode, u_int modifier, 
			    void (*press_action)(class UMServer&, XEvent&),
			    void (*release_action)(class UMServer&, XEvent&)) {
  if (keycode != 0) {
    UMSactionKey akey(keycode, modifier, press_action, release_action);

    action_keys.push_back(akey);

    XGrabKey(xdisplay, keycode, modifier, xrootwin, True, 
	     GrabModeAsync, GrabModeAsync);
  }
}

/* ==================================================== ======== ======= */

void UMServer::setNativeButtonMapping(u_int source_button, 
				      u_int source_modifiers,
				      u_int flow_button,
				      u_int flow_modifiers) {
  UMSbuttonMapping* bm = new UMSbuttonMapping(source_button,
					      source_modifiers,
					      flow_button,
					      flow_modifiers);
  setNativeButtonMapping(bm);
}

void UMServer::setNativeButtonMapping(UMSbuttonMapping* bm) {
  button_map.push_back(bm);
  XGrabButton(xdisplay, bm->source_button, bm->source_modifiers,
	      xrootwin, True, 
	      ButtonPressMask | ButtonReleaseMask | ButtonMotionMask,
	      GrabModeAsync, GrabModeAsync, None, /*Cursor*/None);
}

UMSbuttonMapping* UMServer::getNativeButtonMapping(u_int source_button,
						   u_int source_modifiers) {
  for (u_int k = 0; k < button_map.size(); k++) {
    if (button_map[k]->source_button == source_button
	// NB: button_mask non inclus dans source_modifiers
	&& ((button_map[k]->source_modifiers & source_modifiers) 
	    == source_modifiers)
	)
      return button_map[k];
  }
  return null;
}

/* ==================================================== ======== ======= */

void UMSeventSource::setButtonMapping(u_int source_button, 
				      u_int source_modifiers,
				      u_int flow_button,
				      u_int flow_modifiers) {

  UMSbuttonMapping* bm = new UMSbuttonMapping(source_button,
					      source_modifiers,
					      flow_button,
					      flow_modifiers);
  setButtonMapping(bm);
}

void UMSeventSource::setButtonMapping(UMSbuttonMapping* bm) {
  button_map.push_back(bm);
}

UMSbuttonMapping* UMSeventSource::getButtonMapping(u_int source_button,
						   u_int source_modifiers) {
  for (u_int k = 0; k < button_map.size(); k++) {
    if (button_map[k]->source_button == source_button
	// NB: button_mask non inclus dans source_modifiers
	&& ((button_map[k]->source_modifiers & source_modifiers) == source_modifiers)
	)
      return button_map[k];
  }
  return null;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UMServer::processEvents(XEvent& e) {

  if (calibrating) {
    switch (e.type) {

      case ConfigureNotify:
      // if the calwin is moved or resize
      if (e.xconfigure.window == calwin) {
        configureCalibrationWin(e.xconfigure.x, e.xconfigure.y,
                                e.xconfigure.width, e.xconfigure.height);
        return;
      }

    case Expose:
      // refreshes the calibration window (use for absolute positioning
      // devices such as the MIMIO)
      if (e.xexpose.window == calwin) {
        drawCalibrationPoint();
        return;
      }

    case ButtonPress:       // TEST
      if (e.xbutton.window == calwin) {
        setCalibrationPoint(e.xbutton.x, e.xbutton.y);
        return;
      }
    }
  }

  switch (e.type) {

    // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
    // changes the events generated by the native mouses
    // (typically to emulation of mouse button 2 and 3 events for machines
    //  that do nothave 3 buttons such as the IPAQ):
    // Notes:
    // - concerns ButtonPress, ButtonRelease and MotionNotify

  case MotionNotify:
    // NB: comme c'est lie a un grabButton, ne peut intervenir
    //     que si pseudo_button 2 ou 3_modifier pressed
    if (e.xmotion.window == xrootwin) {
      //cerr << "motion "  << e.xbutton.state << endl;
      getNativeMouseFlow()->mouseMoved(e.xmotion.x, e.xmotion.y);
    }
    return;

  case ButtonPress:
    if (e.xbutton.window == xrootwin) {
      UMSbuttonMapping* mapping = getNativeButtonMapping(e.xbutton.button,
                                                         e.xbutton.state);
      if (mapping) { 
	//cerr << "buttonpress2 " << e.xbutton.state  << endl;
	getNativeMouseFlow()->pressMouse(mapping);
      }
    }
    return;

  case ButtonRelease:
    if (e.xbutton.window == xrootwin) {
      // NON: le release serait perdu si on lache le modifier entre-temps
      //if (getNativeButtonMapping(e.xbutton.button, e.xbutton.state)){

      UMSbuttonMapping* mapping = getNativeButtonMapping(e.xbutton.button,
                                                         0);
      if (mapping) { 
	//cerr << "buttonrelease2 "  << e.xbutton.state << endl;
	getNativeMouseFlow()->releaseMouse(mapping);
      }
    }
    return;

    // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
    // pre-defined keys that perform some action action (such as opening
    // menus or whatever)

  case KeyPress:
    for (u_int k = 0; k < action_keys.size(); k++) {
      if (action_keys[k].keycode == e.xkey.keycode 
	  && (action_keys[k].modifier == 0 
	      || action_keys[k].modifier & e.xkey.state)
	  && action_keys[k].press_action)
	(action_keys[k].press_action)(*this, e);
    }
    return;

  case KeyRelease: 
    for (u_int k = 0; k < action_keys.size(); k++) {
      if (action_keys[k].keycode == e.xkey.keycode 
	  && (action_keys[k].modifier == 0 
	      || action_keys[k].modifier & e.xkey.state)
	  && action_keys[k].release_action)
	(action_keys[k].release_action)(*this, e);
    }
    return;

    // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
    // window management:
    // - ConfigureNotify : stacking order
    // - MapNotify/UnmapNotify : updates the list of visible windows

  case ConfigureNotify: {  // (among other things) stacking order is changed

    // chercher si window et above sont dans la liste
    list<UMSwin*>::iterator window = root_children.end();
    list<UMSwin*>::iterator above = root_children.end();

    for (list<UMSwin*>::iterator k = root_children.begin(); 
	 k != root_children.end(); 
	 k++) {
      if ((*k)->win == e.xconfigure.window) window = k;
      if ((*k)->win == e.xconfigure.above)  above = k;
    }

    if (window == root_children.end()) return;  // not found

    if (e.xconfigure.above == None) {
      // window est au debut de rien => deplacer en debut de liste
      moveWin(root_children, window, root_children.begin());
      //ex: root_children.erase(window);
      //ex: root_children.push_front(e.xconfigure.window);
    }
    else {
      if (above == root_children.end()) return;  // not found
      // window est au dessus de above => deplacer apres above 
      moveWin(root_children, window, ++above);   // deplacer *apres* above
    }
  } return;

  case MapNotify:       // a window is raised
    // mettre en fin de liste : par dessus les autres windows
    // att: ne pas mettre les pointer wins dans la liste !
    if (!isPointer(e.xmap.window)) 
      addWin(root_children, e.xmap.window, true, -1); //init_wname, sock=-1
    return;

  case UnmapNotify:     // a window is iconified
    removeWin(root_children, e.xunmap.window);
    return;
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UMServer::startCalibration(const char* calwin_title, bool full_screen) {
  double screen_size_percent = 1;
  //if (full_screen) screen_size_percent = 1.;
  //else screen_size_percent = UMScalibration::SCREEN_SIZE_PERCENT;

  int w = int(screen_width * screen_size_percent);
  int h = int(screen_height * screen_size_percent);
  int x = (screen_width - w) / 2;
  int y = (screen_height - w) / 2;

  if (calwin_title) XStoreName(xdisplay, calwin, calwin_title);
  //if (calwin_message) {
  //  free(cal.message); cal.message = strdup(calwin_message);
  //} 

  cal.curpoint = 0;
  cal.x = cal.y = 0;
  cal.width = cal.height = 0;

  configureCalibrationWin(x, y, w, h);
  calibrating = true;

  XMoveResizeWindow(xdisplay, calwin, x, y, w, h);
  XMapRaised(xdisplay, calwin);
  XFlush(xdisplay);

  cerr << "Note: the calibration window can be moved of resized before setting the first point" << endl;
}

void UMServer::configureCalibrationWin(int calwin_x, int calwin_y,
                               int calwin_width, int calwin_height){
  if (cal.curpoint > 0) {
     XMoveResizeWindow(xdisplay, calwin, cal.x, cal.y, cal.width, cal.height);
    return;
  }

  // stores the offset (= the position of the calwin on the screen)
  Window childw;
  XTranslateCoordinates(xdisplay, calwin, RootWindowOfScreen(xscreen), 
 		        0, 0, &cal.x, &cal.y, &childw);
  cal.width  = calwin_width;
  cal.height = calwin_height;
  //cerr << "offset " << cal.x << " " << cal.y << endl;

  cal.xp[0] = UMScalibration::CROSS_XYPOS;
  cal.yp[0] = UMScalibration::CROSS_XYPOS;
  cal.xp[1] = calwin_width - UMScalibration::CROSS_XYPOS;
  cal.yp[1] = UMScalibration::CROSS_XYPOS;
  cal.xp[2] = calwin_width - UMScalibration::CROSS_XYPOS;
  cal.yp[2] = calwin_height - UMScalibration::CROSS_XYPOS;
  cal.xp[3] = UMScalibration::CROSS_XYPOS;
  cal.yp[3] = calwin_height - UMScalibration::CROSS_XYPOS;
}

/* ==================================================== ======== ======= */

void UMServer::drawCalibrationPoint() {
  if (!calibrating || cal.curpoint >= cal.POINT_COUNT) return;
  XClearWindow(xdisplay, calwin);

  // draws a cross
  XDrawLine(xdisplay, calwin, calgc, 
	    cal.xp[cal.curpoint]-cal.CROSS_SIZE/2, cal.yp[cal.curpoint], 
	    cal.xp[cal.curpoint]+cal.CROSS_SIZE/2, cal.yp[cal.curpoint]);

  XDrawLine(xdisplay, calwin, calgc, 
	    cal.xp[cal.curpoint], cal.yp[cal.curpoint]-cal.CROSS_SIZE/2, 
	    cal.xp[cal.curpoint], cal.yp[cal.curpoint]+cal.CROSS_SIZE/2);

  XFlush(xdisplay);
}

/* ==================================================== ======== ======= */

void UMServer::setCalibrationPoint(double x, double y) {
  if (!calibrating || cal.curpoint >= cal.POINT_COUNT) return;

  cal.xcm[cal.curpoint] = x;
  cal.ycm[cal.curpoint] = y;
  cal.curpoint++;

  if (cal.curpoint < cal.POINT_COUNT)
    drawCalibrationPoint();
  else {
    //cerr << "final " << cal.x << " " << cal.y << endl;
    XUnmapWindow(xdisplay, calwin);
  }
}

/* ==================================================== ======== ======= */

bool UMServer::isCalibrationCompleted() const {
  return (calibrating && cal.curpoint >= cal.POINT_COUNT);
}

const UMScalibration& UMServer::getCalibration() const {
  return cal;
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
