/* ==================================================== ======== ======= *
 *
 *  uupdate.cpp
 *  Ubit Project [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:01] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uupdate.cpp	ubit:03.06.04"
#include <iostream>
#include <config.h>
#include <udefs.hpp>
#include <ubrick.hpp>
#include <uerror.hpp>
#include <ucall.hpp>
#include <ustr.hpp>
#include <ubox.hpp>
#include <uboxImpl.hpp>
#include <uwin.hpp>
#include <uview.hpp>
#include <uviewImpl.hpp>
#include <uevent.hpp>
#include <ugraph.hpp>
#include <ucontext.hpp>
#include <update.hpp>
using namespace std;

const UUpdate UUpdate::layout(LAYOUT); 
const UUpdate UUpdate::paint(PAINT); 
const UUpdate UUpdate::all(ALL); 

UUpdate::UUpdate(UUpdate::Mode m) : ix(m) {
  always  = false;
  doublebuf = false;
  delta_x = delta_y = 0;
  elem    = null;
  region  = null;
}

void UUpdate::layoutIfNotShown(bool state) {
  always = state;
}

void UUpdate::paintDoubleBuffered(bool state) {
  doublebuf = state;
}

void UUpdate::paintTitle(const UStr *_elem) {
  ix = TITLE;
  elem = _elem;
}

void UUpdate::paintElem(const UElem *_elem) {
  ix = PAINT;
  elem = _elem;
  delta_x = 0; //nb: this is only useful for strings in uflows
  delta_y = 0;
}

// will only paint an enclosing subpart of this string
//
void UUpdate::paintStr(const UStr *_elem, int strpos1, int strpos2) {
  ix = PAINT;
  elem = _elem;
  delta_x = strpos1; //nb: this is only useful for strings in uflows
  delta_y = strpos2;
}

void UUpdate::paintRegion(const URegion *_region) {
  ix = PAINT;
  region = _region;
}

void UUpdate::setOffset(u_pos d_x, u_pos d_y) {
  delta_x = d_x;
  delta_y = d_y;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UGroup::updateView(UEvent&, UView*, const UUpdate&) {
  // comme group n'a pas de views il se contente de faire un update des parents
  parents.updateParents(UUpdate::paint);
}

void UBox::updateView(UEvent& e, UView *view, const UUpdate &upmode) {
  if (!view) {
    error("updateView", "[internal] null event or null view!");
    return;
  } 
  // !!ATT: verifier que la Window a ete cree
  // il ne s'agit pas d'une erreur ou d'une incoherence mais du simple fait
  // que les champs "visible" ne sont pas modifies recursivement:
  // ainsi un element peut etre "visible" alors qu'un parent ne l'est pas
  // et n'a pas ete realize.

  UWinGraph* g = view->getWinGraph();
  if (!g || !g->isWinRealized()) return;

  // pas visible du tout ==> rien a faire (sauf si show/hide)
  if (!e.sp.redrawStatus
      && upmode.ix != UUpdate::SHOW && upmode.ix != UUpdate::HIDE) {
    view->updateWinData(e.sp.redrawClip);
    return;    // !att: return necessaire pour sauter default_case
  }

  //*************SCROLL****************
  //*************SCROLL****************

  else if (upmode.ix == UUpdate::SCROLL) {
    // Just update data, do not paint
    // !att: c'est necessaire pour remettre entierement a jour les coordonnees
    // des objets scrolles et EN PARTICULIER la zone qui a ete directement
    // recopiee sans regeneration du graphique
    //
    // DE PLUS: ce mode permet de savoir s'il n'ya pas un overlapping
    // avec un ou des objets superposes (voir ci-dessous)

    // !att: ne pas utiliser 2 fois de suite le meme winctx !
    int overlapp = view->updateWinData(e.sp.redrawClip);

    // s'il y a un objet (TRANSP ou non) qui OVERLAPPE (= qui est au dessus
    // le la zone SCROLLEE ==> on ne peut pas recopier directement cette zone
    // (sinon le machin par dessus va aussi etre recopie)
    // ==> le plus simple dans ce cas est de tout regenerer et reafficher
    //    (tant pis pour les perfs, au moins l'affichage sera correct)

#ifdef WITH_GL
    // tout reafficher dans ce mode, le reaff par partie marche mal
    view->updateWinPaint(e.sp.redrawClip, e.sp.opaqueView, 
			 upmode.doublebuf, false);
    return;    // !att: return necessaire pour sauter default_case
#else
    // ex: if (overlapp > 1) {
    if (overlapp > 0) {
      view->updateWinPaint(e.sp.redrawClip, e.sp.opaqueView, 
			   upmode.doublebuf, false);
      return;    // !att: return necessaire pour sauter default_case
    }
    else {
      // ON RECOPIE directement ce qu'on peut en changeant l'offset vertical
      // ou horizontal (selon le cas, un seul ne devant changer qu' ala fois)
      // PUIS on regenere et on affiche la zone manquante
      // ==> avantage : BEAUCOUP plus rapide, presque plus de flicking

      UView* winview = g->getHardwin()->getWinView(view->getDisp()); //???
      if (!winview) return;
     
      URegion clip(e.sp.redrawClip);
      URegion manquant;
      int src_x, src_y, dst_x, dst_y;

      // ATTENTION:
      // - SEULE une des 2 directions doit changer a la fois
      // - ne RIEN faire qunad delta_* == 0, de maniere a ce que la dimension
      //   invariante (par exple: width) ne soit pas mise a 0 stupidement

      if (upmode.delta_x == 0) {
	// rien a faire dans ce cas: sans ce test les micro-deplacements 
	// entraineraient un reaffichage complet de toute la page !
	if (upmode.delta_y == 0) return;

	//clip.width inchange
	src_x = clip.x;
	dst_x = clip.x;
	manquant.x = clip.x;
	manquant.width = clip.width;
      }
      else if (upmode.delta_x > 0) {
	clip.width -= upmode.delta_x;
	src_x = clip.x + upmode.delta_x;
	dst_x = clip.x;
	manquant.x = clip.x + clip.width;
	manquant.width = upmode.delta_x;
      }
      else {
	clip.width += upmode.delta_x; // en fait une soustraction
	src_x = clip.x;
	dst_x = clip.x - upmode.delta_x;
	manquant.x = clip.x;
	manquant.width = -upmode.delta_x;
      }

      if (upmode.delta_y == 0) {
	//clip.height inchange
	src_y = clip.y;
	dst_y = clip.y;
	manquant.y = clip.y;
	manquant.height = clip.height;
      }
      else if (upmode.delta_y > 0) {
	clip.height -= upmode.delta_y;
	src_y = clip.y + upmode.delta_y;
	dst_y = clip.y;
	manquant.y = clip.y + clip.height -1;
	manquant.height = upmode.delta_y;
      }
      else {
	clip.height += upmode.delta_y; // en fait une soustraction
	src_y = clip.y;
	dst_y = clip.y - upmode.delta_y;
	manquant.y = clip.y;
	manquant.height = -upmode.delta_y;
      }
      
      // eviter pbms (en particulier si le delta_* depasse la largeur ou
      // la hauteur de la zone a afficher, ce qui peut arriver
      clip.setInter(winview); 

      if (clip.width <= 0 || clip.height <= 0) {
	// dans ce cas il faut tout reafficher: le slider est alle trop vite
	// et il n'y a aucune partie commune avec ce qui precedait
	// (donc rien a recopier et tout a reafficher)
	goto default_case;
      }

      // copier ce qui n'a pas change
      // note: CopyArea can't copy obscured parts of the component
      // but will generate refresh events form repainting the missing parts
      // as its last arg. is true

      g->begin(*winview); // ne pas oublier!
      g->copyArea(src_x, src_y, clip.width, clip.height,
		 -upmode.delta_x, -upmode.delta_y, true);
      g->end();

      // puis generer et afficher la partie manquante

      // permet de corriger un regrettable bug: le copyArea precedent
      // recopie la dernier ligne qui est malheureusement blanche
      // car ecrasee qq part par une fonction indelicate. en affichant
      // une partie manquante un peu plus grande on resoud donc le pbm

      manquant.x--;
      manquant.width++; 
      manquant.y--;
      manquant.height++;

      view->updateWinPaint(manquant, e.sp.opaqueView, upmode.doublebuf, false);
    }
    return;    // !att: return necessaire pour sauter default_case
#endif
  }


  //************* MOVE ****************
  //************* MOVE ****************

  else if (upmode.ix == UUpdate::MOVE) {
    // faire en sorte que les coords. de CLIP soient TOUJOURS les nouvelles
    // coords en testant si celles de VIEW ont deja ete mises a jour
    // (OBSOLETE_COORDS => view = anciennes valeurs (*avant* le move), 
    // sinon view = nouvelles valeurs (*apres* le move)
    // NOTES:
    // -- voir notes dans UPos::set()  (floating boxes)
    // -- ce mecanisme est necessaire car l'ordre de remise a jour des coords
    // des floating views est indetermine du fait d'eventuelles dependances 
    // entre les views (dans les cas ou le floating est partage)
    // -- la maj. est faite par UView::doUpdate() qui est appelee
    // recursivement en fonction du view tree et des occlusions
    URegion clip(*view);

    /* comprend rien...
    if (view->isDef(UView::OBS_COORDS)) {
      clip.x += upmode.delta_x;
      clip.y += upmode.delta_y;
    }
    */

    // on n'utilise le double buffering que si demande ou si objet TRANSPARENT
    bool doublebuf = 
      upmode.doublebuf || !e.sp.opaqueView || e.sp.opaqueView != view;

    if (doublebuf) {
      if (upmode.delta_x > 0) {
	clip.width += upmode.delta_x;
      }
      else {
	clip.x += upmode.delta_x;  // delta_x < 0
	clip.width -= upmode.delta_x;
      }

      if (upmode.delta_y > 0) {
	clip.height += upmode.delta_y;
      }
      else {
	clip.y += upmode.delta_y;  // delta_y < 0
	clip.height -= upmode.delta_y;
      }
      
      UView *parview = view->getParentView();
      if (parview) {
	// intersection du clip avec parent pour eviter de reafficher
	// (inutilement) ce qui est en dehors du parent
	//// clip.setInter(parview);
	parview->updateWinPaint(clip, e.sp.opaqueView, doublebuf, false);
      }
      else {
	view->updateWinPaint(clip, e.sp.opaqueView, doublebuf, false);
      }
    }

    else {  // NOT doublebuf

      //ce decoupage en 3 parties evite de reafficher SOUS le Floating
      //quand il est OPAQUE. Noter que 'xstrip' inclue aussi le 'coin'
      //(l'intersection entre la bande verticale et la bande horizontale)
      //et que ce coin est donc exclu de 'ystrip'.

      //  zzyy     dx>0          yyzz     dx<0    Note: zz et xx sont
      //  zzyy     dy>0          yyzz     dy>0    regroupes dans 'xstrip'
      //  xx****               ****xx             yy = 'ystrip'
      //  xx****               ****xx
      //    ****               ****
      //    ****               ****
      
      //    ****               ****
      //    ****               ****
      //  xx****               ****xx
      //  xx****               ****xx
      //  zzyy     dx>0          yyzz     dx<0
      //  zzyy     dy<0          yyzz     dy<0
      
      URegion xstrip, ystrip;
      xstrip.y = clip.y - upmode.delta_y;
      xstrip.height = clip.height;
      
      if (upmode.delta_x > 0) {
	xstrip.x = clip.x - upmode.delta_x;
	xstrip.width = upmode.delta_x;
	ystrip.x = clip.x;
	ystrip.width = clip.width - upmode.delta_x;
      }
      else {
	xstrip.x = clip.x + clip.width; 
	xstrip.width  = -upmode.delta_x;   //nb: deltax <= 0 !
	//ystrip.x = clip.x - upmode.delta_x;
	ystrip.x = clip.x + upmode.delta_x;
	//ystrip.width  = clip.width + upmode.delta_x;
	ystrip.width  = clip.width - upmode.delta_x;
      }

      if (upmode.delta_y > 0) {
	ystrip.y = clip.y - upmode.delta_y;
	ystrip.height  = upmode.delta_y;
      }
      else {
	ystrip.y = clip.y + clip.height;
	ystrip.height = -upmode.delta_y;
      }
      
      UView *parview = view->getParentView();
      if (parview) clip.setInter(parview);

      view->updateWinPaint(clip, e.sp.opaqueView, doublebuf, false);
      
      if (parview) {
	if (xstrip.width != 0)
	  parview->updateWinPaint(xstrip, e.sp.opaqueView,
				  upmode.doublebuf,false);
	if (ystrip.height != 0)
	  parview->updateWinPaint(ystrip, e.sp.opaqueView, 
				upmode.doublebuf, false);
      }
    }

    return;    // !att: return necessaire pour sauter default_case
  }  //endif (upmode.ix == UUpdate::MOVE)


  //*************CAS STANDARD****************
  //*************CAS STANDARD****************

 default_case:   //PAINT, LAYOUT...
 // Notes:
 // 1. normalement redrawClip est defini SAUF si show/hide
 //    c'est en particulier necessaire pour le scrolling (sinon l'objet 
 //    serait affiche en entier et deborderait sur ses voisins)
 // 2. le mode PAINT_DAMAGED est est incompatible avec le DoubleBuffering
 //    (lequel entraine donc tjrs un PAINT_ALL dans updateWinPaint())

 if (upmode.region) {
   // ne reafficher que la region demandee 
   view->updateWinPaint(*upmode.region, e.sp.opaqueView, 
			upmode.doublebuf, false);
 }
 else if (e.sp.redrawStatus) {
   // reafficher clipZone (si definie)
   view->updateWinPaint(e.sp.redrawClip, e.sp.opaqueView, 
			upmode.doublebuf, false);
 }
 else {
   // sinon reafficher zone de l'objet en entier
   view->updateWinPaint(*view, e.sp.opaqueView, upmode.doublebuf, false);
 }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// Note: isShown() ne veut pas dire grand chose (sauf pour les Window)
// a cause des scrollpanes et autres clipping: une box peut-etre "SHOWN"
// sans apparaitre reellement a l'ecran (car hors zone de clipping
// du viewport)

void UGroup::show(bool state) {

  // EX: faux if (isShown() == state) return;
  // FAUX: isShown est egalement relatif aux parents!

  if (isShowable() == state) return;

  setCmodes(UMode::CAN_SHOW, state);
  if (state) {
    //fire(&e, UOn::show); // pas recursif! => faire autrement
    UUpdate upd(UUpdate::SHOW);
    update(upd);
  }
  else {
    //fire(&e, UOn::hide);
    UUpdate upd(UUpdate::HIDE);
    update(upd);
  }
}

void UGroup::update(UUpdate upd) {
  parents.updateParents(upd);
}
void UGroup::update() {
  parents.updateParents(UUpdate::all);
}

/* ==================================================== ======== ======= */
//Notes: 
// 1. cette fct est virtuelle et c'est la redefinition UWin::update(...)
//    qui doit etre appelee pour les wins
// 2. peut faire apparaitre ou disparaitre this (show appelle cette fct)

void UBox::update() {
  update(UUpdate::all);
}

void UBox::update(UUpdate upmode) {
  // REMARQUE IMPORTANTE: les hide/show ne sont pas traites de la meme facon
  // si l'objet est floating ou non:
  // - dans le 1er cas, pas besoin de recalculer le layout, un repaint suffit
  // - dans le 2nd cas, il faut recalculer le Layout - et actuellement -
  //   c'est meme celui DE LA FENETRE qu'on recalcule ce qui est A REVOIR!!!

  UWin *updated_win = null;
  for (ULinkLink *llk = parents.first(); llk != null; llk = llk->getNext()){

    //NB: au moins (UBoxLink*) par construction
    UBoxLink *link = static_cast<UBoxLink*>(llk->link());

    for (UView* view = link->getViews(); view!=null; view = view->getNext()){
      UWin*  hardwin = null;
      UView* hardwin_view = null;

      // check views[k]!=null (some views may have been deleted)
      if (view && view->isRealized() 
	  && (hardwin = view->getHardwin())
	  && (hardwin_view = hardwin->getWinView(view->getDisp()))
	  // ne rien faire si la window n'est pas visible (sauf si always)
	  // ce qui implique de ne pas appliquer directement cette version 
	  // de update() sur une window             !!POSE PBM AVEC SOFTWINS!!
	  && (hardwin->isShowable() || upmode.always)
	  ){

	UEvent e(UEvent::viewPaint, null, hardwin_view, null);

	switch (upmode.ix) {
	case UUpdate::MOVE: 
	  // on veut reafficher la zone minimale du parent de view
	  // (= la reunion de l'ancienne et de la nouvelle position de view)
	  e.locateSource(view);
	  updateView(e, view, upmode); 

	  // ce cas est particulier pour les UUpdate::move des floating objects:
	  // comme la position absolue des floating est mise a jour dans
	  // getSizeHints et doUpdate, il est possible, dans le cas de vues
	  // multiples crees au fur et a mesure, que les coords de certaines
	  // vues n'aient jamais ete mises a jour si elles sont sorties de
	  // la zone visible du parent a un moment donne.
	  // ==> un updateView() direct sur la View de ce floating ne ferait rien
	  // ==> il faut faire une updateView() sur la vue du parent pour
	  //     remettre a jour les corrds de la vue du floating (et le
	  //     redessiner par la meme occasion)
	  //
	  //  if (!e.sourceView) {
	  //    UView* parview = view->getParentView();
	  //    parview->getBox()->updateView(&e, parview);
	  //  } 
	  //  else updateView(&e, view, &upmode);
	  break;

	case UUpdate::PAINT: {
	  if (!isShowable() && !upmode.always) break;
	  e.locateSource(view);
	  // le cas !e.sourceView signifie que cette view n'est pas visible
	  // a l'ecran actuellement car cachee par un ScrollPane, etc.
	  if (!e.getView()) return;

	  // si elem est specifie, essayer de ne reafficher QUE la region
	  // contenant elem (au lieu de la box entiere)
	  // NB: noter que elem n'est pas necessairement l'elem situe
	  // sous la souris (ie. ce n'est pas toujours e->getElem())

	  UElemProps itd; // !!ATTENTION: declarer itd ICI pour que itd.region
	  // reste correctement alloue dans l'appel a updateView()
	  // (itd.region etant pointe par upmode.region)

	  if (upmode.elem) {
	    //NOTE: delta_x, delta_y used for strpos1, strpos2
	    UElem *elem = e.searchElem(itd, upmode.elem, 
				       upmode.delta_x, upmode.delta_y);
	    if (elem && elem == upmode.elem)
	      upmode.region = &itd.region; // !!ATTENTION: voir note ci-dessus
	  }

	  // NB: optimisation (dans updateView) qui utilise e.redrawClip
	  // et redrawStatus (initialises par locateSource) de maniere a ne
	  // reafficher que le clip concerne (ou juste upmode.region si definie)
	  updateView(e, view, upmode);
	  }break;

	case UUpdate::TITLE:                  // !!A_REVOIR POUR SOFTWINS!!!
	  // commande de differente nature => a ne pas compter dans
	  // le "updated_win" sinon il y aura des cas de double modif
	  // ou soit le titre soit le contenu n'auront pas ete mis a jour
	  //OBS: win->update(upmode);    // !att: upmode PAS UUpdate::title
	                          //car upmode contient des data!
	  break;

	case UUpdate::SHOW:
	case UUpdate::HIDE:
	  setCmodes(UMode::CAN_SHOW, (upmode.ix == UUpdate::SHOW));
	  //cas floatings
	  if (isDef(0,UMode::FLOATING|UMode::SOFTWIN)/*||link->floatingCast()*/){
	    UView *fromview = view->getParentView();
	    if (!fromview) fromview = hardwin_view;
   	    //keep the same size (un show/hide ne doit pas retailler le parent!)
	    u_dim ww, hh; fromview->getSize(ww, hh);
	    updateImpl(upmode, hardwin_view, fromview, view, true, ww, hh);
	  }
	  else {
	    //devrait etre comme LAYOUT mais le pbm c'est que e.firstLayoutView 
	    //est indetermine
	    if (hardwin != updated_win) {
	      hardwin->update(UUpdate::all);  // pas upmode!
	      updated_win = hardwin;
	    }
	  }
	  break;

	case UUpdate::LAYOUT:
	case UUpdate::ALL:
	default:                       //SCROLL mais passe jamais par la
	  // ici il y a un pbm si la region est ponctuelle sans epaisseur
	  // (par ex une ubox vide) car locateSource renverra null
	  // (du fait d'une intersection vide) => on met a 1,1
	  if (view->getWidth() == 0)  view->setWidth(1);
	  if (view->getHeight() == 0) view->setHeight(1);

	  e.locateSource(view);

	    // fromview = from which view we must perform the lay out
	    // (de telle sorte que les parents se reajustent correctement
	    // qunad c'est necessaire)

	    UView *fromview = e.sp.layoutView;
	    if (!fromview) fromview = hardwin_view;

	    u_dim ww, hh; fromview->getSize(ww, hh);  //keep the same size
	    
	    if (isShowable() && e.getView()) {
	      if (upmode.ix == UUpdate::LAYOUT)
		//que layout, pas paint => showiew = null = null 
		updateImpl(upmode, hardwin_view, fromview, null, true, ww, hh);
	      else
		updateImpl(upmode, hardwin_view, fromview, fromview, true, ww, hh);
	    }
	    else if (upmode.always) {
	      //toujours faire layout (mais paint inutile) si always
	      bool can_show = isDef(0, UMode::CAN_SHOW);
	      setCmodes(UMode::CAN_SHOW, true);
	      updateImpl(upmode, hardwin_view, fromview, null, true, ww, hh);
	      setCmodes(UMode::CAN_SHOW, can_show);
	    }
	    //}
	  break;
	}
      }
    }
  }
}

/* ==================================================== ======== ======= */
// layoutview : view a partir de laquelle on recalcule le layout
// showview: view a reafficher
// en general layoutview == showview sauf pour show
// showview = nll => que layout, pas de paint!

void UBox::updateImpl(const UUpdate& upmode, 
		      UView* winview, UView* layoutview, UView* showview,
		      bool impose_size, u_dim w, u_dim h) {
  UViewLayout vl;
  UWinContext winctx1(winview);
  bool mustLayoutAgain = layoutview->doLayout(winctx1, vl);
  
 AGAIN:
  // forcer valeurs de w et h, indep de ce que veut layout()
  if (impose_size) layoutview->resize(w, h);

  // dans ce cas (typiquement pour show() d'un floating) il faut recalculer
  // le layout a partir de layoutview (= par ex. le parent du floating)
  // mais ne reafficher in fine que showview (= le floating)
  if (layoutview && layoutview != showview) {
    layoutview->updateWinData(*layoutview);
    if (impose_size) layoutview->resize(w, h);
  }

  if (!mustLayoutAgain) {
    if (showview) // showview = nll => que layout, pas de paint!
      winview->updateWinPaint(*showview, null, upmode.doublebuf, true);

    //optimisations possibles pour pas reafficher tous les layers
    //fromview->updateWinPaint(fromview, false/*doublebuf*/, 
    //        e.firstTranspView *de fromview!*, false);
  }
  else {
    layoutview->updateWinData(*layoutview);
    if (impose_size) layoutview->resize(w, h);
    
    UWinContext winctx2(winview); //nb: not the same WinContext!
    layoutview->doLayout(winctx2, vl);
    mustLayoutAgain = false;
    goto AGAIN;
  }
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
