/* ==================================================== ======== ======= *
 *
 *  uuctrl.cpp
 *  Ubit Project [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uuctrl.cpp	ubit:03.05.05"
#include <iostream>
#include <udefs.hpp>
#include <ubrick.hpp>
#include <ucall.hpp>
#include <uprop.hpp>
#include <ustr.hpp>
#include <ucond.hpp>
#include <uctrl.hpp>
#include <ucursor.hpp>
#include <ubox.hpp>
#include <uwin.hpp>
#include <update.hpp>
#include <uview.hpp>
#include <ucontext.hpp>
#include <uevent.hpp>
#include <uappli.hpp>
#include <uflow.hpp>
#include <ucontext.hpp>
#include <ustyle.hpp>
#include <umenuImpl.hpp>
#include <unatgraph.hpp>
#include <X11/keysym.h>
using namespace std;

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UCtrl::UCtrl()  {
  cmodes = 0;  // bmodes mis a 0 par UBrick
  istate = UOn::IDLE;
}

/* ==================================================== ======== ======= */
// NB: preserves existing modes (except those being redefined)

void UCtrl::setCmodes(u_modes m, bool onoff) {
  if (onoff) cmodes = cmodes | m;
  else cmodes = cmodes & ~m;
}
//inherited: void UCtrl::setBmodes(u_modes m, bool onoff) {

void UCtrl::setState(u_state state) {
  istate = state;
}

bool UCtrl::isEnabled() const  {return istate != UOn::DISABLED;}

/* ==================================================== ======== ======= */

bool UCtrl::isShown() const {
  if (!isShowable()) return false;

  for (ULinkLink *l = parents.first(); l!=null; l = l->getNext()) {

    // NB: par == null if no other parent (UFrame or similar)
    UGroup *par = l->link()->getParent();
    if (par) {
      // continuer recursivement (on retourne direct si OK
      // on continue en sequence grace au for() sinon
      if (par->isShown()) return true;
    }
  }

  // aucune racine valide parmi tous les chemins des parents
  return false;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UCtrl::select(bool state, bool call_cbs) {
  if (state == isSelected()) return;   // empecher boucles infinies!

  setCmodes(UMode::SELECTED, state);

  if (call_cbs) {
    UEvent e( state ? UEvent::select : UEvent::unselect,  
	      null, null, NULL);  //NB: win_view = null!
 
   if (groupCast()) e.setSource(groupCast());
    if (state) fire(e, UOn::select); else fire(e, UOn::unselect);

    // 26jt02: UOn::change supprime car ambigu
    // fire(e, UOn::change);
    
    // 07jun03: UOn::action deja appele par ailleurs
    // d'autre part, action est reserve aux cas ou il y a une action directe
    // sur l'objet. les actions indirectes (eg. deselection parce que l'on a 
    // selectionne un autre bouton) ne doivent pas etre prises en compte
    // pour eviter incoherences et diverses bizarreries
    // fire(e, UOn::action);
  }

  update(UUpdate::paint); // !!
}

void UCtrl::enable(bool state, bool call_cbs) {
  if (state == isEnabled()) return; // 27avril02 boucles infinies !
  //setCmodes(UMode::ENABLED, state);

  if (state) setState(UOn::IDLE);
  else setState(UOn::DISABLED);

  update(UUpdate::paint); // !!
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UCtrl::enterBehavior(UFlow&f, UEvent& e, int bstyle) {
  
  // comportement standard: le redraw et le change mouse ne sont faits 
  // que si l'on rentre dans l'objet avec la souris non enfoncee

                          // !! MAIS ATT: DnD: le cursor change si btn>0 !!!
  if (bstyle == STANDARD) {
    // changer le cursor si necessaire
    // ATT: ca ne depend uniquement pas de OWN_CURSOR: la view precedente
    // peut avoir affecte son propre cursor qu'il faut donc enlever

    /*  
    if (e.sp.cursor != f.lastCursor) {
      f.lastCursor = e.sp.cursor;
      e.sourceView->wg().setCursor(f.lastCursor);
    }
    */
    f.setCursor(e, e.sp.cursor);

    // ne changer l'ACTION et ne reafficher QUE si necessaire
    // ATT: ne pas changer l'action si ces modes sont off car 
    // on aurait alors un changement de couleur des reafficahges partiel 
    // (par exple dans le Text)

    // jamais execute si disabled
    if (istate != UOn::DISABLED
	&& isDef(0, UMode::ENTER_HIGHLIGHT/*| UMode::ENTER_HIGHBORDER*/)) {
                                // !!NB: il faudra distinguer les 2 cas !!
      setState(UOn::ENTERED);

      // PBM: affiche des zones inutiles ce qui fait merder en doublebuf
      //e.sourceView->updateWinPaint(e.firstTranspView);

      // PBM: utiliser e.redrawClip, ce qui est OK pour reafficher juste
      // la zone utile mais pose pbm si le callaback agrandit l'objet
      if (e.sourceView) updateView(e, e.sourceView, UUpdate::paint);
    }
  }

  // appeler qu'un button soit pressed ou non  (et meme si DND ???)
  /*if (isDef(UMode::ENTER_CB, 0))*/
  fire(e, UOn::enter); // ordre???

  // cas du Browsing ou la soue.redrawClipris a ete enfoncee ailleurs
  if (istate != UOn::DISABLED && bstyle == BROWSING) {

    if (!f.dragSource)		// just browsing
      armBehavior(f, e, BROWSING);
    else			// DND being performed
      if (isDef(0, UMode::CAN_DROP)) {
	f.dropTarget = this;
	setState(UOn::DROP_ENTERED);
	if (e.sourceView) updateView(e, e.sourceView, UUpdate::paint);
	fire(e, UOn::dropEnter);
      }
  }
}

/* ==================================================== ======== ======= */

void UCtrl::leaveBehavior(UFlow& f, UEvent& e) {

  // jamais execute SAUF CALLBACK si disabled
  if (istate != UOn::DISABLED) {

    if (f.dragSource /*&& bstyle == BROWSING*/) {  // DND being performed
      
      if (isDef(0, UMode::CAN_DROP) && istate == UOn::DROP_ENTERED) {
	f.dropTarget = null;
	setState(UOn::IDLE);
	if (e.sourceView) updateView(e, e.sourceView, UUpdate::paint);
	fire(e, UOn::dropLeave);
      }
    }

    else {  // NOT dnd
      
      if (!disarmBehavior(f, e, true))    // true -> means browsing
	return;     // si la vue a ete detruite, sortir au plus vite !
      
      // si istate vaut IDLE c'est que l'on etait en mode Browsing et 
      // que disarmB() a fait le travail. sinon disarm() a remit en mode
      // ENTER et il faut donc reafficher a la sortie de soursi
    
      if (istate != UOn::IDLE) {
	setState(UOn::IDLE);

	// ne reafficher QUE si necessaire
	if (isDef(0, UMode::ENTER_HIGHLIGHT /*|UMode::ENTER_HIGHBORDER*/)) {
                                // !!NB: il faudra distinguer les 2 cas !!
	  if (e.sourceView) updateView(e, e.sourceView, UUpdate::paint);
	}
      }
    }
  }

  // appeler que btn soit pressed ou non (et meme DND)
  /*if (isDef(UMode::ENTER_CB, 0))*/
  fire(e, UOn::leave);  // cf ordre!!
}

/* ==================================================== ======== ======= */

void UCtrl::armBehavior(UFlow& f, UEvent& e, int bstyle) {

  // jamais execute si disabled
  if (istate == UOn::DISABLED) return;

  if (isDef(0, UMode::CAN_DRAG1|UMode::CAN_DRAG2)) {	// Drag and Drop

    if (bstyle == STANDARD 
	&& ((e.getButtons() == UEvent::MButton2)       // RENDRE PARAMETRABLE!!
	    || (isDef(0, UMode::CAN_DRAG1) && e.getButtons() == UEvent::MButton1)
	    )
	) {
      // !ATT: ne commence JAMAIS en mode Browsing
      // (sinon, entrerait en mode DND des qu'on survolerait un objet 
      // CAN_DRAG meme si on n'est pas parti de celui-ci)

      f.dragSource = this;
      f.dropTarget = null;

      fire(e, UOn::dragStart);

      // changer curseur!     //NB: pourrait avoir cursor propre 
      // e.sourceView->wg().setCursor(f.lastCursor = &UCursor::drag);
      f.setCursor(e, &UCursor::drag);
    }
  }

  if (f.lastPressed != this 
      || !isDef(UMode::MOUSE_CLICK_CB,0)
      || e.time >= f.mclickTime + UAppli::getDefaults().mouse_click_delay) 
    f.mclickCount = 0;
 
  // set click pos (necessaire pour mclick)
  f.mclickX = e.xmouse;
  f.mclickY = e.ymouse;

  bool armable =
    isDef(0, UMode::CAN_ARM)
    //   ... discutable et embrouille : a revoir ...
    //|| (isDef(0, UMode::CAN_ACTION) && !isDef(0, UMode::CAN_EDIT_TEXT))
    ;

  // ATT: pas de else: un objet peut etre a la fois canArm ET canDrag !
  if (armable) {

    if (bstyle == STANDARD) {
      if (e.getButtons() == UEvent::MButton1) {
	f.lastArmed = this;
	setState(UOn::ARMED);
	
	if (e.sourceView) {
	  updateView(e, e.sourceView, UUpdate::paint);
	  fire(e, UOn::arm);
	}
      }
    } 

    else if (bstyle == BROWSING) {

      // n'importe quel button dans un menu system, que btn1 sinon
      if ((e.sp.inMenu
          || e.getButtons() == UEvent::MButton1
           )
          && (
              // coming back to the button that was initially pressed
              (f.lastPressed == this && armable)
              ||    // OR: this object is in the current Browsing group
              (f.browsing_group && f.browsing_group == e.sp.parentBrowsingGroup)
            )
          ) { 
        f.lastArmed = this;
	setState(UOn::ARMED);          
	if (e.sourceView) {
	  updateView(e, e.sourceView, UUpdate::paint);
	  fire(e, UOn::arm);
	}
      } 
    }

    else if (bstyle == TYPING) { // this action was provoked by a keyPress
      f.lastArmed = this;
      setState(UOn::ARMED);

      if (e.sourceView) {
	updateView(e, e.sourceView, UUpdate::paint);
	fire(e, UOn::arm);
      }
    }
  }
}

/* ==================================================== ======== ======= */

bool UCtrl::disarmBehavior(UFlow& f, UEvent& e, int bstyle) {
  // jamais execute si disabled
  if (istate == UOn::DISABLED) return true;

  if (f.dragSource) {   // DND being performed

    if (f.dropTarget) // si on est toujours au dessus de dropTarget
      f.dropTarget->fire(e, UOn::dropDone);

    f.dragSource->setState(UOn::IDLE);
    f.dragSource->update(UUpdate::paint);  // on pourrait sauver dragSourceView
    f.dragSource->fire(e, UOn::dragDone);

    if (e.sourceView) updateView(e, e.sourceView, UUpdate::paint);

    // retablir curseur
    //e.sourceView->wg().setCursor(f.lastCursor = e.sp.cursor);
    f.setCursor(e, e.sp.cursor);
    f.dragSource = null;
    f.dropTarget = null;
  }

  else {
    // dans le cas normal du Disarm provoqu par un mrelease
    // ==> mettre le BrowsingGroup  a null  (cas typique de la List
    //     ou on relache le bouton de la souris)
    // Exception: les menus ou le fait de relacher le menu opener
    //     ne doit pas desactiver le browsing puisque celui-ci
    //     doit continuer a se faire dans le menu affiche

    // si c'est bien l'objet sur lequel on avait appuye

    // 7AUG02: deplace et relatif a lastPressed et non lastArmed
    // (sinon pas effectue si objet par armable)

    bool multiclick = false;

    if (f.lastPressed == this) {

      if (isDef(UMode::MOUSE_CLICK_CB,0)
	  && e.xmouse > f.mclickX - UAppli::getDefaults().mouse_click_radius
	  && e.xmouse < f.mclickX + UAppli::getDefaults().mouse_click_radius
	  && e.ymouse > f.mclickY - UAppli::getDefaults().mouse_click_radius
	  && e.ymouse < f.mclickY + UAppli::getDefaults().mouse_click_radius) {

	if (f.mclickCount == 0) {    // single click
	  e.detail = f.mclickCount = 1;
	  f.mclickTime = e.time;
	  fire(e, UOn::mclick);
	}

	// cas multi-click :
	// verifier delai et position entre clics successifs
	else 
	  if (f.mclickCount > 0 
	      && e.time < f.mclickTime + UAppli::getDefaults().mouse_click_delay) {
	    
	    // initialise le champ detail du UEvent recupere par 
	    // methode getClickCount()
	    // (!att: detail sert a plusieurs choses)
	    e.detail = ++f.mclickCount;
	    f.mclickTime = e.time;
	    multiclick = true;

	    switch (f.mclickCount) {
	    case 2: fire(e, UOn::mbiclick); break;
	    case 3: fire(e, UOn::mtriclick); break;
	    case 4: fire(e, UOn::mquadclick); break;
	    }
	  }
      }
    }
    
    // si c'est bien l'objet que l'on avait arme
    
    if (f.lastArmed && f.lastArmed == this) {
      //ex: f.lastArmed = null; mal place

      // NB: on ne lance pas actionBehavior si bi/tri/quadclick !
      if (bstyle != BROWSING && !multiclick) {
	actionBehavior(f, e);
      }

      // !SI la vue a ete detruite => sortir au plus vite !
      if (!f.lastArmed) return false;

      // reaffichage final en mode IDLE si on est en browsing
      // et en mode ENTER sinon (car ds ce cas le press et le release
      // n'ont pu se faire qu'a l'interieur d'un objet!
      // --noter egalement que ds le cas general non Browsing actionB() 
      // ne reaffiche pas l'objet et que c'est donc disarm qui le fait

      //f.lastArmed = null; deplace plus bas
      if (bstyle == BROWSING)  setState(UOn::IDLE);
      else setState(UOn::ENTERED);
  
      if (e.sourceView) {
	updateView(e, e.sourceView, UUpdate::paint);
	fire(e, UOn::disarm);
      }

      if (!f.lastArmed) return false;
      else f.lastArmed = null; 
    }
  }

  return true;  //pas detruit, continuer normalement
}

/* ==================================================== ======== ======= */

bool UCtrl::actionBehavior(UFlow& f, UEvent& e) {

  f.inAction = this;

  // mis en premier sinon fire(action) est appele avec une valeur 
  // de isSelected() qui est obsolete

  if (isSelectable()) {
    // il faut selectionner au DISARM et non au ARM pour avoir un
    // comportement conforme a l'usage
 
    setCmodes(UMode::SELECTED, !isSelected());
    if (isSelected())  fire(e, UOn::select);
    else fire(e, UOn::unselect);

    // change est appele apres select/unselect et avant action
    fire(e, UOn::change);
   }

  setState(UOn::ACTIONED);

  //if (isDef(0, UMode::ACTION_WATCH)
  //  changeCursor(...)

  if (isDef(0, UMode::ACTION_HIGHLIGHT /*|UMode::ACTION_HIGHBORDER*/)) {
    if (e.sourceView) updateView(e, e.sourceView, UUpdate::paint);
  }

  if (e.sourceView) fire(e, UOn::action);

  //if (isDef(0, UMode::ACTION_WATCH)
  //  changeCursor(...)

  //NB: normalement le repaint est effectue dans le disarm
  // (sauf si appel depuis typeB() auquel cas il y aura un pbm
  // si on met UMode::ACTION_HIGHLIGHT

  //!Attention: la vue a ete detruite => sortir au plus vite !
  if (!f.lastArmed) return false;

  // IDLE doit etre la car actionB() n'est pas seulement appele de armB() 
  // mais aussi de typeB() et aussi pour le update des selectables

  setState(UOn::IDLE);

  if (isSelectable()) {
    //remet a jour toutes les "views" en mode IDLE
    //NB: le redraw simple ne suffit pas car objet modal 
    update(UUpdate::paint);
 }

  //f.inAction = null;
  return true;  //pas detruit, continuer normalement
}

/* ==================================================== ======== ======= */

void UCtrl::keyPressBehavior(UFlow& f, UEvent& e) {
  // jamais execute si disabled
  if (istate == UOn::DISABLED) return;

  // Attention (X11 doc) :
  // The creation of XComposeStatus structures is implementation-dependent;
  // a portable program must pass NULL for this argument.
  //
  //XComposeStatus compose;
  //int count = XLookupString(keyevent, buf, sizeof(buf), &keysym, &compose);

  KeySym keysym = 0; 
  char buf[1] = {'\0'};
  int count = XLookupString(&(e.xev->xkey), buf, sizeof(buf), &keysym, NULL);

  // the 'detail' field is used by getTypedChar()
  // (!att: detail sert a plusieurs choses)
  e.detail = buf[0];

  if (isDef(UMode::KEY_CB, 0)) fire(e, UOn::kpress);

  if (count > 0) {
    // n'est appele que pour les touches "imprimables"
    //!! NB: faudrait composer les touches !!!
    if (isDef(UMode::KEY_CB, 0)) 
      fire(e, UOn::ktype);

    if (isDef(0, UMode::CAN_EDIT_TEXT)) {
      if (keysym == XK_Return)           // rendre configurable!!
	actionBehavior(f, e);
    }
    else {
      if (keysym == XK_space)            // rendre configurable!!
	armBehavior(f, e, TYPING);
    }
  }
}

/* ==================================================== ======== ======= */
// NB: certaines implementation pourraient etre nettement plus compliquees,
// le UOn::type pouvant eventuellement etre appele apres le keyReleaseBehavior

void UCtrl::keyReleaseBehavior(UFlow& f, UEvent& e) {
  if (istate == UOn::DISABLED) return;

  if (isDef(UMode::KEY_CB, 0)){
    //see above
    //XLookupString(&(e.xev->xkey), buf, sizeof(buf), &keysym, &compose);

    KeySym keysym = 0; 
    char buf[1] = {'\0'};
    XLookupString(&(e.xev->xkey), buf, sizeof(buf), &keysym, NULL);

    e.detail = buf[0];

    fire(e, UOn::krelease);
  }
  disarmBehavior(f, e, TYPING);
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */

