/* ==================================================== ======== ======= *
 *
 *  umode.hpp
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _umode_hpp_
#define	_umode_hpp_
//pragma ident	"@(#)umode.hh	ubit:03.05.05"
#include <ubit/ubrick.hpp>

/** Objects' modes.
 * this class is implementation dependant.
 * only its static fields should be used by clients.
 */
class UMode: public UBrick {
protected:
  friend class UCtrl;
  // NB: bmodes is inherited from UBrick
  u_modes cmodes, bmask, cmask;

public:
  static UMode 
    /// selects the parents (if applicable, see: isSelectable())
    selected,

    /// unselects the parents (if applicable, see: isSelectable())
    unselected,

    /// parents can be selected (see isSelectable())
    canSelect, //canSelectChildren,

    /// parents can be armed
    canArm,

    /// parents are highlighted when entered
    enterHighlight, enterHighborder, 

    /// parents are highlighted when activated
    actionHighlight, actionHighborder, //actionWatch,

    /// parents can start a drag and drop action with button 1
    canDrag1, 

    /// parents can start a drag and drop action with button 2 (or equivalent)
    canDrag2, 

    /// parents can receive a drag and drop action
    canDrop,

    /// parents will ignore events
    ignoreEvents;

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  enum BMODES {
    /// this object was created by 'new' and can be destroyed by 'delete' 
    CAN_DEL        = 1<<0,
    
    /// this object is constant
    UCONST         = 1<<1,
    
    // this object is being destructed (by its destructors)
    DESTRUCTING    = 1<<2,
    DESTRUCTED     = 1<<3,

    /// these modes should not be propagated to parents
    DONT_PROPAGATE = CAN_DEL | DESTRUCTED | DESTRUCTING | UCONST,   // !dont' forget!

    // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    /// update() is NOT invoked when set(), add() or remove() are called
    NO_AUTO_UPDATE = 1<<4,

    /// this object is part of a text selection
    IN_TEXTSEL     = 1<<5,

    /// this object has a vertical orientation (horizontal otherwise)
    IS_VERTICAL	   = 1<<6,

    /// this object has a backwards orientation (forwards otherwise)
    //IS_BACKWARDS   = 1<<,

    /// events go through this object
    //RESEND_EVENTS  = 1<<,

    /// this object ignores events (its parent will get them)
    IGNORE_EVENTS  = 1<<9,

    // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    // callbacks to which this object is sensitive

    DESTRUCT_CB      = 1<<10,
    CHANGE_CB        = 1<<11,
    CONTAINER_CB     = 1<<12,
    CHILD_CHANGE_CB  = 1<<13,
    //CHILD_SELECT_CB  = 1<<, plus utilise
    VIEW_PAINT_CB    = 1<<15,
    VIEW_CHANGE_CB   = 1<<16,
    // -- Key: UOn::kpress, UOn::krelease and UOn::ktype
    KEY_CB           = 1<<17,
    // -- Mouse: UOn::mpress, UOn::mrelease (NB: not used)
    MOUSE_CB         = 1<<18,
    // -- Mouse: UOn::mclick, UOn::mbiclick, etc. (NB: not used)
    MOUSE_CLICK_CB   = 1<<19,
    // -- Enter, Leave and Cursor: UOn::enter and UOn::leave
    MOUSE_CROSS_CB   = 1<<20,
    // -- Drag & Move: UOn::mdrag and UOn::mmove
    MOUSE_DRAG_CB    = 1<<21,
    MOUSE_MOVE_CB    = 1<<22,

    // -- UOn::message, UOn::preChildEvent, UOn::anyEvent
    UMESSAGE_CB      = 1<<27,
    PRE_CHILD_EVENT_CB = 1<<28,
    RAW_EVENT_CB     = 1<<29,
    NOTIFY_EVENT_CB  = 1<<30,

    CALLBACKS = 
    DESTRUCT_CB|CHANGE_CB|CONTAINER_CB|CHILD_CHANGE_CB    // |CHILD_SELECT_CB
    |VIEW_PAINT_CB|VIEW_CHANGE_CB
    |KEY_CB|MOUSE_CB|MOUSE_CLICK_CB|MOUSE_CROSS_CB|MOUSE_DRAG_CB|MOUSE_MOVE_CB
    |UMESSAGE_CB|PRE_CHILD_EVENT_CB|RAW_EVENT_CB|NOTIFY_EVENT_CB
    //!no comma after last item!

    //!BEWARE total size must be coherent with typedef u_modes in udefs.hh!
  };

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  enum CMODES {
    //--- Internal MetaTypes
    GROUP             = 1<<0, ///< logical grouping container
    BOX               = 1<<1, ///< container that manage views
    SOFTWIN  	      = 1<<2, ///< simulated "soft" window (NB: softwin => floating)
    HARDWIN           = 1<<3, ///< "hard" (native) window
    INCRUST           = 1<<4, ///< incrusted hard window (see UIncrust)

    FLOATING          = 1<<5, ///< untiled box or softwin: its position is controlled by an UPos
    MENU              = 1<<6, ///< this object is a Menu (or a subclass)
    FAKE_PARENT       = 1<<7, ///< not taken into account when auto_deleting children
    BORDER_LIST       = 1<<8, ///< list of border frame elements
    SOFTWIN_LIST      = 1<<9, ///< list of softwins controlled by a hardwin

    /// refer to isShowable() and isShown() methods for details.
    CAN_SHOW          = 1<<10,

    /// object and chidren painted in double buffer mode
    DOUBLE_BUFFER     = 1<<11,

    /// this object is modal when shown
    MODAL             = 1<<12,

    //ENABLED (ISTATE::DISABLED used instead)

    /// this object is in the selected mode
    SELECTED          = 1<<13,

    /// can be selected (in the same way as a checkbox)
    CAN_SELECT        = 1<<14, 

    /// can select its ARMable children (in the same way as a listbox)
    CAN_SELECT_CHILDREN = 1<<15, 

    /// can select included text
    CAN_SELECT_TEXT   = 1<<16,
    /// can edit included text (ie. the group contains a UEdit brick)
    CAN_EDIT_TEXT     = 1<<17,

    /// can browse children (for lists and menus)
    CAN_BROWSE_CHILDREN = 1<<19, 
    
    /// can be dragged (with button 1 or button2) or dropped
    CAN_DRAG1         = 1<<20,
    CAN_DRAG2         = 1<<21,
    CAN_DROP          = 1<<22,

    /// can be armed (meaning it will be repaint on Uon::arm conditions
    /// and that corresponding object's callbacks will be called)
    CAN_ARM           = 1<<23,
    //CAN_ACTION        = 1<<,

    // ** Higligting:
    // -- on UOn::action conditions
    ACTION_HIGHLIGHT  = 1<<24,
    //ACTION_HIGHBORDER = 1<<,
    //ACTION_WATCH      = 1<<,

    // -- on UOn::enter and UOn::leave conditions
    ENTER_HIGHLIGHT   = 1<<25,
    //ENTER_HIGHBORDER  = 1<<,

    //--- Box attributes

    /// this object has a specific renderer (= a UViewStyle object)
    HAS_RENDERER      = 1<<26,

    /// this object has a specific cursor
    HAS_CURSOR        = 1<<27,

    /// if HAS_CLOSE_MENU_MODE is true, CLOSE_MENU_MODE defines
    /// this behavior (which is then dynamically inherited)
    HAS_CLOSE_MENU_MODE  = 1<<28,
    CLOSE_MENU_MODE      = 1<<29,

    //--- Win and menu attributes

    /// this win is mapped on the screen
    WIN_MAPPED           = 1<<30,

    /// this win is automatically opened if its parent is ARMable
    WIN_AUTO_OPEN_MODE   = 1<<31

    // !no comma after last item!
    // !BEWARE total size must be coherent with typedef u_modes in udefs.hh !!
  };

  UMode(u_modes bmodes, u_modes cmodes, bool on_off);
  UMode(u_modes bmodes = 0);
  virtual ~UMode() {destructs();}

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void setParentModes(UCtrl *parent) const;
  virtual void addingTo(class ULink *selflink, UGroup *parent);
  virtual void removingFrom(class ULink *selflink, UGroup *parent);
  ///< NOTE that this function require a specific destructor.
};

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
#endif

