/* ==================================================== ======== ======= *
 *
 *  ugroup.hpp : Logical "grouping" container.
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _ugroup_hpp_
#define	_ugroup_hpp_
//pragma ident	"@(#)ugroup.hpp	ubit:03.06.04"
#include <ubit/uargs.hpp>
#include <ubit/uctrl.hpp>
#include <ubit/ucall.hpp>


/** Position in the child list.
 * see: UGroup::getChild(UListPos&);
 */
class UListPos {
  friend class UGroup;
  class ULink* link;
  int pos;
public:
  static UListPos none;
  UListPos();
  void reset();
  int  getPos() const;
};

/* ==================================================== ======== ======= */
/** Logical "grouping" container.
 *
 * This class is used for grouping objects. it is a very lightweight
 * and general container. by opposition to the UBox class (and subclasses)
 * UGroup objects do not have Views and, thus, do not correspond to any
 * specific location on the Screen (there are just genuine containers)
 *
 * See:
 * - class UBrick and UBrick::~UBrick for notes on recursive deletion.
 * - class UBox for info on how to combine boxes and interactors.
 * - class UCtrl as many methods are inherited from this class.
 */
class UGroup: public UCtrl {
public:
  static UStyle *style;   ///< corresponding Ubit UStyle

  UGroup(const UArgs& a = UArgs::none);
  /**<
    * constructor; see also <em>creator shortcut</em> ugroup().
    */

  friend UGroup& ugroup(const UArgs& a = UArgs::none) {return *new UGroup(a);}
  /**<
    * <em>creator shortcut</em> that is equivalent to *new UGroup().
    * Note: watch the case: UGroup is a class while ugroup() is a function!
    */

  virtual ~UGroup();
  /**<
    * see <b>important notes</b> on <b>recursive deletion</b>.
    * refer to: UBrick::~UBrick()
    */

  virtual const UStyle& getStyle(UContext*) const {return makeStyle();}
  /**<
   * returns the contextual UStyle of this object.
   *  This virtual function calls the makeStyle() static function that
   *  was redefined for this specific class
   * - see also: UStyle and makeStyle() functions */

  static const UStyle& makeStyle();
  /**<
   * creates the Style of this object.
   * This static function is redefined by each class that derives from UGroup.
   * It is called by the virtual function UGroup::getStyle()
   * - note that certain classes have several makeStyle() functions (in which case
   *   they also have a specific getStyle() function)
   * - see also: UStyle and UGroup::getStyle()
    */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // child management 

  UGroup& addlist(const UArgs&);
  /**<
   * adds a <b>list</b> of children to the end of the object's child list.
   * Notes:
   * - this function is NOT virtual and returns the object
   * - this function updates graphics if the list contains UProp, UElem, 
   *   or UGroup objects
   * Exemple:
   * - UGroup& g = ugroup(); g.addlist(UColor::red + UFont::bold + "abcd"):
   */

  virtual void add(UBrick* child, bool update = true);
  virtual void add(UBrick& child, bool update = true);
  virtual void add(ULink* child,  bool update = true);
  virtual void add(ULink& child,  bool update = true);
  /**<
   * adds a 'child' to the end of the object's child list.
   * Arguments:
   * - graphics are updated if 'update' is true.
   *   clients should call the update() function later otherwise.
   *   (this is useful when many children are added: graphics update
   *    then take place only one time)
   * - the same child can be added several times to the child list.
   */

  virtual void addOnce(UBrick* child, bool update = true);
  virtual void addOnce(UBrick& child, bool update = true);
  /**<
   * adds a 'child' if not already in the child list.
   */

  virtual void insert(int pos, UBrick* child, bool update = true);
  virtual void insert(int pos, UBrick& child, bool update = true);
  virtual void insert(int pos, ULink* child,  bool update = true);
  virtual void insert(int pos, ULink& child,  bool update = true);
  /**<
   * insert a 'child' into the object's child list at this position.
   * Arguments:
   * - pos =  0  means "before the first child"
   * - pos = -1  means "after the last child"
   * - a non fatal error if produced if 'pos' is out of range
   *
   * - this function updates graphics if 'update' is true.
   *   clients should call the update() function later otherwise.
   *   (this is useful when many children are added: graphics update
   *    then take place only one time)
   */
  
  virtual void remove(UBrick* child, 
		      bool auto_delete = true, bool update = true);
  virtual void remove(UBrick& child, 
		      bool auto_delete = true, bool update = true);
  /**<
   * removes 'child' OR <b>deletes</b> 'child' and its descendants.
   * Arguments:
   *  - if <em>auto_delete</em> is false:
   *    the <b>first</b> occurence of 'child' is removed from the object's
   *    child list (other occurences are not removed if 'child' appears
   *    several times in the child list). 'child' is <b>not</b> deleted
   *
   *  - if <em>auto_delete</em> is true:
   *    the <b>first</b> occurence of 'child' is removed AND DELETED
   *    if it has no other parent and is not pointed by an uptr() 
   *
   *    -# this option is generally preferable as this is the only
   *        way to free unreferenced objects
   *    -# when 'child' is deleted, its descendants are also be deleted
   *       (if they have no other parent and are not pointed by an uptr()).
   *       See:  UGroup::~UGroup for details.
   *    -# 'child' is not deleted if it appears several times in the 
   *        object's child list
   *
   * - this function updates graphics if 'update' is true.
   *   clients should call the update() function later otherwise.
   *   (this is useful when many children are removed: graphics update
   *    then take place only one time)
   */

  virtual void remove(int pos, bool auto_delete = true, bool update = true);
  /**<
   * removes OR <b>deletes</b> the child at position 'pos' and its descendants.
   * Works in the same way as:  remove(UBrick&, bool, bool) except that:
   * - 'pos' =  0  means "first child" 
   * - 'pos' = -1  means "last child"
   * - a non fatal error if produced if 'pos' is out of range
   */

  virtual void removeAll(bool auto_delete = true, bool update = true);
  /**<
   * removes all children OR <b>deletes</b> all children and their descendants.
   * Works in the same way as:  remove(UBrick&, bool, bool).
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // access to children

  virtual int getChildCount() const;
  /**< returns the number of children */

  virtual UBrick** getChildren() const;
  virtual UBrick** getChildren(int& child_count) const;
  /**< 
   * returns a copy of the child list.
   * Returned value:
   * - returns null if there are no children.
   * - returns the child list if there is at least one child.
   *   This list must be destroyed by clients by calling delete[].
   *
   * Note: 
   * - Ubit objects can be added several times to the same parent.
   *   So, be careful when you delete children not to delete them twice
   *   (or more!) as some of them may appear several times in the child list.
   * See also:
   * - getChild() and other variants.
   */

  virtual int getChildren(std::vector<UBrick*>&) const;
  /**< 
   * returns a copy of the child list.
   * returns the number of children and intializes the vector of children.
   * Note: child_vect is not cleared : children are added to the 
   *       child_vect argument if its is not empty
   * See also: UBrick** getChildren() and getChild() (and other variants).
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  
  virtual UBrick* getChild(int pos) const;
  /**< 
   * returns the child at this position in the child list.
   * Arguments:
   * - 'pos' = 0 means "first child" and 'pos' = -1 means "last child"
   * - returns null if 'pos' is out of range (doesn't generate an error).
   *
   * Note:
   *   use getChild(UListPos&) or getChildren() is you need to access many 
   *   children in the child list: this will be more efficient than multiple 
   *   calls to getChild(int pos)
   */

  virtual UBrick* getChild(UListPos&) const;
  /**<
   *  efficient API to retreive children iteratively.
   *  Notes: 
   *  - use UListPos::reset() to rescan the list
   *
   *  Example:
   *  <pre><tt>
   *      UBrick* child = null;
   *      UListPos lpos;
   *
   *      while ((child = parent.getChild(lpos)))  {
   *          cout << "Class: " << child->getClassName() << endl;
   *      }
   *
   *      // rescan the child list from the beginning
   *      lpos.reset();
   *      while ((child = parent.getChild(lpos)))  {....}
   *  </tt></pre>
   */

  virtual UBrick* getChild(bool(predicate)(const UBrick* child),
                           UListPos& = UListPos::none) const;
  virtual UBrick* getChild(UListCall& predicate, bool& status,
                           UListPos& = UListPos::none) const;
  /**<
    * searches for children that statisfy a certain condition.
    * usage:
    * - returns the first child that verifies the condition if
    *   UListPos == none (the default)
    * - returns the next child that verifies the condition otherwise
    * - use UListPos::reset() to rescan the list
    * 
    * Examples:
    *  <pre><tt>
    *      UBrick* child = parent.getChild(&UBrick::isInstance<UButton>);
    * or:
    *      UListPos lpos;
    *      while ((child = parent.getChild(&UBrick::isInstance<UStr>, lpos)))  {....}
    *  </tt></pre>   
    */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual int getChildPos(const UBrick& obj, int nth = 0) const;
  virtual int getChildPos(const UBrick* obj, int nth = 0) const;
  /**<
    * returns the position of the Nth occurence of 'obj' in the child list (returns -1 if not found).
    * 'Nth' = 0 means "search first occurence", 'Nth' = 1 "search second occurence", etc.
    *
    *  See also: getChildPos(const UBrick*, UListPos&) for multiple calls.
    */

  virtual int getChildPos(const UBrick* obj, UListPos&) const;
  /**<
    * returns the position of the next occurence of 'obj' in the child list (returns -1 if not found).
    * works in the same way as getChild(UListPos&).
    */
  
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual UBrick** getAttrs() const;
  virtual UBrick** getAttrs(int& child_count) const;
  ///< returns a copy of the attribute list (works in the same way as getChildren()).

  virtual int getAttrCount() const;
  ///< returns the number of attributes.

  virtual void addAttr(UBrick* child, bool update = true);
  virtual void addAttr(UBrick& child, bool update = true);
  virtual void addAttr(ULink& child,  bool update = true);
  ///< adds an attribute (works in the same way as add()).

  virtual void insertAttr(int pos, UBrick* child, bool update = true);
  virtual void insertAttr(int pos, UBrick& child, bool update = true);
  virtual void insertAttr(int pos, ULink& child,  bool update = true);
  ///< adds an attribute (works in the same way as insert()).

  virtual void removeAttr(UBrick* child, bool auto_delete = true,
			  bool update = true);
  virtual void removeAttr(UBrick& child, bool auto_delete = true, 
			  bool update = true);
  ///< removes an attribute (works in the same way as remove()).

  virtual void removeAllAttr(bool auto_delete = true, bool update = true);
  ///< removes all attributes (works in the same way as removeAll()).

  virtual UBrick* getAttrChild(UListPos& from) const;
  virtual UBrick* getAttrChild(bool(predicate)(const UBrick* child),
                               UListPos& = UListPos::none) const;
  virtual UBrick* getAttrChild(UListCall& predicate, bool& stat, UListPos& from) const;
  ///< searches in the attribute list (works in the same way as getChild()).

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual UBrick* getAnyChild(UListPos& from) const;
  virtual UBrick* getAnyChild(bool(predicate)(const UBrick*),
			      UListPos& from = UListPos::none) const;
  virtual UBrick* getAnyChild(UListCall& predc, bool& stat,
			      UListPos& from = UListPos::none) const;
  ///< searches in the attribute then in the element list (works in the same way as getChild()).

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // other methods

  virtual UStr copyText(bool recursive = true) const;
  virtual void copyText(UStr&, bool recursive = true) const;
  /**<
   *  collates and returns the text that is enclosed in this object.
   *  'recursive' means that the text included in descendants is also collated.
   *
   *  Notes:
   *  - copyTextTo() is more efficient than copyText()
   *  - getTextSeparator() is used for separating enclosed children when
   *    'recursive' is true.
   */

  virtual int getViews(std::vector<UView*>&) const;

  virtual const UStr* getTextSeparator() const;
  /**< 
   * text separator used by retreiveText() for separating enclosed children
   */

  virtual void show(bool = true);
  /**<
   * shows the object if argument is true; hides it if false.
   * see also: hide(), isShown(), isShowable()
   */

  virtual void hide() {show(false);}
  /**<
   * same as: show(false)
   */

  virtual void highlight(bool state);
  /**<
   * highlight the object (and brings it to the top if it is a window).
   */

  virtual void update();
  virtual void update(UUpdate upmode); 
  /**< 
   * updates this object's paint and/or layout.
   * <p>Arguments:
   * - no argument: updates layout then repaints
   * - 'upmode' argument : updates according to this argument. 
   *   see class UUpdate for details
   * <p>Note:
   * - update() must be called after add() or remove() if their last argument
   *   was set to 'false'
   */

  virtual void close(int status);
  /**< 
   * closes this object
   * Default behavior: hides the object (same as show(false)).
   * 
   * Can be redefined by subclasses for appropriate behaviors (such as object 
   * deletion, saving or freeing data, etc.)
   * 
   * Note: status is only meaningful for UWin subclasses.
   * <p>See also: UBox::closeWin(), UWin::close()
   */

  static void  closeWin(class UEvent&, int status);
  virtual void closeWin(int status);
  /**<
   * closes the first window (UDialog, UMenu...) that contains this object.
   * Notes:
   * - status should be >= 0 (see UWin::lock())
   * - calls UWin::closeWin()
   * - called by callback object: ucloseWin()
   */

  friend UCall& ucloseWin(int status = 0)
  {return ucall(status, &UGroup::closeWin);}
  /**<
   * callback that closes the first window (UDialog, UMenu...) that contains this object.
   * same effect as: UGroup::closeWin()
   */
  
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

  enum ChildListType {ATTR_LIST, ELEM_LIST};
  enum RemoveMode {REMOVE_FROM_PARENTS=-1, NO_DEL=false, AUTO_DEL=true};

  virtual void addImpl(ChildListType, UBrick* child, class ULink* childlink,
                       int pos, bool update, bool* should_update);

  virtual void removeImpl(ChildListType, UBrick* child, 
			  class ULink* prevlink, RemoveMode remove_mode, 
			  bool update, bool* should_update);
  
  ULink* getChildImpl(ChildListType, const UBrick* child, int nth,
                      class ULink*& prevlink, int* pos) const;
  ULink* getChildImpl(ChildListType, int pos, class ULink*& prevlink) const;
  ULink* getChildImpl(ChildListType, UListPos& from) const;
  ULink* getChildImpl(ChildListType, UListCall&, bool& status, UListPos& from) const;
  /**<
    * [impl] implementation of public add/remove/getChild methods.
    * these methods are called by add(), remove(), getChild()
    * and operator delete. They can can possibly be redefined by subclasses.
    * Note however its is generally necessary to redefine all of them
    * simultaneously as these methods call each other
    */

  virtual void fire(class UEvent&, const class UOn&) const; 
  ///< [impl] fires callback functions and callback methods.

  virtual void destructs();
  /**<
    * [impl] see <b>important note</b> on <b>subclassing</b>.
    * destructs() unlinks the object from its parents and destroys children
    * (when applicable).
    * Any class that redefines removingFrom() MUST HAVE A DESTRUCTOR
    * that calls destructs().
    */
  
#ifndef NO_DOC
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  friend class UBrick;
  friend class UWin;

  virtual class UGroup* groupCast() {return this;}

  class ULink* getAttrLinks() const;
  class ULink* getChildLinks() const;
  ///< [impl] returns internal child links.

  bool isBrowsingGroup() {return (cmodes & UMode::CAN_BROWSE_CHILDREN) != 0;}
  virtual UGroup* getBrowsingGroup() {return (cmodes & UMode::CAN_BROWSE_CHILDREN) ? this : null;}
  ///< [impl] children managed in BROWSING mode behavior if not null (see UCtrl).

  virtual void initView(ULink *selflink, UView *ancestor_view);
  virtual void initChildViews(UGroup*, ULink *childlink);
  virtual void updateView(UEvent&, UView*, const UUpdate&);

protected:
  class UChain children;	  // child list

  virtual int   getTextLength(bool recursive) const;
  virtual char* getTextData(char *ptr, bool recursive) const;
  ///<  [impl] intermediate functions for retrieveText()

  virtual void deleteRelatedViews(class UView* parview, class ULink*);
  virtual void deleteRelatedViews2(class UView* parview);

#endif
};


#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
