/* ==================================================== ======== ======= *
 *
 *  udoc.hpp
 *  Ubit Project [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _udoc_hpp_
#define _udoc_hpp_
//pragma ident	"@(#)udoc.hpp	ubit:03.06.03"
#include <vector>
#include <ubit/ustr.hpp>
#include <ubit/ubox.hpp>

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/** Abstract Document
*/
class UDoc : public UFlowbox {
public:
  
  struct Object {
    virtual UStr* getSrcPath() const = 0;
    virtual bool isLoaded() const = 0;
    virtual int load() = 0;
  };
  typedef std::list<Object*> Objects;
  
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  struct Link {
    virtual UStr* getNamePath() const = 0;
    virtual UStr* getHRefPath() const = 0;
   };
  typedef std::list<Link*> Links;

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  struct Callbacks {
    virtual void notify(UEvent&, UStr* path, UDoc*) = 0;
  };

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  struct Errors {
    int stat;
    UStr messages;
    void set(int);
    void set(int, const UStr&);
  };

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  
  struct Reader {
    Reader() {}
    virtual ~Reader() {}
    virtual UDoc* read(const UStr& pathname, UDoc::Callbacks*, UDoc::Errors*) = 0;
  };

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  static UDoc* readDoc(const UStr& pathname, Callbacks* = 0, Errors* = 0);
  /**< reads and creates a Document from a file.
   * details:
   * - this function returns an instance of the appropriate UDoc subclass
   *   depending on the file format (specified by its suffix).
   * - null is returned if the document can't be created.
   * - 'Callbacks' and 'Errors' arguments can be null.
   * see also:
   * - addReader(), removeReader(), setDefaultReader()
   * default suffixes and formats:
   * - xpm, jpg, jpeg, gif files creates a UImaDoc instance
   * - txt, java, c, h, cpp, hpp, cc, hh files creates a UTextDoc instance
   * - pdf, doc, ppt, xls files are managed by the UExternalDoc::Reader.
   *   This Reader calls an external application (acroread or soffice)
   *   that displays these documents in a separate window.
   * - xhtml files can be (optionally) managed by the UHtmlDoc::Reader
   *   (see xml/uhtml.hpp)
   */

  static void addReader(const char* suffix, UDoc::Reader*);
  static void removeReader(const char* suffix, UDoc::Reader*);
  /**< adds/removes a Document reader.
   * details:
   * - this "Reader" will be called by readDoc() for all files that
   *   have the corresponding "suffix".
   * - a "Reader" creates an instance of its corresponding UDoc subclass
   *   (for instance: UImaDoc::Reader creates an UImaDoc instance)
   * - when several Readers are registered for the same suffix, the last
   *   one is taken into account
   * see also:
   * - setDefaultReader()
   * - UImaDoc::Reader, UTextDoc::Reader, UExternalDoc::Reader
   * - UHtmlDoc::Reader (in xml/uhtml.hpp)
   */

  static void setDefaultReader(UDoc::Reader*);
   /**< specifies the default Document reader.
    * details:
    * - the default Document reader is used to create Documents whose
    *   suffix is unknown (i.e. no specific Reader has been specified
    *   for this suffix)
    * - such documents are not created if "Reader" is null
    * see also:
    * - addReader(), removeReader(), setDefaultReader()
    */

  //virtual UBrick* getIcon(const UStr& pathname);

  virtual int loadMissingObjects() {return 0;}
  ///< returns the # of objects that are still missing

  virtual const Objects* getObjects() const {return null;}
  /**< retreives the list of attached objects (such as images in HTML docs).
   * Note: these objects are implicitely destroyed when the document
   * is deleted (except if pointed by an uptr<> in client code)
   */

  virtual const Links* getLinks() const {return null;}
  /**< retreives the list of links in the doc (such as HTML <a> tags).
   * Note: these links are implicitely destroyed when the document
   * is deleted (except if pointed by an uptr<> in client code)
   */

  virtual ~UDoc() {}

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

protected:
  static class UDocManager* manager;
  UStr pathname;
  UDoc(const UStr& _pathname);

private:  // these operators are forbidden
  UDoc(const UDoc&);
  UDoc& operator=(const UDoc&);
};

/* ==================================================== ======== ======= */

class UTextDoc : public UDoc {
public:
  
  struct Reader : public UDoc::Reader {
    virtual UDoc* read(const UStr& pathname, UDoc::Callbacks*, UDoc::Errors*);
  };

  UTextDoc(const UStr& pathname, UStr& content);
};

/* ==================================================== ======== ======= */

class UImaDoc : public UDoc {
public:
  
  struct Reader : public UDoc::Reader {
    virtual UDoc* read(const UStr& pathname, UDoc::Callbacks*, UDoc::Errors*);
  };

  UImaDoc(const UStr& pathname, UIma& content);
};

/* ==================================================== ======== ======= */

class UExternalDoc : public UDoc {
public:
  
  struct Reader : public UDoc::Reader {
  public:
    Reader(const char* _doctype, const char* command);
    virtual UDoc* read(const UStr& pathname, UDoc::Callbacks*, UDoc::Errors*);
  protected:  
    UStr doctype, command;
  };

  UExternalDoc(const UStr& pathname, const UStr& doctype, const UStr& command);
  virtual ~UExternalDoc() {}

protected:
  UStr doctype, command, status;
  virtual void openDoc();
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

class UDocbox : public UVbox {
public:
  UDocbox(const UArgs& a = UArgs::none);

  //virtual int readContent(const UStr& name, const UStr& pathname);
  /**< reads content from a file.
    *   returned value: UFilestat or UXmlStat
    */

  virtual void setContent(const UStr& name, UDoc&);
  /**< sets name and content.
    * Note: 
    * content is not copied and should not be destroyed while 
    * the UDoc is used. It will be destroyed automatically when the 
    * UDoc is destroyed if its is not part of the instance graph
    * nor pointed by an uptr()
    */

  virtual void clearContent();

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual const UStr& getName() const;
  virtual void setName(const UStr&);

  virtual UDoc* getDoc()     {return pdoc;}
  virtual UBox* getGlass()   {return pglass;}
  virtual UScale* getScale()  {return pscale;}
  virtual void setScale(int);       ///< absolute zoom
  virtual void incrScale(int);      ///< relative zoom  

  UBox&   glass()    {return *pglass;}
  UBox&   content()  {return *pcontent;}
  UStr&   name()     {return *pname;}
  UScale& scale()    {return *pscale;}

 // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
protected:
  uptr<UDoc>     pdoc;
  uptr<UBox>     pcontent;
  uptr<UBox>     pglass;
  uptr<UScale>   pscale;
  uptr<UStr>     pname;
};
  
#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
