/* ==================================================== ======== ======= *
 *
 *  uufont.cc
 *  Ubit Project  [Elc][beta1][2001]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:01] ======= *
 * ==================================================== ======== ======= */

//#pragma ident	"@(#)uufont.cc	ubit:b1.10.2"
#include <ubrick.hh>
#include <ucontext.hh>
#include <uctrl.hh>
#include <ufont.hh>
#include <uappli.hh>
#include <ubox.hh>

// minimum, medium and maximum LOGICAL size (not a point size!)
const int UFont::MIN_LSIZE      = 1;   //NB: MUST be 1
const int UFont::MEDIUM_LSIZE   = 7;   //NB: MUST be > 3 1
const int UFont::MAX_LSIZE      = 15;
//OBS: const int UFont::MAX_LSIZE = sizeof(UFont::FONT_SIZES) / sizeof(UFont::FONT_SIZES[0]);

// 4 variants: plain, bold, italic, boldItalic
const int UFont::FONTMAP_SIZE   = 4 * (UFont::MAX_LSIZE + 1);

static const UFont::FontSize _DEFAULT_FONT_SIZES[UFont::MAX_LSIZE] = {
  {1,2,"2"}, 
  {2,2,"2"}, 
  {3,2,"2"}, 
  {4,6,"6"}, 
  {5,8,"8"}, 
  {6,10,"10"}, 
  {7,12,"12"}, 
  {8,14,"14"}, 
  {9,16,"16"}, 
  {10,18,"18"}, 
  {11,24,"24"}, 
  {12,30,"30"}, 
  {13,40,"40"}, 
  {14,50,"50"}, 
  {15,60,"60"}
};

const UFont::FontSize*
UFontFamily::DEFAULT_FONT_SIZES = _DEFAULT_FONT_SIZES;


/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// *** Predefined FontFamilies ***

//Problemes:
// -- la specif 'normal' deconne sous certains Linux
// -- italique est tantot specifie par 'o' ou par 'i'
// -- fixed n'a pas forcement d'italique (ou alors d'une taille delirante)

UFontFamily 
UFontFamily::helvetica("helvetica",
		       "-*-helvetica-medium-r-normal-*-12-*-*-*-*-*-*-1",
		       "medium", "bold", "o");
UFontFamily
UFontFamily::courier("courier",
		     "-*-courier-medium-r-normal-*-12-*-*-*-*-*-*-1",
		     "medium", "bold", "o");
UFontFamily 
UFontFamily::times("times",
		   "-*-times-medium-r-normal-*-12-*-*-*-*-*-*-1",
		   "medium", "bold", "i");
UFontFamily 
UFontFamily::fixed("fixed",
		   //NB: pbm avec Linux avec 'normal'
		   //"-*-fixed-medium-r-normal-*-12-*-*-*-*-*-*-1",
		   "-*-fixed-medium-r-*-*-12-*-*-*-*-*-*-1",
		   //NB: pas d'italique pour fixed
		   "medium", "bold", "r");
UFontFamily 
UFontFamily::symbol("symbol",
		    "-*-symbol-*-r-*-*-12-*-*-*-*-*-*-1",
		    "medium", null, null);

// The Standard Font Family. Should always be available and preloaded
UFontFamily 
UFontFamily::standard("standard", helvetica);

// The "Any" Font Family. 
// Can design any font family in UFont() specifications (see below)
UFontFamily 
UFontFamily::any("any", standard);


/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

int UFontFamily::familyCount = 0;

// private constr: Creates a FontFamily alias
UFontFamily::UFontFamily(const char *fname, UFontFamily &ff) {
  name        = fname;
  ix          = ff.familyCount;
  natName     = ff.natName;
  normalStyle = ff.normalStyle;
  boldStyle   = ff.boldStyle;
  italicStyle = ff.italicStyle;
  fontSizes   = ff.fontSizes;
}

// Creates a new FontFamily from a native system font name
UFontFamily::UFontFamily(const char *ubit_family_name,
			 const char *native_family_name_specs,
			 const char *normal_style_specs,
			 const char *bold_style_specs, 
			 const char *italic_style_specs,
			 const UFont::FontSize *font_sizes) {
  ix          = familyCount++; //!!ATT: reference count
  name	      = ubit_family_name;
  natName     = native_family_name_specs;
  normalStyle = normal_style_specs;
  boldStyle   = bold_style_specs;
  italicStyle = italic_style_specs;
  if (font_sizes) fontSizes = font_sizes;
  else fontSizes = DEFAULT_FONT_SIZES;
}

// simplified API: assumes default options
UFontFamily::UFontFamily(const char *fname, const char *native_name) {
  ix          = familyCount++; //!!ATT: reference count
  name        = fname;
  natName     = native_name;
  normalStyle  = "medium";
  boldStyle   = "bold";
  italicStyle = "o";   //!! pas fini !!!
  fontSizes   = DEFAULT_FONT_SIZES;
}

const UFont::FontSize* UFontFamily::getDefaultFontSizes() {
  return DEFAULT_FONT_SIZES;
}

// get scale factor related to medium standard size
float UFontFamily::getDefaultFontScale(int logical_size) {
  return (float)DEFAULT_FONT_SIZES[logical_size-1].ptsize //att -1 SHIFT !
    / DEFAULT_FONT_SIZES[UFont::MEDIUM_LSIZE-1].ptsize;
}

// there is usually no need to call this function except if you want
// to change the default font sizes, in which case it must be called
// at initialization time; Argument is an array of MAX_LSIZE
/***  VOIR UConfig
void UFontFamily::setDefaultFontSizes(const UFont::FontSize* fs) {
  DEFAULT_FONT_SIZES = fs;
}
***/

int UFontFamily::lsizeToPtsize(int _lsize) const { // LOGICAL size 
  if (_lsize <= 0) return 0;
  else if (_lsize > UFont::MAX_LSIZE) _lsize = UFont::MAX_LSIZE;
  return fontSizes[_lsize-1].ptsize;
}

int UFontFamily::ptsizeToLsize(int _ptsize) const {
  if (_ptsize <= 0) return 0;
  int mind = 99999;
  int mink = 0;
  for (int k = 0; k < UFont::MAX_LSIZE; k++) {
    int d = _ptsize - fontSizes[k].ptsize;
    if (d < 0) d = -d; //abs
    if (d > mind) return fontSizes[mink].logsize;
    else {mind = d; mink = k;} 
  }

  return fontSizes[mink].logsize; // securite
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// *** Fonts ***

const UClass UFont::uclass("UFont");

// NOT bold nor Italic
UFont UFont::normal(UFontFamily::any,  -BOLD|-ITALIC, 0, UMode::UCONST);
UFont UFont::bold(UFontFamily::any,      BOLD,        0, UMode::UCONST);
UFont UFont::italic(UFontFamily::any,    ITALIC,      0, UMode::UCONST);
UFont UFont::underline(UFontFamily::any, UNDERLINE,   0, UMode::UCONST);

// NOT bold
UFont UFont::_bold(UFontFamily::any,     -BOLD,        0, UMode::UCONST);
// NOT italic
UFont UFont::_italic(UFontFamily::any,   -ITALIC,      0, UMode::UCONST);
// NOT underlined
UFont UFont::_underline(UFontFamily::any,-UNDERLINE,   0, UMode::UCONST);

// sizes
UFont UFont::xx_small(UFontFamily::any,  0, MEDIUM_LSIZE-3, UMode::UCONST);
UFont UFont::x_small(UFontFamily::any,   0, MEDIUM_LSIZE-2, UMode::UCONST);
UFont UFont::small(UFontFamily::any,     0, MEDIUM_LSIZE-1, UMode::UCONST);
// Default size is MEDIUM_LSIZE
UFont UFont::medium(UFontFamily::any,    0, MEDIUM_LSIZE,   UMode::UCONST);
UFont UFont::large(UFontFamily::any,     0, MEDIUM_LSIZE+1, UMode::UCONST);
UFont UFont::x_large(UFontFamily::any,   0, MEDIUM_LSIZE+2, UMode::UCONST);
UFont UFont::xx_large(UFontFamily::any,  0, MEDIUM_LSIZE+3, UMode::UCONST);

// 'inherit' means that the font is inherited from the parent box
UFont UFont::inherit(UFontFamily::standard,    0,0, UMode::UCONST);
// standard imposes the default size
UFont UFont::standard(UFontFamily::standard,   0, MEDIUM_LSIZE, UMode::UCONST);

UFont UFont::helvetica(UFontFamily::helvetica, 0,0, UMode::UCONST);
UFont UFont::times(UFontFamily::times,         0,0, UMode::UCONST);
UFont UFont::courier(UFontFamily::courier,     0,0, UMode::UCONST);
UFont UFont::fixed(UFontFamily::fixed,         0,0, UMode::UCONST);
UFont UFont::symbol(UFontFamily::symbol,       0,0, UMode::UCONST);

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// Arguments:
// - family is a previously created font family (see above)
// - style is an integer bitmask of BOLD, ITALIC, etc. 0 means default
// - lsize is the LOGICAL size: from MIN_LSIZE to MAX_LSIZE
//   * Default size is MEDIUM_LSIZE
//   * 0 means "unspecified, don't impose a specific size".
//
UFont::UFont(const UFontFamily &ff, int _styles, int _lsize, u_modes m) 
  : UProp(m) {
  family = &ff;
  setLsize(_lsize, false);		// NB: 0 means default
  setStyle(_styles, false);
}

UFont::UFont(const UFont &f, u_modes m) : UProp(m) {
  family   = f.family;
  lsize    = f.lsize;
  onStyles = f.onStyles;
  offStyles= f.offStyles;
}

UFont& ufont(const UFont &f) {
  return *(new UFont(f));
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

u_bool UFont::equals(const UFont &f) const {
  return (family == f.family
	  && lsize == f.lsize
	  && onStyles == f.onStyles
	  && offStyles == f.offStyles);
}

u_bool UFont::equals(const UFont *f) const {
  return equals(*f);
}

void UFont::set(const UFont &f, u_bool upd) {
  if (equals(f)) return;
  family    = f.family;
  lsize     = f.lsize;
  onStyles  = f.onStyles;
  offStyles = f.offStyles;
  changed(upd);
}

void UFont::set(const UFont *f, u_bool upd) {
  if (!f) error("set", "Null UFont!");
  else set(*f, upd);
}

void UFont::set(const UFontDesc *f, u_bool upd) {
  family   = f->family;
  lsize    = f->lsize;
  onStyles = f->styles;
  offStyles= 0;
  changed(upd);
}

/* ==================================================== ======== ======= */
// merge font characteristics

void UFont::merge(const UFont &f) {
  if (f.family != &UFontFamily::any)  family = f.family;
  // 0 means don't impose size
  if (f.lsize > 0) lsize = f.lsize;

  // combines styles
  onStyles = onStyles | f.onStyles;
  offStyles = offStyles | f.offStyles;
  changed(true);
}

void UFont::merge(const UFont *f) {
  if (!f) error("merge", "Null UFont!");
  else merge(*f);
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

void UFont::setFamily(const UFont &f, u_bool upd) {
  family = f.family;
  changed(upd);
}

void UFont::setFamily(const UFontFamily &ff, u_bool upd) {
  family = &ff;
  changed(upd);
}

void UFont::setStyle(const UFont &f, u_bool upd) {
  onStyles  = f.onStyles;
  offStyles = f.offStyles;
  changed(upd);
}

void UFont::setStyle(int _styles, u_bool upd) {
  if (_styles > 0) {
    onStyles  = _styles;
    offStyles = 0;
  }
  else {
    onStyles  = 0;
    offStyles = -_styles;
  }
  changed(upd);
}

void UFont::setSize(const UFont &f, u_bool upd) {
  lsize  = f.lsize;
  changed(upd);
}

// LOGICAL size NOT point size; 0 = default
void UFont::setLsize(int _lsize, u_bool upd) {
  // 0 means don't impose a specific size
  if (_lsize <= 0) _lsize = 0;
  else if (_lsize > MAX_LSIZE) _lsize = MAX_LSIZE;
  lsize = _lsize;
  changed(upd);
}

void UFont::setPtsize(int _ptsize, u_bool upd) {
  setLsize(family->ptsizeToLsize(_ptsize), upd);
}

int UFont::getLsize()  const {return lsize;}
int UFont::getPtsize() const {return family->lsizeToPtsize(lsize);}

u_bool UFont::isBold()   const {return (onStyles & BOLD) != 0;}
u_bool UFont::isItalic() const {return (onStyles & ITALIC) != 0;}

const UFontFamily* UFont::getFamily() const {return family;}

void UFont::update() {
  // size changed in both directions
  parents.updateParents(UUpdate::layout);
}

// This method initializes a UFont for a given UDisplay (or UAppli)
// It returns true if the font could be found and false otherwise
// (a default font will be used in this case)
// NOTE: it is not necessary to call this function explicitely since
// version 99.10 (it will automatically be called the first time 
// this UFont is used for drawing a string on the screen)
//
u_bool UFont::realize(UDisp *d) {
  return d->realizeFont(this);
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

void UFont::putProp(UContext *props, UCtrl *state) {
  UFontDesc &df = props->fontdesc;
  if (family != &UFontFamily::any) {
    df.family = family;
    //df.family_ix = family->ix;
  }
  // ajouter les 'onStyles' et enlever les 'offStyles'
  df.styles = (df.styles | onStyles) & ~offStyles;

  //NB: lsize = 0 means "any size" and thus, does not change df.size 
  if (lsize > 0) {
    df.lsize = lsize + props->lscale;   //NB: lscale = LOGICAL scale
    if (df.lsize < 1) df.lsize = 1;
    else if (df.lsize > MAX_LSIZE) df.lsize = MAX_LSIZE;
  }

  df.font_ix = (df.lsize << 2) + (df.styles & BOLD) + (df.styles & ITALIC);
  //printf("putProp family=%s  df.font_ix=%d maxsize=%d\n",	 family->name, df.font_ix,  UFont::FONTMAP_SIZE);
}

/* ==================================================== ======== ======= */

void UFontDesc::set(const class UFont*f) {
  family  = f->family;
  styles  = f->onStyles;
  lsize   = f->lsize;
  font_ix = (lsize << 2) + (styles & UFont::BOLD) + (styles & UFont::ITALIC);
}

void UFontDesc::set(const class UFont*f, int delta_scale) {
  family  = f->family;
  styles  = f->onStyles;
  lsize   = f->lsize + delta_scale;
  if (lsize < 1) lsize = 1;
  else if (lsize > UFont::MAX_LSIZE) lsize = UFont::MAX_LSIZE;

  font_ix = (lsize << 2) + (styles & UFont::BOLD) + (styles & UFont::ITALIC);
}

void UFontDesc::merge(const class UFont *f, int delta_scale) {
  if (f->family != &UFontFamily::any)  family = f->family;
  if (f->lsize > 0) lsize = f->lsize;
  lsize += delta_scale;

  // combines styles
  styles = styles | f->onStyles;
  //offStyles = offStyles | f.ffStyles;
  //  if (f->onStyles)  styles  = f->onStyles;

  font_ix = (lsize << 2) + (styles & UFont::BOLD) + (styles & UFont::ITALIC);
}

void UFontDesc::incrScale(int delta_scale) {
  lsize += delta_scale;
  if (lsize < 1) lsize = 1;
  else if (lsize > UFont::MAX_LSIZE) lsize = UFont::MAX_LSIZE;

  font_ix = (lsize << 2) + (styles & UFont::BOLD) + (styles & UFont::ITALIC);
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:01] ======= */
