/* ==================================================== ======== ======= *
 *
 *  ugraph.hh
 *  Ubit Project [Elc][beta1][2001]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:01] ======= *
 * ==================================================== ======== ======= */

#ifndef _ugraph_hh
#define	_ugraph_hh
#include <uappli.hh>
//pragma ident	"@(#)ugraph.hh	ubit:b1.11.5"

class UGraph {
  friend class UNatGraph;
  friend class UNatWin;
protected:
  UView *boxview;  // logical View where the graphics are currently being drawn
  class UDisp     *disp;
  class UNatGraph *natgraph;	// pointer to (hidden) Native Graphic Context
  class UNatWin   *natwin;	// pointer to (hidden) Native Window
  class UNatWin   *doublebuf;	// used when double buffering is active
  class UNatWin   *transpbuf;	// used when drawing transparent objects
  u_bool xorMode, subwin;
  short thickness;		// !!a revoir avec UPen !!

public:
  // IMPORTANT NOTICE:
  // 1. Creating a new UGraph object allocates physical resources
  //    and locks drawing on the corresponding View
  //    ==> thus, UGraph objects should be deleted as soon as possible.
  //
  // 2. The Font and the foreground and background Colors are undefined 
  //    when the UGraph is created and must be set explicitely (when needed)
  //
  // 3. The default drawing mode (called 'PaintMode') overwrites pixels.
  //    Function setXORMode() sets the 'XORMode' that alternates the current
  //    foreground and background Colors.

  UGraph(class UView*);
  UGraph();
  virtual ~UGraph();

  // ==== Related objects ==================================================

  // returns the current View (= the View where the graphics are being drawn)
  UView *getView() const {return boxview;}

  //##obs. desormais uniquement dans UWinGraph
  //UView *getWinView();
  //UWin *getWin();

  // returns the logical Display of this View
  UDisp* getDisp() const {return disp;}

  // returns the Application this View belongs to
  UAppli* getAppli() const {return disp->getAppli();}

  // Acces to implementation-dependent hidden objects:
  // NB: these "Native Interfaces" encapsulate native platform-dependant data
  //     their implementation may change or be different on various platforms
  // ==> avoid accessing these objects directly
  //
  // -- Native Window Interface
  class UNatWin  *getNatWin()  const {return natwin;}
  // -- Native Display Interface
  class UNatDisp *getNatDisp() const {return disp->getNatDisp();}
  // -- Native Graphics Interface
  class UNatGraph *getNatGraph() const {return natgraph;}

  // ==== Modes and Attributes =============================================

  // these functions set the current foreground and background Colors
  // !NOTE that these values are undefined when the UGraph is created!
  void setColor(const UColor* c);
  void setColor(const UColor& c) {setColor(&c);}

  void setBgcolor(const UColor* c);
  void setBgcolor(const UColor& c) {setBgcolor(&c);}

  // these functions set the current Font (undefined when UGraph is created)
  // - UFontDesc is an internal representation of UFont (see file ufont.hh)
  void setFont(const UFont*);
  void setFont(const class UFontDesc*);

  // this function changes the Mouse Cursor
  void setCursor(const UCursor*);

  // these functions set the drawing mode:
  // - PaintMode (the default) overwrites pixels with the foreground Color
  // - XORMode alternates the current foreground and background Colors
  void setPaintMode();
  void setXORMode();

  // Changes line thickness when drawing lines, etc.
  // - the default value 0 sets thickness to 1 pixel and optimizes drawing
  void setThickness(int = 0);

  // !!attention : ces fonctions marchent pour UWinGraph mais n'ont pas 
  // ete testee pour UGraph !
  void beginDoubleBuffering(const URegion &clip);
  void endDoubleBuffering(const URegion &clip);
  u_bool isDoubleBuffered() {return doublebuf != null;}

  // ==== Clip and Flush ==================================================

  // these functions set the clipping zone inside the current View
  // -- setClip: coordinates are relative to the origin of the VIEW
  void setClip(const URegion&);
  void setClip(u_pos x, u_pos y, u_dim width, u_dim height);

  // -- setWinClip: coordinates are relative to the origin of the WINDOW
  void setWinClip(const URegion&);
  void setWinClip(u_pos x, u_pos y, u_dim width, u_dim height);

  // this function flushes drawing requests (when applicable)
  void flush();

  // physical grab on the corresponding window (only for menus!)
  int grabPointer(class UCursor* = null, u_bool confine_to_window = false); 
  // this function ungrabs the X Server (ie. the mouse and/or the keyboard)
  void ungrabPointer();

  // ==== Coordinates ======================================================

  // width and heigh of the current View
  u_dim getWidth();
  u_dim getHeight();

  // conversion from local View coords. to global Window coords.
  u_pos XToXwin(u_pos x_in_view);
  u_pos YToYwin(u_pos y_in_view);

  // conversion from global Window coords. to local View coords.
  u_pos XwinToX(u_pos x_in_win);
  u_pos YwintoY(u_pos y_in_win);

  // ==== Clearing and Copying routines ====================================

  // -- if 'repaint_view' is true : repaints the normal content of the view
  //    in the specified region
  // -- otherwise: fills the area with the UGraph's background Color
  // --> see also: UView::updatePaint() in uview.hh for more options
  // Note: this function is not altered by the XOR mode
  void clearRect(u_pos x, u_pos y, u_dim width, u_dim height,
		 u_bool repaint_view = false);

  // same as clearRect but clears the entire view
  void clearView(u_bool repaint_view = false);

  // Copies rectangular area at location (x,y) and size (width,height)
  // to a location shifted by delta_x, delta_y
  // -- note: CopyArea can't copy obscured parts of the component
  //    but will generate refresh events form repainting the missing parts
  //    if the last arg. is true
  void copyArea(u_pos x, u_pos y, u_dim width, u_dim height,
		u_dim delta_x, u_dim delta_y,
		u_bool generate_refresh_events_when_obscured);

  // ==== Drawing routines =================================================
  // Note: these functions only use the Foreground Color in PaintMode
  //       they also use the Background Color in XORMode

  // --> SEE ALSO the Pixmap and Image sections below !

  // x,y,width,height : top/left position and size of the enclosing rectangle
  // angles are in degrees
  void drawArc(u_pos x, u_pos y, u_dim width, u_dim height, 
	       float from_angle, float arc_angle);

  void drawLine(u_pos x1, u_pos y1, u_pos x2, u_pos y2);

  void drawPoint(u_pos x, u_pos y);

  // drawPolyline draws a sequence of connected lines:
  void drawPolyline(u_pos xpoints[], u_pos ypoints[], u_card card);

  // drawPolygon draws a polygon (which is automatically closed):
  void drawPolygon(u_pos xpoints[], u_pos ypoints[], u_card card);  

  void drawRect(u_pos x, u_pos y, u_dim width, u_dim height);

  // Note: these functions also use the current Font
  void drawString(const UStr *s, u_pos x, u_pos y);
  void drawString(const char *str, int str_length, u_pos x, u_pos y);

  // ==== Filling routines =================================================
  // Note: these functions only use the Foreground Color in PaintMode
  //       they also use the Background Color in XORMode

  // x,y,width,height : top/left position and size of the enclosing rectangle
  // angles are in degrees
  void fillArc(u_pos x, u_pos y, u_dim width, u_dim height,
	       float from_angle, float arc_angle);

  // this function fills a polygon (which is automatically closed)
  // argument 'mode' can be set for optimisation purpose: 
  //   - by default, mode=0 and the polygon can have an arbitrary shape
  //   - mode=2 assumes a Convex polygon. 
  //   - mode=1 assumes a non Convex but non Intersecting polygon.
  //
  void fillPolygon(u_pos xpoints[], u_pos ypoints[], u_card card, int mode=0);

  // fills the area 
  void fillRect(u_pos x, u_pos y, u_dim width, u_dim height);
  // fills the whole View
  void fillView(); 

  // ==== Images and Pixmaps =============================================

  //nb: the following method can aslo be used for UPix objects
  void drawIma(const class UIma*, u_pos x, u_pos y);
  void drawIma(const class UNatIma*, u_pos x, u_pos y);
  void drawIma(const class UNatPix*, u_pos x, u_pos y);

  static void getImaSize(class UNatIma*, u_dim &w, u_dim &h);
  static void getPixSize(class UNatPix*, u_dim &w, u_dim &h);

  // ==== Font Metrics ====================================================

  int  getTextHeight(const class UFontDesc*);
  int  getTextWidth(const class UFontDesc*, const char *str, int len);
  void getTextSize(const class UFontDesc*, const char *str, int len, 
		   u_dim *width, u_dim *height);

  // returns the length and size of the enclosed substring
  int getSubTextSize(const class UFontDesc*, const char *s, int len, 
		     u_dim maxwidth, u_dim chw, u_dim *w, u_dim *h);

  // returns the char pos in the string and vice-versa
  int getCharPos(const class UFontDesc*, const char *s, int len, u_pos x);
  int getXPos(const class UFontDesc*, const char *s, int len, int charpos);

  // === Ubit Intrinsics ===
  class UNatWin* createAltBuffer(const URegion &r, const char* errmess);
  void deleteAltBuffer(class UNatWin *&altbuf);
  void beginTranspBuffering(const URegion &r);
  void endTranspBuffering(const URegion &r);
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// !CAUTION!
// - use the UGraph object for standard drawing do NOT use UWinGraph !
// - UWinGraph is an optimized version of UGraph for drawing directly
//   in the Window Views. Its methods use absolute Window coordinate
//   instead of relative coordinates in the current View
// - besides UWinGraph does not use the same internal GCs than UGraph

class UWinGraph : public UGraph {
  friend class UNatWin;
  friend class UView;
protected:
  class UWin* hardwin;  //the corresponding hard window
public:
  enum {OFF=0, ON=1, SHIFTED, BLENDED};

  UWinGraph(class UWin*);
  virtual ~UWinGraph();

  //NB: init() does not create the X Window (this action will be
  //is performed by UWin::realize)
  u_bool init(UDisp*, UView* window_view);

  //is the X Window created ?
  u_bool isWinRealized() const;

  // returns the Window that contains the current View
  class UWin* getHardwin() const {return hardwin;}

  int begin(const URegion&);
  void end();

  //sets a shift (for X incrusted sub hardwins)
  int beginShifted(const URegion&, u_pos xshift, u_pos yshift);

  //starts blending (for transparency)
  int beginBlended(const URegion &clip, const URegion &r_obj);

  // this method must be called with the value that was returned by 
  // one of the 3 'begin' functions when the drawing is finished
  void endAll(int begin_return_value, const URegion &r_obj);


  // setClip and setWinClip are similar for this class: 
  // -- coordinates are relative to the origin of the WINDOW
  // -- the view is no taken into account
  void setClip(const URegion&);
  void setClip(u_pos x, u_pos y, u_dim width, u_dim height);
  void setWinClip(const URegion &r);
  void setWinClip(u_pos x, u_pos y, u_dim width, u_dim height);
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
/* NOT USED in this version
class UOffScreen {
  friend class UGraph;
protected:
  u_dim width, height;
  //class UNatDisp *natdisp;
  class UDisp   *disp;
  class UNatWin *natwin;
public:
  UOffScreen(UDisp*, u_dim height, u_dim width);
  ~UOffScreen();
  u_dim getWidth()  {return width;}
  u_dim getHeight() {return height;}
};
*/

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:01] ======= */
