/* ==================================================== ======== ======= *
 *
 *  uappli.hh
 *  Ubit Project  [Elc][beta1][2001]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:01] ======= *
 * ==================================================== ======== ======= */

#ifndef _uappli_hh
#define	_uappli_hh
//pragma ident	"@(#)uappli.hh	ubit:b1.11.2"
#include <uevent.hh>

class UDisp {
  friend class UWin;
  friend class UGraph;
  friend class UWinGraph;
  friend class UNatDisp;
protected:
  class UNatDisp *natdisp;	// Native Display
  class UAppli   *appli;	// Corresponding UAppli
  UChain hardwins;		// list of hardwins that are event managed
public:
  static const int pseudoColor, trueColor;

  UDisp(class UAppli*, u_bool init);

  virtual void realize();
  virtual u_bool isRealized();

  // Searchs if Screen 'screen_no' supports this depth with this visual_class
  // -- usually 'screen_no' = 0 (except if you have several screens on your display)
  // -- 'visual_class' = UDisp::pseudoColor, UDisp::trueColor, etc.
  // -- if 'depth_hint' is not supported with this 'visual_class', the function
  //    tries to to find the closest depth that is compatible with this 'visual_class'
  //    and it returns it
  // -- returned value:
  //    depth_hint if supported, the closest supported depth otherwise
  //    and 0 if this visual_class is not supported at all by this screen

  int matchVisualProps(int visual_class, int depth_hint, int screen_no = 0);

  // Set the requested visual props if they are supported on this screen
  // -- this fct. calls matchVisualProps() first and it works in the same way
  // -- the Colormap arg. is optional and its is automatically created if == None
  // -- this fct. must be called before realizing X windows 
  //    (ie. before appli->add(), appli->realize() and mainLoop())

  int setVisualProps(int visual_class, int depth_hint, int screen_no = 0);

  // NOTE: other matchVisualProps and setVisualProps functions are available
  // through the lower-level UNatDisp API

  // depth of *this* UDisp
  int getDepth() const;

  // default depth of the Screen
  int getScreenDefaultDepth() const;

  // size of the Screen
  int getScreenWidth() const;
  int getScreenHeight() const;

  class UAppli   *getAppli()   const {return appli;}
  class UNatDisp *getNatDisp() const {return natdisp;}

  //==== Ubit Intrinsics ===================================================

  // will manage the events of this window if it is realized and shown
  void addHardwin(class UWin*);
  void removeHardwin(class UWin*);

  // returns true if OK; use default otherwise
  u_bool realizeFont(const class UFont*);
  u_bool realizeColor(class UColor*);
  u_bool realizeCursor(class UCursor*);
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UAppli: public UDisp {
  friend class UWin;
  friend class UMenu;
  friend class UCtrl;
  friend class UNatWin;
  friend class UNatDisp;
  friend class UNatAppli;
public:
  // standard constructor for creating a new application
  UAppli(int *argc, char *argv[],
	 // Style file (not used yet)
	 const char *style_file = null,
	 // X Display where the application should be opened
	 // (can also be specified by option -display in command line)
	 const char *display_name = null);

  virtual ~UAppli();

  // IMPORTANT:
  // - at least one Window (typically an UFrame) must be added to
  //   the UAppli instance
  // - the "Main Frame" is the first UFrame that is added to the UAppli
  // - the "Main Frame" is automatically opened (the 'show' method must
  //   be called to open the other children of the UAppli)

  virtual void add(class UWin*);
  virtual void add(class UWin&);

  // same args as UGroup::remove
  virtual UWin* remove(UWin &child, u_bool delete_win);
  virtual UWin* remove(UWin *child, u_bool delete_win);

  // starts the event main loop of the application
  // -- returns the status argument given as an arg. of the exit() method
  int mainLoop();

  // quits the main loop of the application if status >= 0
  void quit(int status = 0);

  // changes the title of the Main Frame (if any)
  // (the main frame is the first UFrame that is added to the UAppli)
  void setTitle(const UStr &title);

  // returns the Main Frame (if any)
  class UFrame *getMainFrame() const;

  // Timers
  // -- methods for adding timers to the application
  virtual void add(class UTimer*);
  virtual void add(class UTimer&);

  // Updates all windows
  // - updates layout then repaints
  virtual void updateAll();
  // - updates according to mode (see UUpdate class)
  virtual void updateAll(UUpdate upmode); 

  // *** UAppli variables :

  // retreives a value from a variable name
  // -- searches in the local UAppli database, then in the SHELL environment
  //    if not found and if last arg. is true (note that variables are then 
  //    copied in the local database and never searched again in the SHELL)
  const char* getVar(const char *name, u_bool get_from_shell = true);

  // adds a variable/value pair to the local variable database of the UAppli
  // -- note that 'name' and 'value' are duplicated.
  //    this fct. returns the duplicated value or null if memory is exhausted
  const char* setVar(const char *name, const char *value);

  // NOTE: a typical usage is for defining image prefixes: for instance,
  // $MY_THEME would be replaced by its actual value in expression:
  //    uima("$MY_THEME/my_image.gif")

  // the UIMA_PATH variable defines the DEFAULT location of images
  // (this variable is undefined by default)

  const char* getImaPath();
  const char* setImaPath(const char* value);
  // obsolete synomym
  const char* setPixmapPaths(const char* value);

  // creates a complete image path 
  // -- prefixes filename with UIMA_PATH value if it does not start by . / or $
  // -- expands variable prefixes (starting by $, for instance: $MY_THEME)
  char *makeImaPath(const char* filename);

  // initializes actual X data 
  // Notes: 
  // -- this function does not need to be explicitely called in the general case)
  // -- argc, argv can be null
  virtual void realize();
  virtual void realize(int *argc, char *argv[]);
  virtual u_bool isRealized();

  // get the defaut appli (as there is usually only one UAppli)
  static UAppli* getAppli()  {return defaultAppli;}

  //==== Ubit Intrinsics ===================================================

  // returns the text selection of the application
  class UTextsel& getTextsel() {return *textsel;}

  // declares that the appli owns the selection of the X Server
  // -- the UEvent must be a button event
  u_bool ownServerSelection(class UEvent&);

  // inserts the selection of the X Server in a UStr
  // -- the UEvent must be a button event
  void insertServerSelection(class UEvent&, class UStr*, int pos);

  // gets/sets current browse group
  UGroup* getBrowseGroup() {return browseGroup;}
  void setBrowseGroup (UGroup *grp) {browseGroup = grp;}

  class UMenuCtrl *getMenuCtrl() {return menuCtrl;}

  // returns the Native Appli object (implementation specific data)
  class UNatAppli *getNatAppli() const {return (UNatAppli*)natdisp;}

  // notifies the application that a view has been deleted
  void deleteNotify(class UView* deleted_view);

protected:
  static UAppli* defaultAppli;
  void *varDB;			// variable DataBase of the uappli
  char *pixmapPath, *bitmapPath;
  char *appli_name, *style_name, *display_name;
  UStyle *defaultStyle;
  class UFrame *mainFrame;  // the main frame of the appli (if any)
  class UGroup *winlist;

  static int mclickRadius, mclickDelay;
  u_card mclickCount;
  u_time mclickTime;
  u_pos  mclickX, mclickY;

  // last entered, pressed, armed, browsed object in the Display
  UCtrl *lastPressed, *lastArmed, *dragSource, *dropTarget;
  UView *lastEnteredView;
  UBox  *lastEnteredBox;
  UEvent lastEnteredEvent, lastPressedEvent;

  // current browsing group (for Menus, Lists, Choices...)
  // (null if no browsing group currently activated)
  UGroup *browseGroup;

  // current modal dialog in the Display
  // (null if no modal dialog currently activated)
  UGroup *modalDialog;

  // controls the menus and menubars of the application
  class UMenuCtrl *menuCtrl;

  // cursor being currently shown
  const class UCursor *lastCursor;

  // controls the text selection of the application
  class UTextsel *textsel;
  // the UStr where the Server selection must be inserted
  class UStr *server_selection_str;
  int server_selection_pos;

  void winMousePress(UWin *win, UEvent&);
  void winMouseRelease(UWin *win, UEvent&);
  void winMouseMove(UWin *win, UEvent&);
  void winKeyPress(UWin *win, UEvent&);
  void winKeyRelease(UWin *win, UEvent&);
  void winLeave(UWin *win, UEvent&);
  void winDestroy(UWin *win, UEvent&);

  void boxMousePress(UEvent&);
  void boxMouseRelease(UEvent&);
  void boxMouseMove(UEvent&);
  void boxKeyPress(UEvent&);
  void boxKeyRelease(UEvent&);
  inline void boxEnter(UEvent&, UBox*, int bstyle);
  inline void boxLeave(UEvent&);
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:01] ======= */
