#include "std_standardPkg.hh"
#include "Signal.hh"
#include "ImplicitSignal.hh"
#include "Variable.hh"
#include "SignalNetinfo.hh"
#include "STDTypes.hh"
#include <cmath>
#include <sstream>

using std::ostringstream;

RealType::RealType() : ScalarType() {}

RealType::RealType(const RealType& value): ScalarType(value) {
  range = value.range;
}

RealType&
RealType::operator = (const RealType& value) {
  ((ScalarType &) *this) = (const ScalarType &) value;
  range = (const RealTypeInfo &) value.range;
  return *this;
}

//new constructors for folding

RealType::RealType(ObjectBase::ObjectType objType,
		   const TypeInfo& tInfo): ScalarType(objType) {
  ASSERT ( tInfo.getKind() == REAL_INFO );
  const RealTypeInfo& rInfo = (const RealTypeInfo &) tInfo;
  
  UniversalReal initialValue(rInfo.get_left());
  
  switch(objType) {
  case ObjectBase::SIGNAL: 
    setObject( new Signal<UniversalReal>(initialValue) );
    break;
  case ObjectBase::IMPLICIT_SIGNAL:
    setObject( new ImplicitSignal<UniversalReal>(initialValue));
    break;
  case ObjectBase::VARIABLE:
    setObject( new Variable<UniversalReal>(initialValue) );
    break;
  case ObjectBase::SIGNAL_NETINFO:
    setObject( new SignalNetinfo );
    break;
  default:
    cerr << "ERROR: unknown object type" << endl;
    exit(-1);
  }
  //Copying the range Info
  range = rInfo;
} 

RealType::RealType(ObjectBase::ObjectType objType, 
		   const UniversalReal& val,
		   const TypeInfo& tInfo): 
  ScalarType(objType) {
  switch(objType) {
  case ObjectBase::SIGNAL : 
    setObject( new Signal<UniversalReal>(val) ); 
    break;
  case ObjectBase::IMPLICIT_SIGNAL :
    setObject( new ImplicitSignal<UniversalReal>(val) ); 
    break;
  case ObjectBase::VARIABLE :
    setObject( new Variable<UniversalReal>(val) ); 
    break;
  case ObjectBase::SIGNAL_NETINFO :
    setObject( new SignalNetinfo ); 
    break;
  default:
    cerr << "ERROR: unknown object type" << endl;
    exit(-1);
  }

  ASSERT ( tInfo.getKind() == REAL_INFO );
  range = (const RealTypeInfo &) tInfo;
} 

RealType::RealType(ObjectBase::ObjectType objType, 
		   const RealType& value,
		   const TypeInfo& rInfo): 
  ScalarType(objType) {
  UniversalReal val = (UniversalReal&) value.readVal();
  switch(objType) {
  case ObjectBase::SIGNAL : 
    setObject( new Signal<UniversalReal>(val) ); 
    break;
  case ObjectBase::IMPLICIT_SIGNAL :
    setObject( new ImplicitSignal<UniversalReal>(val) );
    break;
  case ObjectBase::VARIABLE :
    setObject( new Variable<UniversalReal>(val) ); 
    break;
  case ObjectBase::SIGNAL_NETINFO :
    setObject( new SignalNetinfo ); 
    break;
  default:
    cerr << "ERROR: unknown object type" << endl;
    exit(-1);
  }

  ASSERT  ( rInfo.getKind() == REAL_INFO );
  range = (const RealTypeInfo &) rInfo;
} 


RealType::RealType(const RealType& value, 
		   const TypeInfo& rInfo): ScalarType(value) { 
  range = (const RealTypeInfo &) rInfo;
}


RealType::RealType(bool alias, const RealType& actual,
		   const TypeInfo& rInfo) 
  : ScalarType(actual.getObject()->getKind(), alias) {
  setObject( actual.getObject() );
  range = (const RealTypeInfo &) rInfo;
}


Type 
RealType::get_kind() const {
  return REAL_TYPE;
}

RealType
RealType::vhdlPlus( const RValue &rhs ) const {
  UniversalReal retval( getDoubleValue() + rhs.getDoubleValue() );
  return RealType(ObjectBase::VARIABLE, retval, getTypeInfo());
}

RealType
RealType::vhdlMinus( const RValue &rhs ) const {
  UniversalReal retval( getDoubleValue() - rhs.getDoubleValue() );
  return RealType(ObjectBase::VARIABLE, retval, getTypeInfo());
}

RealType
RealType::vhdlUnaryPlus() const {
  return *this;
}

RealType 
RealType::vhdlUnaryMinus() const {
  UniversalReal retval( -1.0 * getDoubleValue() );
  return RealType( ObjectBase::VARIABLE, retval, getTypeInfo() );
}

RealType
RealType::vhdlMultiply( const RValue &rhs ) const {
  UniversalReal retval( getDoubleValue() * rhs.getDoubleValue() );
  return RealType(ObjectBase::VARIABLE, retval, getTypeInfo());
}

PhysicalType
RealType::vhdlMultiply( const PhysicalType &rhs ) const {
  UniversalLongLongInteger retval( getDoubleValue() * rhs.getInt64Value() );
  return PhysicalType(ObjectBase::VARIABLE, retval, getTypeInfo());
}

RealType
RealType::vhdlDivide( const RValue &rhs ) const {
  UniversalReal retval( getDoubleValue() / rhs.getDoubleValue() );
  return RealType(ObjectBase::VARIABLE, retval, getTypeInfo());
}

RealType
RealType::vhdlPow( const RValue &rhs ) const {
  UniversalReal retval( ::pow( getDoubleValue(), rhs.getDoubleValue() ) );
  return RealType(ObjectBase::VARIABLE, retval, getTypeInfo());
}

RealType
RealType::vhdlAbs() const {
  UniversalReal retval( fabs( getDoubleValue() ) );

  return RealType(ObjectBase::VARIABLE, retval, getTypeInfo());
}

//The attributes of RealType

RealType 
RealType::LEFT(const TypeInfo& rInfo) {
  ASSERT ( rInfo.getKind() == REAL_INFO );
  return RealType(ObjectBase::VARIABLE, ((const RealTypeInfo &) rInfo).get_left(), rInfo);
}

RealType
RealType::RIGHT(const TypeInfo& rInfo) {
  ASSERT ( rInfo.getKind() ==  REAL_INFO ); 
  return RealType(ObjectBase::VARIABLE, ((const RealTypeInfo &) rInfo).get_right(), rInfo);
}

RealType
RealType::HIGH(const TypeInfo& tInfo) {
  ASSERT ( tInfo.getKind() == REAL_INFO );
  const RealTypeInfo& rInfo = (const RealTypeInfo &) tInfo;
  
  if (rInfo.get_left() > rInfo.get_right()) {
    return RealType(ObjectBase::VARIABLE, rInfo.get_left(), tInfo);
  }
  else {
    return RealType(ObjectBase::VARIABLE, rInfo.get_right(), tInfo);
  }
}

RealType
RealType::LOW(const TypeInfo& tInfo) {
  ASSERT ( tInfo.getKind() == REAL_INFO );
  const RealTypeInfo& rInfo = (const RealTypeInfo &) tInfo;
  
  if (rInfo.get_left() > rInfo.get_right()) {
    return RealType(ObjectBase::VARIABLE, rInfo.get_right(), tInfo);
  }
  else {
    return RealType(ObjectBase::VARIABLE, rInfo.get_left(), tInfo);
  }
}

EnumerationType
RealType::ASCENDING(const TypeInfo& tInfo){
  ASSERT ( tInfo.getKind() == REAL_INFO );
  const RealTypeInfo& rInfo = (const RealTypeInfo &) tInfo;
  
  if (rInfo.get_direction() == ArrayInfo::to) {
    return SAVANT_BOOLEAN_TRUE;
  }
  else {
    return SAVANT_BOOLEAN_FALSE;
  }
}

ArrayType
RealType::IMAGE(const RealType& x){
  UniversalInteger val;
  val = x.getObject()->readVal();
  ostringstream valString;
  valString << val;
  return ArrayType(ObjectBase::VARIABLE, SavantstringType_info, -1, valString.str().c_str());
}

RealType
RealType::VALUE(const ArrayType& x, const TypeInfo& tInfo) {
  ASSERT ( tInfo.getKind() == REAL_INFO );
  const RealTypeInfo& rInfo = (const RealTypeInfo &) tInfo;
  
  int length = x.get_number_of_elements();
  ostringstream value;
  for(int i=0; i < length; i++) {
    x.get_element(i).print(value);
  }

  int endPos;
  double val = stringToDouble( value.str(), endPos );
  ASSERT( endPos > 0 );

  return RealType(ObjectBase::VARIABLE, UniversalReal(val), rInfo);
}

RealType
RealType::VAL( const RealType &x, const TypeInfo &tInfo ){
  ASSERT( tInfo.getKind() == REAL_INFO );
  
  UniversalReal val = x.getObject()->readVal();
  if((x.range.get_direction() == ArrayInfo::to && val >= x.range.get_left() &&
      val <= x.range.get_right()) || 
     (x.range.get_direction() == ArrayInfo::downto && val <= x.range.get_left() &&
      val >= x.range.get_right())){
    return RealType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
  else {
    cerr << "The Result of `VAL attribute not withing range";abort();
    return RealType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
}

RealType
RealType::POS(const RealType& x, const TypeInfo&){
  UniversalReal val = x.getObject()->readVal();
  return RealType(ObjectBase::VARIABLE, val, x.getTypeInfo());
}


const VHDLData&
RealType::leftValue() {
  static UniversalReal retValue(0.0);

  retValue = UniversalReal(range.get_left());

  return retValue;
}

TypeInfo&
RealType::getTypeInfo() const {
  return (TypeInfo &) range;
}
