#ifndef SIGNALBASE_HH
#define SIGNALBASE_HH
//---------------------------------------------------------------------------
// Copyright (c) 1995-1999 Ohio Board of Regents and the University of
// Cincinnati.  All Rights Reserved.

// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

//---------------------------------------------------------------------------

#include "VTime.hh"
#include "vhdl.hh"
#include "ArrayInfo.hh"
#include "ObjectBase.hh"
#include "DefaultAttributes.hh"
#include "Block.hh"
#include "SignalNetinfo.hh"

class VHDLKernelBase;
class SourceBase;

class SignalBase : public ObjectBase {
public:

  SigType type;      // signal type
  int id;            // global signal ID
  const char* name;  // human-readable signal name
  bool sensitive;    // does it belong to the processes sensitivity list?
  int busResFn;      // index of busResFn for this signal
  int numSources;

  // fanout list.  This contains the simulation object IDs that this
  // signal gets transmitted to.  This data should never change during a
  // simulation, so it is allocated only once, during initState, and
  // deleted after the simulation is over.  The copy constructor and
  // operator= simply copy the pointer to the new object.
  int fanout;
  FanDest_t *fanDest;
  // The attributeList is handled in the same way as the fanout list.
  int numAttributes;
  Attribute *attributeList;
  DefaultAttributes allAttributes;

  Block *downTypeConversionFnListPtr;


  // we should search for updating the Drivers.
  Block *additionalDriverList;

  SignalBase() {		// Default initialization.
    sensitive = false;
    busResFn = -1;
    fanout = 0;
    numAttributes = 0;
    numSources=-1;
    downTypeConversionFnListPtr = NULL;
    additionalDriverList = NULL;
    id = -1;
  }
  
  SignalBase(const char* newname): name(newname), numSources(-1) {
    sensitive = false;
    busResFn = -1;
    fanout = 0;
    numAttributes = 0;
    downTypeConversionFnListPtr = NULL;
    additionalDriverList = NULL;
    id = -1;
  }
  
  SignalBase(int ns): name(NULL), numSources(ns) {
    sensitive = false;
    busResFn = -1;
    fanout = 0;
    numAttributes = 0;
    downTypeConversionFnListPtr = NULL;
    additionalDriverList = NULL;
    id = -1;
  }
  
  SignalBase(char* newname, int ns) : name(newname), numSources(ns) {
    type = EXPLICIT;
    sensitive = false;
    busResFn = -1;
    fanout = 0;
    numAttributes = 0;
    downTypeConversionFnListPtr = NULL;
    additionalDriverList = NULL;
    id = -1;
  }
  
  SignalBase(const SignalBase& sb) : name(sb.name), numSources(sb.numSources) {
    type = sb.type;
    id = sb.id;
    sensitive = sb.sensitive;
    busResFn = sb.busResFn;
    fanout = sb.fanout;
    numAttributes = sb.numAttributes;
    
    downTypeConversionFnListPtr = sb.downTypeConversionFnListPtr;
    
    if ( sb.additionalDriverList != NULL ){
      additionalDriverList = new Block();
      *additionalDriverList = *(sb.additionalDriverList);
    }
  }
  
  virtual ~SignalBase() {}
  
  SignalBase& operator=(const SignalBase& sb) {
    type          = sb.type;
    id            = sb.id;
    name          = sb.name;
    sensitive     = sb.sensitive;
    busResFn      = sb.busResFn;
    fanout        = sb.fanout;
    numAttributes = sb.numAttributes;
    numSources    = sb.numSources;
    fanDest       = sb.fanDest;
    attributeList = sb.attributeList;
    allAttributes = sb.allAttributes;

    downTypeConversionFnListPtr = sb.downTypeConversionFnListPtr;
    
    if ( sb.additionalDriverList != NULL ) {
      additionalDriverList = new Block();
      *additionalDriverList = *(sb.additionalDriverList);
    }

    parentCompositeType = sb.parentCompositeType;
    
    return *this;
  }

  virtual bool _is_resolved() const {
    cerr << "SignalBase::_is_resolved() calle.  Aborting." << endl;
    abort();
    return false;
  }

  bool _is_signal() const {
    return true;
  }
  
  void setBusKind() { 
    type = G_BUS;
  }

  virtual void print(ostream& os = cout) const {
    os << "SignalBase " << name << " id " << id;
    os.flush();
  }
  
  virtual ArrayInfo* readArrayInfo() {
    cerr << name << " readArrayInfo ERROR!" << endl;
    return NULL;
  }
  
  virtual void updateDriver(const int, const int, VHDLData*, const ArrayInfo*,
			    const ArrayInfo* = NULL) { 
    cerr << name << " updateDriver ERROR!" << endl;
  }

  virtual void disconnectDriver(const int) { 
    cerr << name << " disconnectDriver ERROR!" << endl;
  }
  
  virtual VHDLType* readDriver(int) { 
    cerr << name << " readDriver ERROR!" << endl;
    return NULL;
  }
  
  virtual void updateDrvVal(const VHDLData*, const ArrayInfo* = NULL,
			    const ArrayInfo* = NULL) { 
    cerr << name << " updateDrvVal ERROR!" << endl;
  }
  
  virtual VHDLData* readDrvVal() { 
    cerr << name << " readDrvVal ERROR!" << endl;
    return NULL;
  }
  
  virtual void updateEffVal(const VHDLData&, const ArrayInfo* = NULL,
			    const ArrayInfo* = NULL) { 
    cerr << name << " updateEffVal ERROR!" << endl;
  }
  
  virtual void updateEffVal(const VHDLData*, const ArrayInfo* = NULL,
			    const ArrayInfo* = NULL) { 
    cerr << name << " updateEffVal ERROR!" << endl;
  }

  virtual VHDLData* readEffVal() { 
    cerr << name << " readEffVal ERROR!" << endl;
    return NULL;
  }
  
  virtual VTime& readTimeVal() {
    cerr << name << " readTimeVal ERROR!" << endl;
    // this is bogus, but it should never be called
    VTime* dummyTime = new VTime(PINFINITY);
    return (*dummyTime);
  }
  
  virtual VTime& updateTimeVal(const VTime& v = PINFINITY) {
    cerr << name << " updateTimeVal ERROR!" << endl;
    return (VTime &) v;
  }

  virtual void setSource(SourceBase*) {
   cerr << "ERROR: SignalBase::setSource(SourceBase *) called" << endl;
   abort();
  } 

  void setElaborationInfo(ObjectBase& obj){
    ASSERT (obj.getKind() == SIGNAL_NETINFO);
    SignalNetinfo &temp_obj = (SignalNetinfo &) obj;
    id = temp_obj.id;
    setSource(temp_obj.source);
    downTypeConversionFnListPtr = &(temp_obj.downTypeConversionFnList);
    sensitive = true;
    fanout = temp_obj.fanout;
    fanDest = temp_obj.obj_ids;
    //copy also the additionalDriverList
    if ( temp_obj.getAdditionalDriverList() != NULL ){
      additionalDriverList = new Block();
      *additionalDriverList = *(temp_obj.getAdditionalDriverList());
    }
  }

  void setNumAttributes(int noAttribs) {
    numAttributes = noAttribs;
    attributeList = new Attribute[noAttribs];
    for (int index = 0; index < noAttribs; index++) {
      attributeList[index].attrib = INVALID;
      attributeList[index].value = NULL;
    }
  }
  
  void setAttrib(AttribType typ, VHDLType& attrib);

  // resolve is called only for explicit signals.  For any other derived
  // class of SignalBase, this empty function will be called and no action
  // will be performed.
  virtual VHDLType* resolve(VHDLKernelBase*) { return NULL; };

  virtual bool isCompositeResolvedSignal() {
    cerr << "ERROR: SignalBase::isCompositeResolvedSignal() called " << endl;
    return false;
  }

  virtual SignalBase* findSigInBlock(int sigId, int srcId);
  virtual void set_sourcebase_delete_flag(bool flag);

  virtual SourceBase* getSource();
};

#endif
