//---------------------------------------------------------------------------
// Copyright (c) 1995-1999 Ohio Board of Regents and the University of
// Cincinnati.  All Rights Reserved.

// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

//---------------------------------------------------------------------------

#ifndef STDTYPES_HH
#define STDTYPES_HH

#include <iostream.h>
#include <string.h>

#include "VHDLData.hh"
#include "bool.hh"
#include "Types.hh"

#define FILE_DELIMITER '|'

template<class Type> class AccessVariable;
class SavantlineType;

// Some of the Linux machines do not have these predefined.
#ifndef LLONG_MIN
#define LLONG_MIN       (-9223372036854775807LL-1LL)
#endif

#ifndef LLONG_MAX
#define LLONG_MAX       9223372036854775807LL
#endif

void extractToDelimiter(char *source, int delimiter, char *destination);

enum SeverityLevel_t { NOTE, WARNING, ERROR, FAILURE };
enum FileOpenKind_t { READ_MODE, WRITE_MODE, APPEND_MODE };
enum FileOpenStatus_t { OPEN_OK, STATUS_ERROR, NAME_ERROR, MODE_ERROR };
enum Side_t { LEFT, RIGHT };
enum TimeUnit_t { FS, PS, NS, US, MS, SECONDS, MIN, HR};

struct UniversalBoolean : public VHDLData {
  friend inline ostream& operator<<(ostream&, const UniversalBoolean&);

  bool val;

  virtual ~UniversalBoolean() {};

  virtual VHDLData::UniversalType get_universal_kind() const{
    return UNIVERSAL_BOOLEAN;
  }

  UniversalBoolean() { 
    val = false; 
  }

  UniversalBoolean(bool b) { 
    val = b;
  }

  UniversalBoolean(int value) {
    if(value == 0) {
      this->val = false;
    }
    else {
      this->val = true;
    }
  }

  UniversalBoolean(char c) {
    switch(c) {
    case '0':
      val = false;
      break;
    case '1':
      val = true;
      break;
    default:
      cerr << "Wrong value passed" << endl;
      abort();
      break;
    }
  }

  UniversalBoolean(double value) {
    val = false;
    value = value;
    cerr << "Wrong type passed" << endl;
    abort();
   }

  UniversalBoolean(const UniversalBoolean& b) { 
    val = b.val;
  }
  
  UniversalBoolean(const VHDLData& v);

  UniversalBoolean& operator=(bool b) {
    val = b;
    return *this;
  }

  VHDLData& operator=(const VHDLData& d) {
    val = ((UniversalBoolean*)&d)->val;
    return *this;
  }

  UniversalBoolean& operator=(const UniversalBoolean& d) {
    val = d.val;
    return *this;
  }
  bool operator!=(const VHDLData& d) const {
    UniversalBoolean b = *(UniversalBoolean*)&d;
    return INT_TO_BOOL(val != b.val);
  }

  operator bool() { return val; }

  UniversalBoolean assign(char ch) { // Useful in Vector.
    val = UniversalBoolean(ch);
    return val;
  }

  int savantwrite(AccessVariable <char*> &) const;
  int savantwrite(SavantlineType &) const;
  int savantread(AccessVariable <char*> &);
  int savantread(SavantlineType &);
  int savantread(char *) {
    return NORMAL_RETURN;
  }
  
  int savantwrite(ostrstream &os) const {
    os << this->val;
    return NORMAL_RETURN;
  }
  VHDLData* clone() const {
    return new UniversalBoolean(*this);
  }
  
  int getSize() const { return sizeof(*this); }
  void print(ostream& os = cout) const { os << *this; os.flush();}
};

extern const UniversalInteger UNIVERSAL_TRUE;
extern const UniversalInteger UNIVERSAL_FALSE;

extern bool savantAnd(const UniversalBoolean& , const UniversalBoolean& );
extern bool savantOr(const UniversalBoolean& , const UniversalBoolean& );
extern bool savantNand(const UniversalBoolean& , const UniversalBoolean& );
extern bool savantNor(const UniversalBoolean& , const UniversalBoolean& );
extern bool savantXor(const UniversalBoolean& , const UniversalBoolean& );
extern bool savantXnor(const UniversalBoolean& , const UniversalBoolean& );
extern bool savantNot(const UniversalBoolean& );
extern bool savantEqual(const UniversalBoolean&, const UniversalBoolean&);
extern bool savantNotEqual(const UniversalBoolean&, const UniversalBoolean&);
extern bool savantLessThan(const UniversalBoolean&, const UniversalBoolean&);
extern bool savantLessThanOrEqual(const UniversalBoolean&, const UniversalBoolean&);
extern bool savantGreaterThan(const UniversalBoolean&, const UniversalBoolean&);
extern bool savantGreaterThanOrEqual(const UniversalBoolean&, const UniversalBoolean&);


struct UniversalCharacter : public VHDLData {
  friend inline ostream& operator<<(ostream&, const UniversalCharacter&);

  char val;

  virtual VHDLData::UniversalType get_universal_kind() const{
    return UNIVERSAL_CHARACTER;
  }

  UniversalCharacter() { val = '\0'; }
  UniversalCharacter(char c) { val = c;}

  UniversalCharacter(bool b) {
    if(b == true) {
      val = '1';
    }
    else {
      val = '0';
    }
  }

  UniversalCharacter(int value) {
    val = value % 256;
  }

  UniversalCharacter(double value) {
    value = value;
    cerr << "wrong type passed " << endl;
    abort();
  }

  UniversalCharacter(const UniversalCharacter& c) { val = c.val; }
  UniversalCharacter(const VHDLData& v);

  bool operator!=(const VHDLData& d) const {
    UniversalCharacter c = *(UniversalCharacter*)&d;
    if (val == c.val) return false;
    return true;
  }

  UniversalCharacter& operator=(const UniversalCharacter& c) {
    val = c.val;
    return *this;
  }

  VHDLData& operator=(const VHDLData& c) {
    val = (*(UniversalCharacter*)&c).val;
    return *this;
  }

  operator char() { return val;}

  UniversalCharacter assign(char ch) { // Useful in Vector.
    val = UniversalCharacter(ch);
    return val;
  }

  int savantwrite(AccessVariable <char*> &) const;
  int savantwrite(SavantlineType &) const;
  int savantread(AccessVariable <char*> &);
  int savantread(SavantlineType &);
  int savantread(char *) {
    return NORMAL_RETURN;
  }
  int savantwrite(ostrstream &os) const {
    os << this->val;
    return NORMAL_RETURN;
  }
  
  VHDLData* clone() const {
    return new UniversalCharacter(*this);
  }
  
  int getSize() const { return sizeof(*this); }
  void print(ostream& os = cout) const { os << *this; os.flush(); }
};

extern bool savantEqual(const UniversalCharacter&, const UniversalCharacter&);
extern bool savantNotEqual(const UniversalCharacter&, const UniversalCharacter&);
extern bool savantLessThan(const UniversalCharacter&, const UniversalCharacter&);
extern bool savantLessThanOrEqual(const UniversalCharacter&, const UniversalCharacter&);
extern bool savantGreaterThan(const UniversalCharacter&, const UniversalCharacter&);
extern bool savantGreaterThanOrEqual(const UniversalCharacter&, const UniversalCharacter&);

struct _savant_severity_level : public VHDLData {
  friend inline ostream& operator<<(ostream&, const _savant_severity_level&);
  
  SeverityLevel_t val;

  virtual VHDLData::UniversalType get_universal_kind() const{
    return SEVERITY_LEVEL;
  }

  _savant_severity_level() { val = NOTE;}
  _savant_severity_level(SeverityLevel_t s) { val = s;}
  _savant_severity_level(const _savant_severity_level& s) { val = s.val;}
//   operator bool() { return val; }
  
  bool operator!=(const VHDLData& d) const {
    _savant_severity_level c = *(_savant_severity_level *)&d;
    return (INT_TO_BOOL(val != c.val));
  }

  int getSize() const { return sizeof(*this);}
  void print(ostream& os = cout) const { os << *this; os.flush();}


};
extern bool savantEqual(const _savant_severity_level&, const _savant_severity_level&);
extern bool savantNotEqual(const _savant_severity_level&, const _savant_severity_level&);
extern bool savantLessThan(const _savant_severity_level&, const _savant_severity_level&);
extern bool savantLessThanOrEqual(const _savant_severity_level&, const _savant_severity_level&);
extern bool savantGreaterThan(const _savant_severity_level&, const _savant_severity_level&);
extern bool savantGreaterThanOrEqual(const _savant_severity_level&, const _savant_severity_level&);

struct UniversalReal;
struct UniversalLongLongInteger;

struct UniversalInteger : public VHDLData {
  friend inline ostream& operator<<(ostream&, const UniversalInteger&);

  int val;
  bool overflow_flag;
  
  static const UniversalInteger MAX;
  static const UniversalInteger MIN;

  virtual VHDLData::UniversalType get_universal_kind() const{
    return UNIVERSAL_INTEGER;
  }

  UniversalInteger() { val = 0; overflow_flag = false; }
  UniversalInteger(int i) { val = i; overflow_flag = false; }
  UniversalInteger(LONG i) { val = i; overflow_flag = false; }
  UniversalInteger(char c) { val = c - '0'; overflow_flag = false; }
  UniversalInteger(bool b) {val = (true == b ? 1 : 0); overflow_flag = false; }
  UniversalInteger(double f) {val = (LONG) f; overflow_flag = false; }

  UniversalInteger(const VHDLData& v);

//   UniversalInteger(UniversalBoolean& value) {
//     val = (int) value;
//   }

  UniversalInteger(const UniversalInteger& i) {
    val = i.val;
    overflow_flag = false;
  }
  ~UniversalInteger(){}
  UniversalInteger(const UniversalLongLongInteger& i);
  UniversalInteger(const UniversalReal&);

  VHDLData& operator=(const VHDLData &);
  UniversalInteger& operator=(const UniversalInteger &);
  bool operator!=(const VHDLData& d) const;

  operator int() const { return val; }

  UniversalInteger assign(char ch) { // Useful in Vector.
    val = UniversalInteger(ch - '0');
    return val;
  }

  UniversalInteger& increment(void) {
    int oldval = val;
    val++;

    if (val < oldval) {
      overflow_flag = true;
    }
    else {
      overflow_flag = false;
    }

    return *this;
  }

  UniversalInteger& decrement(void) {
    int oldval = val;
    val--;

    if (val > oldval) {
      overflow_flag = true;
    }
    else {
      overflow_flag = false;
    }
    
    return *this;
  }

  bool overflow() const { return overflow_flag; }
  
  int savantwrite(AccessVariable <char*> &) const;
  int savantwrite(SavantlineType &) const;
  int savantread(AccessVariable <char*> &);
  int savantread(SavantlineType &);
  int savantread(char *) {
    return NORMAL_RETURN;
  }
  int savantwrite(ostrstream &os) const {
    os << this->val;
    return NORMAL_RETURN;
  }
  
  VHDLData* clone() const {
    UniversalInteger *tempInt = new UniversalInteger(*this);
    *tempInt = *this;
    return tempInt;
  }
  
  int getSize() const { return sizeof(*this); }
  void print(ostream& os = cout) const { os << *this; os.flush();}

  static UniversalInteger typeCast(VHDLData& toCast);
};


extern bool savantEqual(const UniversalInteger&, const UniversalInteger&);
extern bool savantNotEqual(const UniversalInteger&, const UniversalInteger&);
extern bool savantLessThan(const UniversalInteger&, const UniversalInteger&);
extern bool savantLessThanOrEqual(const UniversalInteger&, const UniversalInteger&);
extern bool savantGreaterThan(const UniversalInteger&, const UniversalInteger&);
extern bool savantGreaterThanOrEqual(const UniversalInteger&, const UniversalInteger&);
extern UniversalInteger savantUnaryPlus(const UniversalInteger&);
extern UniversalInteger savantUnaryMinus(const UniversalInteger&);
extern UniversalInteger savantAbs(const UniversalInteger&);
extern UniversalInteger savantPlus(const UniversalInteger&,const UniversalInteger&);
extern UniversalInteger savantMinus(const UniversalInteger&,const UniversalInteger&);
extern UniversalInteger savantMultiply(const UniversalInteger&,const UniversalInteger&);
extern UniversalInteger savantDivide(const UniversalInteger&,const UniversalInteger&);
extern UniversalInteger savantMod(const UniversalInteger&,const UniversalInteger&);
extern UniversalInteger savantRem(const UniversalInteger&,const UniversalInteger&);
extern UniversalInteger savantPow(const UniversalInteger&,const UniversalInteger&);

extern const UniversalInteger savantAnd(const UniversalInteger& , const UniversalInteger& );
extern const UniversalInteger savantOr(const UniversalInteger& , const UniversalInteger& );
extern const UniversalInteger savantNand(const UniversalInteger& , const UniversalInteger& );
extern const UniversalInteger savantNor(const UniversalInteger& , const UniversalInteger& );
extern const UniversalInteger savantXor(const UniversalInteger& , const UniversalInteger& );
extern const UniversalInteger savantXnor(const UniversalInteger& , const UniversalInteger& );
extern const UniversalInteger savantNot(const UniversalInteger&);


typedef UniversalInteger savantBit;




struct UniversalLongLongInteger : public VHDLData {
  friend inline ostream& operator<<(ostream&, const UniversalLongLongInteger&);

  LONG val;

  static const UniversalLongLongInteger MAX;
  static const UniversalLongLongInteger MIN;

  virtual VHDLData::UniversalType get_universal_kind() const{
    return UNIVERSAL_LONG_LONG_INTEGER;
  }

  UniversalLongLongInteger() { val = 0; }
  UniversalLongLongInteger(int i) { val = (LONG) i;}
  UniversalLongLongInteger(UniversalInteger UI) { val = (LONG) UI.val;}
  UniversalLongLongInteger(LONG i) { val = i;}
  UniversalLongLongInteger(char c) { val = (LONG) c - '0';}
  UniversalLongLongInteger(bool b) {val = (true == b ? 1LL : 0LL); }
  UniversalLongLongInteger(double f) {val = (LONG) f;}
  UniversalLongLongInteger(float f) {val = (LONG) f;}

  UniversalLongLongInteger(const UniversalLongLongInteger& i) { val = i.val; }
  UniversalLongLongInteger(const UniversalReal&);
  UniversalLongLongInteger(const VHDLData& v);

  VHDLData& operator=(const VHDLData& d) {
    val = ((UniversalLongLongInteger*)&d)->val;
    return *this;
  }
  UniversalLongLongInteger& operator=(const UniversalLongLongInteger& i) {
    val = i.val;
    return *this;
  }
  
  UniversalLongLongInteger& operator=(const LONG value) {
    val = value;
    return *this;
  }
  
  bool operator!=(const VHDLData& d) const {
    UniversalLongLongInteger i = *(UniversalLongLongInteger*)&d;
    return INT_TO_BOOL(val != i.val);
  }

  bool operator==(const UniversalLongLongInteger& d) const {
    return INT_TO_BOOL(val == d.val);
  }

  UniversalLongLongInteger assign(char ch) { // Useful in Vector.
    val = (ch - '0');
    return val;
  }

  operator int() const { return int(val); }

  int savantwrite(AccessVariable <char*> &) const;
  int savantwrite(SavantlineType &) const;
  int savantread(AccessVariable <char*> &);
  int savantread(SavantlineType &);
  int savantread(char *) {
    return NORMAL_RETURN;
  }
  int savantwrite(ostrstream &os) const {
    os << this->val;
    return NORMAL_RETURN;
  }
  
  VHDLData* clone() const {
    return new UniversalLongLongInteger(*this);
  }
  
  int getSize() const { return sizeof(*this); }
  void print(ostream& os = cout) const { os << *this; os.flush();}

  static UniversalLongLongInteger typeCast(VHDLData& toCast);

  friend inline LONG longlongint(UniversalLongLongInteger var) { return var.val; }
};


extern bool savantEqual(const UniversalLongLongInteger&, const UniversalLongLongInteger&);
extern bool savantNotEqual(const UniversalLongLongInteger&, const UniversalLongLongInteger&);
extern bool savantLessThan(const UniversalLongLongInteger&, const UniversalLongLongInteger&);
extern bool savantLessThanOrEqual(const UniversalLongLongInteger&, const UniversalLongLongInteger&);
extern bool savantGreaterThan(const UniversalLongLongInteger&, const UniversalLongLongInteger&);
extern bool savantGreaterThanOrEqual(const UniversalLongLongInteger&, const UniversalLongLongInteger&);
extern UniversalLongLongInteger savantUnaryPlus(const UniversalLongLongInteger&);
extern UniversalLongLongInteger savantUnaryMinus(const UniversalLongLongInteger&);
extern UniversalLongLongInteger savantAbs(const UniversalLongLongInteger&);
extern UniversalLongLongInteger savantPlus(const UniversalLongLongInteger&,const UniversalLongLongInteger&);
extern UniversalLongLongInteger savantMinus(const UniversalLongLongInteger&,const UniversalLongLongInteger&);
extern UniversalLongLongInteger savantMultiply(const UniversalLongLongInteger&,const UniversalLongLongInteger&);
extern UniversalLongLongInteger savantMultiply(const UniversalLongLongInteger&, const UniversalReal&);
extern UniversalLongLongInteger savantMultiply(const UniversalLongLongInteger&, const UniversalInteger&);
extern UniversalLongLongInteger savantDivide(const UniversalLongLongInteger&,const UniversalLongLongInteger&);
extern UniversalLongLongInteger savantDivide(const UniversalLongLongInteger&,const UniversalInteger&);
extern UniversalLongLongInteger savantDivide(const UniversalLongLongInteger&,const UniversalReal&);
extern UniversalLongLongInteger savantMod(const UniversalLongLongInteger&,const UniversalLongLongInteger&);
extern UniversalLongLongInteger savantRem(const UniversalLongLongInteger&,const UniversalLongLongInteger&);
extern UniversalLongLongInteger savantPow(const UniversalLongLongInteger&,const UniversalLongLongInteger&);

extern const UniversalLongLongInteger savantAnd(const UniversalLongLongInteger& , const UniversalLongLongInteger& );
extern const UniversalLongLongInteger savantOr(const UniversalLongLongInteger& , const UniversalLongLongInteger& );
extern const UniversalLongLongInteger savantNand(const UniversalLongLongInteger& , const UniversalLongLongInteger& );
extern const UniversalLongLongInteger savantNor(const UniversalLongLongInteger& , const UniversalLongLongInteger& );
extern const UniversalLongLongInteger savantXor(const UniversalLongLongInteger& , const UniversalLongLongInteger& );
extern const UniversalLongLongInteger savantXnor(const UniversalLongLongInteger& , const UniversalLongLongInteger& );
extern const UniversalLongLongInteger savantNot(const UniversalLongLongInteger&);


struct UniversalReal : public VHDLData {
  friend inline ostream& operator<<(ostream&, const UniversalReal&);

  double val;

  static const UniversalReal MAX;
  static const UniversalReal MIN;

  virtual VHDLData::UniversalType get_universal_kind() const{
    return UNIVERSAL_REAL;
  }

  UniversalReal() { val = 0.0; }
  UniversalReal(double d) { val = d;}
  UniversalReal(float f) { val = (double) f;}
  UniversalReal(const UniversalReal& d) { val = (double) d.val; }
  UniversalReal(int value) { val = value;}
  UniversalReal(char c) { val = c; }
  UniversalReal(bool b) { val = b; }
  UniversalReal(const VHDLData& v);

  VHDLData& operator=(const VHDLData& from) {
    val = (double) ((UniversalReal *) &from)->val;
    return *this;
  }

  UniversalReal& operator=(const UniversalReal& from) {
    val = (double) from.val;
    return *this;
  }

  bool operator!=(const VHDLData& d) const {
    UniversalReal r = *(UniversalReal*)&d;
    return INT_TO_BOOL(val != r.val);
  }

  UniversalReal& operator=(double from) {
    val = from;
    return *this;
  }
  
  operator double() const { return val; }
  
  UniversalReal assign(char ch) { // Useful in Vector.
    val = UniversalReal(ch - '0');
    return val;
  }
  
  int savantwrite(AccessVariable <char*> &) const;
  int savantwrite(SavantlineType &) const;
  int savantread(AccessVariable <char*> &);
  int savantread(SavantlineType &);
  int savantread(char *) {
    return NORMAL_RETURN;
  }
  int savantwrite(ostrstream &os) const {
    os << this->val;
    return NORMAL_RETURN;
  }
  
  VHDLData* clone() const {
    return new UniversalReal(*this);
  }
  
  int getSize() const { return sizeof(*this); }
  void print(ostream& os = cout) const { os << *this; os.flush();}

  static UniversalReal typeCast(VHDLData& toCast);
};
extern bool savantEqual(const UniversalReal&, const UniversalReal&);
extern bool savantNotEqual(const UniversalReal&, const UniversalReal&);
extern bool savantLessThan(const UniversalReal&, const UniversalReal&);
extern bool savantLessThanOrEqual(const UniversalReal&, const UniversalReal&);
extern bool savantGreaterThan(const UniversalReal&, const UniversalReal&);
extern bool savantGreaterThanOrEqual(const UniversalReal&, const UniversalReal&);
extern UniversalReal savantUnaryPlus(const UniversalReal&);
extern UniversalReal savantUnaryMinus(const UniversalReal&);
extern UniversalReal savantAbs(const UniversalReal&);
extern UniversalReal savantPlus(const UniversalReal&,const UniversalReal&);
extern UniversalReal savantMinus(const UniversalReal&,const UniversalReal&);
extern UniversalReal savantMultiply(const UniversalReal&,const UniversalReal&);
extern UniversalReal savantDivide(const UniversalReal&,const UniversalReal&);
extern UniversalReal savantPow(const UniversalReal&,const UniversalInteger&);


struct _savant_file_open_kind: public VHDLData {
  friend inline ostream& operator<<(ostream&, const _savant_file_open_kind&);
  
  FileOpenKind_t val;

  virtual VHDLData::UniversalType get_universal_kind() const{
    return FILE_OPEN_KIND;
  }

  _savant_file_open_kind() { val = READ_MODE; }
  _savant_file_open_kind(const _savant_file_open_kind& s) { val = s.val; }
  _savant_file_open_kind(FileOpenKind_t s) { val = s; }
  operator bool() { return INT_TO_BOOL(val); }

  bool operator!=(const VHDLData& d) const {
    _savant_file_open_kind c = *(_savant_file_open_kind *)&d;
    return INT_TO_BOOL(val != c.val);
  }

  VHDLData& operator=(const VHDLData& d) {
    val = ((_savant_file_open_kind*)&d)->val;
    return *this;
  }
  
  _savant_file_open_kind& operator=(const _savant_file_open_kind& d) {
    val = d.val;
    return *this;
  }
  
  int getSize() const { return sizeof(*this);}
  void print(ostream& os = cout) const { os <<*this; os.flush();}

};

extern bool savantEqual(const _savant_file_open_kind&, const _savant_file_open_kind&);
extern bool savantNotEqual(const _savant_file_open_kind&, const _savant_file_open_kind&);
extern bool savantLessThan(const _savant_file_open_kind&, const _savant_file_open_kind&);
extern bool savantLessThanOrEqual(const _savant_file_open_kind&, const _savant_file_open_kind&);
extern bool savantGreaterThan(const _savant_file_open_kind&, const _savant_file_open_kind&);
extern bool savantGreaterThanOrEqual(const _savant_file_open_kind&, const _savant_file_open_kind&);


struct _savant_file_open_status: public VHDLData {
  friend inline ostream& operator<<(ostream&, const _savant_file_open_status&);
  
  FileOpenStatus_t val;

  virtual VHDLData::UniversalType get_universal_kind() const{
    return FILE_OPEN_STATUS;
  }

  _savant_file_open_status() { val = OPEN_OK; }
  _savant_file_open_status(const _savant_file_open_status& s) { val = s.val; }
  _savant_file_open_status(FileOpenStatus_t s) { val =s; }
  operator bool() { return INT_TO_BOOL(val); }

  bool operator!=(const VHDLData& d) const {
    _savant_file_open_status c = *(_savant_file_open_status *)&d;
    return INT_TO_BOOL(val != c.val);
  }

  int getSize() const { return sizeof(*this);}
  void print(ostream& os = cout) const { os <<*this; os.flush();}

};

extern bool savantEqual(const _savant_file_open_status&, const _savant_file_open_status&);
extern bool savantNotEqual(const _savant_file_open_status&, const _savant_file_open_status&);
extern bool savantLessThan(const _savant_file_open_status&, const _savant_file_open_status&);
extern bool savantLessThanOrEqual(const _savant_file_open_status&, const _savant_file_open_status&);
extern bool savantGreaterThan(const _savant_file_open_status&, const _savant_file_open_status&);
extern bool savantGreaterThanOrEqual(const _savant_file_open_status&, const _savant_file_open_status&);
  

inline ostream& operator<<(ostream& os, const _savant_severity_level& sl) {
  switch (sl.val) {
  case NOTE:
    os << "NOTE";
    break;
  case WARNING:
    os << "WARNING";
    break;
  case ERROR:
    os << "ERROR";
    break;
  case FAILURE:
    os << "FAILURE";
    break;
  default:
    os << "(invalid SeverityLevel type)";
    break;
  }
  return os;
}


inline ostream& operator<<(ostream& os, const _savant_file_open_kind& fok) {
  switch (fok.val) {
  case READ_MODE:
    os << "READ_MODE";
    break;
  case WRITE_MODE:
    os << "WRITE_MODE";
    break;
  case APPEND_MODE:
    os << "APPEND_MODE";
    break;
  default:
    os << "(invalid FileOpenKind type)";
    break;
  }
  return os;
}


inline ostream& operator<<(ostream& os, const _savant_file_open_status& fos) {
  switch (fos.val) {
  case OPEN_OK:
    os << "OPEN_OK";
    break;
  case STATUS_ERROR:
    os << "STATUS_ERROR";
    break;
  case NAME_ERROR:
    os << "NAME_ERROR";
    break;
  case MODE_ERROR:
    os << "MODE_ERROR";
    break;
  default:
    os << "(invalid FileOpenStatus type)";
    break;
  }
  return os;
}


inline ostream& operator<<(ostream& os, const Side_t& s) {
  switch (s) {
  case LEFT:
    os << "LEFT";
    break;
  case RIGHT:
    os << "RIGHT";
    break;
  default:
    os << "(invalid Side type)";
    break;
  }
  return os;
}


inline ostream& operator<<(ostream& os, const UniversalBoolean& b) {
  switch (b.val) {
  case false:
    os << "FALSE";
    break;
  case true:
    os << "TRUE";
    break;
  default:
    os << "(invalid UniversalBoolean type)";
    break;
  }
  return os;
}


inline ostream& operator<<(ostream& os, const UniversalInteger& i) {
  os << i.val;
  return os;
}

inline ostream& operator<<(ostream& os, const UniversalLongLongInteger& i) {
  os << i.val;
  return os;
}

inline ostream& operator<<(ostream& os, const UniversalReal& d) {
  os << d.val;
  return os;
}

inline ostream& operator<<(ostream& os, const UniversalCharacter& c) {
  os << c.val;
  return os;
}

inline bool savantAnd(bool lhs, bool rhs) {
  return INT_TO_BOOL(lhs && rhs);
}

inline bool savantOr(bool lhs, bool rhs) {
  return INT_TO_BOOL(lhs || rhs);
}

inline bool savantNand(bool lhs, bool rhs) {
  return INT_TO_BOOL(!(lhs && rhs));
}

inline bool savantNor(bool lhs, bool rhs) {
  return INT_TO_BOOL(!(lhs || rhs));
}

inline bool savantXor(bool lhs, bool rhs) {
  return INT_TO_BOOL((lhs && !rhs) || (!lhs && rhs));
}

inline bool savantXnor(bool lhs, bool rhs) {
  return INT_TO_BOOL(!((lhs && !rhs) || (!lhs && rhs)));
}

inline bool savantNot(bool rhs) {
  return INT_TO_BOOL(!rhs);
}

#endif
