//-*-c++-*-

#include <list>
#include <cmath>
#include <typeinfo>
#include <string>

#if (__GNUC__ < 3)
#include <hash_map>
#else
#include <ext/hash_map>
#endif

#include <time.h>

#include <GL/glu.h>

#include <tulip/LayoutProxy.h>
#include <tulip/MetricProxy.h>
#include <tulip/StringProxy.h>
#include <tulip/SelectionProxy.h>
#include <tulip/SizesProxy.h>
#include <tulip/IntProxy.h>
#include <tulip/ColorsProxy.h>
#include <tulip/MetaGraphProxy.h>
#include <tulip/SuperGraph.h>
#include <tulip/TemplateFactory.h>
#include <tulip/DrawingTools.h>

#include "tulip/GlGraph.h"
#include "tulip/Color.h"
#include "tulip/Coord.h"
#include "tulip/Size.h"
#include "tulip/EpsFunction.h"
#include "tulip/TlpTools.h"
#include "tulip/Glyph.h"
#include "tulip/PluginLoader.h"
#include "tulip/TextRenderer.h"

using namespace std;


//test 
#include <sys/timeb.h>
//====================================================
#ifdef _WIN32 
#ifdef DLL_EXPORT
TemplateFactory<GlyphFactory,Glyph,GlyphContext *> *GlGraph::glyphFactory;
stdext::hash_map<int,std::string>  GlGraph::idToName;
stdext::hash_map<std::string, int> GlGraph::nameToId;
#endif
#else
TemplateFactory<GlyphFactory,Glyph,GlyphContext *> *GlGraph::glyphFactory;
stdext::hash_map<int,std::string>  GlGraph::idToName;
stdext::hash_map<std::string, int> GlGraph::nameToId;
#endif
//====================================================
std::string GlGraph::glyphName(int id) {
  if (idToName.find(id)!=idToName.end())
    return idToName[id];
  else {
    cerr << __PRETTY_FUNCTION__ << endl;
    cerr << "Invalid glyph id" << endl;
    return string("invalid");
  }
}
//====================================================
int GlGraph::glyphId(std::string name) {
  if (nameToId.find(name)!=nameToId.end())
    return nameToId[name];
  else {
    cerr << __PRETTY_FUNCTION__ << endl;
    cerr << "Invalid glyph name" << endl;
    return 0;
  }
}
//====================================================
const int GlGraph::edgeShapesCount = 3;
int GlGraph::edgeShapeIds[GlGraph::edgeShapesCount] =
  {BEZIERSHAPE, POLYLINESHAPE, SPLINESHAPE};
std::string GlGraph::edgeShapeName(int id) {
  switch(id) {
  case POLYLINESHAPE:
    return std::string("Polyline");
  case BEZIERSHAPE:
    return std::string("Bezier Curve");
  case SPLINESHAPE:
    return std::string("Spline Curve");
  defaut:
    cerr << __PRETTY_FUNCTION__ << endl;
    cerr << "Invalid edge shape id" << endl;
    return std::string("invalid shape id");
  }
}
//====================================================
int GlGraph::edgeShapeId(std::string name) {
  if (name == edgeShapeName(POLYLINESHAPE))
    return POLYLINESHAPE;
  if (name == edgeShapeName(BEZIERSHAPE))
    return BEZIERSHAPE;
  if (name == edgeShapeName(SPLINESHAPE))
    return SPLINESHAPE;
  cerr << __PRETTY_FUNCTION__ << endl;
  cerr << "Invalid edge shape name" << endl;
  return -1;
}
//====================================================
void GlGraph::loadPlugins(string dir, PluginLoader *plug) {
  initFactory();
  GlGraph::glyphFactory->load(dir + "glyph", "Glyph", plug);
  idToName.clear();
  nameToId.clear();
  Iterator<string> *itS=glyphFactory->availablePlugins();
  while (itS->hasNext()) {
  	string pluginName=itS->next();
    int pluginId=glyphFactory->objMap[pluginName]->getId();
    idToName[pluginId]=pluginName;
    nameToId[pluginName]=pluginId;
  } delete itS;
}
//====================================================
void GlGraph::loadPlugins(PluginLoader *plug) {
  string::const_iterator begin=tlp::TulipPluginsPath.begin();
  string::const_iterator end=begin;
  while (end!=tlp::TulipPluginsPath.end())
    if ((*end)==tlp::PATH_DELIMITER) {
      if (begin!=end) 
	loadPlugins(string(begin,end)+"/",plug);
      ++end;
      begin=end;
    } else
      ++end;
  if (begin!=end) 
    loadPlugins(string(begin,end)+"/",plug);
}
//====================================================
GlGraph::GlGraph():
  _superGraph(0),
  backgroundColor(65,65,65),
  _viewArrow(false), _viewLabel(false),
  _viewMetaLabel(false), _viewStrahler(false),
  _viewAutoScale(true),  _incrementalRendering(true),_edgeColorInterpolate(true), _edge3D(false),
  _viewOrtho(true), _FontsType(0),
  elementColor(0), elementSize(0),elementShape(0),elementSelected(0),elementLabel(0),
  elementLayout(0),elementMetaGraph(0), elementTexture(0),
  selectBuf(0),// objectGraph(0),objectCube(0), 
  selectionDL(0),
  metaGraphDL(0),
  arrowDL(0),
  sceneTranslation(0,0,0),
  sceneRotation(Coord(180,0,0)),
  _displayEdges(true),
  maxNumberOfNodeToDraw(500),maxNumberOfEdgeToDraw(500),
  drawNodesIterator(0),
  drawEdgesIterator(0),
  drawLabelsIterator(0),
  drawSelectedLabelsIterator(0),
  drawEdgeLabelsIterator(0),
  drawEdgeSelectedLabelsIterator(0),
  winX(0), winY(0),
  winH(480), winW(640),
  labelsBorder(2),
  layoutName("viewLayout")
{
  initFactory();
  initProxies();
  TRACE_EXEC();
  fontsPath = tlp::TulipLibDir + "tlp/bitmaps/";
  fontRenderer = new TextRenderer();
}
//====================================================
GlGraph::GlGraph(const GlGraph &g) :
  _superGraph(g._superGraph),
  backgroundColor(g.backgroundColor),
  _viewArrow(g._viewArrow), _viewLabel(g._viewLabel),
  _viewMetaLabel(g._viewMetaLabel), _viewStrahler(g._viewStrahler),
  _viewAutoScale(g._viewAutoScale), _incrementalRendering(g._incrementalRendering),
  _edgeColorInterpolate(g._edgeColorInterpolate), _edge3D(g._edge3D),
  _viewOrtho(g._viewOrtho), _FontsType(g._FontsType),
  //texturesMap not initialized... don't know if we should, and how
  selectBuf(0),
  sceneTranslation(g.sceneTranslation),
  sceneRotation(g.sceneRotation),
  cameraEyes(g.cameraEyes), cameraCenter(g.cameraCenter), cameraUp(g.cameraUp),
  cameraZoomFactor(g.cameraZoomFactor),
  distCam(g.distCam),
  maxNumberOfNodeToDraw(g.maxNumberOfNodeToDraw),
  maxNumberOfEdgeToDraw(g.maxNumberOfEdgeToDraw),
  drawNodesIterator(0),
  drawEdgesIterator(0),
  drawLabelsIterator(0),
  drawSelectedLabelsIterator(0),
  drawEdgeLabelsIterator(0),
  drawEdgeSelectedLabelsIterator(0),
  selectionDL(0),
  metaGraphDL(0),
  arrowDL(0),
  winH(g.winH), winW(g.winW),
  winX(g.winX), winY(g.winY),
  labelsBorder(g.labelsBorder) {
  initFactory();
  initProxies();
  for (int i=0; i<4; ++i) viewportArray[i] = g.viewportArray[i];
  _superGraph->addObserver(this);
  if (isViewStrahler()) {
    buildOrderedList();
  }
  fontsPath = tlp::TulipLibDir + "tlp/bitmaps/";
  fontRenderer = new TextRenderer();
}
//====================================================
GlGraph::~GlGraph() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  notifyDestroy(this);
  removeObservers();
  if(_superGraph != 0)
    _superGraph->removeObserver(this);
  deleteDisplayLists();
  delete fontRenderer;
}
//====================================================
void GlGraph::initializeGL() {
  buildDisplayLists();
}
//==================================================
void GlGraph::centerScene() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  sceneRotation.set(180,0,0);
  if (_superGraph==0) return;
  TRACE_EXEC();
  goodScale();
}
//====================================================
void GlGraph::changeViewport( unsigned int x, unsigned int y, unsigned int w, unsigned int h ) {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  TRACE_EXEC();
  makeCurrent();
  winX=x;
  winY=y;
  winH=h;
  winW=w;
  glViewport(winX,winY,winW,winH);
  glGetIntegerv(GL_VIEWPORT, viewportArray);
}
//====================================================
SuperGraph* GlGraph::getSuperGraph() const {
  return _superGraph;
}
//====================================================
void GlGraph::setSuperGraph(SuperGraph *superGraph) {
  //   cerr << __PRETTY_FUNCTION__ << endl;
  if ( superGraph == _superGraph ) return;
  TRACE_EXEC();
  timerStop();
  if (_superGraph!=0)
    _superGraph->removeObserver(this);
  if (superGraph!=_superGraph)
    nbElementsFrame.setAll(200);
  _superGraph=superGraph;
  if (_superGraph == 0) return;
  initProxies();
  _superGraph->addObserver(this);
  setViewStrahler(false);
  if (_viewStrahler) {
    setViewStrahler(true);
  }
}
//==================================================
void GlGraph::draw() {
  //cerr << __PRETTY_FUNCTION__ << endl;
  Observable::holdObservers();
  timerStop();
  makeCurrent();
  nodesRenderedAsPoint.setAll(false);
  occlusionTest.reset();
  initGlParameter();
  initLights();
  initProjection();
  initModelView();
  drawState = 0;
  currentFrame = 1;
  if (_superGraph==0) {
    Observable::unholdObservers();
    return;
  }
  initIterators();
  if (!_incrementalRendering) {
    drawNodes(_superGraph->numberOfNodes(),drawNodesIterator);
    if (_displayEdges) 
      drawEdges(_superGraph->numberOfEdges(),drawEdgesIterator);
    if (isViewLabel()) {
      drawNodeLabels(_superGraph->numberOfNodes(),drawSelectedLabelsIterator,true);
      drawEdgeLabels(_superGraph->numberOfEdges(),drawEdgeSelectedLabelsIterator,true);
      drawNodeLabels(_superGraph->numberOfNodes(),drawLabelsIterator,false);
      drawEdgeLabels(_superGraph->numberOfEdges(),drawEdgeLabelsIterator,false);
    }
  }
  else {
    timerStart(0);
  }
  mPaint();
  notifyDraw(this);
  Observable::unholdObservers();
 }
//====================================================
int getTime() {
  return (10000*clock())/CLOCKS_PER_SEC;
}
//====================================================
bool GlGraph::drawPart() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  if (_superGraph==0) return true;
  Observable::holdObservers();
  makeCurrent();
  initLights();
  initProjection();
  initModelView();

  bool finished=false;
  int startTime,endTime;
  startTime = getTime();
  unsigned int elementToDraw = nbElementsFrame.get(currentFrame);
  switch (drawState) {
  case DRAWNODE :
  if (!drawNodesIterator->hasNext())
      drawState ++;
    else {
      elementToDraw -= drawNodes(elementToDraw, drawNodesIterator);
      if (drawNodesIterator->hasNext()) break;
      else drawState++;
     }
  case DRAWEDGE :
   if (!_displayEdges || !drawEdgesIterator->hasNext()) 
      drawState++;
    else {
      elementToDraw -= drawEdges(elementToDraw, drawEdgesIterator);
      if (drawEdgesIterator->hasNext()) break;
      else drawState++;
    }
  case DRAWSELECTEDNODELABELS :
     if (!isViewLabel() || !drawSelectedLabelsIterator->hasNext())
      drawState++;
    else {
      elementToDraw -= drawNodeLabels(elementToDraw, drawSelectedLabelsIterator,true);
      if (drawSelectedLabelsIterator->hasNext()) break;
      else drawState++;
    }
  case DRAWSELECTEDEDGELABELS :
     if (!isViewLabel() || !_displayEdges || !drawEdgeSelectedLabelsIterator->hasNext())
      drawState++;
    else {
      elementToDraw -= drawEdgeLabels(elementToDraw, drawEdgeSelectedLabelsIterator,true);
      if (drawEdgeSelectedLabelsIterator->hasNext()) break;
      else drawState++;
    }
  case DRAWNODELABELS :
    if (!isViewLabel() || !drawLabelsIterator->hasNext())
      drawState++;
    else {
      elementToDraw -= drawNodeLabels(elementToDraw, drawLabelsIterator,false);
      if (drawLabelsIterator->hasNext()) break;
      else drawState++;
    }
  case DRAWEDGELABELS :
    if (!isViewLabel() || !_displayEdges || !drawEdgeLabelsIterator->hasNext())
      drawState++;
    else {
      elementToDraw -= drawEdgeLabels(elementToDraw, drawEdgeLabelsIterator,false);
      if (drawEdgeLabelsIterator->hasNext()) break;
      else drawState++;
    }
  case DRAWEND :
    if (drawState==DRAWEND) finished=true;
  }
  endTime=getTime();
  unsigned int elapsedTime = endTime - startTime;
  unsigned int elementDrawn = nbElementsFrame.get(currentFrame) - elementToDraw;
  if (elapsedTime > 0 && elementDrawn > 0)
    nbElementsFrame.set(currentFrame, int(rint(1000.0*double(elementDrawn)/(double)elapsedTime)));
  else
    nbElementsFrame.set(currentFrame, 1000);
  currentFrame ++;
  if (finished) mPaint();
  Observable::unholdObservers();
  return finished;
}
//====================================================
void GlGraph::goodScale() {
  timerStop();
  if (_superGraph==0) return;
  initProxies();
  //  cerr << __PRETTY_FUNCTION__ << endl;
  
  pair<Coord, Coord> bboxes = tlp::computeBoundingBox(_superGraph, elementLayout, elementSize, elementRotation);
  Coord maxC = bboxes.first;
  Coord minC = bboxes.second;

  double dx = maxC[0] - minC[0];
  double dy = maxC[1] - minC[1];
  double dz = maxC[2] - minC[2];

  Coord center = (maxC + minC) / -2.0;
  sceneTranslation = center;

  if ((dx==0) && (dy==0) && (dz==0))
    dx=dy=dz=10;
  distCam = sqrt(dx*dx+dy*dy+dz*dz)/2; //radius of the sphere hull of the layout bounding box
  cameraEyes.set(0,0,-distCam);
  cameraCenter.set(0,0,0);
  cameraUp.set(0,1,0);
  if (_viewOrtho) 
    cameraZoomFactor=0.5;
  else 
    cameraZoomFactor=0.5;
}
//====================================================
void GlGraph::outputEPS(int size, int doSort, const char *filename) {
  TRACE_EXEC();
  timerStop();
  makeCurrent();
  GLfloat *feedbackBuffer;
  GLint returned;
  FILE *file;
  //backup unsuported rendering setting
  bool saveLabelState=_viewLabel;
  if (_FontsType!=0)
    _viewLabel=false;
  feedbackBuffer = (GLfloat *)calloc(size, sizeof(GLfloat));
  glFeedbackBuffer(size, GL_3D_COLOR, feedbackBuffer);
  glRenderMode(GL_FEEDBACK);
  initProjection();
  initModelView();
  initGlParameter();
  Iterator<node> *drawNodesIterator=_superGraph->getNodes();
  drawNodes(_superGraph->numberOfNodes(), drawNodesIterator);  
  delete drawNodesIterator;
  Iterator<edge> *drawEdgesIterator=_superGraph->getEdges();
  if (_displayEdges)
    drawEdges(_superGraph->numberOfEdges(), drawEdgesIterator);  
  delete drawEdgesIterator;
  drawNodesIterator=_superGraph->getNodes();
  if (_viewLabel)
    drawNodeLabels(_superGraph->numberOfNodes(), drawNodesIterator, true);
  delete drawNodesIterator;
  drawNodesIterator=_superGraph->getNodes();
  if (_viewLabel)
    drawNodeLabels(_superGraph->numberOfNodes(), drawNodesIterator, false);
  delete drawNodesIterator;
  glFlush();
  glFinish();
  returned = glRenderMode(GL_RENDER);
  if (filename) {
    file = fopen(filename, "w");
    if (file) 
      {spewWireFrameEPS(file, doSort, returned, feedbackBuffer, "rendereps");} 
    else 
      {printf("Could not open %s\n", filename);}
  } 
  else 
    {printBuffer(returned, feedbackBuffer);}
  free(feedbackBuffer);
  _viewLabel=saveLabelState;
}
//====================================================
unsigned char * GlGraph::getImage(int &w, int &h) {
  TRACE_EXEC();
  timerStop();
  makeCurrent();
  bool saveState=_incrementalRendering;
  w=winW;
  h=winH;
  unsigned char *image=(unsigned char *)malloc(w*h*3*sizeof(unsigned char));
  _incrementalRendering=false;
  draw();
  makeCurrent(); //necessary because notify inside draw, the openGL context can change.
  glFlush();
  glFinish();
  glPixelStorei(GL_PACK_ALIGNMENT,1);
  glReadPixels(winX,winY,winW,winH,GL_RGB,GL_UNSIGNED_BYTE,image);
  _incrementalRendering=saveState;
  return image;
}
//====================================================
void GlGraph::screenTo3DWorld(float& x, float& y, float& z) {
  TRACE_EXEC();
  makeCurrent();
  double X=x,Y=y;
  double xScr,yScr,zScr;
  GLint realy;
  GLint viewport[4];
  GLdouble modelview[16], projection[16];
  glGetIntegerv(GL_VIEWPORT, viewport);
  glGetDoublev (GL_MODELVIEW_MATRIX, modelview);
  glGetDoublev (GL_PROJECTION_MATRIX, projection);
  realy = viewport[3] - (GLint) Y - 1;
  gluProject(0,0,0,modelview,projection,viewport,&xScr,&yScr,&zScr);
  double _x, _y, _z;
  gluUnProject((GLdouble)winW - X, (GLdouble) realy, zScr, modelview, projection, viewport, 
	       &_x, &_y, &_z);
  x = _x; y = _y; z=_z;
}
//====================================================
void GlGraph::worldTo2DScreen(float& x, float& y, float& z) {
  TRACE_EXEC();
  makeCurrent();
  GLint viewport[4];
  GLdouble modelview[16], projection[16];
  glGetIntegerv(GL_VIEWPORT, viewport);
  glGetDoublev (GL_MODELVIEW_MATRIX, modelview);
  glGetDoublev (GL_PROJECTION_MATRIX, projection);
  double _x, _y, _z;
  gluProject(x, y, z, modelview, projection, viewport, &_x, &_y, &_z);
  x = _x; y = _y; z=_z;
}
//====================================================
class LessThanNode {
public:
  MetricProxy *metric;
  bool operator() (node n1,node n2)  {
    return (metric->getNodeValue(n1) > metric->getNodeValue(n2));
  } 
};
//====================================================
class LessThanEdge{
public:
  MetricProxy *metric;
  SuperGraph *sp;
  bool operator() (edge e1,edge e2) {
    return (metric->getNodeValue(sp->target(e1))>metric->getNodeValue(sp->target(e2)));
  } 
};
//====================================================
void GlGraph::buildOrderedList() {
  TRACE_EXEC();
  orderedNode.clear();
  orderedEdge.clear();
  if (!_viewStrahler) return;
  bool cached,resultBool;string erreurMsg;
  MetricProxy *metric=new MetricProxy(_superGraph);
  _superGraph->computeProperty("StrahlerGeneral",metric,erreurMsg);
  Iterator<node> *itN=_superGraph->getNodes();
  while (itN->hasNext()) 
    orderedNode.push_back(itN->next());
  delete itN;
  LessThanNode comp;
  comp.metric=metric;
  orderedNode.sort(comp);
  Iterator<edge> *itE=_superGraph->getEdges();
  while (itE->hasNext()) {
    orderedEdge.push_back(itE->next());
  } delete itE;
  LessThanEdge comp2;
  comp2.metric=metric;
  comp2.sp=_superGraph;
  orderedEdge.sort(comp2);
  delete metric;
}
//====================================================
/**
 * Management of the graph modification, necessary when we store ordered list
 * of graph elements
 */
//====================================================
void GlGraph::addNode (SuperGraph *, const node n) {
  if (_viewStrahler)
    orderedNode.push_back(n);
}
//====================================================
void GlGraph::addEdge (SuperGraph *, const edge e) {
  if (_viewStrahler)
    orderedEdge.push_back(e);
}
//====================================================
void GlGraph::delNode (SuperGraph *, const node n) {
  if (_viewStrahler)
    orderedNode.remove(n);
}
//====================================================
void GlGraph::delEdge (SuperGraph *, const edge e) {
  if (_viewStrahler)
    orderedEdge.remove(e);
}
//====================================================
void GlGraph::destroy (SuperGraph *graph) {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  if (graph->getRoot()!=graph) {
    setSuperGraph(graph->getRoot());
    goodScale();
    centerScene();
    draw();
  }
  else
    setSuperGraph(0);
}
//====================================================
/**
 *Management of the iterators on the graph elements
 */
//====================================================
void GlGraph::deleteIterators() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  if (drawNodesIterator!=0) {
    delete drawNodesIterator;
    drawNodesIterator=0;
  }
  if (drawEdgesIterator!=0) {
    delete drawEdgesIterator;
    drawEdgesIterator=0;
  }
  if (drawLabelsIterator!=0) {
    delete drawLabelsIterator;
    drawLabelsIterator=0;
  }
  if (drawSelectedLabelsIterator!=0) {
    delete drawSelectedLabelsIterator;
    drawSelectedLabelsIterator=0;
  }
  if (drawEdgeLabelsIterator!=0) {
    delete drawEdgeLabelsIterator;
    drawEdgeLabelsIterator=0;
  }
  if (drawEdgeSelectedLabelsIterator!=0) {
    delete drawEdgeSelectedLabelsIterator;
    drawEdgeSelectedLabelsIterator=0;
  }
}
//====================================================
void GlGraph::initProxies() {
  if (_superGraph != NULL) {
    elementRotation   = _superGraph->getProperty<MetricProxy>("viewRotation");
    elementSelected   = _superGraph->getProperty<SelectionProxy>("viewSelection");
    elementLabel      = _superGraph->getProperty<StringProxy>("viewLabel");
    elementLabelColor = _superGraph->getProperty<ColorsProxy>("viewLabelColor");
    elementColor      = _superGraph->getProperty<ColorsProxy>("viewColor");
    elementShape      = _superGraph->getProperty<IntProxy>("viewShape");
    elementSize       = _superGraph->getProperty<SizesProxy>("viewSize");
    elementLayout     = _superGraph->getProperty<LayoutProxy>(layoutName);
    elementMetaGraph  = _superGraph->getProperty<MetaGraphProxy>("viewMetaGraph");
    elementTexture    = _superGraph->getProperty<StringProxy>("viewTexture");
  }
}
//====================================================
void GlGraph::initIterators() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  deleteIterators();
  if (isViewStrahler()) {
    typedef std::list<node>::const_iterator  ListNodeIt;
    typedef std::list<edge>::const_iterator  ListEdgeIt;
    drawNodesIterator = new stlListIterator<node, ListNodeIt > (orderedNode.begin(),orderedNode.end());
    drawEdgesIterator = new stlListIterator<edge, ListEdgeIt > (orderedEdge.begin(),orderedEdge.end());
    if (_viewLabel) {
      drawLabelsIterator = new stlListIterator<node, ListNodeIt > (orderedNode.begin(),orderedNode.end());
      drawSelectedLabelsIterator = new stlListIterator<node, ListNodeIt > (orderedNode.begin(),orderedNode.end());
      drawEdgeLabelsIterator = new stlListIterator<edge, ListEdgeIt > (orderedEdge.begin(),orderedEdge.end());
      drawEdgeSelectedLabelsIterator = new stlListIterator<edge, ListEdgeIt > (orderedEdge.begin(),orderedEdge.end());
    }
  }
  else {
    drawNodesIterator=_superGraph->getNodes();
    drawEdgesIterator=_superGraph->getEdges();
    if (_viewLabel) {
      drawLabelsIterator=_superGraph->getNodes();
      drawSelectedLabelsIterator=_superGraph->getNodes();
      drawEdgeLabelsIterator=_superGraph->getEdges();
      drawEdgeSelectedLabelsIterator=_superGraph->getEdges();
    }
  }
}
//====================================================
