/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    http://www.travis-analyzer.de/

    Copyright (c) 2009-2014 Martin Brehm
                  2012-2014 Martin Thomas

    This file written by Martin Thomas.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/

#include "ir.h"

#include "globalvar.h"
#include "maintools.h"
#include "timestep.h"

#include <math.h>

#define BUF_SIZE 4096

static CxObArray g_PowerObserv;

CPowerObservation::CPowerObservation(bool global) {
	int i;
	
	try { _atoms = new CAtomGroup(); } catch(...) { _atoms = NULL; }
	if(_atoms == NULL) NewException((double)sizeof(CAtomGroup), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	
	if(global) {
		m_iShowMol = -1;
		m_iShowMolCount = g_oaSingleMolecules.GetSize();
		_name = new char[7];
		sprintf(_name, "global");
	} else {
		char buf[BUF_SIZE];
		char buf2[BUF_SIZE];
		size_t remaining = BUF_SIZE;
		if(g_oaMolecules.GetSize() > 1) {
#ifdef TARGET_LINUX
			remaining -= snprintf(buf, remaining, "    Which molecule should be observed (");
#else
			remaining -= sprintf(buf, "    Which molecule should be observed (");
#endif
			for(i = 0; i < g_oaMolecules.GetSize(); i++) {
				if(remaining < 1)
					break;
#ifdef TARGET_LINUX
				size_t length = snprintf(buf2, remaining, "%s=%d", ((CMolecule *)g_oaMolecules[i])->m_sName, i+1);
#else
				size_t length = sprintf(buf2, "%s=%d", ((CMolecule *)g_oaMolecules[i])->m_sName, i+1);
#endif
				strncat(buf, buf2, remaining - 1);
				remaining -= length;
				if(i < g_oaMolecules.GetSize() - 1) {
#ifdef TARGET_LINUX
					length = snprintf(buf2, remaining, ", ");
#else
					length = sprintf(buf2, ", ");
#endif
					strncat(buf, buf2, remaining - 1);
					remaining -= length;
				}
			}
			strncat(buf, ")? ", remaining - 1);
			m_iShowMol = AskRangeInteger_ND(buf, 1, g_oaMolecules.GetSize()) - 1;
		} else {
			m_iShowMol = 0;
			mprintf("    Observing molecule %s.\n", ((CMolecule *)g_oaMolecules[m_iShowMol])->m_sName);
		}
		
		while(true) {
			mprintf("    Which atom(s) to observe (e.g. \"C1,C3-5,H\", \"*\"=all)? [*] ");
			inpprintf("! Which atom(s) to observe (e.g. \"C1,C3-5,H\", \"*\"=all)? [*]\n");
			char buf[BUF_SIZE];
			myget(buf);
			if(strlen(buf) == 0) {
				if(!_atoms->ParseAtoms((CMolecule *)g_oaMolecules[m_iShowMol], "*")) {
					eprintf("Weird error.\n");
					continue;
				}
			} else if(!_atoms->ParseAtoms((CMolecule *)g_oaMolecules[m_iShowMol], buf)) {
				continue;
			}
			break;
		}
		mprintf("\n    Observing %d atoms of molecule %s.\n", _atoms->m_iAtomGes, ((CMolecule *)g_oaMolecules[m_iShowMol])->m_sName);
		m_iShowMolCount = ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize();
		_name = new char[strlen(((CMolecule *)g_oaMolecules[m_iShowMol])->m_sName) + strlen(_atoms->m_sName) + 4];
		sprintf(_name, "[%s_%s]", ((CMolecule *)g_oaMolecules[m_iShowMol])->m_sName, _atoms->m_sName);
		mprintf("\n");
	}
	
	if(g_iTrajSteps != -1) {
		_correlationDepth = 0.75 * g_iTrajSteps;
		if(_correlationDepth > 4096)
			_correlationDepth = 4096;
		if(g_fTimestepLength > 1.0f)
			_correlationDepth = 2048;
		if(g_fTimestepLength > 2.0f)
			_correlationDepth = 1024;
		_correlationDepth = AskUnsignedInteger("    Enter the resolution (=depth) of the ACF (in time steps): [%d] ", _correlationDepth, _correlationDepth);
	} else {
		_correlationDepth = AskUnsignedInteger("    Enter the resolution (=depth) of the ACF (in time steps): [256] ", 256);
	}
	int size = CalcFFTSize(_correlationDepth, false);
	if(_correlationDepth != size) {
		mprintf(WHITE, "    The next \"fast\" size for FFT is %d. Using this instead of %d as depth.\n", size, _correlationDepth);
		_correlationDepth = size;
	}
	
	if(g_bAdvanced2) {
		_windowFunction = AskRangeInteger("    Window function: cos^2(a*t) (1), exp(-t/a) (2), exp(-(t/a)^2) (3) [1] ", 1, 3, 1);
		if(_windowFunction == 1) {
			mprintf("    The parameter \"a\" is chosen according to the correlation depth.\n");
			_windowFunctionParameter = 0;
		} else if(_windowFunction == 2) {
			_windowFunctionParameter = AskUnsignedInteger("    Parameter \"a\" (in time steps): [%d] ", _correlationDepth / 4, _correlationDepth / 4);
		} else if(_windowFunction == 3) {
			_windowFunctionParameter = AskUnsignedInteger("    Parameter \"a\" (in time steps): [%d] ", _correlationDepth / 2, _correlationDepth / 2);
		} else {
			eprintf("This is impossible.\n");
			abort();
		}
	} else {
		_windowFunction = 1;
		_windowFunctionParameter = 0;
	}
	
	if(g_bAdvanced2) {
		_zeroPadding = AskUnsignedInteger("    Zero Padding: How many zeros to insert? [%d] ", _correlationDepth * 3, _correlationDepth * 3);
		size = CalcFFTSize(_correlationDepth + _zeroPadding, false);
		if(_correlationDepth + _zeroPadding != size) {
			mprintf(WHITE, "    The next \"fast\" size for FFT is %d. Using %d zeros for zero padding.\n", size, size-_correlationDepth);
			_zeroPadding = size-_correlationDepth;
		}
	} else {
		_zeroPadding = _correlationDepth * 3;
		size = CalcFFTSize(_correlationDepth + _zeroPadding, false);
		mprintf("    Using cos^2 window function; inserting %d zeros for zero padding.\n",_zeroPadding);
		if(_correlationDepth + _zeroPadding != size) {
			mprintf(WHITE, "    The next \"fast\" size for FFT is %d. Using %d zeros for zero padding.\n", size, size-_correlationDepth);
			_zeroPadding = size-_correlationDepth;
		}
	}
	
	float possibleRange = 33356.41f / g_fTimestepLength / 2.0f;
	_specResolution = possibleRange / (_correlationDepth + _zeroPadding);
	mprintf("    This results in a spectral resolution of %.2f cm^-1.\n", _specResolution);
	mprintf("\n    A time step length of %.2f fs allows a spectral range up to %.2f cm^-1.\n", g_fTimestepLength, possibleRange);
	float specLimit = AskRangeFloat("\n    Calculate spectrum up to which wave number (cm^-1)? [%.2f cm^-1] ", 0, possibleRange, (possibleRange < 5000.0f) ? possibleRange : 5000.0f, (possibleRange < 5000.0f) ? possibleRange : 5000.0f);
	_specSize = specLimit / _specResolution;
	mprintf("\n");
	
	_massWeighting = AskYesNo("    Weight autocorrelation functions by atomic mass (y/n)? [yes] ", true);
	mprintf("\n");
	
	if(g_bAdvanced2) {
		_saveACF = AskYesNo("    Save autocorrelation function (y/n)? [no] ", false);
		mprintf("\n");
	} else {
		_saveACF = false;
	}
}


CPowerObservation::~CPowerObservation() {
	delete[] _name;
	delete _atoms;
}


void CPowerObservation::initialize() {
	int i, j, k;
	int n;
	if(g_iTrajSteps != -1)
		n = 1.1 * g_iTrajSteps / g_iStride;
	else
		n = 10000;
	
	if(m_iShowMol == -1) {
		mprintf("    Velocity cache: Trying to allocate %s of memory...\n", FormatBytes((double)g_iGesAtomCount * n * sizeof(CxVector3)));
		for(i = 0; i < g_iGesAtomCount; i++) {
			CxVec3Array *a;
			try { a = new CxVec3Array(); } catch(...) { a = NULL; }
			if(a == NULL) NewException((double)sizeof(CxVec3Array), __FILE__, __LINE__, __PRETTY_FUNCTION__);
			a->SetMaxSize(n);
			a->SetGrow(n / 10);
			_velocityCache.Add(a);
		}
	} else {
		mprintf("    Velocity cache: Trying to allocate %s of memory...\n", FormatBytes((double)m_iShowMolCount * _atoms->m_iAtomGes * n * sizeof(CxVector3)));
		for(i = 0; i < m_iShowMolCount * _atoms->m_iAtomGes; i++) {
			CxVec3Array *a;
			try { a = new CxVec3Array(); } catch(...) { a = NULL; }
			if(a == NULL) NewException((double)sizeof(CxVec3Array), __FILE__, __LINE__, __PRETTY_FUNCTION__);
			a->SetMaxSize(n);
			a->SetGrow(n / 10);
			_velocityCache.Add(a);
		}
	}
	
	if(m_iShowMol == -1) {
		_masses.SetSize(g_iGesAtomCount);
		if(_massWeighting) {
			for(i = 0; i < g_iGesAtomCount; i++) {
				_masses[i] = ((CAtom *)g_oaAtoms[g_waAtomRealElement[i]])->m_pElement->m_fMass;
			}
		} else {
			for(i = 0; i < g_iGesAtomCount; i++) {
				_masses[i] = 1.0f;
			}
		}
	} else {
		_masses.SetSize(m_iShowMolCount * _atoms->m_iAtomGes);
		if(_massWeighting) {
			for(i = 0; i < m_iShowMolCount; i++) {
				int n = 0;
				for(j = 0; j < _atoms->m_baRealAtomType.GetSize(); j++) {
					CxIntArray *a = (CxIntArray *)_atoms->m_oaAtoms[j];
					for(k = 0; k < a->GetSize(); k++) {
						_masses[i * _atoms->m_iAtomGes + n] = ((CAtom *)g_oaAtoms[_atoms->m_baRealAtomType[j]])->m_pElement->m_fMass;
						n++;
					}
				}
			}
		} else {
			for(i = 0; i < m_iShowMolCount * _atoms->m_iAtomGes; i++) {
				_masses[i] = 1.0f;
			}
		}
	}
}


void CPowerObservation::process(CTimeStep *ts) {
	int i, j, k;
	if(m_iShowMol == -1) {
		for(i = 0; i < g_iGesAtomCount; i++) {
			((CxVec3Array *)_velocityCache[i])->Add(ts->m_vaVelocities[i]);
		}
	} else {
		for(i = 0; i < m_iShowMolCount; i++) {
			CSingleMolecule *sm = (CSingleMolecule *)g_oaSingleMolecules[((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex[i]];
			int n = 0;
			for(j = 0; j < _atoms->m_baAtomType.GetSize(); j++) {
				CxIntArray *a = (CxIntArray *)_atoms->m_oaAtoms[j];
				for(k = 0; k < a->GetSize(); k++) {
					((CxVec3Array *)_velocityCache[i * _atoms->m_iAtomGes + n])->Add(ts->m_vaVelocities[((CxIntArray *)sm->m_oaAtomOffset[_atoms->m_baAtomType[j]])->GetAt(a->GetAt(k))]);
					n++;
				}
			}
		}
	}
}


void CPowerObservation::finalize() {
	int i, j, k, l;
	int n = ((CxVec3Array *)_velocityCache[0])->GetSize();
	float step;
	if(m_iShowMol == -1) {
		step = (float)g_iGesAtomCount / 20.0f;
	} else {
		step = (float)m_iShowMolCount * _atoms->m_iAtomGes / 20.0f;
	}
	
	mprintf("    Calculating autocorrelation...\n");
	CxFloatArray *acf;
	try { acf = new CxFloatArray(); } catch(...) { acf = NULL; }
	if(acf == NULL) NewException((double)sizeof(CxFloatArray), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	acf->SetSize(_correlationDepth);
	for(i = 0; i < _correlationDepth; i++) {
		acf->GetAt(i) = 0.0f;
	}
	CAutoCorrelation *ac;
	try { ac = new CAutoCorrelation(); } catch(...) { ac = NULL; }
	if(ac == NULL) NewException((double)sizeof(CAutoCorrelation), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	ac->Init(n, _correlationDepth, g_bACFFFT);
	
	CxFloatArray *temp;
	try { temp = new CxFloatArray(); } catch(...) { temp = NULL; }
	if(temp == NULL) NewException((double)sizeof(CxFloatArray), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	temp->SetSize(n);
	CxFloatArray *temp2;
	try { temp2 = new CxFloatArray(); } catch(...) { temp2 = NULL; }
	if(temp2 == NULL) NewException((double)sizeof(CxFloatArray), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	temp2->SetSize(_correlationDepth);
	
	mprintf(WHITE, "     [");
	if(m_iShowMol == -1) {
		for(i = 0; i < g_iGesAtomCount; i++) {
			if(fmodf(i, step) < 1.0f)
				mprintf(WHITE, "#");
			for(j = 0; j < 3; j++) {
				for(k = 0; k < n; k++) {
					temp->GetAt(k) = ((CxVec3Array *)_velocityCache[i])->GetAt(k)[j];
				}
				ac->AutoCorrelate(temp, temp2);
				for(k = 0; k < _correlationDepth; k++) {
					acf->GetAt(k) += temp2->GetAt(k) * _masses[i];
				}
			}
		}
	} else {
		for(i = 0; i < m_iShowMolCount; i++) {
			for(j = 0; j < _atoms->m_iAtomGes; j++) {
				if(fmodf(i * _atoms->m_iAtomGes + j, step) < 1.0f)
					mprintf(WHITE, "#");
				for(k = 0; k < 3; k++) {
					for(l = 0; l < n; l++) {
						temp->GetAt(l) = ((CxVec3Array *)_velocityCache[i * _atoms->m_iAtomGes + j])->GetAt(l)[k];
					}
					ac->AutoCorrelate(temp, temp2);
					for(l = 0; l < _correlationDepth; l++) {
						acf->GetAt(l) += temp2->GetAt(l) * _masses[i * _atoms->m_iAtomGes + j];
					}
				}
			}
		}
	}
	mprintf(WHITE, "]\n");
	
	if(m_iShowMol != -1) {
		for(i = 0; i < _correlationDepth; i++) {
			acf->GetAt(i) /= (float)m_iShowMolCount;
		}
	}
	
	delete ac;
	delete temp2;
	
	if(_saveACF) {
		char filename[BUF_SIZE];
#ifdef TARGET_LINUX
		snprintf(filename, BUF_SIZE, "power_acf_%s.csv", _name);
#else
		sprintf(filename, "power_acf_%s.csv", _name);
#endif
		mprintf("    Saving autocorrelation function as %s...\n", filename);
		FILE *acfFile = OpenFileWrite(filename, false);
		for(i = 0; i < _correlationDepth; i++) {
			fprintf(acfFile, "%.2f; %.8G\n", i * g_fTimestepLength, acf->GetAt(i));
		}
		fclose(acfFile);
	}
	
	temp->CopyFrom(acf);
	
	if(_windowFunction == 1) {
		for(i = 0; i < temp->GetSize(); i++) {
			temp->GetAt(i) *= powf(cosf((float)i / (temp->GetSize() - 1) / 2.0f * Pi), 2.0f);
		}
	} else if(_windowFunction == 2) {
		for(i = 0; i < temp->GetSize(); i++) {
			temp->GetAt(i) *= expf(-(float)i / _windowFunctionParameter);
		}
	} else if(_windowFunction == 3) {
		for(i = 0; i < temp->GetSize(); i++) {
			temp->GetAt(i) *= expf(-(float)i * i / _windowFunctionParameter / _windowFunctionParameter);
		}
	} else if(_windowFunction != 0) {
		eprintf("Unknown window function.\n");
		abort();
	}
	
	if(_saveACF) {
		char filename[BUF_SIZE];
#ifdef TARGET_LINUX
		snprintf(filename, BUF_SIZE, "power_acf_windowed_%s.csv", _name);
#else
		sprintf(filename, "power_acf_windowed_%s.csv", _name);
#endif
		mprintf("    Saving windowed autocorrelation function as %s...\n", filename);
		FILE *acfFile = OpenFileWrite(filename, false);
		for(i = 0; i < _correlationDepth; i++) {
			fprintf(acfFile, "%.2f; %.8G\n", i * g_fTimestepLength, temp->GetAt(i));
		}
		fclose(acfFile);
	}
	
	if(_zeroPadding > 0) {
		for(i = 0; i < _zeroPadding; i++) {
			temp->Add(0.0f);
		}
	}
	
	int oldSize = temp->GetSize();
	temp->SetSize(2 * oldSize);
	for(i = 1; i < oldSize; i++) {
		temp->GetAt(oldSize + i) = temp->GetAt(oldSize - i);
	}
	temp->GetAt(oldSize) = 0.0f;
	
	mprintf("    Fourier transforming autocorrelation function...\n");
	CFFT *fft;
	try { fft = new CFFT(); } catch(...) { fft = NULL; }
	if(fft == NULL) NewException((double)sizeof(CFFT), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	fft->PrepareFFT_C2C(temp->GetSize());
	for(i = 0; i < temp->GetSize(); i++) {
		fft->m_pInput[2*i] = temp->GetAt(i);
		fft->m_pInput[2*i+1] = 0.0f;
	}
	fft->DoFFT();
	
	CxFloatArray *spectrum;
	try { spectrum = new CxFloatArray(); } catch(...) { spectrum = NULL; }
	if(spectrum == NULL) NewException((double)sizeof(CxFloatArray), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	spectrum->SetSize(_specSize);
	for(i = 0; i < _specSize; i++) {
		spectrum->GetAt(i) = 7.211349e-9f * fft->m_pOutput[2*i] * g_fTimestepLength; // Output in K*cm
	}
	
	char filename[BUF_SIZE];
#ifdef TARGET_LINUX
	snprintf(filename, BUF_SIZE, "power_spectrum_%s.csv", _name);
#else
	sprintf(filename, "power_spectrum_%s.csv", _name);
#endif
	mprintf("    Saving spectrum as %s...\n", filename);
	FILE *specFile = OpenFileWrite(filename, false);
	fprintf(specFile, "#Wavenumber (cm^-1); Spectrum (K*cm); Integral (K)\n");
	double integral = 0.0;
	for(i = 0; i < _specSize; i++) {
		integral += (double)spectrum->GetAt(i) * _specResolution;
		fprintf(specFile, "%.2f; %.8G; %.14G\n", _specResolution * i, spectrum->GetAt(i), integral);
	}
	fclose(specFile);
	
	if(m_iShowMol == -1)
		mprintf("\n    Assuming %d degrees of freedom, the average temperature is %.2f K\n", 3 * g_iGesAtomCount, integral / (3.0 * g_iGesAtomCount));
	else
		mprintf("\n    Assuming %d degrees of freedom, the average temperature is %.2f K\n", 3 * _atoms->m_iAtomGes, integral / (3.0 * _atoms->m_iAtomGes));
	
	delete acf;
	delete temp;
	delete fft;
	delete spectrum;
	
	for(i = 0; i < _velocityCache.GetSize(); i++) {
		delete (CxVec3Array *)_velocityCache[i];
	}
}
	

bool gatherPowerSpectrum() {
	g_bUseVelocities = true;
	
	while(true) {
		mprintf(YELLOW, "\n>>> Power Spectrum %d >>>\n\n", g_PowerObserv.GetSize() + 1);
		
		CPowerObservation *obs;
		try { obs = new CPowerObservation(); } catch(...) { obs = NULL; }
		if(obs == NULL) NewException((double)sizeof(CPowerObservation), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		g_PowerObserv.Add(obs);
		
		mprintf(YELLOW, "<<< End of Power Spectrum %d <<<\n\n", g_PowerObserv.GetSize());
		
		if(!AskYesNo("    Add another observation (y/n)? [no] ", false))
			break;
		mprintf("\n");
	}
	
	if(AskYesNo("\n    Compute power spectrum of whole system (y/n) [no] ", false)) {
		mprintf(YELLOW, "\n>>> Global Power Spectrum >>>\n\n");
		
		CPowerObservation *obs;
		try { obs = new CPowerObservation(true); } catch(...) { obs = NULL; }
		if(obs == NULL) NewException((double)sizeof(CPowerObservation), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		g_PowerObserv.Add(obs);
		
		mprintf(YELLOW, "<<< End of Global Power Spectrum <<<\n\n");
	}
	
	return true;
}


bool initializePowerSpectrum() {
	int i;
	for(i = 0; i < g_PowerObserv.GetSize(); i++) {
		((CPowerObservation *)g_PowerObserv[i])->initialize();
	}
	return true;
}


void processPowerSpectrum(CTimeStep* ts) {
	int i;
	for(i = 0; i < g_PowerObserv.GetSize(); i++) {
		((CPowerObservation *)g_PowerObserv[i])->process(ts);
	}
}


void finalizePowerSpectrum() {
	int i;
	for(i = 0; i < g_PowerObserv.GetSize(); i++) {
		mprintf(YELLOW, "\n>>> Power Spectrum %d >>>\n\n", i+1);
		((CPowerObservation *)g_PowerObserv[i])->finalize();
		delete (CPowerObservation *)g_PowerObserv[i];
		mprintf(YELLOW, "\n<<< End of Power Spectrum %d <<<\n\n", i+1);
	}
}


static CxObArray g_IRObserv;


CIRObservation::CIRObservation(bool global) {
	int i;
	if(global) {
		m_iShowMol = -1;
		m_iShowMolCount = g_oaSingleMolecules.GetSize();
		_name = new char[7];
		sprintf(_name, "global");
	} else {
		char buf[BUF_SIZE];
		char buf2[BUF_SIZE];
		size_t remaining = BUF_SIZE;
		if(g_oaMolecules.GetSize() > 1) {
#ifdef TARGET_LINUX
			remaining -= snprintf(buf, remaining, "    Which molecule should be observed (");
#else
			remaining -= sprintf(buf, "    Which molecule should be observed (");
#endif
			for(i = 0; i < g_oaMolecules.GetSize(); i++) {
				if(remaining < 1)
					break;
#ifdef TARGET_LINUX
					size_t length = snprintf(buf2, remaining, "%s=%d", ((CMolecule *)g_oaMolecules[i])->m_sName, i+1);
#else
					size_t length = sprintf(buf2, "%s=%d", ((CMolecule *)g_oaMolecules[i])->m_sName, i+1);
#endif
					strncat(buf, buf2, remaining - 1);
					remaining -= length;
					if(i < g_oaMolecules.GetSize() - 1) {
#ifdef TARGET_LINUX
						length = snprintf(buf2, remaining, ", ");
#else
						length = sprintf(buf2, ", ");
#endif
						strncat(buf, buf2, remaining - 1);
						remaining -= length;
					}
			}
			strncat(buf, ")? ", remaining - 1);
			m_iShowMol = AskRangeInteger_ND(buf, 1, g_oaMolecules.GetSize()) - 1;
		} else {
			m_iShowMol = 0;
			mprintf("    Observing molecule %s.\n", ((CMolecule *)g_oaMolecules[m_iShowMol])->m_sName);
		}
		m_iShowMolCount = ((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex.GetSize();
		_name = new char[strlen(((CMolecule *)g_oaMolecules[m_iShowMol])->m_sName) + 1];
		strcpy(_name, ((CMolecule *)g_oaMolecules[m_iShowMol])->m_sName);
		mprintf("\n");
	}
	
	if(g_iTrajSteps != -1) {
		_correlationDepth = 0.75 * g_iTrajSteps;
		if(_correlationDepth > 4096)
			_correlationDepth = 4096;
		if(g_fTimestepLength > 1.0f)
			_correlationDepth = 2048;
		if(g_fTimestepLength > 2.0f)
			_correlationDepth = 1024;
		_correlationDepth = AskUnsignedInteger("    Enter the resolution (=depth) of the ACF (in time steps): [%d] ", _correlationDepth, _correlationDepth);
	} else {
		_correlationDepth = AskUnsignedInteger("    Enter the resolution (=depth) of the ACF (in time steps): [256] ", 256);
	}
	int size = CalcFFTSize(_correlationDepth, false);
	if(_correlationDepth != size) {
		mprintf(WHITE, "    The next \"fast\" size for FFT is %d. Using this instead of %d as depth.\n", size, _correlationDepth);
		_correlationDepth = size;
	}
	
	if(g_bAdvanced2) {
		_windowFunction = AskRangeInteger("    Window function: cos^2(a*t) (1), exp(-t/a) (2), exp(-(t/a)^2) (3) [1] ", 1, 3, 1);
		if(_windowFunction == 1) {
			mprintf("    The parameter \"a\" is chosen according to the correlation depth.\n");
			_windowFunctionParameter = 0;
		} else if(_windowFunction == 2) {
			_windowFunctionParameter = AskUnsignedInteger("    Parameter \"a\" (in time steps): [%d] ", _correlationDepth / 4, _correlationDepth / 4);
		} else if(_windowFunction == 3) {
			_windowFunctionParameter = AskUnsignedInteger("    Parameter \"a\" (in time steps): [%d] ", _correlationDepth / 2, _correlationDepth / 2);
		} else {
			eprintf("This is impossible.\n");
			abort();
		}
	} else {
		_windowFunction = 1;
		_windowFunctionParameter = 0;
	}
	
	if(g_bAdvanced2) {
		_zeroPadding = AskUnsignedInteger("    Zero Padding: How many zeros to insert? [%d] ", _correlationDepth * 3, _correlationDepth * 3);
		size = CalcFFTSize(_correlationDepth + _zeroPadding, false);
		if(_correlationDepth + _zeroPadding != size) {
			mprintf(WHITE, "    The next \"fast\" size for FFT is %d. Using %d zeros for zero padding.\n", size, size-_correlationDepth);
			_zeroPadding = size-_correlationDepth;
		}
	} else {
		_zeroPadding = _correlationDepth * 3;
		mprintf("    Using cos^2 window function; inserting %d zeros for zero padding.\n",_zeroPadding);
		size = CalcFFTSize(_correlationDepth + _zeroPadding, false);
		if(_correlationDepth + _zeroPadding != size) {
			mprintf(WHITE, "    The next \"fast\" size for FFT is %d. Using %d zeros for zero padding.\n", size, size-_correlationDepth);
			_zeroPadding = size-_correlationDepth;
		}
	}
	
	float possibleRange = 33356.41f / g_fTimestepLength / 2.0f;
	_specResolution = possibleRange / (_correlationDepth + _zeroPadding);
	mprintf("    This results in a spectral resolution of %.2f cm^-1.\n", _specResolution);
	mprintf("\n    A time step length of %.2f fs allows a spectral range up to %.2f cm^-1.\n", g_fTimestepLength, possibleRange);
	float specLimit = AskRangeFloat("\n    Calculate spectrum up to which wave number (cm^-1)? [%.2f cm^-1] ", 0, possibleRange, (possibleRange < 5000.0f) ? possibleRange : 5000.0f, (possibleRange < 5000.0f) ? possibleRange : 5000.0f);
	_specSize = specLimit / _specResolution;
	mprintf("\n");
	
	if(g_bAdvanced2) {
		_finiteDifferenceCorrection = AskYesNo("    Apply finite difference correction (y/n)? [yes] ", true);
		mprintf("\n");
	} else {
		_finiteDifferenceCorrection = true;
	}
	
	if(g_bAdvanced2) {
		_saveACF = AskYesNo("    Save autocorrelation function (y/n)? [no] ", false);
		mprintf("\n");
	} else {
		_saveACF = false;
	}
	
	if(g_bAdvanced2 && m_iShowMol == -1) {
		_includeCross = AskYesNo("    Include also cross-correlations (y/n)? [no] ", false);
		mprintf("\n");
	} else {
		_includeCross = false;
	}
	
	_quantumCorrection = 1;
}

CIRObservation::~CIRObservation() {
	delete[] _name;
}

void CIRObservation::initialize() {
	int n;
	if(g_iTrajSteps != -1)
		n = 1.1 * g_iTrajSteps / g_iStride;
	else
		n = 10000;
	
	mprintf("    Moment cache: Trying to allocate %s of memory...\n", FormatBytes((double)m_iShowMolCount * n * sizeof(CxVector3)));
	int i;
	for(i = 0; i < m_iShowMolCount; i++) {
		CxVec3Array *a;
		try { a = new CxVec3Array(); } catch(...) { a = NULL; }
		if(a == NULL) NewException((double)sizeof(CxVec3Array), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		a->SetMaxSize(n);
		a->SetGrow(n / 10);
		_dipoleCache.Add(a);
	}
}

void CIRObservation::process() {
	int i;
	if(m_iShowMol == -1) {
		for(i = 0; i < m_iShowMolCount; i++) {
			CSingleMolecule *sm = (CSingleMolecule *)g_oaSingleMolecules[i];
			((CxVec3Array *)_dipoleCache[i])->Add(sm->m_vDipole);
		}
	} else {
		for(i = 0; i < m_iShowMolCount; i++) {
			CSingleMolecule *sm = (CSingleMolecule *)g_oaSingleMolecules[((CMolecule *)g_oaMolecules[m_iShowMol])->m_laSingleMolIndex[i]];
			((CxVec3Array *)_dipoleCache[i])->Add(sm->m_vDipole);
		}
	}
}

void CIRObservation::finalize() {
	mprintf("    Deriving dipole moments...\n");
	int n = ((CxVec3Array *)_dipoleCache[0])->GetSize() - 2;
	float step = (float)m_iShowMolCount / 20.0f;
	mprintf(WHITE, "     [");
	int i, j, k, l;
	for(i = 0; i < m_iShowMolCount; i++) {
		if(fmodf(i, step) < 1.0f)
			mprintf(WHITE, "#");
		for(j = 0; j < n; j++) {
			((CxVec3Array *)_dipoleCache[i])->GetAt(j) = 0.5f * (((CxVec3Array *)_dipoleCache[i])->GetAt(j+2) - ((CxVec3Array *)_dipoleCache[i])->GetAt(j)) / g_fTimestepLength;
		}
	}
	mprintf(WHITE, "]\n");
	
	mprintf("    Calculating autocorrelation...\n");
	CxFloatArray *acf;
	try { acf = new CxFloatArray(); } catch(...) { acf = NULL; }
	if(acf == NULL) NewException((double)sizeof(CxFloatArray), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	acf->SetSize(_correlationDepth);
	for(i = 0; i < _correlationDepth; i++) {
		acf->GetAt(i) = 0.0f;
	}
	CAutoCorrelation *ac;
	try { ac = new CAutoCorrelation(); } catch(...) { ac = NULL; }
	if(ac == NULL) NewException((double)sizeof(CAutoCorrelation), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	ac->Init(n, _correlationDepth, g_bACFFFT);
	
	CxFloatArray *temp;
	try { temp = new CxFloatArray(); } catch(...) { temp = NULL; }
	if(temp == NULL) NewException((double)sizeof(CxFloatArray), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	temp->SetSize(n);
	CxFloatArray *temp2;
	try { temp2 = new CxFloatArray(); } catch(...) { temp2 = NULL; }
	if(temp2 == NULL) NewException((double)sizeof(CxFloatArray), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	temp2->SetSize(n);
	
	mprintf(WHITE, "     [");
	for(i = 0; i < m_iShowMolCount; i++) {
		if(fmodf(i, step) < 1.0f)
			mprintf(WHITE, "#");
		for(j = 0; j < 3; j++) {
			for(k = 0; k < n; k++) {
				temp->GetAt(k) = ((CxVec3Array *)_dipoleCache[i])->GetAt(k)[j];
			}
			ac->AutoCorrelate(temp, temp2);
			for(k = 0; k < _correlationDepth; k++) {
				acf->GetAt(k) += temp2->GetAt(k);
			}
		}
	}
	mprintf(WHITE, "]\n");
// 	if(m_iShowMol == -1) {
// 		for(i = 0; i < _correlationDepth; i++) {
// 			acf->GetAt(i) /= 3.0f;
// 		}
// 	} else {
// 		for(i = 0; i < _correlationDepth; i++) {
// 			acf->GetAt(i) /= 3.0f * m_iShowMolCount;
// 		}
// 	}
// The 3.0f is included in the final normalization
	if(m_iShowMol != -1) {
		for(i = 0; i < _correlationDepth; i++) {
			acf->GetAt(i) /= (float)m_iShowMolCount;
		}
	}
	
	delete ac;
	
	CxFloatArray *ccf = NULL;
	if(_includeCross) {
		mprintf("    Calculating cross-correlation...\n");
		try { ccf = new CxFloatArray(); } catch(...) { ccf = NULL; }
		if(ccf == NULL) NewException((double)sizeof(CxFloatArray), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		ccf->SetSize(_correlationDepth);
		for(i = 0; i < _correlationDepth; i++) {
			ccf->GetAt(i) = 0.0f;
		}
		CCrossCorrelation *cc;
		try { cc = new CCrossCorrelation(); } catch(...) { cc = NULL; }
		if(cc == NULL) NewException((double)sizeof(CxFloatArray), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		cc->Init(n, _correlationDepth, g_bACFFFT);
		CxFloatArray *temp3;
		try { temp3 = new CxFloatArray(); } catch(...) { temp3 = NULL; }
		if(temp3 == NULL) NewException((double)sizeof(CxFloatArray), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		temp3->SetSize(n);
		temp->SetSize(n);
		temp2->SetSize(n);
		
		mprintf(WHITE, "     [");
		step = (float)m_iShowMolCount * (m_iShowMolCount - 1) / 2.0f / 20.0f;
		int c = 0;
		for(i = 0; i < m_iShowMolCount; i++) {
			for(j = i + 1; j < m_iShowMolCount; j++) {
				if(fmodf((float)c++, step) < 1.0f)
					mprintf(WHITE, "#");
				for(k = 0; k < 3; k++) {
					for(l = 0; l < n; l++) {
						temp->GetAt(l) = ((CxVec3Array *)_dipoleCache[i])->GetAt(l)[k];
						temp2->GetAt(l) = ((CxVec3Array *)_dipoleCache[j])->GetAt(l)[k];
					}
					cc->CrossCorrelate(temp, temp2, temp3);
					for(l = 0; l < _correlationDepth; l++) {
						ccf->GetAt(l) += 2.0f * temp3->GetAt(l);
					}
				}
			}
		}
		mprintf(WHITE, "]\n");
// 		for(i = 0; i < _correlationDepth; i++) {
// 			ccf->GetAt(i) /= 3.0f;
// 		}
		
		delete cc;
		delete temp3;
	}
	
	delete temp2;
	
	if(_saveACF) {
		char filename[BUF_SIZE];
#ifdef TARGET_LINUX
		snprintf(filename, BUF_SIZE, "ir_acf_%s.csv", _name);
#else
		sprintf(filename, "ir_acf_%s.csv", _name);
#endif
		mprintf("    Saving autocorrelation function as %s...\n", filename);
		FILE *acfFile = OpenFileWrite(filename, false);
		for(i = 0; i < _correlationDepth; i++) {
			fprintf(acfFile, "%.2f; %.10G\n", i * g_fTimestepLength, acf->GetAt(i));
		}
		fclose(acfFile);
	}
	
	temp->CopyFrom(acf);
	
	if(_windowFunction == 1) {
		for(i = 0; i < temp->GetSize(); i++) {
			temp->GetAt(i) *= powf(cosf((float)i / (temp->GetSize() - 1) / 2.0f * Pi), 2.0f);
		}
	} else if(_windowFunction == 2) {
		for(i = 0; i < temp->GetSize(); i++) {
			temp->GetAt(i) *= expf(-(float)i / _windowFunctionParameter);
		}
	} else if(_windowFunction == 3) {
		for(i = 0; i < temp->GetSize(); i++) {
			temp->GetAt(i) *= expf(-(float)i * i / _windowFunctionParameter / _windowFunctionParameter);
		}
	} else if(_windowFunction != 0) {
		eprintf("Unknown window function.\n");
		abort();
	}
	
	if(_saveACF) {
		char filename[BUF_SIZE];
#ifdef TARGET_LINUX
		snprintf(filename, BUF_SIZE, "ir_acf_windowed_%s.csv", _name);
#else
		sprintf(filename, "ir_acf_windowed_%s.csv", _name);
#endif
		mprintf("    Saving windowed autocorrelation function as %s...\n", filename);
		FILE *acfFile = OpenFileWrite(filename, false);
		for(i = 0; i < _correlationDepth; i++) {
			fprintf(acfFile, "%.2f; %.10G\n", i * g_fTimestepLength, temp->GetAt(i));
		}
		fclose(acfFile);
	}
	
	if(_zeroPadding > 0) {
		for(i = 0; i < _zeroPadding; i++) {
			temp->Add(0.0f);
		}
	}
	
	int oldSize = temp->GetSize();
	temp->SetSize(2 * oldSize);
	for(i = 1; i < oldSize; i++) {
		temp->GetAt(oldSize + i) = temp->GetAt(oldSize - i);
	}
	temp->GetAt(oldSize) = 0.0f;
	
	mprintf("    Fourier transforming autocorrelation function...\n");
	CFFT *fft;
	try { fft = new CFFT(); } catch(...) { fft = NULL; }
	if(fft == NULL) NewException((double)sizeof(CFFT), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	fft->PrepareFFT_C2C(temp->GetSize());
	for(i = 0; i < temp->GetSize(); i++) {
		fft->m_pInput[2*i] = temp->GetAt(i);
		fft->m_pInput[2*i+1] = 0.0f;
	}
	fft->DoFFT();
	
	CxFloatArray *spectrum;
	try { spectrum = new CxFloatArray(); } catch(...) { spectrum = NULL; }
	if(spectrum == NULL) NewException((double)sizeof(CxFloatArray), __FILE__, __LINE__, __PRETTY_FUNCTION__);
	spectrum->SetSize(_specSize);
	for(i = 0; i < _specSize; i++) {
		spectrum->GetAt(i) = 3047.2310f * fft->m_pOutput[2*i] * g_fTimestepLength; // Output in K*cm*km/mol
	}
	
	if(_finiteDifferenceCorrection) {
		float f = _specResolution * g_fTimestepLength * 1.883652e-4f;
		for(i = 1; i < _specSize; i++) {
			spectrum->GetAt(i) *= powf(f * i / sinf(f * i), 2.0f); // Divide by sinc function to correct finite difference derivation
		}
	}
	
	if(_quantumCorrection != 1) {
		for(i = 0; i < _specSize; i++) {
			float factor = 1.0f;
			if(_quantumCorrection == 2) {
				factor = 2.0f / (1.0f + expf(-1.438777f * _specResolution * i / _quantumCorrectionTemperature)) / (1.438777f * _specResolution * i / _quantumCorrectionTemperature) * (1.0f - expf(-1.438777f * _specResolution * i / _quantumCorrectionTemperature));
			} else if(_quantumCorrection == 3) {
				factor = expf(0.719388f * _specResolution * i / _quantumCorrectionTemperature) / (1.438777f * _specResolution * i / _quantumCorrectionTemperature) * (1.0f - expf(-1.438777f * _specResolution * i / _quantumCorrectionTemperature));
			}
			spectrum->GetAt(i) *= factor;
		}
	}
	
	if(_includeCross) {
		char filename[BUF_SIZE];
#ifdef TARGET_LINUX
		snprintf(filename, BUF_SIZE, "ir_spectrum_auto_%s.csv", _name);
#else
		sprintf(filename, "ir_spectrum_auto_%s.csv", _name);
#endif
		mprintf("    Saving autocorrelation spectrum as %s...\n", filename);
		FILE *specFile = OpenFileWrite(filename, false);
		fprintf(specFile, "#Wavenumber (cm^-1); Spectrum (K*cm*km*mol^-1); Integral (K*km*mol^-1)\n");
		double integral = 0.0;
		for(i = 0; i < _specSize; i++) {
			integral += (double)spectrum->GetAt(i) * _specResolution;
			fprintf(specFile, "%.2f; %.8G; %.14G\n", _specResolution * i, spectrum->GetAt(i), integral);
		}
		fclose(specFile);
		
		if(_saveACF) {
			char filename[BUF_SIZE];
#ifdef TARGET_LINUX
			snprintf(filename, BUF_SIZE, "ir_ccf_%s.csv", _name);
#else
			sprintf(filename, "ir_ccf_%s.csv", _name);
#endif
			mprintf("    Saving cross-correlation function as %s...\n", filename);
			FILE *ccfFile = OpenFileWrite(filename, false);
			for(i = 0; i < _correlationDepth; i++) {
				fprintf(ccfFile, "%.2f; %.10G\n", i * g_fTimestepLength, ccf->GetAt(i));
			}
			fclose(ccfFile);
		}
		
		temp->CopyFrom(ccf);
		
		if(_windowFunction == 1) {
			for(i = 0; i < temp->GetSize(); i++) {
				temp->GetAt(i) *= powf(cosf((float)i / temp->GetSize() / 2.0f * Pi), 2.0f);
			}
		} else if(_windowFunction == 2) {
			for(i = 0; i < temp->GetSize(); i++) {
				temp->GetAt(i) *= expf(-(float)i / _windowFunctionParameter);
			}
		} else if(_windowFunction == 3) {
			for(i = 0; i < temp->GetSize(); i++) {
				temp->GetAt(i) *= expf(-(float)i * i / _windowFunctionParameter / _windowFunctionParameter);
			}
		} else if(_windowFunction != 0) {
			eprintf("Unknown window function.\n");
			abort();
		}
		
		if(_saveACF) {
			char filename[BUF_SIZE];
#ifdef TARGET_LINUX
			snprintf(filename, BUF_SIZE, "ir_ccf_windowed_%s.csv", _name);
#else
			sprintf(filename, "ir_ccf_windowed_%s.csv", _name);
#endif
			mprintf("    Saving windowed cross-correlation function as %s...\n", filename);
			FILE *ccfFile = OpenFileWrite(filename, false);
			for(i = 0; i < _correlationDepth; i++) {
				fprintf(ccfFile, "%.2f; %.10G\n", i * g_fTimestepLength, ccf->GetAt(i));
			}
			fclose(ccfFile);
		}
		
		if(_zeroPadding > 0) {
			for(i = 0; i < _zeroPadding; i++) {
				temp->Add(0.0f);
			}
		}
		
		int oldSize = temp->GetSize();
		temp->SetSize(2 * oldSize);
		for(i = 1; i < oldSize; i++) {
			temp->GetAt(oldSize + i) = temp->GetAt(oldSize - i);
		}
		temp->GetAt(oldSize) = 0.0f;
		
		mprintf("    Fourier transforming cross-correlation function...\n");
		fft->PrepareFFT_C2C(temp->GetSize());
		for(i = 0; i < temp->GetSize(); i++) {
			fft->m_pInput[2*i] = temp->GetAt(i);
			fft->m_pInput[2*i+1] = 0.0f;
		}
		fft->DoFFT();
		
		CxFloatArray *ccSpectrum;
		try { ccSpectrum = new CxFloatArray(); } catch(...) { ccSpectrum = NULL; }
		if(ccSpectrum == NULL) NewException((double)sizeof(CxFloatArray), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		ccSpectrum->SetSize(_specSize);
		for(i = 0; i < _specSize; i++) {
			ccSpectrum->GetAt(i) = 3047.2310f * fft->m_pOutput[2*i] * g_fTimestepLength;
		}
		
		if(_quantumCorrection != 1) {
			for(i = 0; i < _specSize; i++) {
				float factor = 1.0f;
				if(_quantumCorrection == 2) {
					factor = 2.0f / (1.0f + expf(-1.438777f * _specResolution * i / _quantumCorrectionTemperature)) / (1.438777f * _specResolution * i / _quantumCorrectionTemperature) * (1.0f - expf(-1.438777f * _specResolution * i / _quantumCorrectionTemperature));
				} else if(_quantumCorrection == 3) {
					factor = expf(0.719388f * _specResolution * i / _quantumCorrectionTemperature) / (1.438777f * _specResolution * i / _quantumCorrectionTemperature) * (1.0f - expf(-1.438777f * _specResolution * i / _quantumCorrectionTemperature));
				}
				ccSpectrum->GetAt(i) *= factor;
			}
		}
		
#ifdef TARGET_LINUX
		snprintf(filename, BUF_SIZE, "ir_spectrum_cross_%s.csv", _name);
#else
		sprintf(filename, "ir_spectrum_cross_%s.csv", _name);
#endif
		mprintf("    Saving cross-correlation spectrum as %s...\n", filename);
		specFile = OpenFileWrite(filename, false);
		fprintf(specFile, "#Wavenumber (cm^-1); Spectrum (K*cm*km*mol^-1); Integral (K*km*mol^-1)\n");
		integral = 0.0;
		for(i = 0; i < _specSize; i++) {
			integral += ccSpectrum->GetAt(i) * _specResolution;
			fprintf(specFile, "%.2f; %.8G; %.14G\n", _specResolution * i, ccSpectrum->GetAt(i), integral);
		}
		fclose(specFile);
		
		for(i = 0; i < _specSize; i++) {
			spectrum->GetAt(i) += ccSpectrum->GetAt(i);
		}
		
		delete ccSpectrum;
	}
	
	char filename[BUF_SIZE];
#ifdef TARGET_LINUX
	snprintf(filename, BUF_SIZE, "ir_spectrum_%s.csv", _name);
#else
	sprintf(filename, "ir_spectrum_%s.csv", _name);
#endif
	mprintf("    Saving spectrum as %s...\n", filename);
	FILE *specFile = OpenFileWrite(filename, false);
	fprintf(specFile, "#Wavenumber (cm^-1); Spectrum (K*cm*km*mol^-1); Integral (K*km*mol^-1)\n");
	double integral = 0.0;
	for(i = 0; i < _specSize; i++) {
		integral += spectrum->GetAt(i) * _specResolution;
		fprintf(specFile, "%.2f; %.8G; %.14G\n", _specResolution * i, spectrum->GetAt(i), integral);
	}
	fclose(specFile);
	
	delete acf;
	if(_includeCross) {
		delete ccf;
	}
	delete temp;
	delete fft;
	delete spectrum;
	
	for(i = 0; i < _dipoleCache.GetSize(); i++)
		delete (CxVec3Array *)_dipoleCache[i];
}

bool gatherIR() {
	g_bDipole = true;
	ParseDipole();
	
	while(true) {
		mprintf(YELLOW, "\n>>> IR Observation %d >>>\n\n", g_IRObserv.GetSize() + 1);
		
		CIRObservation *obs;
		try { obs = new CIRObservation(); } catch(...) { obs = NULL; }
		if(obs == NULL) NewException((double)sizeof(CIRObservation), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		g_IRObserv.Add(obs);
		
		mprintf(YELLOW, "<<< End of IR Observation %d <<<\n\n", g_IRObserv.GetSize());
		
		if(!AskYesNo("    Add another observation (y/n)? [no] ", false))
			break;
		mprintf("\n");
	}
	
	if(AskYesNo("\n    Compute IR spectrum of whole system (y/n) [no] ", false)) {
		mprintf(YELLOW, "\n>>> Global IR Observation >>>\n\n");
		
		CIRObservation *obs;
		try { obs = new CIRObservation(true); } catch(...) { obs = NULL; }
		if(obs == NULL) NewException((double)sizeof(CIRObservation), __FILE__, __LINE__, __PRETTY_FUNCTION__);
		g_IRObserv.Add(obs);
		
		mprintf(YELLOW, "<<< End of Global IR Observation <<<\n\n");
	}
	
	return true;
}

bool initializeIR() {
	int i;
	for(i = 0; i < g_IRObserv.GetSize(); i++) {
		((CIRObservation *)g_IRObserv[i])->initialize();
	}
	return true;
}

void processIR(CTimeStep* ts) {
	(void)ts;
	int i;
	for(i = 0; i < g_IRObserv.GetSize(); i++) {
		((CIRObservation *)g_IRObserv[i])->process();
	}
}

void finalizeIR() {
	int i;
	for(i = 0; i < g_IRObserv.GetSize(); i++) {
		mprintf(YELLOW, "\n>>> IR Observation %d >>>\n\n", i+1);
		((CIRObservation *)g_IRObserv[i])->finalize();
		delete (CIRObservation *)g_IRObserv[i];
		mprintf(YELLOW, "\n<<< End of IR Observation %d <<<\n\n", i+1);
	}
}

