/*
 * Ticker.app - A ticker dockapp

 * Copyright (c) 1999 Jiro Sekiba<jir@sekiba.com>
 * Copyright (c) 2001,2002 Seiichi SATO <ssato@sh.rim.or.jp>

 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#if HAVE_X11_XLOCALE_H
# include <X11/Xlocale.h>
#else
#  if HAVE_LOCALE_H
#    include <locale.h>
#  endif
#endif

#include "dockapp.h"
#include "message.h"
#include "font.h"
#include "popup.h"
#include "backdrop.xpm"
#include "buttons.xpm"

#define POPUP_OPENING 0x01
#define POPUP_CLOSED  0x00
#define DEFAULT_FONT   "-*-helvetica-bold-r-normal-*-17-*-100-100-p-*"
#define DEFAULT_FONT_P "-*-helvetica-bold-r-normal-*-24-*-75-75-p-*"
#define SCROLL_SPEED_FAST 10
#define SCROLL_SPEED_MID  20
#define SCROLL_SPEED_SLOW 30
#define NOT_UPDATE_COUNT 20

/* kinds of button */
#define BUTTON_NONE	0
#define BUTTON_POPUP	1
#define BUTTON_NEXT	2
#define BUTTON_EXIT	3
#define BUTTON_KNOB	4
#define SCROLLER_L	5
#define SCROLLER_R	6
#define BUTTON_ALL	7

#define SIZE	    56
#define WINDOWED_BG "  \tc #AEAAAE"

/* private variables */
static Pixmap pix_buffer;
static Pixmap pix_backdrop;
static Pixmap pix_buttons;
static Pixmap pix_msg;
static Pixmap mask;
static char *display_name = "";
static char *bg_color = NULL;
static char *fg_color = NULL;
static char *filename = NULL;
static char *fontname = NULL;
static char *fontname_popup = NULL;
static char *popup_position = NULL;
static int mode_tail = False;
static int popup_state = POPUP_CLOSED;
static int pre_pressed_btn = BUTTON_NONE;
static FontI18N *font;
static char *message = NULL;
static int msg_draw_position_x = 52;
static int current_scroller_knob_pos = 0;	/* 0 to 33 */
static int scroll_speed = SCROLL_SPEED_MID;
static int pix_msg_width = -52;

/* private function prototype */
static void msg_update(void);
static void msg_redraw(void);
static int get_pointer_place(int x, int y);
static void redraw_button(int button, int pressed);
static void redraw_scroller(void);
static void parse_arguments(int argc, char **argv);
static void print_help(char *prog);

int
main(int argc, char **argv)
{
    XEvent event;
    int pressed_pos_on_knob = BUTTON_NONE;
    XpmColorSymbol colors = { "BG", NULL, 0 };
    int ncolor = 0;
    int not_update_count = 0;

#if defined(HAVE_SETLOCALE) || defined(HAVE_XSETLOCALE)
    /* set program's locale */
    setlocale(LC_ALL, "");

    /* check X server (locale support?) */
    if (!XSupportsLocale())
	setlocale(LC_ALL, "C");
#endif

    /* Parse Command-Line */
    parse_arguments(argc, argv);

    /* Initialize Application */
    dockapp_open_window(display_name, PACKAGE, SIZE, SIZE, argc, argv);
    dockapp_set_eventmask(ButtonPressMask | ButtonReleaseMask |
			  Button1MotionMask);
    if (bg_color) {
	colors.pixel = dockapp_getcolor(bg_color);
	ncolor = 1;
    }

    /* change raw xpm data to pixmap */
    if (dockapp_iswindowed)
	backdrop_xpm[1] = WINDOWED_BG;
    dockapp_xpm2pixmap(backdrop_xpm, &pix_backdrop, &mask, &colors, ncolor);
    dockapp_xpm2pixmap(buttons_xpm, &pix_buttons, NULL, NULL, 0);
    /* shape window */
    if (!dockapp_iswindowed)
	dockapp_setshape(mask, 0, 0);
    /* pix_buffer : draw area */
    pix_buffer = dockapp_XCreatePixmap(SIZE, SIZE);

    /* Activate Application */
    dockapp_set_background(pix_backdrop);
    dockapp_show();

    /* Initialize viewing area */
    pix_msg = dockapp_XCreatePixmap(52, 23);
    dockapp_copyarea(pix_backdrop, pix_msg, 2, 2, 52, 23, 0, 0);
    font = CreateFont(display, fontname ? fontname : DEFAULT_FONT);
    XSetForeground(display, DefaultGC(display, DefaultScreen(display)),
		   dockapp_getcolor(fg_color ? fg_color : "white"));
    if (font->use_fontset == False)
	XSetFont(display, DefaultGC(display, DefaultScreen(display)),
		 font->fontstruct->fid);

    /* Initialize message */
    Msg_init(filename, mode_tail);

    dockapp_copyarea(pix_backdrop, pix_buffer, 0, 0, SIZE, SIZE, 0, 0);
    redraw_scroller();
    redraw_button(BUTTON_NEXT, False);
    /* Initialize popup window */
    PU_WindowInitialize(display,
			fontname_popup ? fontname_popup : DEFAULT_FONT_P,
			dockapp_getcolor(fg_color ? fg_color : "white"),
			dockapp_getcolor(bg_color ? bg_color : "#525252"));
    PU_SetShowPosition(popup_position);
    msg_update();
    if (popup_state == POPUP_OPENING) {
	PU_Show(message);
	redraw_button(BUTTON_POPUP, True);
    }

    /* Main loop */
    for (;;) {
	if (dockapp_nextevent_or_timeout(&event, scroll_speed) == False) {
	    if (pre_pressed_btn != BUTTON_KNOB &&
		pre_pressed_btn != SCROLLER_R &&
		pre_pressed_btn != SCROLLER_L) {
		if (not_update_count > 0)
		    not_update_count--;
		if (not_update_count == 0){
		    msg_update();
		    dockapp_copy2window(pix_buffer);
		}
	    }
	} else {
	    int btn = get_pointer_place(event.xbutton.x, event.xbutton.y);

	    if (event.type == ButtonPress
		&& event.xbutton.button == Button1) {
		switch (btn) {
		case BUTTON_EXIT:	/* pressed exit button */
		case BUTTON_NEXT:	/* pressed next button */
		    redraw_button(btn, True);
		    dockapp_copy2window(pix_buffer);
		    pre_pressed_btn = btn;
		    break;
		case BUTTON_POPUP:	/* pressed popup button */
		    if (popup_state == POPUP_CLOSED) {
			redraw_button(BUTTON_POPUP, True);
			dockapp_copy2window(pix_buffer);
			popup_state = POPUP_OPENING;
			PU_Show(message);
		    } else {
			redraw_button(BUTTON_POPUP, False);
			dockapp_copy2window(pix_buffer);
			popup_state = POPUP_CLOSED;
			PU_Hide();
		    }
		    break;
		case BUTTON_KNOB:	/* pressed scroller knob */
		    pressed_pos_on_knob =
			event.xbutton.x - current_scroller_knob_pos - 3;
		    pre_pressed_btn = BUTTON_KNOB;
		    break;
		case SCROLLER_L:	/* pressed scroller left */
		    current_scroller_knob_pos -= 8;
		    if (current_scroller_knob_pos < 3)
			current_scroller_knob_pos = 0;
		    redraw_scroller();
		    msg_redraw();
		    dockapp_copy2window(pix_buffer);
		    pre_pressed_btn = SCROLLER_L;
		    break;
		case SCROLLER_R:	/* pressed scroller right */
		    current_scroller_knob_pos += 8;
		    if (current_scroller_knob_pos > 33)
			current_scroller_knob_pos = 33;
		    redraw_scroller();
		    msg_redraw();
		    dockapp_copy2window(pix_buffer);
		    pre_pressed_btn = SCROLLER_R;
		    break;
		default:
		    break;
		}
	    } else if (event.type == ButtonRelease &&
		       event.xbutton.button == Button1) {
		if (btn == pre_pressed_btn) {
		    switch (btn) {	/* released exit button */
		    case BUTTON_EXIT:
			redraw_button(BUTTON_EXIT, False);
			dockapp_copy2window(pix_buffer);
			exit(0);
			break;
		    case BUTTON_NEXT:	/* released next button */
			redraw_button(BUTTON_NEXT, False);
			current_scroller_knob_pos = 0;
			redraw_scroller();
			current_scroller_knob_pos = 33;
			msg_redraw();
			dockapp_copy2window(pix_buffer);
			pre_pressed_btn = BUTTON_NONE;
			break;
		    case BUTTON_KNOB:	/* released scrollknob */
		    case SCROLLER_R:	/* released scrollvier */
		    case SCROLLER_L:	/* released scrollview */
			pre_pressed_btn = BUTTON_NONE;
			break;
		    default:
			break;
		    }
		} else {	/* released button ( != pressed button) */

		    switch (pre_pressed_btn) {
		    case BUTTON_NEXT:
		    case BUTTON_EXIT:
			redraw_button(pre_pressed_btn, False);
			dockapp_copy2window(pix_buffer);
			break;
		    default:
			break;
		    }
		    pre_pressed_btn = BUTTON_NONE;
		}
	    } else if (event.type == MotionNotify) {
		switch (pre_pressed_btn) {
		    int pre_pos;
		case BUTTON_KNOB:
		    pre_pos = current_scroller_knob_pos;
		    current_scroller_knob_pos =
			event.xmotion.x - pressed_pos_on_knob - 3;
		    if (current_scroller_knob_pos < 0)
			current_scroller_knob_pos = 0;
		    if (current_scroller_knob_pos > 33)
			current_scroller_knob_pos = 33;
		    if (pre_pos != current_scroller_knob_pos) {
			redraw_scroller();
			msg_redraw();
			dockapp_copy2window(pix_buffer);
		    }
		    break;
		case BUTTON_EXIT:
		case BUTTON_NEXT:
		    redraw_button(pre_pressed_btn,
				  btn == pre_pressed_btn ? True : False);
		    dockapp_copy2window(pix_buffer);
		    break;
		default:
		    break;
		}
	    } else if (event.type == ButtonPress
		     && event.xbutton.button == Button4) {
	    /* wheel mouse up */
		current_scroller_knob_pos += 2;
		if (current_scroller_knob_pos > 33)
		    current_scroller_knob_pos = 33;
		redraw_scroller();
		msg_redraw();
		dockapp_copy2window(pix_buffer);
		not_update_count = NOT_UPDATE_COUNT;
	    } else if (event.type == ButtonPress
		     && event.xbutton.button == Button5) {
	    /* wheel mouse down */
		current_scroller_knob_pos -= 2;
		if (current_scroller_knob_pos < 0)
		    current_scroller_knob_pos = 0;
		redraw_scroller();
		msg_redraw();
		dockapp_copy2window(pix_buffer);
		not_update_count = NOT_UPDATE_COUNT;
	    }
	}
    }				/* end of main loop */
    return 0;
}

static void chk_file_change(void)
{
    static struct stat st;
    static time_t pre_mtime;
    static time_t pre_ctime;

    if (filename == NULL)
	return;

    if (stat(filename, &st) == 0) {
	if (st.st_mtime != pre_mtime || st.st_ctime != pre_ctime) {
	    Msg_reload(filename, mode_tail);
	    if ((stat(filename, &st)) == 0)
		pre_mtime = st.st_mtime, pre_ctime = st.st_ctime;
	}
    }
}

static void redraw_scroller(void)
{
    dockapp_copyarea(pix_backdrop, pix_buffer, 3, 27, 50, 11, 3, 27);
    dockapp_copyarea(pix_buttons, pix_buffer, 0, 0, 17, 11,
		     current_scroller_knob_pos + 3, 27);
}

static void redraw_button(int button, int pressed)
{
    switch (button) {
    case BUTTON_POPUP:
	/* draw button to buffer */
	if (pressed)
	    dockapp_copyarea(pix_buttons, pix_buffer, 0, 11, 16, 12, 0, 44);
	else
	    dockapp_copyarea(pix_backdrop, pix_buffer, 0, 44, 16, 22, 0, 44);
	break;

    case BUTTON_NEXT:
	if (pressed)
	    dockapp_copyarea(pix_buttons, pix_buffer, 0, 23, 17, 22, 18, 44);
	else
	    dockapp_copyarea(pix_backdrop, pix_buffer, 18, 44, 17, 22, 18, 44);
	break;

    case BUTTON_EXIT:
	if (pressed)
	    dockapp_copyarea(pix_buttons, pix_buffer, 0, 35, 12, 12, 44, 44);
	else
	    dockapp_copyarea(pix_backdrop, pix_buffer, 44, 44, 12, 12, 44, 44);
	break;

    case BUTTON_ALL:
	dockapp_copyarea(pix_backdrop, pix_buffer, 0, 44, 56, 12, 0, 44);
	break;
    }
}

static int get_pointer_place(int x, int y)
{
    if (x >= 18 && x <= 34 && y >= 44 && y <= 55)
	return BUTTON_NEXT;
    if (x >= 0 && x <= 15 && y >= 44 && y <= 55)
	return BUTTON_POPUP;
    if (x >= 44 && x <= 55 && y >= 44 && y <= 55)
	return BUTTON_EXIT;
    if (x >= 3 + current_scroller_knob_pos &&
	x <= 19 + current_scroller_knob_pos && y >= 27 && y <= 37)
	return BUTTON_KNOB;
    if (y >= 27 && y <= 37 && x < current_scroller_knob_pos + 3)
	return SCROLLER_L;
    if (y >= 27 && y <= 37 && x > current_scroller_knob_pos + 19)
	return SCROLLER_R;

    return BUTTON_NONE;
}

/* called by timer */
static void msg_update()
{
    static int pre_pos;

    if (msg_draw_position_x == -pix_msg_width) {
	chk_file_change();
	message = Get_message();
	pix_msg_width = GetStringWidth(font, message, strlen(message));
	if (popup_state == POPUP_OPENING)
	    PU_Show(message);
	msg_draw_position_x = 52;
	current_scroller_knob_pos = 0;
    }

    dockapp_copyarea(pix_backdrop, pix_msg, 2, 2, 52, 23, 0, 0);
    DrawString(display, pix_msg, font,
	       DefaultGC(display, DefaultScreen(display)), msg_draw_position_x,
	       12 - font->height / 2 + font->y, message, strlen(message));
    dockapp_copyarea(pix_msg, pix_buffer, 0, 0, 52, 23, 2, 2);

    pre_pos = current_scroller_knob_pos;
    current_scroller_knob_pos =
	(34 * (52 - msg_draw_position_x)) / (52 + pix_msg_width);
    if (pre_pos != current_scroller_knob_pos)
	redraw_scroller();
    msg_draw_position_x--;
}

static void msg_redraw(void)
{
    msg_draw_position_x =
	52 - (current_scroller_knob_pos * (52 + pix_msg_width) / 33);
    dockapp_copyarea(pix_backdrop, pix_msg, 2, 2, 52, 23, 0, 0);
    DrawString (display, pix_msg, font,
		DefaultGC(display, DefaultScreen(display)),
		msg_draw_position_x, 12 - font->height / 2 + font->y,
		message, strlen(message));
    dockapp_copyarea(pix_msg, pix_buffer, 0, 0, 52, 23, 2, 2);
}

static void parse_arguments(int argc, char **argv)
{
    int i;
    for (i = 1; i < argc; i++) {
	if (!strcmp(argv[i], "--help") || !strcmp(argv[i], "-h"))
	    print_help(argv[0]), exit(0);

	else if (!strcmp(argv[i], "--version") || !strcmp(argv[i], "-v"))
	    printf("%s version %s\n", PACKAGE, VERSION), exit(0);

	else if (!strcmp(argv[i], "--display") || !strcmp(argv[i], "-d"))
	    display_name = i < argc ? argv[i + 1] : NULL, i++;

	else if (!strcmp(argv[i], "--background")
		 || !strcmp(argv[i], "-bg"))
	    bg_color = i < argc ? argv[i + 1] : NULL, i++;

	else if (!strcmp(argv[i], "--foreground")
		 || !strcmp(argv[i], "-fg"))
	    fg_color = i < argc ? argv[i + 1] : NULL, i++;

	else if (!strcmp(argv[i], "--file") || !strcmp(argv[i], "-f"))
	    filename = i < argc ? argv[i + 1] : NULL, i++;

	else if (!strcmp(argv[i], "--font") || !strcmp(argv[i], "-fn"))
	    fontname = i < argc ? argv[i + 1] : NULL, i++;

	else if (!strcmp(argv[i], "--font-popup")
		 || !strcmp(argv[i], "-fp"))
	    fontname_popup = i < argc ? argv[i + 1] : NULL, i++;

	else if (!strcmp(argv[i], "--tail") || !strcmp(argv[i], "-t"))
	    mode_tail = True;

	else if (!strcmp(argv[i], "--windowed") || !strcmp(argv[i], "-w"))
	    dockapp_iswindowed = True;

	else if (!strcmp(argv[i], "--popup-pos")
		 || !strcmp(argv[i], "-pp"))
	    popup_position = i < argc ? argv[i + 1] : NULL, i++;

	else if (!strcmp(argv[i], "--popup") || !strcmp(argv[i], "-p"))
	    popup_state = POPUP_OPENING;

	else if (!strcmp(argv[i], "--speed-fast")
		 || !strcmp(argv[i], "-sf"))
	    scroll_speed = SCROLL_SPEED_FAST;

	else if (!strcmp(argv[i], "--speed-slow")
		 || !strcmp(argv[i], "-ss"))
	    scroll_speed = SCROLL_SPEED_SLOW;

	else if (!strcmp(argv[i], "--broken-wm")
		 || !strcmp(argv[i], "-bw"))
	    dockapp_isbrokenwm = True;

	else {
	    fprintf(stderr, "%s: unrecognized option '%s'\n", argv[0],
		    argv[i]);
	    print_help(argv[0]), exit(1);
	}
    }
}

static void print_help(char *prog)
{
    printf("Usage : %s [OPTIONS]\n", prog);
    printf("Ticker.app - A ticker dockapp\n");
    printf("  -f,  --file <file>            file to display\n");
    printf("  -t,  --tail                   run the application in tail mode\n");
    printf("  -d,  --display <string>       display to use\n");
    printf("  -fg, --foreground <color>     foreground color (white is default)\n");
    printf("  -bg, --background <color>     background color (rgb:52/52/52 is defalt)\n");
    printf("  -p,  --popup                  run the application with popup window\n");
    printf("  -pp, --popup-pos [+|-]x[+|-]y position of popup window\n");
    printf("  -fn, --font <fontname>        font to use\n");
    printf("  -fp, --font-popup <fontname>  font to use in popup window\n");
    printf("  -sf, --speed-fast             fast scrolling\n");
    printf("  -ss, --speed-slow             slow scrolling\n");
    printf("  -bw, --broken-wm              activate broken window manager fix\n");
    printf("  -h,  --help                   show this help text and exit\n");
    printf("  -v,  --version                show program version and exit\n");
    printf("  -w,  --windowed               run the application in windowed mode\n");
}
