/*
 * popup.c - Ticker.app popup window module
 *
 * 2001 Copyright (c) Seiichi SATO <ssato@sh.rim.or.jp>
 *
 * licensed under the GPL.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <X11/Xlib.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <stdio.h>
#if defined(HAVE_STRING_H)
#include <string.h>
#elif defined(HAVE_STRINGS_H)
#include <strings.h>
#endif
#include <unistd.h>

#include "popup.h"
#include "font.h"
#include "mem.h"

/* private variables */
static Window window = None;
static Display *display = NULL;
static GC gc = NULL;
static int depth = 0;
static Visual *visual = NULL;
static FontI18N *font;
static unsigned long bg_color;
static unsigned long fg_color;
static unsigned int pos_x = 0, pos_y = 0;
static int pos_x_sin = -1, pos_y_sin = -1;

/* initialize */
void
PU_WindowInitialize(Display * dpy, char *fontname, unsigned long fg,
		    unsigned long bg)
{
    XSetWindowAttributes attribs;
    unsigned long vmask;

    display = dpy;
    depth = DefaultDepth(dpy, DefaultScreen(dpy));
    visual = DefaultVisual(dpy, DefaultScreen(dpy));
    fg_color = fg;
    bg_color = bg;

    vmask = CWSaveUnder | CWOverrideRedirect | CWColormap |
	CWBackPixel | CWBorderPixel;
    attribs.save_under = True;
    attribs.override_redirect = True;
    attribs.colormap = DefaultColormap(dpy, DefaultScreen(dpy));
    attribs.background_pixel = 0;
    attribs.border_pixel = 0;

    window =
	XCreateWindow(dpy, DefaultRootWindow(dpy), 1, 1, 10, 10, 1, depth,
		      CopyFromParent, visual, vmask, &attribs);

    gc = XCreateGC(display, window, 0, NULL);
    font = CreateFont(dpy, fontname);
    if (!font->use_fontset)
	XSetFont(display, gc, font->fontstruct->fid);
}

/*
 * set position
 *
 * e.g.)
 * "-120+40" ==> pos_x:120, pos_x_sin: -1
 *               pos_y: 40, pos_y_sin:  1
 */
void PU_SetShowPosition(char *position)
{
    char *value_x = NULL, *value_y = NULL;
    char *p;
    size_t len;
    size_t lenx, leny;

    if (position == NULL)
	return;

    len = strlen(position);

    if (strlen(position) == 0)
	return;

    if (position[0] == '-') {
	position[0] = ' ';
	pos_x_sin = -1;
    } else if (position[0] == '+') {
	position[0] = ' ';
	pos_x_sin = 1;
    } else
	pos_x_sin = 1;

    if ((p = strchr(position, '-')) != NULL) {
	lenx = p - position;
	value_x = malloc_wrap(sizeof(char) * (lenx + 1));
	strncpy(value_x, position, lenx);
	value_x[len] = '\0';

	leny = len - lenx;
	value_y = malloc_wrap(sizeof(char) * (leny + 1));
	strncpy(value_y, p, leny + 1);
	pos_y_sin = -1;
	value_y[0] = ' ';
    } else if ((p = strchr(position, '+')) != NULL) {
	lenx = p - position;
	value_x = malloc_wrap(sizeof(char) * (lenx + 1));
	strncpy(value_x, position, lenx);
	value_x[len] = '\0';

	leny = len - lenx;
	value_y = malloc_wrap(sizeof(char) * (leny + 1));
	strncpy(value_y, p, leny + 1);
	pos_y_sin = 1;
	value_y[0] = ' ';
    } else {
	value_x = malloc_wrap(sizeof(char) * len + 1);
	strcpy(value_x, position);
	pos_y_sin = 1;
    }

    if (value_x != NULL) {
	sscanf(value_x, "%u", &pos_x);
	free(value_x);
    }

    if (value_y != NULL) {
	sscanf(value_y, "%u", &pos_y);
	free(value_y);
    }
}

static void
convert_popup_position(unsigned int x, unsigned int y, int x_sin,
		       int y_sin, unsigned int w, unsigned int h,
		       unsigned int *draw_pos_x, unsigned int *draw_pos_y)
{
    unsigned int root_w, root_h;

    Window dummy1;
    int dummy2, dummy3;
    unsigned int dummy4, dummy5;

    XGetGeometry(display, DefaultRootWindow(display),
		 &dummy1, &dummy2, &dummy3, &root_w, &root_h, &dummy4,
		 &dummy5);

    if (x_sin < 0)
	x = root_w - w - x - 1;
    if (y_sin < 0)
	y = root_h - h - y - 1;

    if (x < 0)
	x = 0;
    if (y < 0)
	y = 0;
    if (x > root_w)
	x = root_w - w - 1;
    if (y > root_h)
	y = root_h - h - 1;

    *draw_pos_x = x;
    *draw_pos_y = y;
}

static void draw_balloon(Pixmap pix, GC gc, int x, int y, int w, int h)
{
    int rad = h * 3 / 10;

    XFillArc(display, pix, gc, x, y, rad, rad, 90 * 64, 90 * 64);
    XFillArc(display, pix, gc, x, y + h - 1 - rad, rad, rad, 180 * 64,
	     90 * 64);

    XFillArc(display, pix, gc, x + w - 1 - rad, y, rad, rad, 0 * 64,
	     90 * 64);
    XFillArc(display, pix, gc, x + w - 1 - rad, y + h - 1 - rad, rad, rad,
	     270 * 64, 90 * 64);

    XFillRectangle(display, pix, gc, x, y + rad / 2, w, h - rad);
    XFillRectangle(display, pix, gc, x + rad / 2, y, w - rad, h);
}

void PU_Show(char *text)
{
    Pixmap pixmap;
    Pixmap mask;
    GC mono_gc = NULL;
    unsigned int space;
    unsigned int width;
    unsigned int height;
    unsigned int draw_pos_x;
    unsigned int draw_pos_y;

    space = font->height * 0.5;
    width = GetStringWidth(font, text, strlen(text)) + space;
    height = font->height + space;

    mask = XCreatePixmap(display, window, width, height, 1);
    mono_gc = XCreateGC(display, mask, 0, NULL);
    XSetForeground(display, mono_gc, 0);
    XFillRectangle(display, mask, mono_gc, 0, 0, width, height);
    XSetForeground(display, mono_gc, 1);
    draw_balloon(mask, mono_gc, 0, 0, width, height);

    pixmap = XCreatePixmap(display, window, width, height, depth);
    XSetForeground(display, gc, fg_color);
    XFillRectangle(display, pixmap, gc, 0, 0, width, height);
    XSetForeground(display, gc, bg_color);
    draw_balloon(pixmap, gc, 2, 2, width - 4, height - 4);

    XSetForeground(display, gc, fg_color);
    DrawString(display, pixmap, font, gc, space / 2 + 2,
	       (height - font->height) / 2 + font->y + 1, text,
	       strlen(text));

    XSetWindowBackgroundPixmap(display, window, pixmap);
    XCopyArea(display, pixmap, window, gc, 0, 0, width, height, 0, 0);
    XResizeWindow(display, window, width, height);
    XShapeCombineMask(display, window, ShapeBounding, 0, 0, mask,
		      ShapeSet);

    convert_popup_position(pos_x, pos_y, pos_x_sin, pos_y_sin, width,
			   height, &draw_pos_x, &draw_pos_y);
    XMoveWindow(display, window, draw_pos_x, draw_pos_y);
    XMapRaised(display, window);

    XFlush(display);

    XFreePixmap(display, pixmap);
    XFreePixmap(display, mask);
    XFreeGC(display, mono_gc);
}

void PU_Hide(void)
{
    XUnmapWindow(display, window);
}
