/***************************************************************************
                             th-main.c
                             ---------
    begin                : Mon Nov 08 2004
    copyright            : (C) 2004 by Tim-Philipp Mller
    email                : t.i.m@orange.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
 
#include "th-app-window.h"
#include "th-utils.h"

#include "gst-plugins/th-plugins.h"

#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include <gst/gst.h>
#include <gst/pbutils/pbutils.h>

#include <glib/gi18n.h>

#include <string.h>

/***************************************************************************
 *
 *   check_required_plugins
 *
 ***************************************************************************/

typedef enum {
  TH_GST_PLUGINS_CORE,
  TH_GST_PLUGINS_BASE,
  TH_GST_PLUGINS_GOOD,
  TH_GST_PLUGINS_UGLY
} ThGstPackageName;

static const gchar *ver_reqs[] = {
  GST_PLUGINS_CORE_REQ,
  GST_PLUGINS_BASE_REQ,
  GST_PLUGINS_GOOD_REQ,
  GST_PLUGINS_UGLY_REQ
};

static gboolean
check_required_plugins (void)
{
	struct {
		const gchar      *plugin_name;
		const gchar      *feature_name;
		ThGstPackageName   package;
	} req_plugins[] = { 
	 { "dvdreadsrc", "dvdreadsrc", TH_GST_PLUGINS_UGLY },
	 { "mpegstream", "dvddemux", TH_GST_PLUGINS_UGLY },
	 { "mpeg2dec", "mpeg2dec", TH_GST_PLUGINS_UGLY },
	 { "a52dec", "a52dec", TH_GST_PLUGINS_UGLY },
	 /* { "dtsdec", "dtsdec", TH_GST_PLUGINS_BAD }, */
	 /* { "dvdlpcmdec", "dvdlpcmdec", TH_GST_PLUGINS_UGLY }, */
	 { "theora", "theoraenc", TH_GST_PLUGINS_BASE },
	 { "vorbis", "vorbisenc", TH_GST_PLUGINS_BASE },
	 { "ogg", "oggmux", TH_GST_PLUGINS_BASE },
	 { "ffmpegcolorspace", "ffmpegcolorspace", TH_GST_PLUGINS_BASE },
	 { "gstaudioconvert", "audioconvert", TH_GST_PLUGINS_BASE },
	 { "audioresample", "audioresample", TH_GST_PLUGINS_BASE },
	 { "videobox", "videobox", TH_GST_PLUGINS_GOOD },
	 { "videorate", "videorate", TH_GST_PLUGINS_BASE },
	/* { "deinterlace", "deinterlace", TH_GST_PLUGINS_XYZ }, */
	 { "coreelements", "filesink", TH_GST_PLUGINS_CORE },
	 { "coreelements", "identity", TH_GST_PLUGINS_CORE }
	};
  
	GString *missing, *too_old;
	guint    i;
	
	missing = g_string_new (NULL);
	too_old = g_string_new (NULL);

	/* fill in required versions */
	for (i = 0; i < G_N_ELEMENTS (req_plugins); ++i)
	{
		ThGstPackageName package = req_plugins[i].package;
		guint major = 0;
		guint minor = 10;
		guint micro = 0;

		if (package >= 0 && package < G_N_ELEMENTS (ver_reqs)) {
			sscanf (ver_reqs[req_plugins[i].package],
			        "%u.%u.%u", &major, &minor, &micro);
		}

		if (!gst_default_registry_check_feature_version (req_plugins[i].feature_name,
                                                                 0, 10, 0)) {
			g_string_append_printf (missing, "  %s\n", req_plugins[i].feature_name);
			continue;
		}

		if (!gst_default_registry_check_feature_version (req_plugins[i].feature_name,
                                                                 major, minor, micro)) {
			g_string_append_printf (too_old, "  %s (need v%u.%u.%u)\n", 
			                        req_plugins[i].feature_name,
			                        major, minor, micro);
		}
	}

	if (missing->len > 0 || too_old->len > 0)
	{
		GtkWidget *dlg;
		GString *errmsg;
		 
		errmsg = g_string_new (NULL);
		
		g_string_append_printf (errmsg, "\n<b>%s</b>\n\n", _("Missing or outdated GStreamer plugins"));
		g_string_append (errmsg, _("You seem to be missing some GStreamer plugins, or\n"
		                           "some of your GStreamer plugins are too old, or\n"
		                           "the GStreamer registry needs to be rebuilt by\n"
		                           "running 'gst-register' as root.\n"));
		
		g_string_append (errmsg, "\n\n");
		
		if (missing->len > 0)
		{
			g_string_append_printf (errmsg, "<b>%s</b>\n%s\n\n", _("Missing GStreamer plugins:"), missing->str);
			g_string_append (errmsg, "\n\n");
		}
		
		if (too_old->len > 0)
		{
			g_string_append_printf (errmsg, "<b>%s</b>\n%s\n\n", _("Outdated GStreamer plugins:"), too_old->str);
			g_string_append (errmsg, "\n\n");
		}

		dlg = gtk_message_dialog_new_with_markup (NULL, GTK_DIALOG_MODAL,
		                                          GTK_MESSAGE_ERROR,
		                                          GTK_BUTTONS_CLOSE,
		                                          errmsg->str);
		
		(void) gtk_dialog_run (GTK_DIALOG (dlg));

		gtk_widget_destroy (dlg);

		g_string_free (missing, TRUE);
		g_string_free (too_old, TRUE);
		g_string_free (errmsg, TRUE);
		
		return FALSE;
	}
	
	g_string_free (missing, TRUE);
	g_string_free (too_old, TRUE);
	
	return TRUE;
}

/***************************************************************************
 *
 *   output_log_line
 *
 ***************************************************************************/

static void
output_log_line (const gchar *line, gboolean add_newline)
{
	gchar *s;

	s = g_strdup_printf (" :: %s%s", line, (add_newline) ? "\n" : "");

	g_printerr ("%s", s);

	th_log_add_line (s);

	g_free (s);
}

/***************************************************************************
 *
 *   th_main_log_func
 *
 ***************************************************************************/

static void
th_main_log_func (const gchar *domain, GLogLevelFlags f, const gchar *s, gpointer foo)
{
	gchar **p, **arr, *nl;

	nl = strstr (s, "\n");
	if (nl == NULL)
	{
		output_log_line (s, TRUE);
		return;
	}

	if (nl && *(nl+1) != 0x00)
	{
		arr = g_strsplit (s, "\n", -1);
		for (p = arr;  p && *p;  ++p)
			output_log_line (*p, TRUE);
		g_strfreev (arr);
	}
	else 
	{
		output_log_line (s, FALSE);
	}

	if (((f & G_LOG_LEVEL_INFO) == 0 && f != 0) || g_ascii_strcasecmp (domain, "thoggen"))
		g_log_default_handler (domain, f, s, NULL);
}

/***************************************************************************
 *
 *   th_main_print_func
 *
 ***************************************************************************/

static void
th_main_print_func (const gchar *s)
{
	th_main_log_func ("Thoggen", 0, s, NULL);
}


/***************************************************************************
 *
 *   print_system_info
 *
 ***************************************************************************/

static void
print_system_info (void)
{
	guint  maj, min, micro, nano;
	gchar *out = NULL;

	gst_version (&maj, &min, &micro, &nano);

	th_log ("GStreamer core version: %u.%u.%u.%u\n", maj, min, micro, nano);

	if (g_spawn_command_line_sync ("uname -a", &out, NULL, NULL, NULL))
	{
		g_strdelimit (out, "\n", ' ');
		th_log ("System: %s\n", out);
		g_free (out);
	}
}

static void
plugins_install_result_func (GstInstallPluginsReturn res, gpointer data)
{
  th_log ("Plugin install: %s\n", gst_install_plugins_return_get_name (res));

  if (res == GST_INSTALL_PLUGINS_SUCCESS ||
      res == GST_INSTALL_PLUGINS_PARTIAL_SUCCESS) {
    th_log ("New plugins were installed, updating registry");
    if (!gst_update_registry ()) {
      th_log ("Registry update failed for some reason");
    } else {
      /* not sure what to do now; assuming the installer puts itself on top of
       * the application window, the user couldn't have done much since startup.
       * For ffdeinterlace we don't need to do much, it will be used when we
       * start the encoding (if available). For new audio decoders we'd need to
       * re-read the disc contents for them to be taken into account. */
    }
  }
}

static void
install_missing_desirable_plugins (GtkWidget * appwindow)
{
  const gchar *desirable_features[] = { "dtsdec", "dvdlpcmdec",
      "ffdeinterlace" };
  GstInstallPluginsContext *ctx = NULL;
  GPtrArray *missing = NULL;
  guint i, xid;

  /* must realize the applicatio window, so that widget->window is non-NULL */
  gtk_widget_realize (appwindow);

  ctx = gst_install_plugins_context_new ();
  xid = GDK_WINDOW_XWINDOW (GTK_WIDGET (appwindow)->window);
  gst_install_plugins_context_set_xid (ctx, xid);

  missing = g_ptr_array_new ();
  for (i = 0; i < G_N_ELEMENTS (desirable_features); ++i) {
    const gchar *fname;

    fname = desirable_features[i];
    if (!gst_default_registry_check_feature_version (fname, 0, 10, 0)) {
      GstElement *dummy;
      GstMessage *msg;

      th_log ("GStreamer element '%s' is desirable, but not installed\n", fname);
      dummy = gst_pipeline_new (NULL);
      /* FIXME: use gst_missing_element_installer_detail_new() once
       * gst-plugins-base 0.10.15 is released and we can depend on it */
      msg = gst_missing_element_message_new (dummy, fname);
      g_ptr_array_add (missing,
          gst_missing_plugin_message_get_installer_detail (msg));
      gst_message_unref (msg);
      gst_object_unref (dummy);
    }
  }
  if (missing->len > 0) {
    GstInstallPluginsReturn ret;

    g_ptr_array_add (missing, NULL);

    ret = gst_install_plugins_async ((gchar **) missing->pdata, ctx,
        plugins_install_result_func, NULL);
    if (ret != GST_INSTALL_PLUGINS_STARTED_OK)
      th_log ("Plugin install: %s\n", gst_install_plugins_return_get_name (ret));

    g_ptr_array_foreach (missing, (GFunc) g_free, NULL);
  }
  g_ptr_array_free (missing, TRUE);
  gst_install_plugins_context_free (ctx);
}

/***************************************************************************
 *
 *   main
 *
 ***************************************************************************/

int
main (int argc, char **argv)
{
	union {
		GtkWidget  *appwin;
		gpointer    appwin_ptr;
	} run_info;

	/* must be called before any other GLib function */
	g_thread_init (NULL);

	gst_init (&argc, &argv);
	gtk_init (&argc, &argv);

	if (!check_required_plugins ())
		g_error ("We are missing some required plugins. There's something wrong with the installation.\n");
	
	/* TODO: popup a dialog? */
	if (!th_ensure_static_plugins ())
		g_error ("Eeek, somehow our static plugins didn't get linked in or registered.\n");

#ifdef ENABLE_NLS
	bindtextdomain (GETTEXT_PACKAGE "_iso_639", LOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE "_iso_639", "UTF-8");

	bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

	g_log_set_default_handler (th_main_log_func, NULL);
	g_set_print_handler (th_main_print_func);

	print_system_info ();
	
	run_info.appwin = th_app_window_new ();

	g_object_add_weak_pointer (G_OBJECT (run_info.appwin), &run_info.appwin_ptr);

	/* Try to have plugins that are desirable to have but are currently
	 * missing installed automatically if the distro supports it */
	install_missing_desirable_plugins (run_info.appwin);

	th_app_window_run (TH_APP_WINDOW (run_info.appwin));

	if (run_info.appwin != NULL)
		gtk_widget_destroy (run_info.appwin);

	return EXIT_SUCCESS;
}

