;;; (X)Emacs frontend to forward search with xdvi(k).
;;; See the section on FORWARD SEARCH in the xdvi man page for more
;;; information on forward and reverse search.
;;;
;;; Preliminary test version by Stefan Ulrich <ulrich@cis.uni-muenchen.de>,
;;; 2000/03/13. Tested with Emacs 20.4 and Xemacs 21.1.
;;;
;;; This program is free software; you can redistribute it and/or
;;; modify it under the terms of the GNU General Public License
;;; as published by the Free Software Foundation; either version 2
;;; of the License, or (at your option) any later version.
;;; 
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;; 
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
;;;
;;;
;;; Usage:
;;;
;;; - Add this file to some place where emacs can find it
;;;   (e.g. put it into a directory ~/emacs/, which you add to your
;;;   load path by putting the following line into your .emacs file:
;;;   
;;;      (add-to-list 'load-path (expand-file-name "~/emacs/"))
;;;   
;;;   Then, add the following line to your .emacs file:
;;;   
;;;      (require 'xdvi-search)
;;;   
;;;   After compiling your .tex file with source specials activated, you should
;;;   be able to use
;;;      M-x xdvi-jump-to-line
;;;   to make xdvi jump to the current location of the cursor.
;;;   
;;;   You could also bind this command to a key, e.g. by adding
;;;   a binding to tex-mode-hook:
;;;   
;;;      (add-hook 'tex-mode-hook (lambda ()
;;;                                 (local-set-key "\C-x\C-j" 'xdvi-jump-to-line)))
;;;
;;;   (without AucTeX, or:
;;;
;;;   (add-hook 'LaTeX-mode-hook (lambda ()
;;;                                (local-set-key "\C-x\C-j" 'xdvi-jump-to-line)))
;;;
;;;   with AucTeX).
;;;
;;; - This mode also requires a shell script for starting xdvi (see the variable
;;;   "xdvi-script" below); this shellscript is passed 2 arguments:
;;;   the argument to `-sourceposition', and the name of the master .dvi file.
;;;   Here's a minimal example for such a script:
;;;   
;;;   #---------- startxdvi begin
;;;   #!/bin/sh
;;;   
;;;   ### customizable variables begin
;;;   ###
;;;   ### name of xdvi executable
;;;   XDVI_EXEC="$HOME/xdvi-22.38/xdvi"
;;;   ### set this to /dev/null if you don't want to see the xdvi output:
;;;   XDVI_LOG="$HOME/xdvi-log"
;;;   ###
;;;   ### customizable variables end
;;;   
;;;   #XDVI_CALL="$XDVI_EXEC -name xdvi -debug 32768 -sourceposition"
;;;   XDVI_CALL="$XDVI_EXEC -name xdvi -sourceposition"
;;;   echo "calling $XDVI_CALL \"$1\" $2" > $XDVI_LOG
;;;   $XDVI_CALL "$1" $2 >> $XDVI_LOG 2>&1 &
;;;   exit 0
;;;   #---------- startxdvi end
;;;
;;;   (The reason for using this shell script is that I haven't found
;;;   a convenient way to view the output of the xdvi child process
;;;   (which is forked by the main process when no xdvi is running yet
;;;   on the display).
;;;   When using `start-process' instead of `call-process', the child
;;;   process seems to get killed by emacs as soon as the parent
;;;   exits. Maybe there's an error in how xdvi forks its child.
;;;   Suggestions on this are welcome!)
;;;
;;; Please send bug reports, improvements etc. to
;;; <ulrich@cis.uni-muenchen.de>.
;;;

(defvar xdvi-script "~/shellscripts/startxdvi"
  "*Name of start script for xdvi.")

(defun xdvi-jump-to-line ()
  "Call xdvi-script to perform a `forward search' for current file and line number.
See contents of xdvi-script for details.
If AucTeX is used, the value of TeX-master-file is used as filename
for the master .dvi file; else, the return value of xdvi-master-file-name
is used (which see)."
  (interactive)
  (save-excursion
    (save-restriction
      (widen)
      (beginning-of-line 1)
      (let* (;;; current line in file.
	     ;;; count-lines yields different results at beginning and in the
             ;;; middle of a line, so we normalize by going to BOL first and
             ;;; then adding 1
	     (curr-line (+ 1 (count-lines (point-min) (point))))
	     ;;; name of the `main' .tex file, which is also used as .dvi basename:
	     (master-file (expand-file-name (if (fboundp 'TeX-master-file)
					     (TeX-master-file t)
					   (xdvi-get-masterfile (xdvi-master-file-name)))))
	     ;;; .dvi file name:
	     (dvi-file (concat (file-name-sans-extension master-file) ".dvi"))
	     ;;; current source file name.
	     ;;; for use with TeX patch, which always uses full path names:
;;;	     (filename (expand-file-name (buffer-file-name)))
	     ;;; for use with srcltx.sty, which uses relative path names or filenames only
	     ;;; (fix for relative path names by frisk@isy.liu.se):
	     ;;; if path in master file is shorter than path in buffer file name, use
	     ;;; the remaining path component of buffer file name; else, use buffer file
	     ;;; name:
	     (filename (if (string-match (concat "^" (regexp-quote (file-name-directory master-file)))
					 (buffer-file-name))
			   (substring (buffer-file-name) (match-end 0))
			 (buffer-file-name))))
	(call-process xdvi-script
		      nil t nil ;;; src-args
		      ;;; args for -sourceposition:
		      (concat curr-line " " filename)
		      dvi-file)))))

(defun xdvi-get-masterfile (file)
  "Small helper function for AucTeX compatibility.
Converts the special value t that TeX-master might be set to
into a real file name."
  (if (eq file t)
      (buffer-file-name)
    file))


(defun xdvi-master-file-name ()
  "Emulate AucTeX's TeX-master-file function.
Partly copied from tex.el's TeX-master-file and TeX-add-local-master."
  (if (boundp 'TeX-master)
      TeX-master
    (let ((master-file (read-file-name "Master file (default this file): ")))
      (if (y-or-n-p "Save info as local variable? ")
	  (progn
	    (goto-char (point-max))
	    (if (re-search-backward "^\\([^\n]+\\)Local Variables:" nil t)
		(let* ((prefix (if (match-beginning 1)
				   (buffer-substring (match-beginning 1) (match-end 1))
				 ""))
		      (start (point)))
		  (re-search-forward (regexp-quote (concat prefix "End:")) nil t)
		  (if (re-search-backward (regexp-quote (concat prefix "TeX-master")) start t)
		      ;;; if TeX-master line exists already, replace it
		      (progn
			(beginning-of-line 1)
			(kill-line 1))
		    (beginning-of-line 1))
		  (insert prefix "TeX-master: " (prin1-to-string master-file) "\n"))
	      (insert "\n%%% Local Variables: "
;;; mode is of little use without AucTeX ...
;;;		      "\n%%% mode: " (substring (symbol-name major-mode) 0 -5)
		      "\n%%% TeX-master: " (prin1-to-string master-file)
		      "\n%%% End: \n"))
	    (save-buffer)
	    (message "(local variables written.)"))
	(message "(nothing written.)"))
      (set (make-local-variable 'TeX-master) master-file))))


(provide 'xdvi-search)

