# -*- Mode: ruby; indent-tabs-mode: nil -*-
#
#  $Id: tc_source.rb,v 1.1 2003/09/23 01:20:41 hisa Exp $
#
#  Copyright (c) 2003 FUJIMOTO Hisakuni <hisa@fobj.com>
#

require 'test/unit'
require 'tempura/source'
require 'tempura/charconv'

class TC_Source < Test::Unit::TestCase

  def test_s_new
    # without an argument
    assert_raises(ArgumentError) { Tempura::Source.new }
    # missing path
    assert_raises(Errno::ENOENT) { Tempura::Source.new("hoge.en") }
    # with a path
    src = nil
    assert_nothing_raised { src = Tempura::Source.new("view.en.html") }
    assert_kind_of( REXML::Document, src.fetch_document )
    assert_equal( "view.en.html", src.name )
    assert_equal( File.expand_path("view.en.html"), src.path )
    assert_respond_to( src.charconv, :from_u8 )
    assert_respond_to( src.charconv, :to_u8 )
    assert_nothing_raised { src = Tempura::Source.new("view.en.html") }
    # with a path and a charconv
    assert_nothing_raised { src = Tempura::Source.new( "view.ja.html", Tempura::CharConvEUC ) }
    assert_kind_of( REXML::Document, src.fetch_document )
    assert_equal( "view.ja.html", src.name )
    assert_equal( File.expand_path("view.ja.html"), src.path )
    assert_respond_to( src.charconv, :from_u8 )
    assert_respond_to( src.charconv, :to_u8 )
  end

  def test_s_new_with_string
    view_en_html = "<p>hello world</p>"
    view_ja_html = "<p>ˤ</p>"
    # without an argument
    src = nil
    assert_raises(ArgumentError) { Tempura::Source.new_with_string }
    # with a string
    src = nil
    assert_nothing_raised { src = Tempura::Source.new_with_string( view_en_html ) }
    assert_kind_of( REXML::Document, src.fetch_document )
    assert_nil( src.name )
    assert_nil( src.path )
    assert_respond_to( src.charconv, :from_u8 )
    assert_respond_to( src.charconv, :to_u8 )
    # with a string and a charconv
    assert_nothing_raised { src = Tempura::Source.new_with_string( view_ja_html, Tempura::CharConvEUC ) }
    assert_kind_of( REXML::Document, src.fetch_document )
    assert_nil( src.name )
    assert_nil( src.path )
    assert_respond_to( src.charconv, :from_u8 )
    assert_respond_to( src.charconv, :to_u8 )
  end

  def test_set_name
    src = Tempura::Source.new( "view.en.html" )
    assert_nothing_raised { src.name = "my_template" }
    assert_equal( "my_template", src.name )
  end

  def test_reload
    src = Tempura::Source.new( "view.en.html" )
    assert_nothing_raised { src.reload }
    src = Tempura::Source.new( "view.ja.html", Tempura::CharConvEUC )
    assert_nothing_raised { src.reload }
  end

  # REXML::Documentʣ֤
  def test_fetch_document
    ret = nil
    src = Tempura::Source.new( "view.en.html" )
    assert_nothing_raised { ret = src.fetch_document }
    assert_kind_of( REXML::Document, ret )
 end

  # ƥץ졼Ȥ֤̾
  def test_name
    name = nil
    src = Tempura::Source.new_with_string( "<p>hello world</p>" )
    assert_nothing_raised { name = src.name }
    assert_nil( name )
    src = Tempura::Source.new( "view.en.html" )
    assert_nothing_raised { name = src.name }
    assert_equal( "view.en.html", name )
  end

  # ƥץ졼ȤΥѥ֤
  def test_path
    path = nil
    src = Tempura::Source.new_with_string( "<p>hello world</p>" )
    assert_nothing_raised { path = src.path }
    assert_nil( path )
    src = Tempura::Source.new( "view.en.html" )
    assert_nothing_raised { path = src.path }
    assert_equal( File.expand_path("view.en.html"), path )
  end

  # ʸѴ֤
  def test_charconv
    ret = nil
    src = Tempura::Source.new( "view.en.html" )
    assert_nothing_raised { ret = src.charconv }
    assert_respond_to( src.charconv, :from_u8 )
    assert_respond_to( src.charconv, :to_u8 )
    src = Tempura::Source.new( "view.ja.html", Tempura::CharConvEUC )
    assert_nothing_raised { ret = src.charconv }
    assert_respond_to( src.charconv, :from_u8 )
    assert_respond_to( src.charconv, :to_u8 )
  end

end
