//
//	tardy - a tar post-processor
//	Copyright (C) 1993, 1995, 1996, 1998, 1999, 2001-2004 Peter Miller;
//	All rights reserved.
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
//
// MANIFEST: operating system entry point
//

#include <ac/stdio.h>
#include <ac/stdlib.h>

#include <arglex.h>
#include <error.h>
#include <help.h>
#include <ifmt.h>
#include <progname.h>
#include <tardy.h>
#include <version.h>


enum
{
    arglex_token_blocksize,
    arglex_token_clean,
    arglex_token_clean_meta,
    arglex_token_clean_print,
    arglex_token_clean_space,
    arglex_token_downcase,
    arglex_token_format_input,
    arglex_token_format_output,
    arglex_token_group,
    arglex_token_group_name,
    arglex_token_group_number,
    arglex_token_gunzip,
    arglex_token_gzip,
    arglex_token_list,
    arglex_token_mode_clear,
    arglex_token_mode_set,
    arglex_token_no_directories,
    arglex_token_now,
    arglex_token_old_type,
    arglex_token_prefix,
    arglex_token_progress,
    arglex_token_remove_prefix,
    arglex_token_upcase,
    arglex_token_user,
    arglex_token_user_name,
    arglex_token_user_number
};

static arglex_table_ty argtab[] =
{
    { "-Block_Size",	(arglex_token_ty)arglex_token_blocksize, },
    { "-Clean",		(arglex_token_ty)arglex_token_clean,	},
    { "-Clean_Meta",	(arglex_token_ty)arglex_token_clean_meta, },
    { "-Clean_Print",	(arglex_token_ty)arglex_token_clean_print, },
    { "-Clean_Space",	(arglex_token_ty)arglex_token_clean_space, },
    { "-COmpress",	(arglex_token_ty)arglex_token_gzip,	},
    { "-DeCOmpress",	(arglex_token_ty)arglex_token_gunzip,	},
    { "-DownCase",	(arglex_token_ty)arglex_token_downcase,	},
    { "-Group",		(arglex_token_ty)arglex_token_group,	},
    { "-Group_NAme",	(arglex_token_ty)arglex_token_group_name, },
    { "-Group_NUmber",	(arglex_token_ty)arglex_token_group_number, },
    { "-GUnZip",	(arglex_token_ty)arglex_token_gunzip,	},
    { "-GZip",		(arglex_token_ty)arglex_token_gzip,	},
    { "-Input_ForMaT",	(arglex_token_ty)arglex_token_format_input, },
    { "-List",		(arglex_token_ty)arglex_token_list,	},
    { "-Mode_Clear",	(arglex_token_ty)arglex_token_mode_clear, },
    { "-Mode_Set",	(arglex_token_ty)arglex_token_mode_set,	},
    { "-No_Directories", (arglex_token_ty)arglex_token_no_directories, },
    { "-Now",		(arglex_token_ty)arglex_token_now,	},
    { "-Old_Type",	(arglex_token_ty)arglex_token_old_type,	},
    { "-Output_ForMaT",	(arglex_token_ty)arglex_token_format_output, },
    { "-Prefix",	(arglex_token_ty)arglex_token_prefix,	},
    { "-PROgress",	(arglex_token_ty)arglex_token_progress,	},
    { "-Remove_Prefix",	(arglex_token_ty)arglex_token_remove_prefix, },
    { "-UnCOmpress",	(arglex_token_ty)arglex_token_gunzip,	},
    { "-UpCase",	(arglex_token_ty)arglex_token_upcase,	},
    { "-User",		(arglex_token_ty)arglex_token_user,	},
    { "-User_NAme",	(arglex_token_ty)arglex_token_user_name, },
    { "-User_NUmber",	(arglex_token_ty)arglex_token_user_number, },
    { 0, (arglex_token_ty)0, }, // end marker
};


static void
usage(void)
{
    const char *progname = progname_get();
    fprintf(stderr, "Usage: %s [ <option>... ][ <infile> [ <outfile> ]]\n",
	progname);
    fprintf(stderr, "       %s -Help\n", progname);
    fprintf(stderr, "       %s -VERSion\n", progname);
    exit(1);
}


static void
main_help(void)
{
    static char *text[] =
    {
#include <../man1/tardy.h>
    };

    help(text, SIZEOF(text), usage);
}


int
main(int argc, char **argv)
{
    char            *infile;
    char            *outfile;

    arglex_init(argc, argv, argtab);
    switch (arglex())
    {
    case arglex_token_help:
	main_help();
	exit(0);
    
    case arglex_token_version:
	version();
	exit(0);

    default:
	break;
    }
    infile = 0;
    outfile = 0;
    while (arglex_token != arglex_token_eoln)
    {
	switch (arglex_token)
	{
	default:
	    error
	    (
		"misplaced \"%s\" command line argument",
		arglex_value.alv_string
	    );
	    usage();

	case arglex_token_string:
	    if (!infile)
		infile = arglex_value.alv_string;
	    else if (!outfile)
		outfile = arglex_value.alv_string;
	    else
	    {
		error("too many file names specified");
		usage();
	    }
	    break;

	case arglex_token_stdio:
	    if (!infile)
		infile = "";
	    else if (!outfile)
		outfile = "";
	    else
	    {
		error("too many file names specified");
		usage();
	    }
	    break;

	case arglex_token_user:
	    switch (arglex())
	    {
	    default:
		error("the -User option requires a string or numeric argument");
		usage();

	    case arglex_token_string:
		tardy_user_name_by_string(arglex_value.alv_string);
		tardy_user_number_by_string(arglex_value.alv_string);
		break;

	    case arglex_token_number:
		tardy_user_name_by_number(arglex_value.alv_number);
		tardy_user_number_by_number(arglex_value.alv_number);
		break;
	    }
	    break;

	case arglex_token_user_name:
	    switch (arglex())
	    {
	    default:
		error
		(
		   "the -User_NAme option requires a string or numeric argument"
		);
		usage();

	    case arglex_token_string:
		tardy_user_name_by_string(arglex_value.alv_string);
		break;

	    case arglex_token_number:
		tardy_user_name_by_number(arglex_value.alv_number);
		break;
	    }
	    break;

	case arglex_token_user_number:
	    switch (arglex())
	    {
	    default:
		error
		(
		 "the -User_NUmber option requires a string or numeric argument"
		);
		usage();

	    case arglex_token_string:
		tardy_user_number_by_string(arglex_value.alv_string);
		break;

	    case arglex_token_number:
		tardy_user_number_by_number(arglex_value.alv_number);
		break;
	    }
	    break;

	case arglex_token_group:
	    switch (arglex())
	    {
	    default:
		error
		(
		    "the -Group option requires a string or numeric argument"
		);
		usage();

	    case arglex_token_string:
		tardy_group_name_by_string(arglex_value.alv_string);
		tardy_group_number_by_string(arglex_value.alv_string);
		break;

	    case arglex_token_number:
		tardy_group_name_by_number(arglex_value.alv_number);
		tardy_group_number_by_number(arglex_value.alv_number);
		break;
	    }
	    break;

	case arglex_token_group_name:
	    switch (arglex())
	    {
	    default:
		error
		(
		  "the -Group_NAme option requires a string or numeric argument"
		);
		usage();

	    case arglex_token_string:
		tardy_group_name_by_string(arglex_value.alv_string);
		break;

	    case arglex_token_number:
		tardy_group_name_by_number(arglex_value.alv_number);
		break;
	    }
	    break;

	case arglex_token_group_number:
	    switch (arglex())
	    {
	    default:
		error
		(
		"the -Group_NUmber option requires a string or numeric argument"
		);
		usage();

	    case arglex_token_string:
		tardy_group_number_by_string(arglex_value.alv_string);
		break;

	    case arglex_token_number:
		tardy_group_number_by_number(arglex_value.alv_number);
		break;
	    }
	    break;

	case arglex_token_prefix:
	    if (arglex() != arglex_token_string)
	    {
		error("the -Prefix option requires a string argument");
		usage();
	    }
	    tardy_prefix(arglex_value.alv_string);
	    break;

	case arglex_token_remove_prefix:
	    switch (arglex())
	    {
	    case arglex_token_string:
		tardy_remove_prefix(arglex_value.alv_string);
		break;

	    case arglex_token_number:
		tardy_remove_prefix(arglex_value.alv_number);
		break;

	    default:
		error("the -Remove_Prefix option requires a string argument");
		usage();
		// NOTREACHED
	    }
	    break;

	case arglex_token_mode_set:
	    if (arglex() != arglex_token_number)
	    {
		error
		(
		   "the -Mode_Set option must be followed by a numeric argument"
		);
		usage();
	    }
	    tardy_mode_set(arglex_value.alv_number);
	    break;

	case arglex_token_mode_clear:
	    if (arglex() != arglex_token_number)
	    {
		error
		(
		 "the -Mode_Clear option must be followed by a numeric argument"
		);
		usage();
	    }
	    tardy_mode_clear(arglex_value.alv_number);
	    break;

	case arglex_token_list:
	    tardy_list();
	    break;

	case arglex_token_now:
	    tardy_now();
	    break;

	case arglex_token_old_type:
	    tardy_no_fix_type();
	    break;

	case arglex_token_no_directories:
	    tardy_no_directories();
	    break;

	case arglex_token_clean:
	    tardy_clean_meta();
	    tardy_clean_print();
	    tardy_clean_space();
	    break;

	case arglex_token_clean_meta:
	    tardy_clean_meta();
	    break;

	case arglex_token_clean_print:
	    tardy_clean_print();
	    break;

	case arglex_token_clean_space:
	    tardy_clean_space();
	    break;

	case arglex_token_downcase:
	    tardy_downcase();
	    break;

	case arglex_token_upcase:
	    tardy_upcase();
	    break;

	case arglex_token_progress:
	    ifmt_progress();
	    break;

	case arglex_token_format_input:
	    if (arglex() != arglex_token_string)
	    {
		error("the -Input_ForMaT option requires a string argument");
		usage();
	    }
	    tardy_format_input(arglex_value.alv_string);
	    break;

	case arglex_token_format_output:
	    if (arglex() != arglex_token_string)
	    {
		error("the -Output_ForMaT option requires a string argument");
		usage();
	    }
	    tardy_format_output(arglex_value.alv_string);
	    break;

	case arglex_token_gzip:
	    tardy_gzip();
	    break;

	case arglex_token_gunzip:
	    tardy_gunzip();
	    break;

	case arglex_token_blocksize:
	    if (arglex() != arglex_token_number)
	    {
		error("the -Block-Size option requires a numeric argument");
		usage();
	    }
	    tardy_blocksize(arglex_value.alv_number);
	    break;
	}
	arglex();
    }
    if (outfile && !*outfile)
	outfile = 0;
    if (infile && !*infile)
	infile = 0;
    tardy(infile, outfile);
    exit(0);
    return 0;
}
