/*
 * TDB file functions
 *
 * Copyright (C) 2005  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "TDBFile.h"

#include <fcntl.h>	// O_RDONLY
#include <string.h>	// strlen
#include <errno.h>
#include <assert.h>

using namespace std;

namespace Tagcoll {


TDBFile::TDBFile(const std::string& filename)
	: db(0), _filename(filename)
{
}

TDBFile::~TDBFile()
{
	if (db != 0)
		tdb_close(db);
}

void TDBFile::open(int tdb_flags, int open_flags, mode_t mode)
{
	db = tdb_open(_filename.c_str(), 0, tdb_flags, open_flags, mode);
	if (db == NULL)
		throw SystemException(errno, "opening index file " + _filename);
}

OpSet<string> TDBFile::deserialize_stringset(const TDB_DATA& val)
{
	OpSet<string> res;
	for (char* cur = val.dptr; cur < val.dptr + val.dsize; )
	{
		int len = strlen(cur);
		res.insert(string(cur, len));
		cur += len + 1;
	}
	return res;
}


bool TDBFile::has(const std::string& key) const
{
	TDB_DATA k;

	k.dptr  = const_cast<char*>(key.data());
	k.dsize = key.size();

	return tdb_exists(db, k) == 1;
}

bool TDBFile::getGeneric(const std::string& key, void* buf, unsigned int len) const
{
	assert(db);
	
	TDB_DATA k, v;

	k.dptr  = const_cast<char*>(key.data());
	k.dsize = key.size();

	v = tdb_fetch(db, k);
	if (!v.dptr)
		return false;

	if (v.dsize < len)
		return false;

	memcpy(buf, v.dptr, len);
	free(v.dptr);

	return true;
}


OpSet<string> TDBFile::getStringSet(const std::string& key) const
{
	assert(db);
	
	TDB_DATA k, v;

	k.dptr  = const_cast<char*>(key.data());
	k.dsize = key.size();

	v = tdb_fetch(db, k);
	if (!v.dptr)
		return OpSet<string>();

	// Deserialize the string list
	OpSet<string> res = deserialize_stringset(v);
	free(v.dptr);
	return res;
}

void TDBFile::setGeneric(const std::string& key, const void* buf, unsigned int size)
{
	assert(db);

	TDB_DATA k, v;

	k.dptr  = const_cast<char*>(key.data());
	k.dsize = key.size();

	v.dptr = (char*)buf;
	v.dsize = size;

	if (tdb_store(db, k, v, TDB_REPLACE) == -1)
		throw SystemException(errno, "Writing key " + key + " to TDB file (tdb says: " + tdb_errorstr(db) + ")");
}

void TDBFile::setStringSet(const std::string& key, OpSet<std::string> vals)
{
	assert(db);

	TDB_DATA k, v;
	string payload;

	k.dptr  = const_cast<char*>(key.data());
	k.dsize = key.size();

	// Serialize the string list
	for (OpSet<std::string>::const_iterator i = vals.begin(); i != vals.end(); i++)
		payload += *i + '\0';

	v.dptr = const_cast<char*>(payload.data());
	v.dsize = payload.size();

	if (tdb_store(db, k, v, TDB_REPLACE) == -1)
		throw SystemException(errno, "Writing key " + key + " to on-disk index (tdb says: " + tdb_errorstr(db) + ")");
}

void TDBFile::remove(const std::string& key)
{
	assert(db);

	TDB_DATA k;

	k.dptr  = const_cast<char*>(key.data());
	k.dsize = key.size();

	if (tdb_delete(db, k) == -1)
		throw SystemException(errno, "Deleting key " + key + " from on-disk index (tdb says: " + tdb_errorstr(db) + ")");
}

int TDBFile::traverse(int (*fn)(TDB_CONTEXT *,TDB_DATA,TDB_DATA,void *), void *state) const
{
	assert(db);

	int res = tdb_traverse(db, fn, state);
	if (res == -1)
		throw SystemException(errno, "Traversing database " + _filename + " (tdb says: " + tdb_errorstr(db) + ")");
	return res;
}

}


#ifdef COMPILE_TESTSUITE

#include <tests/test-utils.h>

namespace tut {
using namespace tut_tagcoll;

static const string filename = "tagcoll-tdbfile-test.tdb";

struct tagcoll_tdbfile_shar {
	TDBFile file;

	tagcoll_tdbfile_shar() : file(filename) {
		file.open(0, O_RDWR | O_CREAT);
	}
	~tagcoll_tdbfile_shar() {
		unlink(filename.c_str());
	}
};
TESTGRP(tagcoll_tdbfile);

template<> template<>
void to::test<1>()
{
	gen_ensure_equals(file.filename(), filename);

	gen_ensure(!file.has(""));
	gen_ensure(!file.has("cippo"));

	file.set("cippo", 10);
	gen_ensure(file.has("cippo"));

	int val;
	gen_ensure(file.get("cippo", val));

	gen_ensure_equals(val, 10);

	file.remove("cippo");
	gen_ensure(!file.has("cippo"));
}

}

#endif

// vim:set ts=4 sw=4:
