#ifndef TAGCOLL_COLLECTION_H
#define TAGCOLL_COLLECTION_H

/** \file
 * Interface for all tagged collections
 */

/*
 * Copyright (C) 2003,2004,2005  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <tagcoll/ReadonlyCollection.h>
#include <tagcoll/Consumer.h>

namespace Tagcoll
{
template<typename T1, typename T2> class PatchList;

/**
 * Interface for all collections of tagged items.
 *
 * \note The point of a collection is to track the tags attached to items, and
 * not to store the items themselves.  This means that collections are not
 * required to keep track of items with no tags.
 */
template<typename ITEM, typename TAG>
class Collection : public Consumer<ITEM, TAG>, public ReadonlyCollection<ITEM, TAG>
{
protected:
	/*
	 * Implementation note: to avoid problems with classes implementing only
	 * some of the virtual methods, they are given different names.  The common
	 * 'comsume' methods are just inlined calls to the right virtual functions,
	 * and are a way of keeping the unoverridden methods from being hidden.
	 */

	void consumeItemUntagged(const ITEM&) {}
	void consumeItemsUntagged(const OpSet<ITEM>&) {}

public:
	virtual ~Collection() {}
	
	/**
	 * Apply a patch to the collection
	 *
	 * Example:
	 * \code
	 * void perform(const PatchList<ITEM, TAG>& change)
	 * {
	 *    collection.applyChange(change);
	 *    undo.push_back(change.getReverse());
	 * }
	 * \endcode
	 */
	virtual void applyChange(const PatchList<ITEM, TAG>& change) = 0;
};

};

// vim:set ts=4 sw=4:
#endif
