// Copyright (C) 2009 Martin Sandve Alnes
//
// This file is part of SyFi.
//
// SyFi is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// SyFi is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with SyFi. If not, see <http://www.gnu.org/licenses/>.
//
// First added:  2009-01-01
// Last changed: 2009-04-01
//
// This demo program solves a hyperelasticity problem
// in 3D using the Mooney-Rivlin model.

#include <iostream>
#include <vector>
#include <dolfin.h>
#include "generated_code/HyperElasticityMooneyRivlin.h"

using namespace dolfin; 
using namespace HyperElasticityMooneyRivlin;


// ---------------------------------------- Functions

class Displacement: public Expression 
{
public:
  Displacement() : Expression(3) {}

  void eval(Array<double>& values, const Array<double>& x) const
  {
    const double a = -0.1;
    const double b = 0.05;
    double dy = x[1]-0.5;
    double dz = x[2]-0.5;
    values[0] = x[0]*a;
    values[1] = dy*b;
    values[2] = dz*b;
  }
};

// ---------------------------------------- Subdomains

class DirichletBoundary: public SubDomain
{
  bool inside(const Array<double>& x, bool on_boundary) const
  {
    return on_boundary && (x[0] < DOLFIN_EPS || x[0] > 1.0-DOLFIN_EPS);
    //return on_boundary;
  }
};

// ---------------------------------------- Main program

int main(int argc, char**argv)
{
    // Geometry
    info("Mesh");
    unsigned n = 5;
    UnitCube mesh(n, n, n);



    // Function spaces
    info("Function spaces");
    Form_a::TestSpace V(mesh);
    CoefficientSpace_c1 C(mesh);
    
    // Coefficient functions
    info("Functions");
    
    // Displacement
    Function u(V);
    u.vector().zero();

    // Material parameters
    Constant c1(3.0);
    Constant c2(3.0);
    Constant c3(100.0);

    // Dirichlet BC function
    Displacement ubc;

    // Forms
    info("Forms");
    Form_M M(mesh);
    M.u = u; M.c1 = c1; M.c2 = c2; M.c3 = c3; 
      
    Form_L L(V);
    L.u = u; L.c1 = c1; L.c2 = c2; L.c3 = c3; 
    Form_a a(V, V);
    a.u = u; a.c1 = c1; a.c2 = c2; a.c3 = c3; 

    // Setup boundary conditions
    info("Boundary conditions");

    DirichletBoundary boundary;
    DirichletBC bc(V, ubc, boundary);
    
    // Compute integral of strain energy over domain, should be zero
    double energy = 0.0;
    energy = assemble(M);
    cout << "Energy before solve: " << energy << endl;

    // Solve!
    info("Solving");
    solve(a==L, u, bc); 

    // Compute integral of strain energy over domain
    energy = assemble(M);
    cout << "Energy after solve: " << energy << endl;

    // Write functions to file
    info("Writing to file");

    File ufile("u.pvd");
    ufile << u;

    //plot(u);
    info("Done!");

    return 0;
}

