// Copyright (C) 2006-2009 Kent-Andre Mardal and Simula Research Laboratory
//
// This file is part of SyFi.
//
// SyFi is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// SyFi is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with SyFi. If not, see <http://www.gnu.org/licenses/>.

#ifndef DOFT_IS_INCLUDED
#define DOFT_IS_INCLUDED

#include <map>
#include <vector>
#include <iterator>
#include <iostream>

template <class D, class C>
class DofT
{
	protected:
		bool create_index2dof, create_dof2loc;
		int counter;
		int emax;
		int imax;
		// the structures loc2dof, dof2index, and doc2loc  are completely dynamic
		// they are all initialized and updated by insert_dof(int e, int i, ex Li)

		// (int e, int i) ->  int j
		std::map< std::pair<int,int>, int>          loc2dof;
		// (ex Lj)    -> int j
		std::map<D,int,C>                     dof2index;
		typename std::map<D,int,C>:: iterator iter;

		// (int j)        -> ex Lj
		std::map<int,D>                     index2dof;
		// (ex j)    -> vector< pair<e1, i1>, ..  pair<en, in> >
		std::map <int, std::vector<std::pair<int,int> > >  dof2loc;

	public:
		DofT( bool create_index2dof_ = false, bool create_dof2loc_ = false )
		{
			counter = -1;
			emax = -1;
			imax = -1;
			create_index2dof = create_index2dof_;
			create_dof2loc   = create_dof2loc_;
		}
		~DofT() {}
								 // to update the internal structures
		int insert_dof(int e, int i, D Li);

		// helper functions when the dofs have been set
		// These do not modify the internal structure
		int glob_dof(int e, int i);
		int glob_dof(D Lj);
		D  glob_dof(int j);
		int size() const;
		int num_elements() const { return emax+1; }
		int num_basis_functions() const { return imax+1; }
		std::vector<std::pair<int, int> > glob2loc(int j);
		void clear();
};

template <class D, class C>
int
DofT<D, C> :: size() const
{
	return counter+1;
}


template <class D, class C>
int
DofT<D, C>:: insert_dof(int e, int i, D Li)
{

	if (e > emax) emax = e;
	if (i > imax) imax = i;

	// first we update loc2dof, which always should be updated
	std::pair<int,int> index;
	index.first = e;
	index.second = i;
	int return_dof;

	// check if the dof is new, if so
	// update counter, dof2index and create
	// a new vector in dof2loc
	iter = dof2index.find(Li);
								 //dof is new
	if ( iter == dof2index.end() )
	{
		counter++;
		return_dof = counter;
		dof2index[Li]  = counter;
		loc2dof[index] = counter;
		if ( create_index2dof)
		{
			std::pair<int, D> p(counter, Li);
			index2dof.insert(p);
			//      index2dof[counter] = Li;
			//
		}
		if ( create_dof2loc )
		{
			std::vector<std::pair<int,int> > v;
			dof2loc[counter] = v;
		}
	}							 // dof is not new
	else
	{
		loc2dof[index] = (*iter).second;
		return_dof = (*iter).second;
	}

	// insert (e,i) in dof2loc[Li]
	if (create_dof2loc)
	{
		dof2loc[return_dof].push_back(index);
	}

	return return_dof;
}


template <class D, class C>
int
DofT<D, C>:: glob_dof(int e, int i)
{
	std::pair<int,int> index;
	index.first = e;
	index.second = i;
	if ( loc2dof.find(index) != loc2dof.end())
	{
		return (*(loc2dof.find(index))).second;
	}
	else
	{
		return -1;
	}
}


template <class D, class C>
int
DofT<D, C>:: glob_dof(D Lj)
{
	if ( dof2index.find(Lj) != dof2index.end())
	{
		return (*(dof2index.find(Lj))).second;
	}
	else
	{
		return -1;
	}
}


template <class D, class C>
D
DofT<D, C>:: glob_dof(int j)
{
	if ( create_index2dof)
	{
		if ( index2dof.find(j) != index2dof.end() )
		{
			return (*(index2dof.find(j))).second;
		}
		else
		{
			std::cout <<"not found "<<std::endl;
			return D();
		}
	}
	else
	{
		std::cout <<"This structure has not been created "<<std::endl;
		std::cout <<"You must turn on the create_index2dof flag before initialization!"<<std::endl;
		return D();
	}
}


template <class D, class C>
std::vector<std::pair<int, int> >
DofT<D, C>:: glob2loc(int j)
{
	if ( create_dof2loc )
	{
		return dof2loc[j];
	}
	else
	{
		std::cout <<"This structure has not been created "<<std::endl;
		std::cout <<"You must turn on the create_dof2loc flag before initialization!"<<std::endl;
		return std::vector<std::pair<int,int> >();
	}

}


template <class D, class C>
void
DofT<D,C>:: clear()
{
	counter = -1;
	emax = -1;
	imax = -1;

	loc2dof.clear();
	dof2index.clear();
	index2dof.clear();
	dof2loc.clear();
}
#endif
