
header = """
#ifndef SFC_PTV_IS_INCLUDED
#define SFC_PTV_IS_INCLUDED

#include <functional>
#include <iostream>


namespace sfc
{

  class Ptv
  {
  public:
    unsigned int dim;
    double* v;
  
    static double tol;

  public:
    
    Ptv();
    
    Ptv(unsigned int size_);
    
    Ptv(unsigned int size_, double* v_);
    
    Ptv(double x, double y);
    
    Ptv(double x, double y, double z);
    
    Ptv(const Ptv& p);
    
    ~Ptv();

    void redim(unsigned int size_, double *v_);
    
    void redim(unsigned int size_); 
    
    void fill(double *v_);

    const unsigned int size() const
    { return dim; }

    double* getPtr()
    { return v; }

    const double& operator [] (unsigned int i) const;
    
    double& operator [] (unsigned int i);
    
    Ptv& operator = (const Ptv& p);
    
    bool less(const Ptv& p) const;

    bool operator<(const Ptv& rh) const
    { return less(rh); }

//    friend std::ostream & operator<< ( std::ostream& os, const Ptv& p);
  };

/*
  struct Ptv_is_less : public std::binary_function<sfc::Ptv, sfc::Ptv, bool> {
    bool operator() (const sfc::Ptv &lh, const sfc::Ptv &rh) const { return lh.less(rh);  }
  };
*/

/*
  class Ptv_match : public std::unary_function<sfc::Ptv, bool> {
  protected:
    static double tol;
    unsigned int d;
    double v;
  public:
    Ptv_match();
    Ptv_match(unsigned int d_, double v_);
    virtual ~Ptv_match() {}
    bool operator() (const Ptv &p);
  };
*/

} // namespace sfc

std::ostream & operator<< ( std::ostream& os, const sfc::Ptv& p);

#endif

"""




implementation = """
#include "Ptv.h"
#include <iostream>
#include <math.h>
#include <stdexcept>


using std::ostream;
using std::endl;
using namespace sfc;


double Ptv::tol = 1.0e-9;
//double Ptv_match::tol = 1.0e-9;
//double Ptv::tol = .0;
//double Ptv_match::tol = .0;

Ptv::Ptv():
  dim(0)
{
  v = new double[0];
}

Ptv::Ptv(double x, double y):
  dim(2)
{
  v = new double[dim];
  v[0] = x;
  v[1] = y;
}

Ptv::Ptv(double x, double y, double z):
  dim(3)
{
  v = new double[dim];
  v[0] = x;
  v[1] = y;
  v[2] = z;
}

Ptv::Ptv(unsigned int size_):
  dim(size_)
{
  v = new double[dim];
  for (unsigned int i=0; i< dim; ++i)
  {
    v[i] = 0.0;
  }
}


// FIXME: 
// The constructor which takes unsigned int, double* could/should work
// on the double* provided instead of creating a copy.  
// This however affects the destructor. Since Ptv should
// not delete memory.     
// We could introduce a bool external_storage in Ptv which 
// is used as a test in the destructor. 

Ptv::Ptv(unsigned int size_, double* v_):
  dim(size_) 
{
  v = new double[dim];
  for (unsigned int i=0; i< dim; ++i)
  {
    v[i] = v_[i];
  }
}

Ptv::Ptv(const Ptv& p)
{
  dim = p.size();
  v = new double[dim];
  for (unsigned int i=0; i< dim; ++i) 
  {
    v[i] = p[i];
  }
}

Ptv::~Ptv()
{
  delete [] v;
}

void Ptv::redim(unsigned int size, double* v_)
{
  if (dim != size )
  {
    delete [] v;
    dim = size;
    v = new double[dim];
  }

  for (unsigned int i=0; i< dim; ++i)
  {
    v[i] = v_[i];
  }
}

void Ptv::redim(unsigned int size) 
{
  if (dim != size )
  {
    delete [] v;
    dim = size;
    v = new double[dim];
  }
  for (unsigned int i=0; i< dim; ++i)
  {
    v[i] = 0.0;  
  }
}

void Ptv::fill(double* v_)
{
  for (unsigned int i=0; i< dim; ++i)
  {
    v[i] = v_[i];
  }
}

const double& Ptv::operator [] (unsigned int i) const
{
  // FIXME: should be possible to turn off at compile time
  if ( i < 0 || i >= dim )
  {
    throw std::out_of_range("The index is out of range!");
  }
  return v[i];
}

double& Ptv::operator [] (unsigned int i)
{
  // FIXME: should be possible to turn off at compile time
  if ( i < 0 || i >= dim )
  {
    throw std::out_of_range("The index is out of range!");
  }
  return v[i];
}

Ptv& Ptv::operator = (const Ptv& p)
{
  if ( this != &p)
  {
    if ( dim != p.dim)
    {
      delete [] v;
      dim = p.dim;
      v = new double[dim];
    }
    for (unsigned int i=0; i< dim; ++i)
    {
      v[i] = p[i];
    }
  }
  return *this;
}

inline bool less_with_tol(double a, double b, double tol)
{
  return a + tol < b - tol;
}

bool Ptv::less(const Ptv& p) const
{
  /*
  if ( dim < p.dim ) return true;
  if ( dim > p.dim)  return false;
  */
  if ( dim != p.dim )
  {
    throw std::runtime_error("Non-matching dimensions!");
  }
/*
  for (unsigned int i=dim-1; i>= 0; i--)
  {
    if ( v[i] + tol >= p[i] - tol &&  v[i] - tol <= p[i] + tol )
    {
    }
    else if (v[i] + tol  < p[i] - tol  )
    {
        return true;
    }
    else if ( v[i] - tol > p[i] + tol  )
    { 
        return false;
    }
  }
*/
  for (unsigned int i=0; i<dim; i++)
  {
    bool a_less_b = less_with_tol(v[i], p[i], tol);
    bool b_less_a = less_with_tol(p[i], v[i], tol);

    if ( a_less_b )
    {
        return true;
    }
    else if ( b_less_a  )
    { 
        return false;
    }
  }

  return false;
}



ostream & operator<< ( ostream& os, const Ptv& p)
{
  if (p.dim >= 1)
  {
    os << "[";
    for (unsigned int i=0; i<p.dim-1; ++i)
    {
      os << p[i] << ",";
    }
    os << p[p.dim-1] << "]";
  }
  else
  {
    os << "Ptv not created properly";
  }
  return os;
}



/*
Ptv_match::Ptv_match() {
  d = 0 ; v = 0.0;
}

Ptv_match::Ptv_match(unsigned int d_, double v_) {
  d = d_ ; v = v_;
}

bool Ptv_match:: operator () (const Ptv &p) {
  if ( v + tol >= p[d] && v - tol <= p[d] ) return true;
  else return false;
}
*/

"""

