// Copyright (C) 2009 Martin Sandve Alnes
//
// This file is part of SyFi.
//
// SyFi is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// SyFi is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with SyFi. If not, see <http://www.gnu.org/licenses/>.
//
// First added:  2009-01-01
// Last changed: 2009-03-01
//
// This demo program interpolates functions in lots of different function spaces.

#include <dolfin.h>
#include "generated_code/Elements.h"

using namespace dolfin;

class Scalar2D: public Expression 
{
public:
  void eval(Array<double>& values, const Array<double>& x) const
  {
    values[0] = x[0] + x[1];
  }
};

class Scalar3D: public Expression 
{
public:
  void eval(Array<double>& values, const Array<double>& x) const
  {
    double dz = x[2] - 0.5;
    values[0] = (x[0] + x[1]) * dz;
  }
};

class Vector2D: public Expression 
{
public:
  void eval(Array<double>& values, const Array<double>& x) const
  {
    values[0] = 3.0 + 0.1*x[0] + 0.01*x[1];
    values[1] = 5.0 + 0.1*x[0] + 0.01*x[1];
  }
};

class Vector3D: public Expression 
{
public:
  void eval(Array<double>& values, const Array<double>& x) const
  {
    double dx = x[0] - 0.5;
    double dy = x[1] - 0.5;
    double dz = x[2] - 0.5;
    values[0] = 1.0*dx;
    values[1] = 2.0*dy;
    values[2] = 3.0*dz;
  }
};

class Tensor2D: public Expression 
{
public:
  void eval(Array<double>& values, const Array<double>& x) const
  {
    double dx = x[0] - 0.5;
    double dy = x[1] - 0.5;
    values[0] = 1.0*dx;
    values[1] = 2.0*dx;
    values[2] = 3.0*dy;
    values[3] = 4.0*dy;
  }
};

class Tensor3D: public Expression 
{
public:
  void eval(Array<double>& values, const Array<double>& x) const
  {
    double dx = x[0] - 0.5;
    double dy = x[1] - 0.5;
    double dz = x[2] - 0.5;
    values[0] = 1.0*dx;
    values[1] = 2.0*dx;
    values[2] = 3.0*dx;
    values[3] = 4.0*dy;
    values[4] = 5.0*dy;
    values[5] = 6.0*dy;
    values[6] = 7.0*dz;
    values[7] = 8.0*dz;
    values[8] = 9.0*dz;
  }
};

int main(int argc, char ** argv)
{
  // TODO: The interface between DOLFIN and generated code needs to be more dynamic!

  // --- Meshes

  UnitSquare mesh2D(10, 10);
  UnitCube   mesh3D(5, 5, 5);

  // --- FunctionSpaces

  Elements::CoefficientSpace_s0_2D VS0_2D(mesh2D);
  Elements::CoefficientSpace_s1_2D VS1_2D(mesh2D);
  Elements::CoefficientSpace_s2_2D VS2_2D(mesh2D);
  Elements::CoefficientSpace_s3_2D VS3_2D(mesh2D);

//Elements::CoefficientSpace_s0_3D VS0_3D(mesh3D);
//Elements::CoefficientSpace_s1_3D VS1_3D(mesh3D);
//Elements::CoefficientSpace_s2_3D VS2_3D(mesh3D);
//Elements::CoefficientSpace_s3_3D VS3_3D(mesh3D);

//Elements::CoefficientSpace_v0_2D VV0_2D(mesh2D);
//Elements::CoefficientSpace_v1_2D VV1_2D(mesh2D);
//Elements::CoefficientSpace_v2_2D VV2_2D(mesh2D);
//Elements::CoefficientSpace_v3_2D VV3_2D(mesh2D);

//Elements::CoefficientSpace_v0_3D VV0_3D(mesh3D);
//Elements::CoefficientSpace_v1_3D VV1_3D(mesh3D);
//Elements::CoefficientSpace_v2_3D VV2_3D(mesh3D);
//Elements::CoefficientSpace_v3_3D VV3_3D(mesh3D);

//Elements::CoefficientSpace_t0_2D VT0_2D(mesh2D);
//Elements::CoefficientSpace_t1_2D VT1_2D(mesh2D);
//Elements::CoefficientSpace_t2_2D VT2_2D(mesh2D);
//Elements::CoefficientSpace_t3_2D VT3_2D(mesh2D);

//Elements::CoefficientSpace_t0_3D VT0_3D(mesh3D);
//Elements::CoefficientSpace_t1_3D VT1_3D(mesh3D);
//Elements::CoefficientSpace_t2_3D VT2_3D(mesh3D);
//Elements::CoefficientSpace_t3_3D VT3_3D(mesh3D);

  // --- Source functions

  Scalar2D ss0_2D;
  Scalar2D ss1_2D;
  Scalar2D ss2_2D;
  Scalar2D ss3_2D;

//Scalar3D ss0_3D;
//Scalar3D ss1_3D;
//Scalar3D ss2_3D;
//Scalar3D ss3_3D;

//Vector2D sv0_2D;
//Vector2D sv1_2D;
//Vector2D sv2_2D;
//Vector2D sv3_2D;

//Vector3D sv0_3D;
//Vector3D sv1_3D;
//Vector3D sv2_3D;
//Vector3D sv3_3D;

//Tensor2D st0_2D;
//Tensor2D st1_2D;
//Tensor2D st2_2D;
//Tensor2D st3_2D;

//Tensor3D st0_3D;
//Tensor3D st1_3D;
//Tensor3D st2_3D;
//Tensor3D st3_3D;

  // --- Functions to interpolate into

  Function fs0_2D(VS0_2D);
  Function fs1_2D(VS1_2D);
  Function fs2_2D(VS2_2D);
  Function fs3_2D(VS3_2D);

//Function fs0_3D(VS0_3D);
//Function fs1_3D(VS1_3D);
//Function fs2_3D(VS2_3D);
//Function fs3_3D(VS3_3D);

//Function fv0_2D(VV0_2D);
//Function fv1_2D(VV1_2D);
//Function fv2_2D(VV2_2D);
//Function fv3_2D(VV3_2D);

//Function fv0_3D(VV0_3D);
//Function fv1_3D(VV1_3D);
//Function fv2_3D(VV2_3D);
//Function fv3_3D(VV3_3D);

//Function ft0_2D(VT0_2D);
//Function ft1_2D(VT1_2D);
//Function ft2_2D(VT2_2D);
//Function ft3_2D(VT3_2D);

//Function ft0_3D(VT0_3D);
//Function ft1_3D(VT1_3D);
//Function ft2_3D(VT2_3D);
//Function ft3_3D(VT3_3D);

  // --- Interpolate all functions

  fs0_2D = ss0_2D; 
  fs1_2D = ss1_2D; 
  fs2_2D = ss2_2D; 
  fs3_2D = ss3_2D; 

/*
  fs0_3D = ss0_3D; 
  fs1_3D = ss1_3D; 
  fs2_3D = ss2_3D; 
  fs3_3D = ss3_3D; 

  ......


  */



 // --- Debugging prints
  
  /*
  mesh2D.disp();
  VV2_2D.dofmap().disp();
  fv2_2D.vector().disp();
  */

  // --- Write all functions to file
  
  File file_s0_2D("s0_2D.pvd"); file_s0_2D << fs0_2D;
  File file_s1_2D("s1_2D.pvd"); file_s1_2D << fs1_2D;
  File file_s2_2D("s2_2D.pvd"); file_s2_2D << fs2_2D;
  File file_s3_2D("s3_2D.pvd"); file_s3_2D << fs3_2D;

//File file_s0_3D("s0_3D.pvd"); file_s0_3D << fs0_3D;
//File file_s1_3D("s1_3D.pvd"); file_s1_3D << fs1_3D;
//File file_s2_3D("s2_3D.pvd"); file_s2_3D << fs2_3D;
//File file_s3_3D("s3_3D.pvd"); file_s3_3D << fs3_3D;

//File file_v0_2D("v0_2D.pvd"); file_v0_2D << fv0_2D;
//File file_v1_2D("v1_2D.pvd"); file_v1_2D << fv1_2D;
//File file_v2_2D("v2_2D.pvd"); file_v2_2D << fv2_2D;
//File file_v3_2D("v3_2D.pvd"); file_v3_2D << fv3_2D;

//File file_v0_3D("v0_3D.pvd"); file_v0_3D << fv0_3D;
//File file_v1_3D("v1_3D.pvd"); file_v1_3D << fv1_3D;
//File file_v2_3D("v2_3D.pvd"); file_v2_3D << fv2_3D;
//File file_v3_3D("v3_3D.pvd"); file_v3_3D << fv3_3D;

//File file_t0_2D("t0_2D.pvd"); file_t0_2D << ft0_2D;
//File file_t1_2D("t1_2D.pvd"); file_t1_2D << ft1_2D;
//File file_t2_2D("t2_2D.pvd"); file_t2_2D << ft2_2D;
//File file_t3_2D("t3_2D.pvd"); file_t3_2D << ft3_2D;

//File file_t0_3D("t0_3D.pvd"); file_t0_3D << ft0_3D;
//File file_t1_3D("t1_3D.pvd"); file_t1_3D << ft1_3D;
//File file_t2_3D("t2_3D.pvd"); file_t2_3D << ft2_3D;
//File file_t3_3D("t3_3D.pvd"); file_t3_3D << ft3_3D;

  return 0;
}

