#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ladspa.h"

#include "config.h"
#ifdef EXPLICIT_S
#include <srfftw.h>
#else
#include <rfftw.h>
#endif //EXPLICIT_S

#include "pitchscale.h"

#define FRAME_LENGTH 4096

rfftw_plan aplan = NULL, splan = NULL;

#define PITCHSCALEHQ_MULT              0
#define PITCHSCALEHQ_INPUT             1
#define PITCHSCALEHQ_OUTPUT            2

LADSPA_Descriptor *pitchScaleHQDescriptor = NULL;

typedef struct {
	LADSPA_Data *mult;
	LADSPA_Data *input;
	LADSPA_Data *output;
	sbuffers *   buffers;
	long         sample_rate;
	LADSPA_Data run_adding_gain;
} PitchScaleHQ;

const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {
	switch (index) {
	case 0:
		return pitchScaleHQDescriptor;
	default:
		return NULL;
	}
}

void activatePitchScaleHQ(LADSPA_Handle instance) {
	PitchScaleHQ *plugin_data = (PitchScaleHQ *)instance;
	sbuffers *buffers = plugin_data->buffers;
	long sample_rate = plugin_data->sample_rate;
	memset(buffers->gInFIFO, 0, FRAME_LENGTH*sizeof(float));
	memset(buffers->gOutFIFO, 0, FRAME_LENGTH*sizeof(float));
	memset(buffers->gLastPhase, 0, FRAME_LENGTH*sizeof(float)/2);
	memset(buffers->gSumPhase, 0, FRAME_LENGTH*sizeof(float)/2);
	memset(buffers->gOutputAccum, 0, 2*FRAME_LENGTH*sizeof(float));
	memset(buffers->gAnaFreq, 0, FRAME_LENGTH*sizeof(float));
	memset(buffers->gAnaMagn, 0, FRAME_LENGTH*sizeof(float));
	buffers->gRover = 0;
	sample_rate = sample_rate;
	plugin_data->buffers = buffers;
	plugin_data->sample_rate = sample_rate;

}

void cleanupPitchScaleHQ(LADSPA_Handle instance) {
	free(instance);
}

void connectPortPitchScaleHQ(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	PitchScaleHQ *plugin;

	plugin = (PitchScaleHQ *)instance;
	switch (port) {
	case PITCHSCALEHQ_MULT:
		plugin->mult = data;
		break;
	case PITCHSCALEHQ_INPUT:
		plugin->input = data;
		break;
	case PITCHSCALEHQ_OUTPUT:
		plugin->output = data;
		break;
	}
}

LADSPA_Handle instantiatePitchScaleHQ(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	PitchScaleHQ *plugin_data = (PitchScaleHQ *)malloc(sizeof(PitchScaleHQ));
	sbuffers *buffers = NULL;
	long sample_rate;

	int i;

	buffers = malloc(sizeof(sbuffers));
	sample_rate = s_rate;
	buffers->gInFIFO = malloc(FRAME_LENGTH * sizeof(float));
	buffers->gOutFIFO = malloc(FRAME_LENGTH * sizeof(float));
	buffers->gLastPhase = malloc(FRAME_LENGTH * sizeof(float));
	buffers->gSumPhase = malloc(FRAME_LENGTH * sizeof(float));
	buffers->gOutputAccum = malloc(2*FRAME_LENGTH * sizeof(float));
	buffers->gAnaFreq = malloc(FRAME_LENGTH * sizeof(float));
	buffers->gAnaMagn = malloc(FRAME_LENGTH * sizeof(float));
	buffers->gSynFreq = malloc(FRAME_LENGTH * sizeof(float));
	buffers->gSynMagn = malloc(FRAME_LENGTH * sizeof(float));
	buffers->gWindow = malloc(FRAME_LENGTH * sizeof(float));
	if (aplan == NULL) {
	        aplan = rfftw_create_plan(FRAME_LENGTH, FFTW_REAL_TO_COMPLEX, FFTW_ESTIMATE);
	        splan = rfftw_create_plan(FRAME_LENGTH, FFTW_COMPLEX_TO_REAL, FFTW_ESTIMATE);
	}

	for (i=0; i < FRAME_LENGTH; i++) {
	        buffers->gWindow[i] = -0.5f*cos(2.0f*M_PI*(double)i/(double)FRAME_LENGTH)+.5;
	}

	plugin_data->buffers = buffers;
	plugin_data->sample_rate = sample_rate;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

void runPitchScaleHQ(LADSPA_Handle instance, unsigned long sample_count) {
	PitchScaleHQ *plugin_data = (PitchScaleHQ *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Pitch co-efficient (float value) */
	LADSPA_Data mult = *(plugin_data->mult);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	sbuffers * buffers = plugin_data->buffers;
	long sample_rate = plugin_data->sample_rate;

	pitch_scale(buffers, mult, FRAME_LENGTH, 8, sample_count, sample_rate, input, output, RUN_ADDING, run_adding_gain);
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

void setRunAddingGainPitchScaleHQ(LADSPA_Handle instance, LADSPA_Data gain) {
	((PitchScaleHQ *)instance)->run_adding_gain = gain;
}

void runAddingPitchScaleHQ(LADSPA_Handle instance, unsigned long sample_count) {
	PitchScaleHQ *plugin_data = (PitchScaleHQ *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Pitch co-efficient (float value) */
	LADSPA_Data mult = *(plugin_data->mult);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	sbuffers * buffers = plugin_data->buffers;
	long sample_rate = plugin_data->sample_rate;

	pitch_scale(buffers, mult, FRAME_LENGTH, 8, sample_count, sample_rate, input, output, RUN_ADDING, run_adding_gain);
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

	pitchScaleHQDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (pitchScaleHQDescriptor) {
		pitchScaleHQDescriptor->UniqueID = 1194;
		pitchScaleHQDescriptor->Label = strdup("pitchScaleHQ");
		pitchScaleHQDescriptor->Properties =
		 LADSPA_PROPERTY_HARD_RT_CAPABLE;
		pitchScaleHQDescriptor->Name =
		 strdup("Higher Quality Pitch Scaler");
		pitchScaleHQDescriptor->Maker =
		 strdup("Steve Harris <steve@plugin.org.uk>");
		pitchScaleHQDescriptor->Copyright =
		 strdup("GPL");
		pitchScaleHQDescriptor->PortCount = 3;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(3,
		 sizeof(LADSPA_PortDescriptor));
		pitchScaleHQDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(3,
		 sizeof(LADSPA_PortRangeHint));
		pitchScaleHQDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(3, sizeof(char*));
		pitchScaleHQDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Pitch co-efficient */
		port_descriptors[PITCHSCALEHQ_MULT] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[PITCHSCALEHQ_MULT] =
		 strdup("Pitch co-efficient");
		port_range_hints[PITCHSCALEHQ_MULT].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[PITCHSCALEHQ_MULT].LowerBound = 0.5;
		port_range_hints[PITCHSCALEHQ_MULT].UpperBound = 2;

		/* Parameters for Input */
		port_descriptors[PITCHSCALEHQ_INPUT] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[PITCHSCALEHQ_INPUT] =
		 strdup("Input");
		port_range_hints[PITCHSCALEHQ_INPUT].HintDescriptor = 0;

		/* Parameters for Output */
		port_descriptors[PITCHSCALEHQ_OUTPUT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[PITCHSCALEHQ_OUTPUT] =
		 strdup("Output");
		port_range_hints[PITCHSCALEHQ_OUTPUT].HintDescriptor = 0;

		pitchScaleHQDescriptor->activate = activatePitchScaleHQ;
		pitchScaleHQDescriptor->cleanup = cleanupPitchScaleHQ;
		pitchScaleHQDescriptor->connect_port = connectPortPitchScaleHQ;
		pitchScaleHQDescriptor->deactivate = NULL;
		pitchScaleHQDescriptor->instantiate = instantiatePitchScaleHQ;
		pitchScaleHQDescriptor->run = runPitchScaleHQ;
		pitchScaleHQDescriptor->run_adding = runAddingPitchScaleHQ;
		pitchScaleHQDescriptor->set_run_adding_gain = setRunAddingGainPitchScaleHQ;
	}
}

void _fini() {
	int i;

	if (pitchScaleHQDescriptor) {
		free((char *)pitchScaleHQDescriptor->Label);
		free((char *)pitchScaleHQDescriptor->Name);
		free((char *)pitchScaleHQDescriptor->Maker);
		free((char *)pitchScaleHQDescriptor->Copyright);
		free((LADSPA_PortDescriptor *)pitchScaleHQDescriptor->PortDescriptors);
		for (i = 0; i < pitchScaleHQDescriptor->PortCount; i++)
			free((char *)(pitchScaleHQDescriptor->PortNames[i]));
		free((char **)pitchScaleHQDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)pitchScaleHQDescriptor->PortRangeHints);
		free(pitchScaleHQDescriptor);
	}

}
