/*
 * samplelib_libsndfile.c - SwamiSamplelib plugin for liblibsndfile
 *
 * Swami
 * Copyright (C) 1999-2003 Josh Green <jgreen@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Swami homepage: http://swami.sourceforge.net
 */

#include "config.h"
#include <stdio.h>

#include <glib.h>
#include <gmodule.h>

#include <sndfile.h>

#include <libswami/SwamiObject.h>
#include <libswami/SwamiPlugin.h>
#include <libswami/SwamiSamplelib.h>
#include <libswami/version.h>

#include "samplelib_libsndfile.h"
#include "i18n.h"


static int plugin_libsndfile_init (GModule *module, SwamiPlugin *plugin);
static GType samplelib_libsndfile_get_type (void);
static void samplelib_libsndfile_class_init (SamplelibLibsndfileClass *klass);
static void samplelib_libsndfile_init (SamplelibLibsndfile *samplelib);

static int samplelib_libsndfile_open (SwamiSamplelibHandle *handle);
static void samplelib_libsndfile_close (SwamiSamplelibHandle *handle);
static int samplelib_libsndfile_read (SwamiSamplelibHandle *handle, int frames,
				     gint16 *buf);
static int samplelib_libsndfile_write (SwamiSamplelibHandle *handle, int frames,
				      const gint16 *buf);

/* set plugin information */
SWAMI_PLUGIN_DESC (SWAMI_VERSION_MAJOR, SWAMI_VERSION_MINOR, "libsndfile",
		   N_("libsndfile - libsndfile library plugin"),
		   N_("Josh Green"), "",
		   plugin_libsndfile_init);

static int
plugin_libsndfile_init (GModule *module, SwamiPlugin *plugin)
{
  /* initialize SamplelibLibsndfile type */
  samplelib_libsndfile_get_type ();

  return (SWAMI_OK);
}

static GType
samplelib_libsndfile_get_type (void)
{
  static GType item_type = 0;

  if (!item_type) {
    static const GTypeInfo item_info = {
      sizeof (SamplelibLibsndfileClass),
      NULL,
      NULL,
      (GClassInitFunc) samplelib_libsndfile_class_init,
      NULL,
      NULL,
      sizeof (SamplelibLibsndfile),
      0,
      (GInstanceInitFunc) samplelib_libsndfile_init,
    };

    item_type = g_type_register_static (SWAMI_TYPE_SAMPLELIB,
					"SamplelibLibsndfile",
					&item_info, G_TYPE_FLAG_ABSTRACT);
  }

  return (item_type);
}

static void
samplelib_libsndfile_class_init (SamplelibLibsndfileClass *klass)
{
  SwamiSamplelibClass *samplelib_class;

  samplelib_class = SWAMI_SAMPLELIB_CLASS (klass);
  samplelib_class->open = samplelib_libsndfile_open;
  samplelib_class->close = samplelib_libsndfile_close;
  samplelib_class->read = samplelib_libsndfile_read;
  samplelib_class->write = samplelib_libsndfile_write;
}

static void
samplelib_libsndfile_init (SamplelibLibsndfile *samplelib)
{
}

static int
samplelib_libsndfile_open (SwamiSamplelibHandle *handle)
{
  SwamiSamplelibParams *params = &handle->params;
  SNDFILE *sndfd;
  SF_INFO info;

  if (params->file_type == SWAMI_SAMPLELIB_TYPE_RAW || handle->mode == 'w')
    {
      int major, subtype;

      info.samplerate = params->rate;
      info.channels = params->channels;

      if (params->width == 8 && params->signd)
	subtype = SF_FORMAT_PCM_S8;
      else if (params->width == 8 && !params->signd)
	subtype = SF_FORMAT_PCM_U8;
      else if (params->width == 16 && params->signd)
	subtype = SF_FORMAT_PCM_16;
      else
	{
	  g_critical ("Sample format (width = %d, signed = %d) not supported",
		      params->width, params->signd != 0);
	  return (SWAMI_FAIL);
	}

      if (handle->mode == 'w')	/* write file? */
	{
	  switch (params->file_type)
	    {
	    case SWAMI_SAMPLELIB_TYPE_AIFF:
	      major = SF_FORMAT_AIFF;
	      break;
	    case SWAMI_SAMPLELIB_TYPE_AU:
	      major = SF_FORMAT_AU;
	      break;
	    default:
	      major = SF_FORMAT_WAV;
	      break;
	    }
	}
      else			/** RAW file read mode */
	{
	  major = SF_FORMAT_RAW;
	  subtype |= params->lendian ? SF_ENDIAN_LITTLE : SF_ENDIAN_BIG;
	}

      info.format = major | subtype;
    }

  /* open the sound file */
  sndfd = sf_open (handle->filename,
		   handle->mode == 'w' ? SFM_WRITE : SFM_READ, &info);

  if (!sndfd)
    {
      if (handle->mode == 'w')
	g_critical (_("Failed to open audio file \"%s\" for writing"),
		    handle->filename);
      else
	g_critical (_("Failed to open audio file \"%s\" for reading"),
		    handle->filename);

      return (SWAMI_FAIL);
    }

  if (handle->mode == 'r')
    {
      params->channels = info.channels;
      params->rate = info.samplerate;
      handle->size = info.frames;
    }

  handle->driver_data = sndfd;

  return (SWAMI_OK);
}

/* file close function */
static void
samplelib_libsndfile_close (SwamiSamplelibHandle *handle)
{
  sf_close ((SNDFILE *)(handle->driver_data));
}

/* file read function */
static int
samplelib_libsndfile_read (SwamiSamplelibHandle *handle, int frames,
			  gint16 *buf)
{
  int items;

  items = frames * handle->params.channels;
  items = sf_read_short ((SNDFILE *)(handle->driver_data),
			 (short *)buf, items);
  return (items / handle->params.channels);
}

static int
samplelib_libsndfile_write (SwamiSamplelibHandle *handle, int frames,
			   const gint16 *buf)
{
  int items;

  items = frames * handle->params.channels;
  items = sf_write_short ((SNDFILE *)(handle->driver_data),
			  (short *)buf, items);
  return (items / handle->params.channels);
}
