/*
 * v m . c				-- The STklos Virtual Machine
 * 
 * Copyright  2000-2001 Erick Gallesio - I3S-CNRS/ESSI <eg@unice.fr>
 * 
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 * USA.
 * 
 *           Author: Erick Gallesio [eg@unice.fr]
 *    Creation date:  1-Mar-2000 19:51 (eg)
 * Last file update: 10-Mar-2001 17:54 (eg)
 */

/* #define DEBUG_VM */

#ifdef DEBUG_VM
static int debug_level = 2;	/* 0 is quiet, 1, 2, ... are more verbose */
#endif


#include "stklos.h"
#include "object.h"
#include "vm.h"
#include "vm-instr.h"


#if defined(__GNUC__) && !defined(DEBUG_VM)
   /* Use computed gotos to have better performances */
#  define USE_COMPUTED_GOTO
#  define CASE(x) 	lab_##x:
#  define NEXT		goto *jump_table[fetch_next()]
#else 
   /* Standard C compiler. Use the classic switch statement */
#  define CASE(x)	case x:
#  define NEXT		continue;/* Be sure to not use continue elsewhere */
#endif


#ifdef sparc
#  define FLUSH_REGISTERS_WINDOW()	asm("t 0x3") /* Stolen in Elk 2.0 source */
#else
#  define FLUSH_REGISTERS_WINDOW()
#endif


#define MY_SETJMP(jb) 		(jb.blocked = get_signal_mask(), setjmp(jb.j))
#define MY_LONGJMP(jb, val)	(longjmp((jb).j, val))


static Inline sigset_t get_signal_mask(void)
{
  sigset_t new, old;

  sigemptyset(&new);
  sigprocmask(SIG_BLOCK, &new, &old);
  return old;
}

static Inline void set_signal_mask(sigset_t mask)
{
  sigprocmask(SIG_SETMASK, &mask, NULL);
}


/*===========================================================================*\
 * 
 * 			V M   S T A C K   &   C O D E 
 *
\*===========================================================================*/

#define MAX_EVAL_STACK		100000
static SCM stack[MAX_EVAL_STACK];    /* // FIXME: Make it dynamic or a parameter */
static int stack_len = MAX_EVAL_STACK;

/* ==== Stack access macros ==== */
#define push(v)		(*(--sp) = (v))
#define pop()		(*(sp++))
#define IS_IN_STACKP(a)   ((stack <= (SCM*)(a)) && ((SCM*)(a) < &stack[stack_len]))

/* ==== Code access macros ==== */
#define fetch_next()	(*pc++)
#define fetch_const()	(constants[fetch_next()])
#define fetch_global()	(*(checked_globals[(unsigned) fetch_next()]))


/*===========================================================================*\
 * 
 * 			V M   R E G I S T E R S
 *
\*===========================================================================*/

static STk_instr *pc;		/* Program Counter		*/
static SCM *fp;			/* Frame pointer		*/
static SCM *sp;			/* Stack pointer		*/
static SCM val;			/* Current value register 	*/
static SCM env;			/* Current environment register */
static SCM *constants;		/* Constants of current code 	*/
static SCM *handlers;		/* Exceptions handlers		*/

static SCM r1, r2;			/* general registers		 */
static SCM val2, val3, val4, val5;	/* registers for multiple values */
static SCM valc;			/* # of multiple values 	 */

static jbuf *top_jmp_buf = NULL;  


/*
 * Activation records 
 * 
 */

#define ACT_RECORD_SIZE	   5

#define ACT_VARARG(reg)	   ((void*)(reg)[0]) /* place holder for &rest parameters */
#define ACT_SAVE_ENV(reg)  ((SCM)  (reg)[1])
#define ACT_SAVE_PC(reg)   ((void*)(reg)[2])
#define ACT_SAVE_CST(reg)  ((void*)(reg)[3])
#define ACT_SAVE_FP(reg)   ((void*)(reg)[4])

/*
 * VM state
 *
 */ 
#define VM_STATE_SIZE 5
#define VM_STATE_PC(reg)	((reg)[0])
#define VM_STATE_CST(reg)	((reg)[1])
#define VM_STATE_ENV(reg) 	((reg)[2])
#define VM_STATE_FP(reg)  	((reg)[3])
#define VM_STATE_JUMP_BUF(reg)	((reg)[4])

#define SAVE_VM_STATE()			{		\
  sp 			-= VM_STATE_SIZE;		\
  VM_STATE_PC(sp)       = (SCM) pc;			\
  VM_STATE_CST(sp)      = (SCM) constants;		\
  VM_STATE_ENV(sp)      = (SCM) env;			\
  VM_STATE_FP(sp)       = (SCM) fp;			\
  VM_STATE_JUMP_BUF(sp) = (SCM) top_jmp_buf;		\
}

#define FULL_RESTORE_VM_STATE(p)	{			\
  pc		     = (STk_instr *) VM_STATE_PC(p);		\
  RESTORE_VM_STATE(p);						\
}

#define RESTORE_VM_STATE(p)		{			\
  /* pc is not restored here. See FULL_RESTORE_VM_STATE */	\
  constants          = (SCM *)  VM_STATE_CST(p);		\
  env                = (SCM)    VM_STATE_ENV(p);		\
  fp                 = (SCM *)  VM_STATE_FP(p);			\
  top_jmp_buf 	     = (jbuf *) VM_STATE_JUMP_BUF(p);		\
  sp         	    += VM_STATE_SIZE;				\
}



/*
 * Handlers
 *
 */
#define EXCEPTION_HANDLER_SIZE 3

#define HANDLER_PROC(reg) 	((reg)[0])
#define HANDLER_END(reg)	((reg)[1])
#define HANDLER_PREV(reg)	((reg)[2])


#define SAVE_HANDLER_STATE(proc, addr)  { 		\
  sp 		   -= EXCEPTION_HANDLER_SIZE;		\
  HANDLER_PROC(sp)  =  (SCM) (proc);			\
  HANDLER_END(sp)   =  (SCM) (addr);			\
  HANDLER_PREV(sp)  =  (SCM) handlers;			\
  handlers          = sp;				\
}

#define UNSAVE_HANDLER_STATE()  { 			\
  SCM *old = handlers;					\
							\
  handlers = (SCM *) HANDLER_PREV(handlers);		\
  sp       = old + EXCEPTION_HANDLER_SIZE;		\
}


/*===========================================================================*\
 * 
 * 			C A L L S
 *
\*===========================================================================*/

#define PREP_CALL() {					\
  SCM fp_save = (SCM) fp;				\
							\
  /* Push an activation record on the stack */		\
  sp -= ACT_RECORD_SIZE;				\
  fp  = sp;						\
  ACT_SAVE_FP(fp) = fp_save;				\
  /* Other fields will be initialized later */		\
}


#define RET_CALL() {					\
  sp 	    = fp + ACT_RECORD_SIZE;			\
  env       = ACT_SAVE_ENV(fp);				\
  pc        = ACT_SAVE_PC(fp);				\
  constants = ACT_SAVE_CST(fp);				\
  fp        = ACT_SAVE_FP(fp);				\
}



#define DO_APPLY() {   	       	       	       	       	       	        \
  /* look at last argument */						\
  l   = sp[nargs-1];							\
  len = STk_int_length(l);						\
									\
  if (len < 0) 								\
    STk_error("last argument ~S is not a list", l);			\
  else if (len == 0) {							\
    nargs -= 1;								\
  } else {								\
    len -= 1;								\
    if (len > 0) { 							\
      /* Move the elements which are before the last one toward */	\
      /* the top of stack */						\
      memcpy(sp-len, sp, (nargs-1) * sizeof(SCM));			\
      sp = sp-len;							\
    }									\
    									\
    /* Unfold the last argument in place */				\
    tmp = &sp[nargs-1];							\
    while (!NULLP(l)) {							\
      *tmp++ = CAR(l);							\
      l      = CDR(l);							\
    }									\
    nargs += len;							\
  }									\
}

#define DO_NEXT_METHOD() {               	\
  SCM l;					\
  int i, n;					\
  						\
  val = STk_do_next_method(val, nargs, sp, &l);	\
  n   = STk_int_length(l);			\
  sp += nargs - n; 				\
  for (i = 0; i < n; i++) {			\
    sp[i] = CAR(l);				\
    l = CDR(l);					\
  }						\
  nargs = n;					\
}


/* 
 * 		         M i s c .
 */

#define CHECK_GLOBAL_INIT_SIZE	50
static SCM** checked_globals;
static int   checked_globals_len  = CHECK_GLOBAL_INIT_SIZE;
static int   checked_globals_used = 0;


#define FIRST_BYTE(n)  ((n) >> 8)
#define SECOND_BYTE(n) ((n) & 0xff)




#define PUSH_ENV(nargs, func, next_env)  {	\
    BOXED_TYPE(sp)   = tc_frame;		\
    FRAME_LENGTH(sp) = nargs;			\
    FRAME_NEXT(sp)   = next_env;		\
    FRAME_OWNER(sp)  = func;			\
}

#define CALL_CLOSURE(func) {			\
    pc        = CLOSURE_BCODE(func);		\
    constants = CLOSURE_CONST(func);		\
    env       = (SCM) sp;			\
}


#define RETURN_FROM_PRIMITIVE() {		\
    sp = fp + ACT_RECORD_SIZE;			\
    fp = (SCM *) ACT_SAVE_FP(fp);		\
}


static void run(void);
static void run_vm(STk_instr *code, SCM *constants, SCM envt, int save);


/*===========================================================================*\
 * 
 * 				Utilities
 * 
\*===========================================================================*/

#ifdef DEBUG_VM
void STk_print_vm_registers(char *msg, STk_instr *code)
{
  if (IS_IN_STACKP(env))
    STk_debug("%s VAL=~S PC=%d SP=%d FP=%d CST=%x ENV=%x (%d)", 
	      msg, val, pc-code, sp-stack, fp-stack, constants, env, 
	      (SCM*)env-stack);
  else
    STk_debug("%s VAL=~S PC=%d SP=%d FP=%d CST=%x ENV=%x (%d)", 
	      msg, val, pc-code, sp-stack, fp-stack, constants, env, 
	      (SCM*)env-stack);
}

#endif


#ifdef _FIXME_DEBUG_VM
static int use_debugger = 1;
static int breakpoint = -1;


static void vm_debug(STk_instr *code, short byteop)
{
  int c, prev='\n';
# define DEFINE_NAME_TABLE
# include "vm-instr.h"

  if (!use_debugger) return; 

  /* Print the current instruction and various informations */
  fprintf(stderr, "%03d: %20s  sp=%d fp=%d env=%x", pc-code-1, 
	  name_table[(int)byteop], sp-stack, fp-stack, (int) env);
  if (IS_IN_STACKP(env)) fprintf(stderr, " (%d)", (SCM*)env-stack);
  fprintf(stderr, "\n");


  /* See if we have a breakpoint */
  if (breakpoint > 0) {
    if (breakpoint == pc-code-1) {
      /* We are on the breakpoint */
      breakpoint = -1;
    }
    else {
      return;
    }
  }

  for ( ; ; ) {
    switch (c = getchar()) {
      case 'r':
	STk_print_vm_registers();
	break;
      case 'b':
	printf("Enter the value of PC "); fflush(stdout);
	scanf("%d", &breakpoint);
	return;
      case 'f': {
	int j, i = sp-stack;
	SCM *p; 

	for (p = sp; p < fp; p++) {
	  if (p == env) {
	    STk_debug("%d:     length: %x", i++, FRAME_LENGTH(fp));
	    STk_debug("%d:       next: %x", i++, FRAME_NEXT(fp));
	    STk_debug("%d:      owner: ~S", i++, FRAME_OWNER(fp));
	    p += 3;
	    for (j=0; j < FRAME_LENGTH(fp); j++, p++)
	      STk_debug("%d            : ~S", i++, FRAME_LOCAL(fp, j));
	  } else {
	    STk_debug("%d          ~S", i++, *p++);
	  }
	}
	STk_debug("%d      save env: %x",i++, (SCM*) ACT_SAVE_ENV(fp)-stack);
	STk_debug("%d       save pc: %d",i++, (STk_instr*) ACT_SAVE_PC(fp)-code);
	STk_debug("%d      save cst: %x",i++, (SCM*) ACT_SAVE_CST(fp));
	STk_debug("%d       save fp: %d",i++, (SCM*) ACT_SAVE_FP(fp)-stack);
	break;
      }
      case 'q': 
	use_debugger = 0; 
	/* no break */
      case '\n': if (prev != '\n') break;
      case 'c':
	return;
      default:
	STk_debug("h    -- help");
	STk_debug("r    -- registers");
	STk_debug("f    -- current frame");
	STk_debug("c    -- continue");
	STk_debug("b    -- breakpoint");
	STk_debug("q    -- quit");
	break;
    }
    prev = c;
  }
}
#endif



static Inline SCM listify_top(int n)
{
  SCM res = STk_nil;
  int len = n;

  while (n--) {
    res = STk_cons(sp[n], res);
  }
  sp += len;
  return res;
}


static Inline SCM clone_env(SCM e)
{
  /* clone environment til we find one which is in the heap */
  if (FRAMEP(e) && IS_IN_STACKP(e)) {
    e = STk_clone_frame(e);
    FRAME_NEXT(e) = clone_env((SCM) FRAME_NEXT(e));
  }
  return e;
}


static void error_bad_arity(SCM func, int arity, short given_args)
{
  if (arity >= 0)
    STk_error("%d argument%s required in call to ~S (%d provided)",
	      arity, ((arity>1)? "s": ""), func, given_args);
  else
    STk_error("~S requires at least %d argument%s (%d provided)",
	      func, -arity-1, ((arity>1)? "s" : ""), given_args);
}



static Inline short adjust_arity(SCM func, short nargs)
{
  short arity = CLOSURE_ARITY(func);

  if (arity != nargs) {
    if (arity >= 0) 
      error_bad_arity(func, arity, nargs);
    else {						/* nary procedure call */
      short min_arity = -arity-1;

      if (nargs < min_arity)
	error_bad_arity(func, arity, nargs); 
      else { /* Make a list from the arguments which are on the stack. */
	SCM res = STk_nil;
	
	while (nargs > min_arity) 
	  res = STk_cons(sp[--nargs], res);

	/* Place res in the variable argument. Note that in the special case
	 * where nargs = min_arity, NIL is stored in the special place-holder 
	 * in the activation record.
	 */
	sp[min_arity] = res;
      }
      return -arity;
    }
  }
  return arity;
}


/* Add a new global reference to the table of checked references */
static int add_global(SCM *ref)
{
  if (checked_globals_used >= checked_globals_len) {
    /* resize the checked global array */
    checked_globals_len += checked_globals_len / 2;
    checked_globals      = STk_must_realloc(checked_globals, 
					    checked_globals_len * sizeof(SCM*));
  }
  checked_globals[checked_globals_used] = ref;
  return checked_globals_used++;
}


/*===========================================================================*\
 * 
 * 				      C A L L S
 *
\*===========================================================================*/

/*
<doc  apply
 * (apply proc arg1 ... args)
 *
 * |Proc| must be a procedure and |args| must be a list. Calls |proc| with the
 * elements of the list 
 * @lisp
 * (append (list arg1 ...) args)
 * @end lisp 
 * as the actual arguments.
 * @lisp
 * (apply + (list 3 4))              =>  7
 *
 * (define compose
 *   (lambda (f g)
 *      (lambda args
 *        (f (apply g args)))))
 * 
 * ((compose sqrt *) 12 75)          =>  30
 * @end lisp
doc>
 */
DEFINE_PRIMITIVE("apply", scheme_apply, apply, (int argc, SCM *argv))
{
  SCM l, func, *tmp;
  int len, nargs;

  ENTER_PRIMITIVE(scheme_apply);
  if (argc == 0) STk_error("no function given");
  
  func  = pop();
  nargs = argc-1;
  
  if (nargs > 0) {
    DO_APPLY();
  }
  /* Place the function to apply and its number of arguments in R1 and R2 */
  r1 = func;
  r2 = (SCM) nargs;

  return STk_apply_call;
}


void Inline funcall(int nargs, int tailp)
{
  SCM *old_fp;

apply_proc:

  switch (STYPE(val)) {

    case tc_instance: {
      if (PUREGENERICP(val)) {
	SCM methods = STk_compute_applicable_methods(val, nargs, sp, FALSE);
	
	/* methods is the list of applicable methods. Apply the first
	 * one with the tail of the list as first parameter
	 * (next-method). If methods is STk_nil, that's because the
	 * no-applicable-method triggered didn't yield an error.  
	 */
	if (NULLP(methods)) { val = STk_void; return; }
	
	/* Push next method on stack */
	push(STk_make_next_method(val, nargs, sp, methods));
	val    = INST_SLOT(CAR(methods), S_procedure);
	nargs += 1;
	/* NO BREAK */
      } else {
	SCM gf, args; 
	
	/* Use the MOP and do the call (apply-generic gf args) */
	gf = val; args = listify_top(nargs);
	push(args); push(gf);
	val = STk_lookup(STk_intern("apply-generic"), STk_current_module, 
			 &gf, FALSE);
	nargs = 2;
	goto apply_proc;
      }
    }

    case tc_closure: {
      nargs = adjust_arity(val, nargs);

      if (tailp) {
	/* Tail call: Reuse the old frame for this call.*/
	old_fp = (SCM *) ACT_SAVE_FP(fp);
	
	/* Move the arguments of the function to the old_fp */
	if (nargs) memcpy(old_fp-nargs, sp, nargs*sizeof(SCM));
	fp = old_fp;
	
	/* Push a new environment on the stack */
	sp = fp - nargs - ((sizeof(struct frame_obj) - sizeof(SCM)) / sizeof(SCM));
	PUSH_ENV(nargs, val, CLOSURE_ENV(val));
      } else {
	/* Push a new environment on the stack */
	sp -= (sizeof(struct frame_obj) - sizeof(SCM)) / sizeof(SCM);
	PUSH_ENV(nargs, val, CLOSURE_ENV(val));
      
	/* Finish initialisation of current activation record */
	ACT_SAVE_ENV(fp) = env;
	ACT_SAVE_PC(fp)  = pc;
	ACT_SAVE_CST(fp) = constants;
      }

      /* Do the call */
      CALL_CLOSURE(val);
      return;
    }

    case tc_next_method: {
      DO_NEXT_METHOD();
      goto apply_proc;
    }

    case tc_apply: {
      /* Call the function in the call frame of apply */
      STk_scheme_apply(nargs, sp);
      nargs = (short) r2; 
      val   = r1; 
      goto apply_proc;
    }

    case tc_subr0:
      if (nargs == 0) { val = PRIMITIVE_FUNC(val)();			 break;}
      goto error_invoke;
    case tc_subr1:
      if (nargs == 1) { val = PRIMITIVE_FUNC(val)(sp[0]);		 break;}
      goto error_invoke;
    case tc_subr2:
      if (nargs == 2) { val = PRIMITIVE_FUNC(val)(sp[0], sp[1]);	 break;}
      goto error_invoke;
    case tc_subr3:
      if (nargs == 3) { val = PRIMITIVE_FUNC(val)(sp[0], sp[1], sp[2]);	 break;}
      goto error_invoke;
    case tc_subr4:
      if (nargs==4) { val = PRIMITIVE_FUNC(val)(sp[0],sp[1],sp[2], sp[3]); break;}
      goto error_invoke;
    case tc_subr5:
      if (nargs==5) { val = PRIMITIVE_FUNC(val)(sp[0],sp[1],sp[2],sp[3],sp[4]);
									 break;}
      goto error_invoke;

    case tc_subr01:
      if (nargs == 0) { val = PRIMITIVE_FUNC(val)((SCM) NULL);		 break;}
      if (nargs == 1) { val = PRIMITIVE_FUNC(val)(sp[0]);		 break;}
      goto error_invoke;
    case tc_subr12:
      if (nargs == 1) {val = PRIMITIVE_FUNC(val)(sp[0], (SCM) NULL);	 break;}
      if (nargs == 2) {val = PRIMITIVE_FUNC(val)(sp[0], sp[1]);		 break;}
      goto error_invoke;
    case tc_subr23:
      if (nargs == 2) { val = PRIMITIVE_FUNC(val)(sp[0],sp[1],(SCM)NULL);break;}
      if (nargs == 3) { val = PRIMITIVE_FUNC(val)(sp[0],sp[1],sp[2]);	 break;}
      goto error_invoke;
    case tc_vsubr:
      val = PRIMITIVE_FUNC(val)(nargs, sp);				 break;

    default: 
      STk_error("bad function ~S. Cannot be applied", val);
  }
  /* We are here when we have called a primitive */
  RETURN_FROM_PRIMITIVE();
  return;
error_invoke:
  /* We are here when we had a primitive call with a bad number of parameters */
  STk_error("incorrect number of parameters (%d) in call to ~S", nargs, val);
}



/*===========================================================================*\
 *
 * 				S T k _ C _ a p p l y
 *
 *
 * Execute a Scheme function from C. This function can be used as a 
 * an "excv" or an "execl" function. If nargs is > 0 it is as a Unix "execl" 
 * function: 
 *    STk_C_apply(STk_cons, 2, MAKE_INT(1), MAKE_INT(2)) => (1 . 2)
 * If nargs is < 0, we have something similar to an "execv fucntion
 *    STk_C_apply(STk_cons, -2, Argv)
 * where Argv[0] == MAKE_INT(1) and Argv[1] == MAKE_INT(2) ==> (1 . 2)
 *
\*===========================================================================*/

SCM STk_C_apply(SCM func, int nargs, ...) 
{
  static STk_instr code[]= {INVOKE, 0, END_OF_CODE};
  va_list ap;
  int i;

  va_start(ap, nargs);
  SAVE_VM_STATE();				    /* Save the VM regs */
  PREP_CALL();					    /* PREPARE_CALL */

  if (nargs < 0) {				    /* Push the arguments */
    /* args are in argc/argv form */
    SCM *argv = va_arg(ap, SCM*);

    nargs = -nargs;
    sp   -= nargs;
    for (i = 0; i < nargs; i++) sp[i] = argv[i];
  } else {
    /* We have nargs SCM parameters to read */
    sp -= nargs;
    for (i = 0; i < nargs; i++) sp[i] = va_arg(ap, SCM);
  }

  val     = func;				    /* Store fun in VAL */
  code[1] = (short) nargs;			    /* Patch # of args  */

  run_vm(code, constants, env, 0); 
  FULL_RESTORE_VM_STATE(sp);

  return val;
}


DEFINE_PRIMITIVE("%execute", execute, subr23, (SCM code, SCM consts, SCM envt))
{
  int i, len;
  STk_instr *vinstr, *p;

  ENTER_PRIMITIVE(execute);

  if (!envt) envt = STk_current_module;
  
  if (!VECTORP(code)) 	STk_error("bad code vector ~S", code);
  if (!VECTORP(consts)) STk_error("bad constant list ~S", consts);
  if (!MODULEP(envt))   STk_error("bad module for evaluation ~S", envt);
  
  /* convert code to a vector of instructions */
  len = VECTOR_SIZE(code);
  vinstr = p = STk_must_malloc(len * sizeof(STk_instr));

  for (i = 0; i < len; i++)
    *p++ = (STk_instr) STk_integer_value(VECTOR_DATA(code)[i]);

  SAVE_VM_STATE();
  run_vm(vinstr, VECTOR_DATA(consts), envt, 1);
  FULL_RESTORE_VM_STATE(sp);

  return val;
}


/*===========================================================================*\
 * 
 * 				V A L U E S
 *
\*===========================================================================*/
/*
<doc values
 * (values obj ...)
 *
 * Delivers all of its arguments to its continuation. 
 * @strong{Note}: @rfive{} imposes to use multiple values in the context of 
 * of a |call-with-values|. In @stklos{}, if |values| is not used with 
 * |call-with-values|, only the first value is used (i.e. others values are 
 * @emph{ignored}).
 *
doc>
*/
DEFINE_PRIMITIVE("values", values, vsubr, (int argc, SCM *argv))
{
  switch (argc) {
    case 0:  val = STk_void; break;
    case 1:  val = argv[0]; break;
    case 2:  val = argv[0]; val2 = argv[1]; break;
    case 3:  val = argv[0]; val2 = argv[1]; val3 = argv[2]; break;
    case 4:  val = argv[0]; val2 = argv[1]; val3 = argv[2]; val4 = argv[3]; break;
    case 5:  val = argv[0]; val2 = argv[1]; val3 = argv[2]; val4 = argv[3];
      	     val5= argv[4]; break;
    default: { /* More than 5 values. Use a vector */
	       int i;
	       
	       val  = argv[0]; /*as ussal */
	       val2 = STk_makevect(argc, (SCM) NULL);
	       for (i = 0; i < argc; i++) VECTOR_DATA(val2)[i] = argv[i];
	     }
  }
  /* Retain in valc the number of values */
  valc = argc;
  return val;
}

/*
<doc call-with-values
 * (call-with-values producer consumer)
 *
 * Calls its producer argument with no values and a continuation that, 
 * when passed some values, calls the consumer procedure with those values 
 * as arguments. The continuation for the call to consumer is the 
 * continuation of the call to call-with-values. 
 * @lisp
 * (call-with-values (lambda () (values 4 5))
 *                   (lambda (a b) b))                =>  5
 *
 * (call-with-values * -)                             =>  -1
 * @end lisp
doc>
 */
DEFINE_PRIMITIVE("call-with-values", call_with_values, subr2, (SCM prod, SCM con))
{
  int tmp;
  ENTER_PRIMITIVE(call_with_values);

  if (!STk_procedurep(prod)) STk_error("bad producer", prod);
  if (!STk_procedurep(con))  STk_error("bad consumer", con);
  
  valc = 1;		/* reinitialize valc */
  val  = STk_C_apply(prod, 0);
  tmp  = valc;
  valc = 1;

  switch (tmp) {
    case 0:  return STk_C_apply(con, 0); 
    case 1:  return STk_C_apply(con, 1, val);
    case 2:  return STk_C_apply(con, 2, val, val2);
    case 3:  return STk_C_apply(con, 3, val, val2, val3);
    case 4:  return STk_C_apply(con, 4, val, val2, val3, val4);
    case 5:  return STk_C_apply(con, 5, val, val2, val3, val4, val5);
    default: return STk_C_apply(con, -valc, VECTOR_DATA(val2));
  }
}

SCM STk_2_values(SCM v1, SCM v2)
{
  valc = 2;
  val  = v1;
  val2 = v2;
  return val;
}


#ifdef DEBUG_VM
DEFINE_PRIMITIVE("%vm", set_vm_debug, subr1, (SCM v))
{
  /* 
   * This function is just a placeholder for debugging the VM. It's body is 
   * changed depending of the current bug to track 
   */
  
  return STk_void;
}
#endif 


/*===========================================================================*\
 * 
 *	 	       S T k l o s   V i r t u a l   M a c h i n e 
 *
\*===========================================================================*/

static void run(void)
{
  short offset, nargs=0;
#ifdef USE_COMPUTED_GOTO
#  define  DEFINE_JUMP_TABLE
#  include "vm-instr.h"
  NEXT;
#else
  static STk_instr *code_base = NULL;
  short byteop;
#  ifdef DEBUG_VM
#    define DEFINE_NAME_TABLE
#    include "vm-instr.h"
   if (!code_base) code_base = pc;
#  endif
   
  for ( ; ; ) {		
    /* Execution loop */
    byteop = fetch_next();
#  ifdef DEBUG_VM
    if (debug_level > 1)
      fprintf(stderr, "%08x [%03d]: %20s  sp=%-6d fp=%-6d env=%x\n", 
	      pc - 1,
	      pc-code_base-1, 
	      name_table[(int)byteop], sp-stack, fp-stack, (int) env);
#  endif
    switch (byteop) {
#endif /*  USE_COMPUTED_GOTO */


CASE(NOP) { NEXT; }


CASE(IM_FALSE)  { val = STk_false;       NEXT;}
CASE(IM_TRUE)   { val = STk_true;        NEXT;}
CASE(IM_NIL)    { val = STk_nil;         NEXT;}
CASE(IM_MINUS1) { val = MAKE_INT(-1);    NEXT;}
CASE(IM_ZERO)   { val = MAKE_INT(0);     NEXT;}
CASE(IM_ONE)    { val = MAKE_INT(1);     NEXT;}
CASE(IM_VOID)   { val = STk_void; NEXT;}


CASE(SMALL_INT) { val = MAKE_INT(fetch_next());	NEXT;}
CASE(CONSTANT)  { val = fetch_const();	        NEXT;}


CASE(GLOBAL_REF) {
  SCM ref;

  val = STk_lookup(fetch_const(), env, &ref, TRUE);
  /* patch the code for optimize next accesses */
  pc[-2] = UGLOBAL_REF;
  pc[-1] = add_global(&CDR(ref));
  NEXT;
}
CASE(UGLOBAL_REF) { /* Never produced by compiler */ val = fetch_global(); NEXT; }


CASE(LOCAL_REF0) { val = FRAME_LOCAL(env, 0); NEXT;}
CASE(LOCAL_REF1) { val = FRAME_LOCAL(env, 1); NEXT;}
CASE(LOCAL_REF2) { val = FRAME_LOCAL(env, 2); NEXT;}
CASE(LOCAL_REF3) { val = FRAME_LOCAL(env, 3); NEXT;}
CASE(LOCAL_REF4) { val = FRAME_LOCAL(env, 4); NEXT;}
CASE(LOCAL_REF)  { val = FRAME_LOCAL(env, fetch_next()); NEXT;}
CASE(DEEP_LOCAL_REF) {
  int level, info = fetch_next();
  SCM e = env;

  /* Go down in the dynamic environment */
  for (level = FIRST_BYTE(info); level; level--)
    e = (SCM) FRAME_NEXT(e);

  val = FRAME_LOCAL(e, SECOND_BYTE(info));
  NEXT;
}


CASE(GLOBAL_SET) {
  SCM ref;
  
  STk_lookup(fetch_const(), env, &ref, TRUE);
  CDR(ref) = val;
  val = STk_void;
  /* patch the code for optimize next accesses */
  pc[-2] = UGLOBAL_SET;
  pc[-1] = add_global(&CDR(ref));
  NEXT;
}
CASE(UGLOBAL_SET) { /* Never produced by compiler */
  fetch_global() = val; val = STk_void; NEXT;
}


CASE(LOCAL_SET0) { FRAME_LOCAL(env, 0)         = val; val = STk_void; NEXT;}
CASE(LOCAL_SET1) { FRAME_LOCAL(env, 1)         = val; val = STk_void; NEXT;}
CASE(LOCAL_SET2) { FRAME_LOCAL(env, 2)         = val; val = STk_void; NEXT;}
CASE(LOCAL_SET3) { FRAME_LOCAL(env, 3)         = val; val = STk_void; NEXT;}
CASE(LOCAL_SET4) { FRAME_LOCAL(env, 4)         = val; val = STk_void; NEXT;}
CASE(LOCAL_SET)  { FRAME_LOCAL(env,fetch_next())=val; val = STk_void; NEXT;}
CASE(DEEP_LOCAL_SET) {
  int level, info = fetch_next();
  SCM e = env;

  /* Go down in the dynamic environment */
  for (level = FIRST_BYTE(info); level; level--)
    e = (SCM) FRAME_NEXT(e);

  FRAME_LOCAL(e, SECOND_BYTE(info)) = val;
  val = STk_void;
  NEXT;
}


CASE(GOTO) { offset = fetch_next(); pc += offset; NEXT;}
CASE(JUMP_FALSE) { 
  offset = fetch_next(); 
  if (val == STk_false) pc += offset;
  NEXT;
}
CASE(JUMP_TRUE) {
  offset = fetch_next();
  if (val != STk_false) pc += offset;
  NEXT;
}


CASE(DEFINE_SYMBOL) {
  SCM var = fetch_const();

  STk_define_variable(var, val, STk_current_module);
  if (CLOSUREP(val) && CLOSURE_NAME(val) == STk_false) CLOSURE_NAME(val) = var;
  val   = STk_void;
  val2  = var;
  valc  = 2;
  NEXT;
}


CASE(SET_CUR_MOD) {
  if (!MODULEP(val)) STk_error("bad module ~S", val);
  STk_current_module = env = val;
  NEXT;
}


CASE(POP)     { val = pop(); NEXT; }
CASE(PUSH)    { push(val);   NEXT; }
CASE(PUSH_R1) { push(r1);    NEXT; }


CASE(CREATE_CLOSURE) {
  /* pc[0] = offset; pc[1] = arity ; code of the routine starts in pc+2 */
  env    = clone_env(env);
  val    = STk_make_closure(pc+2, pc[0]-1, pc[1], constants, env);
  pc    += pc[0] + 1;
  NEXT;
}


CASE(PREPARE_CALL) { PREP_CALL(); NEXT; }
CASE(RETURN) 	   { RET_CALL();  NEXT; }
CASE(INVOKE)       {
  nargs = fetch_next();
  funcall(nargs, FALSE);
  NEXT;
}

CASE(TAIL_INVOKE) {
  nargs = fetch_next();
  funcall(nargs, TRUE);
  NEXT;
}


CASE(ENTER_LET_STAR) { sp += (int) pc[1] - 1; /* NO NEXT */ }
CASE(ENTER_LET) {
  nargs = fetch_next();

  /* Push a new env. on the stack. Activation record does not need to be updated  */
  sp -= (sizeof(struct frame_obj) - sizeof(SCM)) / sizeof(SCM);
  PUSH_ENV(nargs, val, env);
  env = (SCM) sp;
  NEXT;
}
CASE(LEAVE_LET) {
  sp  = fp + ACT_RECORD_SIZE;
  env = FRAME_NEXT(env);
  fp  = ACT_SAVE_FP(fp);
  NEXT;
}


CASE(ENTER_TAIL_LET_STAR) { sp += (int) pc[1] - 1; /* NO NEXT */ } 
CASE(ENTER_TAIL_LET) {
  SCM *old_fp = (SCM *) ACT_SAVE_FP(fp);

  nargs = fetch_next();
  /* Move the arguments of the function to the old_fp as in TAIL_INVOKE */
  if (IS_IN_STACKP(env)) {
    if (nargs) memmove((SCM*)env-nargs, sp, nargs*sizeof(SCM));
    fp = old_fp;
    
    /* Push a new environment on the stack */
    sp = (SCM*)env-nargs-((sizeof(struct frame_obj) - sizeof(SCM)) / sizeof(SCM));
  }
  else {
    if (nargs) memmove(old_fp-nargs, sp, nargs*sizeof(SCM));
    fp = old_fp;
    sp = fp - nargs - ((sizeof(struct frame_obj) - sizeof(SCM)) / sizeof(SCM));
  }

  PUSH_ENV(nargs, val, env);
  env  = (SCM) sp;
  NEXT;
}


CASE(PUSH_HANDLER) {
  SCM offset = fetch_next();

  /* place the value in val on the stack as well as the value of handlers */
  if (STk_procedurep(val) == STk_false)
    STk_error("bad exception handler ~S", val);

  SAVE_VM_STATE();
  SAVE_HANDLER_STATE(val, pc+offset);
  NEXT;
}

CASE(POP_HANDLER) {
  UNSAVE_HANDLER_STATE();
  RESTORE_VM_STATE(sp);
  NEXT;
}


CASE(MAKE_EXPANDER) {
  SCM name = fetch_const();
  SCM ref;

  STk_lookup(STk_intern("*expander-list*"), STk_current_module, &ref, TRUE);
  CDR(ref) = STk_cons(STk_cons(name, val), CDR(ref));
  valc = 2;
  val  = STk_void;
  val2 = name;
  NEXT;
}

CASE(END_OF_CODE) { 
  return; 
}

CASE(EVAL_CODE) {
  STk_panic("EVAL-CODE");
}

CASE(EXEC_HANDLER) {
  SCM value, location;

  /* (%%exec kind location value) => value @sp[1], location @sp[0], kind in val */
  location  = pop();
  value     = pop();
  
  STk_execute_current_handler(val, location, value);
}
 
  


/******************************************************************************
 *
 * 			     I n l i n e d   F u n c t i o n s
 *
 ******************************************************************************/
CASE(IN_ADD2)   { val = STk_add2(val, pop());		           NEXT;}
CASE(IN_SUB2)   { val = STk_sub2(val, pop());		           NEXT;}
CASE(IN_MUL2)   { val = STk_mul2(val, pop());		           NEXT;}
CASE(IN_DIV2)   { val = STk_div2(val, pop());		           NEXT;}

CASE(IN_NUMEQ)  { val = MAKE_BOOLEAN(STk_numeq2(val, pop()));      NEXT;}
CASE(IN_NUMLT)  { val = MAKE_BOOLEAN(STk_numlt2(val, pop()));      NEXT;}
CASE(IN_NUMGT)  { val = MAKE_BOOLEAN(STk_numgt2(val, pop()));      NEXT;}
CASE(IN_NUMLE)  { val = MAKE_BOOLEAN(STk_numle2(val, pop()));      NEXT;}
CASE(IN_NUMGE)  { val = MAKE_BOOLEAN(STk_numge2(val, pop()));      NEXT;}

CASE(IN_INCR)   { val = STk_add2(val, MAKE_INT(1));		   NEXT;}
CASE(IN_DECR)   { val = STk_sub2(val, MAKE_INT(1));		   NEXT;}

CASE(IN_CONS)   { val = STk_cons(val, pop());		           NEXT;}
CASE(IN_CAR)    { val = STk_car(val);			           NEXT;}
CASE(IN_CDR)    { val = STk_cdr(val);			           NEXT;}
CASE(IN_NULLP)  { val = MAKE_BOOLEAN(val == STk_nil);	           NEXT;}
CASE(IN_LIST)   { val = listify_top(fetch_next());	           NEXT;}
CASE(IN_NOT)    { val = (val==STk_false) ? STk_true: STk_false;    NEXT;}

CASE(IN_EQUAL)  { val = STk_equal(val, pop());			   NEXT;}
CASE(IN_EQV)    { val = STk_eqv(val, pop());			   NEXT;}
CASE(IN_EQ)     { val = MAKE_BOOLEAN(val == pop());		   NEXT;}

CASE(IN_VREF)   { val = STk_vector_ref(val, pop()); 		   NEXT;}
CASE(IN_SREF)   { val = STk_string_ref(val, pop()); 		   NEXT;}
CASE(IN_VSET)   { 
  SCM arg2 = pop();
  val = STk_vector_set(val, arg2, pop());
  NEXT;
}
CASE(IN_SSET)   { 
  SCM arg2 = pop();
  val = STk_string_set(val, arg2, pop());
  NEXT;
}

CASE(IN_APPLY)   {
  STk_panic("INSTRUCTION IN-APPLY!!!!!!!!!!!!!!!!!!!!!!!");
  NEXT;
}

#ifndef USE_COMPUTED_GOTO
      default:
	STk_panic("INSTRUCTION %d NOT IMPLEMENTED\n", byteop);
    }
  }
#endif
  STk_panic("abnormal exit from the VM");
}


static void run_vm(STk_instr *code, SCM *consts, SCM envt, int save)
{
  jbuf jb, *old_jb;
  
  /* Initialize */
  old_jb      = top_jmp_buf;
  top_jmp_buf = &jb;

  pc          = code;
  constants   = consts;
  env         = envt;

  if (MY_SETJMP(jb)) {
    /* We come back from an error. */
    set_signal_mask(jb.blocked);
  }

  /* Jump in the VM */
  run();
  
  /* Reset the jumbuf we have on entrance */
  top_jmp_buf = old_jb;
}


void STk_execute_current_handler(SCM kind, SCM location, SCM message)
{
  SCM proc;
  SCM *save_vm_state = handlers + EXCEPTION_HANDLER_SIZE;

  if (!handlers) {
    STk_print(LIST3(kind, message, location), STk_stderr, DSP_MODE);
    STk_fprintf(STk_stderr, "**** FATAL ERROR: no handler present!\nABORT\n");
    exit(1);
  }

  /*
   * Grab the handler infos 
   */
  proc  = (SCM) 	HANDLER_PROC(handlers);
  pc    = (STk_instr *) HANDLER_END(handlers);

  UNSAVE_HANDLER_STATE();
  
  RESTORE_VM_STATE(save_vm_state);

  /* Execute the procedure handler on behalf of the old handler (since the 
   * procedure can be itself erroneous). 
   */
  val = STk_C_apply(proc, 3, kind, location, message);

  /* 
   * Return in the good "run_vm" incarnation 
   */
  MY_LONGJMP(*top_jmp_buf, 1);
}


/*===========================================================================*\
 * 
 *			   C O N T I N U A T I O N S
 *
\*===========================================================================*/
DEFINE_PRIMITIVE("%make-continuation", make_continuation, subr0, (void))
{
  SCM z;
  static struct continuation_obj *k;
  static char *p;
  static int csize, ssize;

  p     = (char *) &z;
  ssize =  (stack + stack_len - 1) - sp;

  if (p < GC_stackbottom)
    csize = (SCM) GC_stackbottom - (SCM) p;
  else 
    csize = (SCM) p - (SCM) GC_stackbottom;

  NEWCELL_WITH_LEN(z, continuation, ssize * sizeof(SCM) + csize);

  k = (struct continuation_obj *) z;
  k->ssize 	 = ssize;
  k->csize 	 = csize;
#ifdef STACK_GROWS_DOWN
  k->c_start	 = p;
  k->c_end	 = GC_stackbottom;
#else
  k->c_start	 = GC_stackbottom;
  k->c_end	 = p;
#endif

  k->pc		 = pc;
  k->fp		 = fp;
  k->sp		 = sp;
  k->env	 = env;
  k->constants	 = constants;
  k->handlers	 = handlers;
  k->jb		 = top_jmp_buf;
  
  /* Save the Scheme stack */
  memcpy(k->stack, sp, ssize * sizeof(SCM));

  /* Save the C stack just after */
#ifdef STACK_GROWS_DOWN
  memcpy(k->stack + ssize, p, csize);
#else 
  memcpy(k->stack + ssize, GC_stackbottom, csize);
#endif

  if (MY_SETJMP(k->state) == 0) {
    /* This is the initial call to %make_continuation */
    return z;
  } else {
    /* We come back and restore the continuation */
    return val;
  }
}

static char *s, *d, *end;
static int cpt = 0;
static struct continuation_obj *kont;


void restore_C_stack(struct continuation_obj *k)
{
  union {
    SCM stack_end;
    SCM hole[1024]; /* Reserve 1K-pointers on stack */
  }u;
  

  /* Evaluate room available on the stack. If not enough room, 
   * do a recursive call to allocate a  new hole
   */
#ifdef STACK_GROWS_DOWN
  if (&u.stack_end > (SCM *) k->c_start) restore_C_stack(k);
#else
  if (&u.stack_end < (SCM *) k->c_end)   restore_C_stack(k);
#endif
 
  /* We have enough room. Go for a copy */
  FLUSH_REGISTERS_WINDOW();
  cpt = 0;
  kont = k;
  for (s = (char*)kont->stack + kont->ssize, d = kont->c_start, end = kont->c_end;
       cpt < kont->csize;
       s++, d++) {
    *d = *s;
    cpt++;
  }

  /* Now do a long jmp to go back in the creating %make-continuation call */
  MY_LONGJMP(kont->state, 1);
}  


DEFINE_PRIMITIVE("%restore-continuation", restore_continuation, subr2, 
		 (SCM cont, SCM value))
{
  struct continuation_obj *k;

  ENTER_PRIMITIVE(restore_continuation);
  
  if (!CONTP(cont)) STk_error("bad continuation ~S", cont);

  k = (struct continuation_obj *) cont;

  valc		= 1;
  val		= value;
  
  pc		= k->pc;
  fp		= k->fp;
  sp		= k->sp;
  env		= k->env;
  constants	= k->constants;
  handlers	= k->handlers;
  top_jmp_buf	= k->jb;
  
  /* Restore the Scheme stack */
  memcpy(sp, k->stack, k->ssize * sizeof(SCM));

  /* Restore the C stack */
  restore_C_stack(k);

  return STk_void; /* never reached */
}


static void print_continuation(SCM cont, SCM port, int mode)
{
  STk_fprintf(port, "#[continuation (%d %d) %x]", 
	      ((struct continuation_obj *)cont)->csize,
	      ((struct continuation_obj *)cont)->ssize,
	      (unsigned long) cont);
}

struct extended_type_descr xtype_continuation = {
  "continuation",		/* name */
  print_continuation		/* print function */
};


/*===========================================================================*\
 * 
 *			   Bytecode file dump/load stuff 
 *
\*===========================================================================*/

static int system_has_booted = 0;



/* This function is used to dump the code in a file */
DEFINE_PRIMITIVE("%dump-code", dump_code, subr2, (SCM f, SCM v))
{
  int size, i;
  SCM *tmp;
  STk_instr instr;
  
  ENTER_PRIMITIVE(dump_code);
  if (!FPORTP(f))  STk_error("bad file port ~S", f);
  if (!VECTORP(v)) STk_error("bad code vector ~S", v);

  size = VECTOR_SIZE(v); tmp = VECTOR_DATA(v);

  /* Print size as a Scheme value */
  STk_print(MAKE_INT(size), f, DSP_MODE);
  STk_putc('\n', f);

  /* Print the content of the vector as bytes */
  for (i = 0; i < size; i++) {
    if (!INTP(*tmp)) STk_error("bad value in code vector ~S", v);
   
    instr = (STk_instr) INT_VAL(*tmp++);
    STk_putc(FIRST_BYTE(instr), f);
    STk_putc(SECOND_BYTE(instr), f);
  
  }
  STk_putc('\n', f);
  return STk_void;
}


static Inline STk_instr* read_code(SCM f, int len) /* read a code phrase */
{
  STk_instr *res, *tmp;
  int i, c1, c2;

  tmp = res = STk_must_malloc_atomic(len * sizeof(STk_instr));

  /* skip the separator */
  STk_getc(f);

  /* Read 'len' instruction (coded on 2 bytes) */
  for (i = 0; i < len; i++) {
    c1 = STk_getc(f);
    c2 = STk_getc(f);
    if (c2 == EOF) /* not useful to test c1 */
      STk_error("truncated bytecode file ~S", f);
    
    *tmp++ = (STk_instr) (c1 << 8 | c2);
  }

  return res;  
}

SCM STk_load_bcode_file(SCM f)
{
  SCM consts, code_size, *save_constants, save_env;
  STk_instr *code, *save_pc;
  int size;

  /* Save machine state */
  save_pc = pc; save_constants = constants; save_env = env;

  /* Signature has been skipped during file type analysing */
  for ( ; ; ) {
    consts = STk_read_constant(f, FALSE);	    /* Read  the constants */
    if (consts == STk_eof) break;

    code_size = STk_read(f, FALSE);		    /* Read the code size */
    size      = STk_integer_value(code_size);
    if (size < 0) {
      if (system_has_booted)
	STk_error("Bad bytecode file ~S", f);
      else 
	return STk_false;
    }

    code = read_code(f, size);			    	   /* Read the code */

    run_vm(code, VECTOR_DATA(consts), STk_current_module, 1); /* Execute code read */
  }
  
  /* restore machine state */
  pc = save_pc; constants = save_constants, env = save_env;
  return STk_true;
}


int STk_load_boot(char *filename)
{
  SCM f, tmp;

  f = STk_open_file(filename, "r");
  if (f == STk_false) return -1;

  /* Verify that the file is a bytecode file */
  tmp = STk_read(f, TRUE);
  if (tmp != STk_intern("STklos")) return -2;
  STk_read(f, FALSE); /* Read the version -- unused for now */
  
  tmp = STk_load_bcode_file(f);
  if (tmp == STk_false) return -3;
    
  /* The system has booted on the given file */
  system_has_booted = 1;
  return 0;
}


int STk_init_vm(void)
{
  /* Initialize the VM registers */
  sp       = stack + stack_len;
  fp       = sp;
  val      = STk_void;
  env      = STk_current_module;
  handlers = NULL;

  DEFINE_XTYPE(continuation, &xtype_continuation);

  /* Initialize the table of checked references */
  checked_globals = STk_must_malloc(checked_globals_len * sizeof(SCM));

  /* Add the apply primitive */
  ADD_PRIMITIVE(scheme_apply);
  ADD_PRIMITIVE(execute);
  ADD_PRIMITIVE(dump_code);

  ADD_PRIMITIVE(values);
  ADD_PRIMITIVE(call_with_values);
  ADD_PRIMITIVE(make_continuation);
  ADD_PRIMITIVE(restore_continuation);
#ifdef DEBUG_VM
  ADD_PRIMITIVE(set_vm_debug);
#endif

  return TRUE;
}
