/*
 * Stellarium
 * This file Copyright (C) 2008 Matthew Gates
 * Parts copyright (C) 2016 Georg Zotti (added size transitions)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Suite 500, Boston, MA  02110-1335, USA.
 */


#include "ScreenImageMgr.hpp"
#include "StelMainView.hpp"
#include "StelApp.hpp"
#include "StelFileMgr.hpp"
#include "StelCore.hpp"
#include "StelModuleMgr.hpp"

#include "StelProjector.hpp"
#include "StelModule.hpp"
#include "StelUtils.hpp"
#include "VecMath.hpp"

#include <QGraphicsPixmapItem>
#include <QPixmap>
#include <QString>
#include <QDebug>
#include <QMap>
#include <QTimeLine>
#include <QGraphicsItemAnimation>

///////////////////////
// ScreenImage class //
///////////////////////
ScreenImage::ScreenImage(const QString& filename, float x, float y, bool show, float scale, float alpha, float fadeDuration)
	: tex(Q_NULLPTR), maxAlpha(alpha)
{
	QPixmap pixmap(filename);
	tex = StelMainView::getInstance().scene()->addPixmap(pixmap.scaled(pixmap.size()*scale));
	tex->setPos(x, y);

	anim = new QGraphicsItemAnimation();
	moveTimer = new QTimeLine();
	moveTimer->setCurveShape(QTimeLine::LinearCurve);
	anim->setTimeLine(moveTimer);
	anim->setItem(tex);

	scaleAnim = new QGraphicsItemAnimation();
	scaleTimer = new QTimeLine();
	scaleTimer->setCurveShape(QTimeLine::LinearCurve);
	scaleAnim->setTimeLine(scaleTimer);
	scaleAnim->setItem(tex);
	// we must configure the end for proper rescaling
	scaleAnim->setScaleAt(0., 1., 1.);
	scaleAnim->setScaleAt(1., 1., 1.);

	fadeTimer = new QTimeLine();
	fadeTimer->setCurveShape(QTimeLine::LinearCurve);
	setFadeDuration(fadeDuration);
	connect(fadeTimer, SIGNAL(valueChanged(qreal)), this, SLOT(setOpacity(qreal)));

	// set inital displayed state
	if (show)
		tex->setOpacity(maxAlpha);
	else
		tex->setOpacity(0.);
}

ScreenImage::~ScreenImage()
{
	moveTimer->stop();
	scaleTimer->stop();
	fadeTimer->stop();
	delete anim; anim = Q_NULLPTR;
	delete scaleAnim; scaleAnim = Q_NULLPTR;
	delete moveTimer; moveTimer = Q_NULLPTR;
	delete scaleTimer; scaleTimer = Q_NULLPTR;
	delete fadeTimer; fadeTimer = Q_NULLPTR;
	if (tex!=Q_NULLPTR)
	{
		delete tex;
		tex = Q_NULLPTR;
	}
}

bool ScreenImage::draw(const StelCore*)
{
	return true;
}

void ScreenImage::update(double)
{
}

void ScreenImage::setFadeDuration(float duration)
{
	fadeTimer->setDuration(qMax(1, (int)(duration*1000)));
}

void ScreenImage::setFlagShow(bool b)
{
	if (b)
	{
		fadeTimer->setFrameRange(tex->opacity()*fadeTimer->duration(),fadeTimer->duration());
		fadeTimer->setDirection(QTimeLine::Forward);
	}
	else
	{
		fadeTimer->setFrameRange(tex->opacity()*fadeTimer->duration(),0);
		fadeTimer->setDirection(QTimeLine::Backward);
	}
	fadeTimer->start();
}

bool ScreenImage::getFlagShow(void)
{
	return (tex->opacity() > 0.);
}

void ScreenImage::setAlpha(float a)
{
	maxAlpha = a;
	if (getFlagShow())
	{
		fadeTimer->stop();
		tex->setOpacity(a);
	}
}

void ScreenImage::setXY(float x, float y, float duration)
{
	if (duration<=0.)
	{
		moveTimer->stop();
		tex->setPos(x, y);
	}
	else
	{
		moveTimer->stop();
		moveTimer->setDuration(duration*1000);
		moveTimer->setFrameRange(0,100);
		QPointF p(tex->pos());
		if (p == QPointF(x,y)) return;
		float sX = p.x();
		float sY = p.y();
		float dX = (x-sX) / 200.;
		float dY = (y-sY) / 200.;
		for(int i=0; i<200; i++)
			anim->setPosAt(i/200., QPointF(sX+(dX*i), (sY+(dY*i))));
		anim->setPosAt(1.0, QPointF(x, y));
		moveTimer->start();
	}
}

void ScreenImage::addXY(float x, float y, float duration)
{
	QPointF currentPos = tex->pos();
	setXY(currentPos.x() + x, currentPos.y() + y, duration);
}

int ScreenImage::imageWidth(void)
{
	return tex->pixmap().size().width();
}

int ScreenImage::imageHeight(void)
{
	return tex->pixmap().size().height();
}

void ScreenImage::setOpacity(qreal alpha)
{
	tex->setOpacity(alpha*maxAlpha);
}

void ScreenImage::setScale(float scaleX, float scaleY, float duration)
{
	scaleTimer->stop();

	// Set a least a tiny duration to allow running the "animation"
	scaleTimer->setDuration(qMax(0.001f, duration)*1000);
	scaleTimer->setFrameRange(0,100);

	tex->setTransformationMode(Qt::SmoothTransformation);

	// reconfigure the animation which may have halted at 1 after a previous run, so the current scale is at position 1.
	scaleAnim->setScaleAt(0., scaleAnim->horizontalScaleAt(1.), scaleAnim->verticalScaleAt(1.));
	scaleAnim->setScaleAt(1., scaleX, scaleY);
	scaleTimer->start();
}

float ScreenImage::imageScaleX()
{
	return scaleAnim->horizontalScaleAt(1.);
}

float ScreenImage::imageScaleY()
{
	return scaleAnim->verticalScaleAt(1.);
}


//////////////////////////
// ScreenImageMgr class //
//////////////////////////
ScreenImageMgr::ScreenImageMgr()
{
	setObjectName("ScreenImageMgr");
}
 
ScreenImageMgr::~ScreenImageMgr()
{
}

void ScreenImageMgr::init()
{
}

void ScreenImageMgr::draw(StelCore* core)
{
	for (auto* m : allScreenImages)
		if (m!=Q_NULLPTR)
			m->draw(core);
}

void ScreenImageMgr::createScreenImage(const QString& id, const QString& filename, float x, float y,
	float scale, bool visible, float alpha, float fadeDuration)
{
	// First check to see if there is already an image loaded with the
	// specified ID, and drop it if necessary
	if (allScreenImages.contains(id))
		deleteImage(id);

	QString path = StelFileMgr::findFile("scripts/" + filename);
	if (!path.isEmpty())
	{
		ScreenImage* i = new ScreenImage(path, x, y, visible, scale, alpha, fadeDuration);
		if (i==Q_NULLPTR)
			return;

		allScreenImages[id] = i;
	}
	else
	{
		qWarning() << "Failed to create ScreenImage" << id << ": file not found: " << filename;
	}
}

void ScreenImageMgr::deleteImage(const QString& id)
{
	if (allScreenImages.contains(id))
	{
		if (allScreenImages[id]!=Q_NULLPTR)
		{
			delete allScreenImages[id];
			allScreenImages[id] = Q_NULLPTR;
		}
		allScreenImages.remove(id);
	}
}
	
void ScreenImageMgr::deleteAllImages()
{
	for (auto* m : allScreenImages)
	{
		if (m!=Q_NULLPTR)
		{
			delete m;
			m = Q_NULLPTR;
		}
	}
	allScreenImages.clear();
}

QStringList ScreenImageMgr::getAllImageIDs(void)
{
	return allScreenImages.keys();
}

bool ScreenImageMgr::getShowImage(const QString& id)
{
	if (allScreenImages.contains(id))
		if (allScreenImages[id]!=Q_NULLPTR)
			return allScreenImages[id]->getFlagShow();
	return false;
}

int ScreenImageMgr::getImageWidth(const QString& id)
{
	if (allScreenImages.contains(id))
		if (allScreenImages[id]!=Q_NULLPTR)
			return allScreenImages[id]->imageWidth();
	return 0;
}
	
int ScreenImageMgr::getImageHeight(const QString& id)
{
	if (allScreenImages.contains(id))
		if (allScreenImages[id]!=Q_NULLPTR)
			return allScreenImages[id]->imageHeight();
	return 0;
}

float ScreenImageMgr::getImageScaleX(const QString& id)
{
	if (allScreenImages.contains(id))
		if (allScreenImages[id]!=Q_NULLPTR)
			return allScreenImages[id]->imageScaleX();
	return 0;
}

float ScreenImageMgr::getImageScaleY(const QString& id)
{
	if (allScreenImages.contains(id))
		if (allScreenImages[id]!=Q_NULLPTR)
			return allScreenImages[id]->imageScaleY();
	return 0;
}

void ScreenImageMgr::showImage(const QString& id, bool show)
{
	if (allScreenImages.contains(id))
		if (allScreenImages[id]!=Q_NULLPTR)
			allScreenImages[id]->setFlagShow(show);
}

void ScreenImageMgr::setImageAlpha(const QString& id, float alpha)
{
	if (allScreenImages.contains(id))
		if (allScreenImages[id]!=Q_NULLPTR)
			allScreenImages[id]->setAlpha(alpha);
}

void ScreenImageMgr::setImageXY(const QString& id, float x, float y, float duration)
{
	if (allScreenImages.contains(id))
		if (allScreenImages[id]!=Q_NULLPTR)
			allScreenImages[id]->setXY(x, y, duration);
}

void ScreenImageMgr::addImageXY(const QString& id, float x, float y, float duration)
{
	if (allScreenImages.contains(id))
		if (allScreenImages[id]!=Q_NULLPTR)
			allScreenImages[id]->addXY(x, y, duration);
}

void ScreenImageMgr::setImageScale(const QString& id, float scaleX, float scaleY, float duration)
{
	if (allScreenImages.contains(id))
		if (allScreenImages[id]!=Q_NULLPTR)
			allScreenImages[id]->setScale(scaleX, scaleY, duration);
}


void ScreenImageMgr::update(double deltaTime)
{
	for (auto* m : allScreenImages)
		if (m!=Q_NULLPTR)
			m->update(deltaTime);
}
	
double ScreenImageMgr::getCallOrder(StelModuleActionName actionName) const
{
	if (actionName==StelModule::ActionDraw)
		return StelApp::getInstance().getModuleMgr().getModule("LandscapeMgr")->getCallOrder(actionName)+11;
        return 0;
}
