-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with E_Strings;
with SPARK_IO;
with SP_Symbols;

--# inherit CommandLineData,
--#         Dictionary,
--#         ErrorHandler,
--#         E_Strings,
--#         LexTokenManager,
--#         SparkLex,
--#         SparkMakeErrors,
--#         SPARK_IO,
--#         SP_Symbols;

package TokenManager is

   type Token is record
      Kind  : SP_Symbols.SP_Terminal;
      Value : E_Strings.T;
   end record;

   Null_Token : constant Token := Token'(Kind  => SP_Symbols.SPEND,
                                         Value => E_Strings.Empty_String);

   type Iterator is private;

   Null_Iterator : constant Iterator;

   procedure Get_First_Token (File_Id : in     SPARK_IO.File_Type;
                              It      :    out Iterator);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#           out SparkLex.Curr_Line;
   --# derives ErrorHandler.Error_Context,
   --#         It,
   --#         LexTokenManager.State,
   --#         SparkLex.Curr_Line,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         File_Id,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys;

   procedure Next (It : in out Iterator);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SparkLex.Curr_Line;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         It,
   --#         LexTokenManager.State,
   --#         SparkLex.Curr_Line,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         It,
   --#                                         LexTokenManager.State,
   --#                                         SparkLex.Curr_Line,
   --#                                         SPARK_IO.File_Sys;
   --# pre SparkLex.Curr_Line_Invariant (SparkLex.Curr_Line);
   --# post SparkLex.Curr_Line_Invariant (SparkLex.Curr_Line);

   procedure Look_Ahead (It : in out Iterator);
   --
   -- Allows the client to look at the next symbol after the last returned
   -- by Next or NextIdentifier. Only one lookahead is allowed. Multiple,
   -- consecutive calls to Look_Ahead will thus return the same symbol.
   -- Subsequent calls to Next and NextIdentifier function as normal (i.e. as if
   -- the lookahead had not been made).
   --
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SparkLex.Curr_Line;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         It,
   --#         LexTokenManager.State,
   --#         SparkLex.Curr_Line,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         It,
   --#                                         LexTokenManager.State,
   --#                                         SparkLex.Curr_Line,
   --#                                         SPARK_IO.File_Sys;
   --# pre SparkLex.Curr_Line_Invariant (SparkLex.Curr_Line);
   --# post SparkLex.Curr_Line_Invariant (SparkLex.Curr_Line);

   function Is_Null (It : Iterator) return Boolean;

   function Current (It : Iterator) return Token;

   function To_String (Tok : Token) return E_Strings.T;

private

   type Iterator is record
      File : SPARK_IO.File_Type;
      -- The token we expect
      Current_Token : Token;
      -- Sometimes it is necessary to read a token too far.
      -- If this is the case then Next_Token stores what the token is so it can
      -- be returned on the next call to Next.
      Next_Token : Token;
      -- Allows the user to lookahead.
      Is_Look_Ahead : Boolean;
   end record;

   Null_Iterator : constant Iterator :=
     Iterator'(File          => SPARK_IO.Null_File,
               Current_Token => Null_Token,
               Next_Token    => Null_Token,
               Is_Look_Ahead => False);

end TokenManager;
