-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (ErrorHandler.Conversions.ToString)
procedure SemanticErr (Err_Num          : in     Error_Types.NumericError;
                       With_Explanation : in     Boolean;
                       E_Str            : in out E_Strings.T) is

   procedure SemanticErrExpl (E_Str : in out E_Strings.T)
   --# global in Err_Num;
   --# derives E_Str from *,
   --#                    Err_Num;
      is separate;
   -- Note that the parameter names for this subunit are chosen to make it as easy as
   --      possible to auto-generate the subunit from this, its parent, file.  The
   --      generation requires copying the case statement below, stripping out the
   --      current Append'Thing' statements and adding an AppendString for the
   --      explanatory text that is delineated by --! comments.

   procedure Append_Explanation
   --# global in     Err_Num;
   --#        in     With_Explanation;
   --#        in out E_Str;
   --# derives E_Str from *,
   --#                    Err_Num,
   --#                    With_Explanation;
   is
      Explanation_String : E_Strings.T := E_Strings.Empty_String;
   begin
      if With_Explanation then
         -- we need to at least look for an explanation
         SemanticErrExpl (Explanation_String);
         if E_Strings.Get_Length (E_Str => Explanation_String) > 0 then
            -- there actually is one
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ErrorHandler.Explanation_Prefix);
            E_Strings.Append_Examiner_String (E_Str1 => E_Str,
                                              E_Str2 => Explanation_String);
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ErrorHandler.Explanation_Postfix);
         end if;
      end if;
   end Append_Explanation;

begin
   -- Directives for HTML output:

   --! <NameFormat> <"semantic-"><Name>
   --! <ErrorFormat> <"*** Semantic Error : "><Name><" : "><Error>

   -- To find a spare error number to add a new error, see the
   -- final case at the bottom of this case statement.
   case Err_Num.ErrorNum is
      when 1 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The identifier ");
         if Err_Num.Name2 /= Error_Types.NoName then
            Append_Name (E_Str => E_Str,
                         Name  => Err_Num.Name2,
                         Scope => Err_Num.Scope);
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ".");
         end if;
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is either undeclared or not visible at this point");
         --! If the identifier is declared in a
         --! separate (or parent) package, the package must be included in an inherit clause
         --! and the identifier prefixed with the package name.
         --! Ensure that there are no errors in the declaration of the identifier

      when 2 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " does not denote a formal parameter for ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);

      when 3 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Incorrect number of actual parameters for call of subprogram ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);

      when 4 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "More than one parameter association is given for formal parameter ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);

      when 5 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Illegal use of identifier ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! Usually associated with the use of an identifier other than a package
         --! name as a prefix in a selected component.

      when 6 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Identifier ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not the name of a variable");

      when 7 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Identifier ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not the name of a procedure");

      when 8 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "There is no field named ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " in this entity");
         --! Issued when the selector in a selected component of a record
         --! references a non-existent field.

      when 9 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Selected components are not allowed for ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! Occurs if the prefix to a selected component representing a procedure
         --! in a procedure call statement or a type mark is not a package. Also
         --! occurs if a selector is applied in an expression to an object which
         --! is not a record variable.

      when 10 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Illegal redeclaration of identifier ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);

      when 11 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "There is no package declaration for ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! Issued if a package body is encountered for which there is no
         --! package specification.

      when 12 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " can only be completed by a variable declaration, not a constant");
         --! If the object in question is really a constant, then remove it from
         --! the enclosing package's own variable annotation.

      when 13 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "A body for subprogram ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has already been declared");

      when 14 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Illegal parent unit name");
         --! Issued if the name in a &quot;separate&quot; clause of a subunit does not
         --! correctly identify a compilation unit.  Common causes of this error
         --! are a syntax error in the parent unit or omitting the
         --! parent unit specification and/or parent unit body entries from the index file

      when 15 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The stub for ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is either undeclared or cannot be located");
         --! Common causes of this error are an error in the declaration
         --! of the stub or the omission of the parent unit body from the index file

      when 16 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "A body for package ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has already been declared");

      when 17 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "A body stub for package ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has already been declared");

      when 18 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Identifier ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not the name of a package");

      when 19 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Identifier ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not the name of a procedure");

      when 20 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Illegal operator symbol");
         --! Issued if a renaming declaration contains a non-existent operator.

      when 21 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "This entity is not an array");
         --! Issued if an attempt is made to index into a name which does not
         --! represent an array.

      when 22 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The type in this declaration is not consistent with the previous declaration of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! Occurs when the type given in the Ada declaration of an own variable
         --! differs from that
         --! &quot;announced&quot; in the package's own variable clause.

      when 23 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "No parameter association is given for formal parameter ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);

      when 24 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The identifier ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " (exported by called subprogram) is not visible at this point");
         --! When a procedure is called any global variables exported by that
         --! procedure must be visible at the point of call. This error message
         --! indicates that the global variable concerned is not visible.
         --! It may be that it needs to be added to the global annotation of
         --! the procedure containing the call (or some further enclosing subprogram)
         --! or it may be that an inherit clause is missing from the package containing
         --! the call.

      when 25 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The identifier ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " (imported by called subprogram) is not visible at this point");
         --! When a procedure is called any global variables imported by that
         --! procedure must be visible at the point of call. This error message
         --! indicates that the global variable concerned is not visible.
         --! It may be that it needs to be added to the global annotation of
         --! the subprogram containing the call (or some further enclosing subprogram)
         --! or it may be that an inherit clause is missing from the package containing
         --! the call.

      when 26 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The deferred constant ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " does not have an associated full definition");
         --! Issued at the end of a package specification if no full declaration
         --! has been supplied for a deferred constant declared in the package
         --! specification.

      when 27 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The private type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " does not have an associated full definition");
         --! Issued at the end of a package specification if no full declaration
         --! has been supplied for a private type declared in the package specification.

      when 28 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " does not have a definition");
         --! Issued at the end of a package body if an own variable announced in the
         --! package specification has neither been given an Ada declaration nor refined.

      when 29 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The subprogram ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => ", declared in the package specification, does not have an associated body");
      when 30 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Attribute ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not yet implemented in the Examiner");
         --! The attribute is identified in Annex K of the SPARK 95 report as a
         --! valid SPARK 95
         --! attribute but the Examiner does not currently support it.
         --! It is
         --! possible to work round the omission by putting the use of the attribute
         --! inside a
         --! suitable function which is hidden from the Examiner.

      when 31 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The prefix of this attribute is not an object or type");

      when 32 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Illegal type conversion");
         --! Likely causes are type conversions involving record types or
         --! non-convertible arrays.

      when 33 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Illegal aggregate");
         --! Issued if the prefix of an aggregate is not a composite type.

      when 34 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Illegal procedure call");
         --! Issued if a call is made to a user-defined subprogram in a
         --! package initialization part.

      when 35 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Binary operator is not declared for types ");
         if Err_Num.Name1 /= Error_Types.NoName then
            Append_Name (E_Str => E_Str,
                         Name  => Err_Num.Name1,
                         Scope => Err_Num.Scope);
            if Err_Num.Name2 /= Error_Types.NoName then
               E_Strings.Append_String (E_Str => E_Str,
                                        Str   => " and ");
               Append_Name (E_Str => E_Str,
                            Name  => Err_Num.Name2,
                            Scope => Err_Num.Scope);
            end if;
         end if;
         --! Indicates use of an undeclared binary operator; this message means that
         --! the type on each side
         --! of the operator cannot appear with the operator used. e.g.
         --! attempting to add an integer to
         --! an enumeration literal.

      when 36 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Expression is not static");

      when 37 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Expression is not constant");

      when 38 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Expression is not of the expected type");

      when 39 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Illegal use of unconstrained type");
         --! An unconstrained array type or variable of such a type is
         --! illegally used. Use of
         --! unconstrained arrays in SPARK is limited to passing them as
         --! parameters, indexing into them
         --! and taking attributes of them.  This message also arises if a string
         --! literal is used as an actual parameter where the formal parameter is a string subtype.
         --! In this case, the error can be removed by qualifying the string literal with the subtype
         --! name.

      when 40 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Numeric or Time_Span type required");
         --! This operator is only defined for numeric types and, if the
         --! Ravenscar Profile is
         --! selected, for type Ada.Real_Time.Time_Span.

      when 41 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Array type required");
         --! Issued if a subtype declaration taking the form of a constrained
         --! subtype of an
         --! unconstrained array type is encountered but with a type mark which
         --! does not represent an
         --! array.

      when 42 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Incompatible types");
         --! Issued when a name represents an object which is not of the required type.

      when 43 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Range is not constant");

      when 44 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Scalar type required");
         --! The bounds of an explicit range must be scalar types.

      when 45 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Range is not static");

      when 46 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Discrete type required");

      when 47 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The definition of this type contains errors which may make this array definition invalid");
         --! Issued if an array type definition is encountered where one or
         --! more of the index types used in the definition contained errors in its original declaration.  For
         --! example, SPARK requires array index bounds to be constant (known at compile time) so an attempt
         --! to use an illegal subtype with variable bounds as an array index will generate this message.

      when 48 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Subtypes of private types are not permitted");
         --! Issued if an attempt is made to declare a subtype of a private type in a location where the full
         --! view of the type is not visible.

      when 49 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Attribute ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " takes only one argument");
         --! Only SPARK 95 attributes 'Min and 'Max require two arguments.

      when 50 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Initializing expression must be constant");
         --! To assign a non-constant expression to a variable, an assignment statement
         --! in the body of the program unit (following the 'begin') must be used.

      when 51 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Arrays may not be ordered");
         --! Issued if an ordering operator such as &quot;&lt;&quot; is encountered between
         --! objects of an array
         --! type other than string or a constrained subtype of string.

      when 52 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Only Scalar, String and Time types may be ordered");
         --! Ordering operators are only defined for scalar types and type
         --! String plus, if the
         --! Ravenscar Profile is selected, types Time and Time_Span in package
         --! Ada.Real_Time.

      when 53 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Illegal others clause");
         --! In SPARK record aggregates may not contain an others clause.

      when 54 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Illegal attribute: ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! Issued when an attribute not supported by SPARK is used.

      when 55 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Attribute ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " takes no argument");

      when 56 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Argument expected");

      when 57 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Fixed type definition must have associated range constraint");

      when 58 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " expected, to repeat initial identifier");
         --! Occurs at the end of a package, subprogram, protected type, task type
         --! or loop if the terminal identifier does not
         --! match the name or label originally given.

      when 59 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Composite subtype definition may not have associated range constraint");
         --! A subtype of the form applicable to a subrange of a scalar type has
         --! been encountered
         --! but the type provided is not a scalar type.

      when 60 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Illegal choice in record aggregate");
         --! In SPARK record aggregates may not contain multiple choices, each
         --! field must be
         --! assigned a value individually.

      when 61 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Illegal occurrence of body stub - a body stub may only occur in a compilation unit");

      when 62 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "A body for the embedded package ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is required");
         --! Issued if an embedded package declares subprograms or own variables
         --! and no body is
         --! provided.

      when 63 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not a type mark");

      when 64 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Parameters of function subprograms must be of mode in");

      when 65 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Formal parameters of renamed operators may not be renamed");
         --! The names of the parameters used in renaming declarations may not
         --! be altered from Left,
         --! Right for binary operators and Right for unary operators. These are
         --! the names given for
         --! the parameters in the ARM and the SPARK Definition requires that
         --! parameter names are not
         --! changed.

      when 66 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Unexpected package initialization - no own variables of package ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " require initialization");
         --! Either the package does not have an initializes annotation or all
         --! the own variables
         --! requiring initialization were given values at the point of declaration.

      when 67 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Illegal machine code insertion. Machine code functions are not permitted in SPARK 83.");
         --! This is an Ada 83 rule.  Machine code can only be used in procedures.

      when 68 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Illegal operator renaming - operators are defined on types not subtypes");
         --! Issued if an attempt is made to rename an operator using a subtype
         --! of the type for
         --! which it was originally implicitly declared.

      when 69 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "pragma ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has two parameters");

      when 70 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "pragma ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " expected");
         --! <Error> pragma Import expected.
         --! <Error2> pragma Interface expected.

      when 71 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "This expression does not represent the expected subprogram or variable name ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! Issued if the name supplied in a pragma interface, import or attach_handler
         --! does not match the name of the associated subprogram or variable.

      when 72 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Unexpected pragma ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! <Error> Unexpected pragma Import.
         --! Pragma import may only occur in a body stub, or immediately
         --! after a subprogram declaration in the visible part of a package,
         --! or immediately after a variable declaration.

         --! <Name> 72
         --! <Error> Unexpected pragma Interface.
         --! Pragma interface may only occur in a body stub or immediately
         --! after a subprogram declaration in the visible part of a package.

      when 73 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has already been declared or refined");
         --! Issued if an Ada declaration is given for an own variable which
         --! has been refined, or in
         --! a refinement clause if an own variable is refined more than once.

      when 74 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " does not occur in the package own variable list");
         --! A subject of a refinement definition of a package must be an own
         --! variable of that
         --! package.

      when 75 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Illegal use of inherited package");
         --! Issued if an attempt is made to refine an own variable onto an own
         --! variable of a
         --! non-embedded package.

      when 76 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Identifier ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is already declared and cannot be");
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " the name of an embedded package");
         --! Issued when a refinement clause in a package body attempts to name
         --! an embedded package
         --! own variable as a refinement constituent and the name given for the
         --! embedded package is
         --! already in use.

      when 77 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " should occur in this own variable clause");
         --! Occurs in the own variable clause of a package embedded in another
         --! package if an own
         --! variable which is a refinement constituent of an own variable of the
         --! enclosing package is
         --! omitted.

      when 78 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Initialization of own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is ineffective");
         --! Issued if an own variable occurs in the initialization clause of
         --! an embedded package
         --! and the own variable concerned is a refinement constituent of another
         --! own variable which
         --! is not listed in the initialization specification of its package.

      when 79 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " should occur in this initialization specification");
         --! Occurs in the initialization clause of a package embedded in another
         --! package if an own
         --! variable which is a refinement constituent of an initialized own variable
         --! of the enclosing
         --! package is omitted.

      when 80 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Unexpected own variable clause - ");
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "no variable in this clause ");
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "is a refinement constituent");

      when 81 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Own variable clause expected - ");
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "own variables of this package ");
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "occur as refinement constituents");

      when 82 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Unexpected initialization specification - ");
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "no own variables of this package ");
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "require initialization");
         --! An own variable initialization clause and that of its refinement
         --! constituents must be
         --! consistent.

      when 83 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Initialization specification expected - ");
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "own variables of this package require ");
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "initialization");
         --! Issued if an own variable does not occur in the initialization
         --! clause of an embedded
         --! package and the own variable concerned is a refinement constituent
         --! of another own variable
         --! which is listed in the initialization clause of its package.

      when 84 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The refinement constituent ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " does not have a declaration");
         --! Issued at the end of a package if a refinement constituent of a
         --! refined own variable
         --! has not been given an Ada declaration or further refined.

      when 85 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " is not a constituent of any abstract own variable appearing" &
              " in the earlier global definition for this subprogram");
         --! A variable XXX which has occurred in a refined global annotation
         --! is neither a variable
         --! that occurred in the earlier global definition nor a refinement
         --! constituent of any such
         --! variable.

      when 86 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "At least one constituent of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " was expected in this refined global definition");
         --! If the global annotation of a procedure specification contains
         --! an own variable and that
         --! own variable is later refined then at least one refinement constituent
         --! of the own variable
         --! shall appear in the second global annotation supplied for the procedure
         --! body.

      when 87 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Refined global definition expected for subprogram ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! A global definition containing abstract own variables was given in
         --!  the definition for
         --! subprogram XXX, in a package specification. A refined global definition
         --! is required in the
         --! package body.

      when 88 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not a refinement constituent");

      when 89 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not a private type declared in this package");

      when 90 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "This operator may not be applied to ranges");

      when 91 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Ranges may not be assigned");

      when 92 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Named association may not be used here");

      when 93 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Number of index expressions differs from number of dimensions of array ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);

      when 94 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Condition is not boolean");
         --! Issued anywhere a boolean expression is required (e.g. in<b> if</b>, <b>
         --! exit </b> and <b>while</b> statements) and the expression provided
         --! is not of type boolean.

      when 95 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Type mark expected");

      when 96 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Attribute ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not valid with this prefix");

      when 97 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Attribute BASE may only appear as a prefix");
         --! 'BASE may only be used as a prefix to another attribute.

      when 98 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "This expression is not a range");

      when 99 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Unconstrained array expected");
         --! Occurs if a subtype is declared of an array which is already constrained.

      when 100 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Floating point type mark expected");

      when 101 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Fixed point type mark expected");

      when 102 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "This is not the name of a field of record ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);

      when 103 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "A value has already been supplied for field ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);

      when 104 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "No value has been supplied for field ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
      when 105 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "More values have been supplied than number of fields in record ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);

      when 106 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Range is not of the expected type");

      when 107 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Expression is not of the expected type. Actual type is ");
         if Err_Num.Name1 /= Error_Types.NoName then
            Append_Name (E_Str => E_Str,
                         Name  => Err_Num.Name1,
                         Scope => Err_Num.Scope);
            if Err_Num.Name2 /= Error_Types.NoName then
               E_Strings.Append_String (E_Str => E_Str,
                                        Str   => ". Expected type is ");
               Append_Name (E_Str => E_Str,
                            Name  => Err_Num.Name2,
                            Scope => Err_Num.Scope);
            end if;
         end if;

      when 108 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Expression is not of the expected type. Expected any Integer type");

      when 109 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Expression is not of the expected type. Expected any Real type");

      when 110 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Use type clauses following an embedded package are not currently supported by the Examiner");

      when 111 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Package renaming is not currently supported by the Examiner");

      when 112 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "A use type clause may not appear here.  They are only permitted " &
              "as part of a context clause or directly following an embedded " &
              "package specification");

      when 113 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Private subprogram declarations are not permitted in SPARK 83");
         --! Private subprograms would not be callable in SPARK 83 and are therefore not
         --! permitted; they may be declared and called in SPARK 95.

      when 114 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Subtype mark or Range may not be used in an expression in this context");
         --! A subtype mark or an explicit Range attribute may not be used in a context
         --! where a simple expression is expected.

      when 115 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "In a package body, an own variable annotation must include one or more refinement constituents");
         --! Annotation should be of the form 'own S is A, B, C;'

      when 116 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "View conversion to own type is not permitted in target of an assignment");

      when 117 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Aggregate must be qualified with subtype mark");
         --! Aggregates are qualified expressions so they must be prefixed with a
         --! subtype mark. An exception is made in the case of aggregate assignments to
         --! unconstrained arrays as the rules of Ada do not permit unconstrained array
         --! aggregates to be qualified

      when 118 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Aggregate assignment to unconstrained multi-dimensional array not permitted");
         --! Unqualified aggregates may only be used in assignments to one-dimensional unconstrained
         --! arrays. SPARK does not permit aggregate assignment to multi-dimensional unconstrained
         --! arrays

      when 119 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Unary operator is not declared for type ");
         if Err_Num.Name1 /= Error_Types.NoName then
            Append_Name (E_Str => E_Str,
                         Name  => Err_Num.Name1,
                         Scope => Err_Num.Scope);
         end if;
         --! Indicates use of an undeclared unary operator; this message means that
         --! the type on the right hand side
         --! of the operator cannot appear with the operator used. e.g.
         --! attempting to negate an enumeration literal.

      when 120 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Pragma import not allowed here because variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is already initialized.  See ALRM B.1(24)");

      when 121 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "'Flow_Message' or 'Warning_Message' expected");
         --! The identifier indicating what kind of message to justify must be either
         --! 'Flow_Message' or 'Warning_Message' or some unique abbreviation of them such as
         --! 'Fl' or even 'F'.  Case is ignored.

      when 122 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Error or warning number expected");
         --! This item should be an integer literal representing the error or warning message that
         --! is being marked as expected.

      when 123 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "This warning number may not appear in an accept annotation");
         --! It does not make sense to allow certain warnings to be justified with the accept annotation.
         --! In particular, attempting to justify warnings raised by the justification system itself
         --! could lead to some special kind of recursive hell that we would not wish to enter.

      when 124 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Incorrect number of names in accept annotation: should be 0");
         --! This class of error does not reference any variables, and therefore requires
         --! no names.

      when 125 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Incorrect number of names in accept annotation: should be 1");
         --! This class of error references one variable, and therefore requires
         --! one name.

      when 126 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Incorrect number of names in accept annotation: should be 2");
         --! This class of error references two variables, and therefore requires
         --! two names. Two names are need to justify expected information
         --! flow messages such as &quot;X is not derived from Y&quot;.
         --! Note that for messages of this kind the accept annotation should list the names in the order
         --! &quot;export, import&quot;

      when 127 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Incorrect number of names in accept annotation: should be 0 or 1");
         --! This class of error references either zero or one variable, and therefore
         --! requires either zero or one name. An ineffective assignment error requires
         --! the name of variable being assigned to. An ineffective statement error
         --! has no name associated with it.

      when 128 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Incorrect number of names in accept annotation: should be 1 or 2");
         --! This class of error references either one or two variables, and therefore
         --! requires either one or two names. One name is required when the export
         --! is a function return value.

      when 129 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Assignment to view conversion is not currently implemented");

      when 130 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "A type from the current package should not appear in a use type clause");

      when 131 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The package name ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " should appear in a with clause preceding the use type clause");

      when 132 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The unit name or the name of an enclosing package of the unit should not appear in its with clause");
         --! A package should not 'with' itself and a subunit
         --! should not 'with' the package (or main program) which declares its stub

      when 133 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Name in with clause is locally redeclared");

      when 134 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "A package name should not appear in its own inherit clause");

      when 135 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The package ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " is undeclared or not visible, or there is a circularity in the list of inherited packages");
         --! Possible causes of this error are
         --! an error in the inherited package specification or
         --! omitting an entry for the package specification from the index file or
         --! circular inheritance

      when 136 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " is not declared in the own variable clause of the corresponding package declaration");
         --! A refinement clause of a package body defines the constituent
         --! parts of own variables given in the own variable clause of the
         --! corresponding package declaration

      when 137 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The child package ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is either undeclared or not visible at this point");
         --! Possible causes of this error are
         --! an error in the child package specification or
         --! omitting the child from the parent's component list in the index file
         --! or omitting the child specification entry from the index file

      when 138 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Child package own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is does not appear in the own variable clause of the child package");
         --! A constituent of a refinement clause which is defined in a
         --! child package must be an own variable of the child package

      when 139 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not declared in the own variable clause of this package");
         --! A package can only initialize variables
         --! declared in its own variable clause

      when 140 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The predecessor package ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is either undeclared or not visible at this point");
         --! The parent of a child package must be a library package and
         --! must be declared prior to a child package.  If using an index file
         --! the parent must have an entry in the index file and the child package
         --! must be listed as a component of the parent package

      when 141 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The private type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is either undeclared or not visible at this point");

      when 142 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The subprogram prefix ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is either undeclared or not visible at this point");
         --! The prefix should appear in the inherit clause of the current package

      when 143 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The subprogram ");
         if Err_Num.Name2 /= Error_Types.NoName then
            Append_Name (E_Str => E_Str,
                         Name  => Err_Num.Name2,
                         Scope => Err_Num.Scope);
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ".");
         end if;
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is either undeclared or not visible at this point");

      when 144 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The dotted name ");
         if Err_Num.Name2 /= Error_Types.NoName then
            Append_Name (E_Str => E_Str,
                         Name  => Err_Num.Name2,
                         Scope => Err_Num.Scope);
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ".");
         end if;
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is either undeclared or not visible at this point");
         --! The name must denote an entire variable or an own variable
         --! of a package.  If the variable or own variable is declared in a
         --! separate (or parent) package, the package must be included in an inherit clause
         --! and the identifier prefixed with the package name

      when 145 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The identifier ");
         if Err_Num.Name2 /= Error_Types.NoName then
            Append_Name (E_Str => E_Str,
                         Name  => Err_Num.Name2,
                         Scope => Err_Num.Scope);
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ".");
         end if;
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is either undeclared or not visible at this point");
         --! The identifier should be a typemark.  If the typemark is declared in a
         --! separate (or parent) package, the package must be included in an inherit clause
         --! and the identifier prefixed with the package name.
         --! Ensure that there are no errors in the declaration of the typemark

      when 148 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The abstract proof type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " may not be used to define an own variable in another package");
         --! Own variables may be &quot;type announced&quot; as being of an abstract proof
         --! type only where
         --! that type is declared later in the same package. Thus --# own State :
         --! T; is legal if
         --! --# type T is abstract; appears later in the package; however, --# own
         --! State : P.T; is
         --! illegal if T is an abstract proof type declared in remote package P.

      when 149 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "More than one own variable has been announced as being of type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " which may not therefore be declared as an abstract proof type");
         --! Occurs when an own variable clause announces more than one own variable
         --! as being of
         --! a type XXX and XXX is later declared as being of an abstract proof type.
         --! Each abstract
         --! own variable must be of a unique type.

      when 150 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Entire variable expected. The names of constants never appear in mandatory annotations");
         --! Issued when a the name of a constant is found in a mandatory annotation such as
         --! a global or derives annotation.  Constants should not appear in such annotations.

      when 151 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " does not occur either in the package own variable list or as a refinement constituent");
         --! A variable declared in a package must have been previously announced
         --! as either an own
         --! variable or as a concrete refinement constituent of an own variable.

      when 152 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The number of formal parameters is not consistent with the previous declaration of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
      when 153 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The declaration of formal parameter ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not consistent with the subprogram's previous declaration");
         --! Issued if the name, type or parameter mode of a parameter is different
         --! in the
         --! subprogram body declaration from that declared originally.

      when 154 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The subprogram or task body ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " does not have an annotation");
         --! A subprogram or task body must have a global annotation if it references
         --! global variables; a
         --! procedure or task body must have a dependency relation to perform
         --! information flow analysis.

      when 155 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Unexpected annotation - all annotations required for procedure or task body ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " have already occurred");
         --! Do not repeat global or derives annotations in the body
         --! (or body stub) of a subprogram, entry or task except for
         --! state (own variable) refinement.

      when 156 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Entire variable expected");
         --! Issued when an identifier which SPARK requires to be an entire
         --! variable represents
         --! something other than this. Most commonly this message occurs when
         --! a component of a
         --! structured variable appears in a core annotation.

      when 157 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The name ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " already appears in the global variable list");

      when 158 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is a formal parameter of this subprogram");
         --! Issued in a global annotation if it names a formal parameter of the
         --! subprogram.

      when 159 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The name ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has already appeared as an exported variable");

      when 160 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The name ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " already appears in the list of imported variables");

      when 161 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Exportation of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is incompatible with its parameter mode");
         --! Issued if a parameter appears as an export to a procedure when it is of
         --! parameter mode
         --! in.

      when 162 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Importation of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is incompatible with its parameter mode");
         --! Issued if a parameter appears as an import to a procedure when it is of
         --! parameter mode
         --! out.

      when 163 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Subprogram ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " cannot be called from here");
         --! SPARK contains rules to prevent construction of programs containing
         --! recursive
         --! subprogram calls; this error message occurs if a procedure or function
         --! is called before
         --! its body has been declared. Re-ordering of subprogram bodies in the
         --! package concerned will
         --! be required.

      when 164 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Actual parameter of mode in out or out must be the name of a variable");

      when 165 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "This parameter is overlapped by another one, which is exported");
         --! Violation of the anti-aliasing rule.

      when 166 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "This parameter is overlapped by an exported global variable");
         --! Violation of the anti-aliasing rule.

      when 167 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Imported variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not named in the initialization specification of its package");
         --! Issued when an own variable which is imported into the main program
         --! procedure (or a task when the Ravenscar profile is enabled) has not
         --! been declared as being initialized by its package.  At the main program
         --! level the only
         --! imports that are permitted are initialized own variables of inherited
         --! packages.  There are
         --! two possible cases to consider: (1) the main program should be importing
         --! the variable in
         --! which case it should be annotated in its package with --# initializes
         --! (and, of course, actually
         --! initialized in some way) or be an external variable or protected variable
         --! which is implicitly
         --! initialized; or (2) the own variable concerned is not initialized at
         --! elaboration,
         --! should not therefore be considered an import to the main program and
         --! should be removed from the
         --! main program's import list.

      when 168 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is a loop parameter, whose updating is not allowed");

      when 169 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Global variables of function subprograms must be of mode in");
         --! It is an important property of SPARK  that functions cannot have
         --! side-effects, therefore
         --! only the reading of global variable is permitted.  It is usually
         --! convenient to omit
         --! modes from function global annotations but use of mode 'in' is
         --! permitted.

      when 170 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is a formal parameter of mode in, whose updating is not allowed");

      when 171 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is a formal parameter of mode out, whose value cannot be read");

      when 172 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The actual parameter associated with an exported formal parameter must be an entire variable");
         --! Issued if an actual parameter which is an array element is associated
         --! with an exported
         --! formal parameter in a procedure call. Exported parameters must be either
         --! entire variables
         --! or a record field.

      when 173 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "This exported parameter is named in the global definition of the procedure");
         --! Violation of the anti-aliasing rule.

      when 174 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not an own variable");
         --! Occurs in initialization specifications if something other than a
         --! variable is listed as
         --! being initialized.

      when 176 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " does not have a derives annotation so it may not be called from ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " which is a function or does have a derives annotation");
         --! When analysing with flow=auto, a procedure or entry without a derives annotation
         --! may not be called by a subprogram, task or entry with a derives annotation. This is because
         --! the body of the caller must be checked against its derives annotation.
         --! In order to calculate the correct dependency relation for the body of the caller
         --! there must be derives annotations present on all called procedures or entries.
         --! Functions are considered to have implicit derives annotations.

      when 180 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Entire composite constant expected");
         --! Issued when an identifier which SPARK requires to be an entire
         --! composite constant represents something other than this.

      when 181 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Invalid policy for constant proof rule generation");

      when 182 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Rule Policy for ");
         if Err_Num.Name2 /= Error_Types.NoName then
            Append_Name (E_Str => E_Str,
                         Name  => Err_Num.Name2,
                         Scope => Err_Num.Scope);
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ".");
         end if;
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " already declared in current scope");
         --! Issued when a rule policy has already been declared for this
         --! constant within this declarative region. This rule policy will
         --! be ineffective.

      when 190 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The name ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " already appears in the inherit clause");

      when 191 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The name ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " already appears in the with clause");

      when 200 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The parameter ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is neither imported nor exported");
         --! Each formal parameter of a subprogram shall be imported or exported or both.

      when 201 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The global variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is neither imported nor exported");
         --! Every variable in a global definition must also appear in the associated derives annotation
         --! where it will be either imported or exported or both.

      when 250 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The 'Size value for type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has already been set");

      when 251 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The attribute value for ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "'Size must be of an integer type");

      when 252 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The attribute value for ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "'Size must be a static simple expression");
         --! The value of 'Size must be static and must be of
         --! an integer type

      when 253 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The attribute value for ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "'Size must not be negative");
         --! The value of 'Size must be a positive integer or zero

      when 254 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The Size attribute can only be specified for a first subtype");
         --! Setting 'Size for a user-defined non-first subtype is not permitted.
         --! See Ada95 LRM 13.3(48)

      when 255 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The Address attribute can only be specified for a variable, a constant, or a program unit");
         --! Ada95 LRM Annex N.31 defines a program unit to be either a package, a task unit,
         --! a protected unit, a protected entry, a generic unit, or an explicitly
         --! declared subprogram other than an enumeration literal.

      when 273 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " may not be refined because it was declared with a " &
              "type mark which has not subsequently been declared as " &
              "an abstract proof type");
         --! Where a type mark is included in an own variable declaration it
         --! indicates that the own
         --! variable will either be of a concrete type of that name (which may
         --! be either already
         --! declared or be declared later in the package) or of an abstract proof
         --! type declared in
         --! the package specification.  In the former case the refinement is
         --! illegal because own
         --! variables of concrete Ada types may not be refined.  In the latter case
         --! it is legal;
         --! however, no suitable proof type declaration has been found in this case.

      when 300 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Renaming declarations are not allowed here");
         --! A renaming declaration must be the first declarative item of a package
         --! body or main
         --! program or it must be placed immediately after the declaration of
         --! an embedded package.

      when 301 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Renaming or use type declarations here can only rename subprograms in package ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! A renaming declaration may be placed immediately after the declaration
         --! of an embedded
         --! package; in this case it may only rename subprograms declared in that
         --! package.

      when 302 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The subprogram specification in this renaming declaration " &
              "is not consistent with the declaration of subprogram ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! Issued in a subprogram renaming declaration if it contains parameter
         --! names, numbers or
         --! types which differ from those originally declared.

      when 303 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "An operator can only be renamed by the same operator");
         --! Issued if a renaming declaration has a different operator on each
         --! side of the reserved
         --! word RENAMES.

      when 304 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "A renaming declaration for operator ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not allowed");

      when 305 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The specification in this renaming declaration is not " &
              "consistent with the implicit declaration of operator ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! Issued in an operator renaming declaration if it contains types
         --! which differ from those
         --! applicable to the operator being renamed.

      when 306 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Operator ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is already visible");
         --! Occurs in an operator renaming declaration if an attempt is made
         --! to rename an operator
         --! which is already visible. (The message will also appear as a secondary
         --! consequence of
         --! trying to rename an operator between undeclared types.)

      when 307 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The implicit declaration of this operator does not occur in package ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);

      when 308 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Type is limited");
         --! Issued if an attempt is made to assign a variable of a type which is
         --! limited or which
         --! contains a limited type.

      when 309 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Operator not visible for these types");
         --! This message means that the operator exists between the types on each
         --! side of it but
         --! that it is not visible. The most likely cause is that the types
         --! concerned are defined in
         --! another package and that renaming is required to make the operator visible.

      when 310 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The % operator may only appear in an assert or check statement in a for loop");
         --! The % operator is used to indicate the value of a variable on entry to a for loop.  This is because
         --! the variable may be used in the exit expression of the loop and may also be modified in the body
         --! of the loop.  Since the semantics of Ada require the exit expression to be fixed after evaluation
         --! we require a way of reasoning about the original value of a variable prior to any alteration in
         --! the loop body.  No other situation requires this value so % may not be used anywhere else.

      when 311 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Announced own variable types may not be implemented as unconstrained arrays");
         --! Where an own variable is announced as being of some type,
         --! SPARK requires that type
         --! to be declared; the declaration cannot be in the form of an
         --! unconstrained array because
         --! SPARK prohibits unconstrained variables.

      when 312 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "A subprogram can only be renamed to the same name with the package prefix removed");

      when 313 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Only one main program is permitted");

      when 314 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has been refined and may not appear here");
         --! Issued if an attempt is made to use, in a second annotation, an own
         --! variable which has
         --! been refined. Second annotations should use the appropriate refinement
         --! constituents of the
         --! own variable.

      when 315 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Unsupported proof context");
         --! Certain proof contexts have been included in the syntax of SPARK but
         --! are not yet
         --! supported; this error message results if one is found.

      when 316 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Selected components are not allowed for ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " since type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is private here");
         --! If a type is private, then record field selectors may not be used.
         --! In pre- and post-conditions, a proof function can be declared to
         --! yield the required attribute of a private type.

      when 317 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Tilde, in a function return annotation, may only be applied to an external variable of mode IN");
         --! The tilde decoration indicates the initial value of a variable or
         --! parameter which is both imported and exported. A function may not have an explicit side effect
         --! on a program variable and so cannot be regarded as exporting such a variable.  For modelling purposes
         --! a read of an external (stream) variable is regarded as having a side effect (outside the SPARK
         --! boundary).  Since it may be necessary to refer to the initial value of the external variable, before
         --! this implicit side effect occurs, the use of tilde is allowed only for external variables of mode IN which
         --! are globally referenced by function.
      when 318 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Tilde or Percent may only be applied to variables");
         --! The tilde decoration indicates the initial value of a variable or
         --! parameter which is
         --! both imported and exported. Percent indicates the value of a variable
         --! on entry to a for loop; this message occurs if either operator is applied
         --! to any other object.

      when 319 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Tilde may only be applied to a variable which is both imported and exported");
         --! The tilde decoration indicates the initial value of a variable or
         --! parameter which is
         --! both imported and exported; this message occurs if the variable
         --! concerned is either
         --! exported only or imported only in which case no distinction between
         --! its initial and final
         --! value is required.

      when 320 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Tilde or Percent may only be applied to an entire variable");
         --! Tilde (and %) may not be applied to an element of an array or field of a record.
         --! e.g. to
         --! indicate the initial value of the Ith element of array V use V~(I)
         --! not V(I)~.

      when 321 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Tilde may not appear in pre-conditions");
         --! Since it does not make sense to refer to anything other than the
         --! initial value of a
         --! variable in a pre-condition there is no need to use tilde to distinguish
         --! initial from
         --! final values.

      when 322 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Only imports may be referenced in pre-conditions or return expressions");
         --! Pre-conditions are concerned with the initial values of information
         --! carried into a
         --! subprogram. Since only imports can do this only imports can appear in
         --! pre-condition
         --! expressions.

      when 323 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Updates may only be applied to records or arrays");
         --! The extended SPARK update syntax is only used to express changes to
         --! components of a
         --! structured variable.

      when 324 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Only one field name may appear here");
         --! When using the extended SPARK update syntax for a record, you can not
         --! update more than
         --! one element in each clause of the update. For example, you cannot use
         --! [x,y =&gt; z], you
         --! must instead use [x =&gt; z; y =&gt; z].

      when 325 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has not been declared");
         --! Occurs if a type is &quot;announced&quot; as part of an own variable
         --! clause and the end
         --! of the package is reached without an Ada declaration for a type of
         --! this name being found.

      when 326 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Predicate is not boolean");
         --! Occurs anywhere where a proof context is found not to be a boolean
         --! expression.

      when 327 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is a global variable which may not be updated in a function subprogram");

      when 328 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The identifier ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " (exported by called subprogram) may not be updated in a function subprogram");
         --! Occurs if a function calls a procedure which exports a global variable;
         --! this would
         --! create an illegal side-effect of the function.

      when 329 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Illegal function call");
         --! Issued if a call is made to a user-defined subprogram in a package
         --! initialization part.

      when 330 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Illegal use of an own variable not of this package");
         --! Issued if an attempt is made, in a package initialization part, to
         --! update an own
         --! variable of a non-enclosing package.

      when 331 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Private types may not be unconstrained arrays");

      when 332 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "This private type was not declared as limited");
         --! Issued where the type contains a component which is a limited private
         --! type, but where
         --! the declaration of this type in the visible part of the package does
         --! not specify that the
         --! type is limited.

      when 333 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Initialization of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not announced in the initialization clause of this package");
         --! Issued when an own variable is initialized either by assignment or
         --! by having a pragma Import attached to it when initialization of the variable
         --! is not announced in its package's own variable initialization specification.

      when 334 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Identifier ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not the name of a function");

      when 335 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "This annotation should be placed with the declaration of function ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! Issued if a function is declared in a package specification without an
         --! annotation but
         --! one is then supplied on the function body.

      when 336 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Unexpected annotation - all annotations required for function ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " have already occurred");

      when 337 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Package ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " may not be used as a prefix here");
         --! Selected component notation may not be used in places where an item is
         --! directly
         --! visible.

      when 338 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Scalar parameter ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is of mode in out and must appear as an import");
         --! Parameters passed as mode in out must be listed as imports in the
         --! subprogram's
         --! dependency relation if they are of scalar types. The rule also applies
         --! to a parameter of a
         --! private type if its full declaration is scalar.

      when 339 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Subprogram ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " was not declared in package ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);

      when 340 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Only operators may be renamed in package specifications");
         --! User-declared subprograms may not be renamed in package specifications
         --! although the
         --! implicitly declared function subprograms associated with operators may be.

      when 341 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "A range may not appear here");
         --! Issued if a range is found where a single value is expected, for example,
         --! if an array
         --! slice is constructed.

      when 342 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "This proof annotation should be placed with the declaration of subprogram ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);

         --! Like global and derives annotations, proof annotations should be placed
         --! on the first
         --! appearance of a subprogram.  There may also be a requirement for a
         --! second proof annotation
         --! on a subprogram body where it references an abstract own variable.

      when 343 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Unexpected proof annotation - all annotations required for subprogram ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " have already occurred");
         --! Issued if a second proof annotation for a subprogram is found but
         --! the subprogram does
         --! not reference any abstract own variables.  A second annotation is
         --! only required where it
         --! is necessary to express both an abstract (external) and a refined
         --! (internal) view of an
         --! operation.

      when 399 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Range error in annotation expression");
         --! Issued if a proof annotation contains an expression that would cause
         --! a constraint error
         --! if it were in an executable Ada statement.  For example: &quot;--# post X =
         --! T'Succ(T'Last);&quot;
         --! VCs generated from such malformed predicates would always be unprovable.

      when 400 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Expression contains division by zero");
         --! Issued when a static expression, evaluated using perfect arithmetic,
         --! is found to
         --! contain a division by zero.

      when 401 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Illegal numeric literal");
         --! Issued when a numeric literal is illegal because it contains, for example,
         --! digits not
         --! compatible with its number base.

      when 402 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Constraint_Error will be raised here");
         --! Issued whenever a static expression would cause a constraint error.
         --! e.g. assigning a
         --! value to a constant outside the constant's type range. In SPARK a static
         --! expression may
         --! not yield a value which violates a range constraint.

      when 403 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Argument value is inconsistent with the number of dimensions of array type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! Issued when an array attribute containing an argument is found and the
         --! value of the
         --! argument is inconsistent with the number of dimensions of the array
         --! type to which it is
         --! being applied.

      when 406 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Only subtypes of scalar, non-tagged record, or constrained array types may be declared without a constraint");
         --! Issued if a subtype declaration of the form </i>subtype S is T<i>
         --! is used where </i>T<i> is not a scalar, constrained array or non-tagged record type.
         --! Secondly, T must not be private at the point of this declaration.

      when 407 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "This choice overlaps a previous one");
         --! Choices in case statements and array aggregates may not overlap.

      when 408 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Case statement is incomplete");
         --! A case statement must either explicitly supply choices to cover the
         --! whole range of the
         --! (sub)type of the controlling expression, or it must supply an others choice.

      when 409 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Empty range specified");
         --! In SPARK, no static range is permitted to be null.

      when 410 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Choice out of range");
         --! The choices in case statements and array aggregates must be within
         --! the constraints of
         --! the appropriate (sub)type.

      when 411 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Others clause required");
         --! Issued where an others clause is required to satisfy the Ada language rules.

      when 412 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Explicit boolean range not permitted");
      when 413 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Invalid range constraint");
         --! Issued where a range constraint is outside the range of the (sub)type to
         --! which the
         --! constraint applies.

      when 414 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Array aggregate is incomplete");
         --! An array aggregate must either explicitly supply values for all array
         --! elements or
         --! provide an others clause.

      when 415 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Too many entries in array aggregate");
         --! Issued where an array aggregate using positional association contains
         --! more entries than
         --! required by the array index type.

      when 416 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Type may not have an empty range");

      when 417 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "String subtypes must have a lower index bound of 1");

      when 418 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Index upper and/or lower bounds do not match those expected");
         --! Issued where assignment, association or type conversion is attempted
         --! between two
         --! different constrained subtypes of the same unconstrained array type, and
         --! where the index
         --! bounds do not match.

      when 419 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => ".");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has been renamed locally, so the prefix ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " must not be used");
         --! When an entity is renamed, the fully qualified name is no longer visible,
         --! and so must not be used.

      when 420 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Array index(es) not convertible");
         --! Issued when an attempt is made to convert between two arrays whose indexes
         --! are neither
         --! of the same type nor numeric.

      when 421 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Array components are not of the expected type");
         --! Issued when a type conversion attempts to convert between two array types
         --! whose
         --! components are of different types.

      when 422 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Array component constraints do not match those expected");
         --! Issued when a type conversion attempts to convert between two array types
         --! whose
         --! components are of the same type but do not have constraints which can be
         --! statically
         --! determined to be identical.

      when 423 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Array has different number of dimensions from that expected");
         --! Issued when attempting to convert between two array types which have
         --! different numbers
         --! of dimensions.

      when 425 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "String literals may not be converted");
         --! Issued if the argument of a type conversion is a string literal. A common
         --! cause is an
         --! attempt to type qualify a string and accidentally omitting the tick character.

      when 500 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Mode expected");
         --! Issued when performing data flow analysis only where a subprogram has no
         --! dependency
         --! clause and its global variables have not been given modes in the global
         --! annotation.

      when 501 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Dependency relation expected");
         --! A dependency relation is required for each procedure if information flow
         --! analysis is to
         --! be performed.

      when 502 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Exportation of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is incompatible with its global mode");
         --! Issued when a procedure has both a global annotation with modes and a
         --! dependency
         --! relation, and a global of mode in is listed as an export in the dependency
         --!  relation.

      when 503 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Importation of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is incompatible with its global mode");
         --! Issued when a procedure has both a global annotation with modes and a
         --! dependency
         --! relation, and a global of mode out is listed as an import in the
         --! dependency relation.

      when 504 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Parameter ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is of mode in out and must appear as an import");

      when 505 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Global variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is of mode in out and must appear as an import");
         --! Issued where a procedure has both a global annotation with modes and
         --! a dependency
         --! relation, and a global variable of mode in out is not listed as an
         --! import in the
         --! dependency relation.

      when 506 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Parameter ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is of mode in out and must appear as an export");

      when 507 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Global variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is of mode in out and must appear as an export");
         --! Issued where a procedure has both a global annotation with modes
         --! and a dependency
         --! relation, and a global variable of mode in out is not listed as an
         --! export in the
         --! dependency relation.

      when 508 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "This global variable is a parameter of mode in and can only have the global mode in");

      when 509 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Unexpected refined dependency relation");
         --! When using refinement in automatic flow analysis mode, if there is a
         --! dependency relation on the subprogram specification then there must
         --! also be one on the body. Similarly, if there is no dependency relation
         --! on the specification then the body is not permitted to have one.

      when 550 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "use type clauses may only be used in SPARK95: clause ignored");

      when 551 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "All operators for type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " are already visible");

      when 552 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " already appears in the use type clause");

      when 554 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is a limited private type for which no operators can be made visible");

      when 555 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not mentioned in an earlier with clause of this compilation unit");

      when 600 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "pragma Import has a minimum of 2 and a maximum of 4 parameters");

      when 601 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Convention, Entity, External_Name or Link_Name expected");

      when 602 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "An association for ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has already been given");

      when 603 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "No association for ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " was given");

      when 604 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "This package may not have a body - consider use of pragma Elaborate_Body");
         --! In Ada 95, a package body is illegal unless it is required for the
         --! purpose of providing
         --! a subprogram body, or unless this pragma is used. This error is
         --! issued where a package
         --! body is found for a package whose specification does not require a body.

      when 605 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "pragma Elaborate_Body has one parameter");

      when 606 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "This expression does not represent the expected package name ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! Issued when the parameter to a pragma Elaborate_Body is invalid.

      when 607 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "This package requires a body and must therefore" &
              " include either pragma Elaborate_Body or a subprogram " &
              "declaration");
         --! Issued where a package specification contains no subprogram declarations,
         --! but whose own
         --! variables (as specified in the package annotation) are not all declared
         --! (and initialized
         --! where appropriate) in the package specification. This is because such a
         --! package is not
         --! allowed a body in Ada 95 unless either the pragma is given or a
         --! subprogram declared.

      when 608 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Reduced accuracy subtypes of real numbers are considered obsolescent and are not supported by SPARK");

      when 609 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "This entity cannot be assigned to");

      when 610 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Child packages may not be used in SPARK83");

      when 611 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Illegal use of deferred constant prior to its full declaration");

      when 613 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Illegal name for body stub");
         --! Issued if a dotted name appears in a body stub as in &quot;package body
         --! P.Q is
         --! separate&quot;. No legal stub could ever have such a name.

      when 614 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Child packages may be declared only at library level");
         --! Issued if an attempt is made to declare a child package which is
         --! embedded in a package
         --! or subprogram.

      when 615 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Name does not match name of package");
         --! Issued if the closing identifier of a package has a different number of
         --! identifiers
         --! from the name originally given for the package. For example &quot;package
         --! P.Q is ... end
         --! P.Q.R;&quot;

      when 616 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The private package ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not visible at this point");
         --! Issued if an attempt is made to with or inherit a private package
         --! from the visible part
         --! of a public package.

      when 617 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Public sibling ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not visible at this point");
         --! Arises from attempting to inherit a public sibling child package
         --! from a private child
         --! package.

      when 618 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The owner of the current package does not inherit the package ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! A private child package can only inherit a remote package if its
         --! parent also inherits
         --! it; this is a analogous to the behaviour of embedded packages which
         --!  may also only inherit
         --! a remote package if their enclosing package also does so.

      when 619 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The package ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not owned by the current package");
         --! This message indicates an attempt to claim that own variables of a
         --! package other than a
         --! private child package of the current package are refinement constituents
         --!  of an abstract
         --! own variable of the current package.

      when 620 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Own variables here must be refinement constituents in package owner ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! Own variables of private child packages must appear as refinement
         --! constituents of the
         --! package which owns the child. If the Examiner has seen the owner
         --! package body before
         --! processing the child and has not found the required refinement
         --! constituent then this
         --! message results on processing the child.

      when 621 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " expected as a refinement constituent in this package");
         --! Own variables of private child packages must appear as refinement
         --! constituents of the
         --! package which owns the child. If the Examiner has seen a child package
         --! which declares an
         --! own variable before examining its owner&#146;s body then this message
         --! is issued if the
         --! owner lacks the required refinement constituent declaration.

      when 622 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " did not occur in an initialization specification");
         --! Issued if an own variable appears in an initialization clause and is
         --! also a refinement
         --! constituent of an own variable which is not marked as initialized.

      when 623 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " occurred in an initialization specification");
         --! Issued if an own variable does not appear in an initialization clause
         --! and is also a
         --! refinement constituent of an own variable that is marked as initialized.

      when 624 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "All operators from ancestor package ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " are already visible");
         --! A package must appear in a <b>with</b> clause before types declared in
         --! it can be specified in a <b>use type</b> clause.

         -- nice gap here for generics
      when 630 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not the name of generic subprogram");
         --! Only generic subprogram can be instantiated.

      when 631 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Generic function found where a generic procedure was expected");
         --! Subprogram kind of generic and its instantiation must match

      when 632 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Generic procedure found where a generic function was expected");
         --! Subprogram kind of generic and its instantiation must match

      when 633 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Generic actual part expected, generic unit ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has generic formal parameters");
         --! The number of generic formal and actual parameters must match exactly

      when 634 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Unexpected generic actual part, generic unit ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " does not have any generic formal parameters");
         --! The number of generic formal and actual parameters must match exactly

      when 635 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Incorrect number of generic actual parameters for instantiation of generic unit ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! The number of generic formal and actual parameters must match exactly

      when 636 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not compatible with generic formal parameter ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         --! See ALRM 12.5.  Each generic formal type parameter must be supplied with an actual type
         --! which is of a compatible class.  Note that SPARK does not have default values for such
         --! associations

      when 637 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "User-defined generic units are not permitted in SPARK 83");
         --! There are weaknesses in the generic type model of Ada 83 that prevent the implementation
         --! of a safe subset of generics in SPARK 83.  These deficiencies are overcome in Ada 95.
         --! SPARK 83 users may employ the predefined unit Unchecked_Conversion only.

      when 638 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Unexpected global annotation.  A generic subprogram  may not reference or update global variables");
         --! A standalone generic subprogram may not have a global annotation.  Note that a subprogram in a
         --! generic package may have a global annotation as long as it only refers to own variables that
         --! are local to the package.

      when 639 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "A generic formal object may only have default mode or mode in");
         --! SPARK restricts formal objects to being constants in order to avoid concealed information
         --! flows.

      when 640 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "A generic formal object may only be instantiated with a constant expression");
         --! SPARK restricts formal objects to being constants in order to avoid concealed information
         --! flows.

      when 641 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "There is no generic subprogram declaration named ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " so a generic body of that name cannot be declared here");
         --! A generic body must be preceded by a generic declaration of the same name.

      when 642 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " is not the name of a generic subprogram so a generic body cannot be declared here");
         --! A generic body must be preceded by a generic declaration of the same name.

      when 643 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " is a generic function so a generic procedure body cannot be declared here");
         --! A generic body must be preceded by a generic declaration of the same name and kind.

      when 644 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " is a generic procedure so a generic function body cannot be declared here");
         --! A generic body must be preceded by a generic declaration of the same name and kind.

      when 645 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Actual array element ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not compatible with the element type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " of the generic formal parameter");
         --! See ALRM 12.5.  Each generic formal type parameter must be supplied with an actual type
         --! which is of a compatible class.  Note that SPARK does not have default values for such
         --! associations

      when 646 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Actual array index ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not compatible with the index type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " of the generic formal parameter");
         --! See ALRM 12.5.  Each generic formal type parameter must be supplied with an actual type
         --! which is of a compatible class.  Note that SPARK does not have default values for such
         --! associations

      when 647 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Actual array ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has more dimensions than formal array ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         --! See ALRM 12.5.  Each generic formal type parameter must be supplied with an actual type
         --! which is of a compatible class.  Note that SPARK does not have default values for such
         --! associations

      when 648 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Actual array ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has fewer dimensions than formal array ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         --! See ALRM 12.5.  Each generic formal type parameter must be supplied with an actual type
         --! which is of a compatible class.  Note that SPARK does not have default values for such
         --! associations

      when 649 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Actual array ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is constrained but the associated formal ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is unconstrained");
         --! See ALRM 12.5.  Each generic formal type parameter must be supplied with an actual type
         --! which is of a compatible class.  Note that SPARK does not have default values for such
         --! associations

      when 650 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Actual array ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is unconstrained but the associated formal ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is constrained");
         --! See ALRM 12.5.  Each generic formal type parameter must be supplied with an actual type
         --! which is of a compatible class.  Note that SPARK does not have default values for such
         --! associations

      when 651 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Variables of generic types may not be initialized at declaration");
         --! In non-generic code we statically know the value being assigned to the variable and can
         --! check that it is in range.  In the case of a generic we cannot do this because we do not
         --! know the bounds of the variable's type.  The variable may, however, be assigned to in the sequence
         --! of statements in the generic body because generation of run-time checks will provide
         --! suitable protection from out-of-range values.

      when 652 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Subtypes of generic types are not permitted");
         --! In non-generic code we statically know the values being used as the range bounds for
         --! a subtype and can check that they are in range.  In the case of a generic we cannot
         --! do this because we do not know the bounds of the variable's type.

      when 653 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Constants of generic types are not permitted");
         --! In non-generic code we statically know the value being assigned to the constant and can
         --! check that it is in range.  In the case of a generic we cannot do this because we do not
         --! know the bounds of the constant's type.  A variable, assigned to in the sequence
         --! of statements in the generic body, may be a suitable substitute for such a constant.
      when 654 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " is a generic subprogram which must be instantiated before it can be called");
         --! Generic units provide a template for creating callable units and are not directly
         --! callable.
      when 655 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Invalid prefix, ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is a generic package");
         --! Components of generic packages cannot be accessed directly.  First instantiate the package
         --! and then access components of the instantiation.
      when 656 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The only currently supported attribute in this context is 'Always_Valid");
      when 657 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "A 'Always_Valid assertion requires a variable here");
         --! The 'Always_Valid assertion can only be applied to variables or
         --! to components of record variables.
      when 658 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The object in this assertion must be scalar or a non-tagged aggregation of scalar components");
         --! The 'Always_Valid assertion can only be applied to objects which are:
         --! (1) of a scalar type,
         --! (2) a one dimensional array of scalar components,
         --! (3) an entire record variable of a non-tagged type with all
         --! components that are either scalar or an array of scalar components,
         --! (4) an array variable whose components are records satisfying (3).
         --! Additionally a field of a record satisfying these constraints may
         --! be marked individually as always valid.
      when 659 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "A 'Always_Valid assertion must be in the same declarative " &
              "region as contains the declaration of the variable to " &
              "which it refers");
      when 660 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "A 'Always_Valid assertion must not be applied to an object already marked as always valid");
      when 662 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Only Mode in own variables and constituents can be marked using 'Always_Valid");
         --! The 'Always_Valid assertion can only be applied to variables which are
         --! own variables with the mode in, or to subcomponents of records
         --! which are mode in own variables.

      when 700 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Mode 'in out' may not be applied to own variables or their refinement constituents");
         --! Own variables may be given a mode to indicate that they are system level
         --! inputs
         --! or outputs (i.e. they obtain values from or pass values to the external
         --! environment).  Since effective SPARK design strictly separates inputs from
         --! outputs the mode 'in out' is not permitted.

      when 701 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The mode of this refinement constituent is not consistent with its subject:  ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! If an abstract own variable is given a mode then its refinement
         --! constituents must
         --! all be of the same mode.

      when 702 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " must be given the mode 'in' to match its earlier announcement ");
         --! Issued if an own variable of an embedded package is not given the
         --! same mode as
         --! the earlier refinement constituent that announced it would exist.

      when 703 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " must be given the mode 'out' to match its earlier announcement ");
         --! Issued if an own variable of an embedded package is not given the same
         --! mode as
         --! the earlier refinement constituent that announced it would exist.

      when 704 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " may not have a mode because one was not present in its earlier announcement ");
         --! Issued if an own variable of an embedded package is given a mode when
         --! the earlier refinement constituent that announced it would exist did not
         --! have one.

      when 705 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Refinement constituent ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " must be given the mode 'in' to match the child package own variable with which it is being associated");
         --! If a refinement constituent is an own variable of a private package then the
         --! constituent must have the same mode as the own variable to which it refers.

      when 706 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Refinement constituent ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " must be given the mode 'out' to match the child package own variable with which it is being associated");
         --! If a refinement constituent is an own variable of a private package then the
         --! constituent must have the same mode as the own variable to which it refers.

      when 707 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Refinement constituent ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " may not have a mode because one was not present on the child package " &
              "own variable with which it is being associated");
         --! If a refinement constituent is an own variable of a private package then the
         --! constituent can only be given a mode if the own variable to which it
         --!  refers has one.

      when 708 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has a mode and may not appear in an initializes clause");
         --! Mode own variables (stream variables) are implicitly initialized by the
         --! environment
         --! to which they are connected and may not appear in initializes clauses
         --! since this
         --! would require their explicit initialization.

      when 709 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Own variable or constituent ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has mode 'out' and may not be referenced by a function");
         --! Functions are permitted to reference own variables that are either unmoded
         --! or of
         --! mode 'in'.  Since mode 'out' own variables represent outputs to the
         --! environment,
         --! reading them in a function does not make sense and is not allowed.

      when 710 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The own variable or constituent ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is of mode 'in' and can only have global mode 'in'");
         --! Global modes, if given, must be consistent with the modes of own variables
         --! that appear
         --! in the global list.

      when 711 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The own variable or constituent ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is of mode 'out' and can only have global mode 'out'");
         --! Global modes, if given, must be consistent with the modes of own
         --! variables that appear
         --! in the global list.

      when 712 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The own variable or constituent ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " is of either mode 'in' or mode 'out' and  may not have global mode 'in out'");
         --! Global modes, if given, must be consistent with the modes of own variables
         --! that appear
         --! in the global list.

      when 713 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The own variable or constituent ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is of mode 'in' and may not appear in a dependency clause as an export");
         --! Own variables with mode 'in' denote system-level inputs; their exportation
         --! is
         --! not allowed.

      when 714 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The own variable or constituent ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is of mode 'out' and may not appear in a dependency clause as an import");
         --! Own variables with mode 'out' denote system-level outputs; their
         --! importation is
         --! not allowed.

      when 715 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Function ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " references external (stream) variables and may only appear directly" &
              " in an assignment or return statement");
         --! To avoid ordering effects, functions which globally access own
         --! variables which
         --! have modes (indicating that they are connected to the external
         --! environment) may
         --! only appear directly in assignment or return statements.
         --! They may not appear as
         --! actual parameters or in any other form of expression.

      when 716 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "External (stream) variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " may only appear directly" &
              " in an assignment or return statement; or as an actual parameter to an unchecked conversion");
         --! To avoid ordering effects, own variables which
         --! have modes (indicating that they are connected to the external
         --! environment) may
         --! only appear directly in assignment or return statements. They may not
         --! appear as
         --! actual parameters (other than to instantiations of Unchecked_Conversion) or in any other form of expression.

      when 717 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "External (stream) variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is of mode 'in' and may not be assigned to");
         --! Own variables with mode 'in' represent inputs to the system from the
         --! external
         --! environment. As such, assigning to them does not make sense and is not
         --! permitted.

      when 718 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "External (stream) variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is of mode 'out' and may not be referenced");
         --! Own variables with mode 'out' represent outputs to the external
         --! environment from the system. As such, referencing them does not make sense
         --! and is not permitted.

      when 719 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "External (stream) variables may not be referenced or updated during package elaboration");
         --! Own variables with modes represent inputs and outputs between the external
         --! environment and the system. Referencing or updating them during package
         --! elaboration would introduce ordering effects and is not permitted.

      when 720 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " is an external (stream) variable and may not be initialized at declaration");
         --! Own variables with modes represent inputs and outputs between the external
         --! environment and the system. Referencing or updating them during package
         --! elaboration would introduce ordering effects and is not permitted.

      when 721 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "This refined function global annotation may not reference ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " because it is an external (stream) variable whose abstract subject ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " does not have a mode");
         --! Functions may be used to reference external (stream) variables and the
         --! Examiner
         --! generates the appropriate information flow to show that the value returned
         --!  by
         --! the function is 'volatile'.  If the abstract view of the same function
         --! shows it
         --! referencing an own variable which is not an external stream then the
         --! volatility
         --! of the function is concealed.  The error can be removed either by making the
         --! abstract own variable a mode 'in' stream or by using a procedure instead
         --! of a
         --! function to read the refined stream variable.

      when 722 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The mode on abstract global variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " must be made 'in out' to make it consistent with the referencing of" &
              " mode 'in' external (stream) constituent ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " in the refined global annotation");
         --! Where a procedure references an external (stream) variable of mode 'in' the
         --! Examiner constructs appropriate information flow to show that the input
         --! stream
         --! is 'volatile'. If the abstract view shows that the procedure obtains its
         --! result
         --! by simply reading an own variable which is not an external stream then the
         --! volatility is concealed.  The error can be removed either by making the
         --! global
         --! mode of XXX 'in out' or making XXX an external (stream) variable of mode
         --! 'in'.

      when 723 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " must appear in this refined global annotation");
         --! Issued when a global variable which is present in the first (abstract)
         --! global annotation
         --! is omitted from the second (refined) one.

      when 724 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Exit label must match the label of the most closely enclosing loop statement");
         --! If an exit statement names a loop label, then the most closely enclosing
         --! loop statement must have a matching label

      when 725 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Protected function or variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " may only appear directly in an assignment or return statement");
         --! To avoid ordering effects, protected functions  may
         --! only appear directly in assignment or return statements. They may not
         --! appear as
         --! actual parameters or in any other form of expression.  Ordering effects
         --! occur
         --! because the global state referenced by the protected function may be updated
         --! by another process during expression evaluation.

      when 730 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "A loop with no iteration scheme or exit statements may only " &
              "appear as the last " &
              "statement in the outermost scope of the main subprogram (or a task " &
              "body when using the Ravenscar profile)");
         --! If a loop has neither an iteration scheme nor any exit statements then it
         --! will
         --! run forever. Any statements following it will be unreachable. SPARK only
         --! allows
         --! one such loop which must be the last statement of the main program

      when 750 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The identifier ");
         if Err_Num.Name2 /= Error_Types.NoName then
            Append_Name (E_Str => E_Str,
                         Name  => Err_Num.Name2,
                         Scope => Err_Num.Scope);
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ".");
         end if;
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is either undeclared or not visible at this point");
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => ". An array type may not be used as its own index type");
         --! The type mark used for the index of an array type declaration must not
         --! be the same as the name of the array type being declared

      when 751 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The identifier ");
         if Err_Num.Name2 /= Error_Types.NoName then
            Append_Name (E_Str => E_Str,
                         Name  => Err_Num.Name2,
                         Scope => Err_Num.Scope);
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ".");
         end if;
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is either undeclared or not visible at this point");
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => ". A record type may not include fields of its own type");
         --! The type mark given for a field in a record type declaration must
         --! not be the same as the name of the record type being declared

      when 752 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The identifier ");
         if Err_Num.Name2 /= Error_Types.NoName then
            Append_Name (E_Str => E_Str,
                         Name  => Err_Num.Name2,
                         Scope => Err_Num.Scope);
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ".");
         end if;
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is either undeclared or not visible at this point");
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => ". This identifier must appear in a preceding legal global annotation or formal parameter list");
         --! For an identifier to appear legally as an import in a derives annotation,
         --! it must be a formal parameter or must appear legally in a
         --! preceding global annotation and must be of mode 'in' or mode 'in out'

      when 753 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The identifier ");
         if Err_Num.Name2 /= Error_Types.NoName then
            Append_Name (E_Str => E_Str,
                         Name  => Err_Num.Name2,
                         Scope => Err_Num.Scope);
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ".");
         end if;
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is either undeclared or not visible at this point");
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => ". This identifier must appear in a preceding legal global annotation or formal parameter list");
         --! For an identifier to appear legally as an export in a derives annotation,
         --! it must be a formal parameter or must appear legally in a
         --! preceding global annotation and must be of mode 'out' or mode 'in out'

      when 754 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The identifier ");
         if Err_Num.Name2 /= Error_Types.NoName then
            Append_Name (E_Str => E_Str,
                         Name  => Err_Num.Name2,
                         Scope => Err_Num.Scope);
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ".");
         end if;
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is either undeclared or not visible at this point");
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => ". This package must be both inherited and withed to be visible here");
         --! For a package name to be visible in Ada context, it must appear in
         --! both the inherit clause and the with clause of the enclosing package

      when 755 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The identifier ");
         if Err_Num.Name2 /= Error_Types.NoName then
            Append_Name (E_Str => E_Str,
                         Name  => Err_Num.Name2,
                         Scope => Err_Num.Scope);
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ".");
         end if;
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is either undeclared or not visible at this point");
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => ". A parent of a child package must be inherited to be visible here");
         --! A parent of a child package must be inherited (but not withed)
         --! to be visible in that child.

      when 756 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The identifier ");
         if Err_Num.Name2 /= Error_Types.NoName then
            Append_Name (E_Str => E_Str,
                         Name  => Err_Num.Name2,
                         Scope => Err_Num.Scope);
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ".");
         end if;
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is either undeclared or not visible at this point");
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => ". The grandparent of a child package should not appear in this prefix.");
         --! A grandparent of a child package should not be included in prefixes
         --! referencing a declaration of the child package

      when 757 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The identifer ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is either undeclared or not visible at this point");
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => ". A record field name cannot be the same as its indicated type");

      when 770 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "If Any_Priority is defined, Priority and Interrupt_Priority must also be defined");
         --! If the type Any_Priority is defined in package System, then the subtypes
         --! Priority and
         --! Interrupt_Priority must also be defined; if support for tasking is not
         --! required, then
         --! the definition of Any_Priority may be removed

      when 771 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The parent type of this subtype must be Any_Priority");
         --! Ada 95 requires that both Priority and Interrupt_Priority be immediate
         --! subtypes
         --! of Any_Priority.

      when 772 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The range of Priority must contain at least 30 values; LRM D.1(26)");
         --! Ada 95 requires that the range of the subtype Priority include at least
         --! 30 values;
         --! this requirement is stated in the Ada 95 Language Reference Manual at
         --! D.1(26)

      when 773 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Priority'First must equal Any_Priority'First; LRM D.1(10)");
         --! Ada 95 requires that task priority types meet the following criteria,
         --! the second of which is relevant to this error:
         --! <li>subtype Any_Priority is Integer range implementation-defined;
         --! <li>subtype Priority is Any_Priority range Any_Priority'First .. implementation-defined;
         --! <li>subtype Interrupt_Priority is Any_Priority range Priority'Last+1 .. Any_Priority'Last

      when 774 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Interrupt_Priority'First must equal Priority'Last + 1; LRM D.1(10)");
         --! Ada 95 requires that task priority types meet the following criteria,
         --! the third of which is relevant to this error:
         --! <li>subtype Any_Priority is Integer range implementation-defined;
         --! <li>subtype Priority is Any_Priority range Any_Priority'First .. implementation-defined;
         --! <li>subtype Interrupt_Priority is Any_Priority range Priority'Last+1 .. Any_Priority'Last

      when 775 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Interrupt_Priority'Last must equal Any_Priority'Last; LRM D.1(10)");
         --! Ada 95 requires that task priority types meet the following criteria,
         --! the third of which is relevant to this error:
         --! <li>subtype Any_Priority is Integer range implementation-defined;
         --! <li>subtype Priority is Any_Priority range Any_Priority'First .. implementation-defined;
         --! <li>subtype Interrupt_Priority is Any_Priority range Priority'Last+1 .. Any_Priority'Last

      when 776 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "In SPARK95 mode, only packages Standard, System, Ada.Real_Time " &
              "and Ada.Interrupts may be specified in the config file");
         --! In SPARK95 mode, the packages that may be specified in the target
         --! configuration file are: Standard, System, Ada.Real_Time and Ada.Interrupts.
         --! The latter two are ignored unless the Ravenscar profile is selected.

      when 777 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "In package System, Priority must be an immediate subtype of Integer");
         --! Ada 95, and hence SPARK95, defines Priority as being an immediate
         --! subtype of Integer

      when 778 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "This identifier is not valid at this point in the target configuration file");
         --! The specified identifier cannot be used here; it is most probably
         --! either not valid in the target configuration file at all, or might
         --! be valid in a different package, but not here.

      when 779 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Definition of this package in the target configuration file is not allowed in SPARK83 mode");
         --! In SPARK83 mode, only package Standard may be specified in the
         --! target configuration file.

      when 780 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " must be private");
         --! This type may only be declared as private in the target configuration
         --! file.

      when 781 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The lower bound of a signed integer type declaration must be greater than or equal to System.Min_Int");
         --! This error can only be generated in SPARK95 mode when the configuration
         --! file specifies
         --! a value for System.Min_Int

      when 782 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The upper bound of a signed integer type declaration must be less than or equal to System.Max_Int");
         --! This error can only be generated in SPARK95 mode when the configuration
         --! file specifies
         --! a value for System.Max_Int

      when 783 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Modulus must be less than or equal to System.Max_Binary_Modulus");
         --! This error can only be generated in SPARK95 mode when the configuration
         --! file specifies
         --! a value for System.Max_Binary_Modulus

      when 784 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "System.Max_Binary_Modulus must be a positive power of 2");

      when 785 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The number of digits specified exceeds the value defined for System.Max_Digits");
         --! The maximum decimal precision for a floating point type, where
         --! a range specification has not been included, is defined
         --! by System.Max_Digits

      when 786 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The number of digits specified exceeds the value defined for System.Max_Base_Digits");
         --! The maximum decimal precision for a floating point type, where
         --! a range specification has been included, is defined
         --! by System.Max_Base_Digits

      when 787 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Digits value must be positive");

      when 788 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Delta value must be positive");

      when 789 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The only currently supported type attribute in this context is 'Base");

      when 790 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "A base type assertion requires a type here");

      when 791 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The base type in this assertion must be a predefined type");
         --! Predefined types are those defined either by the language, or in package
         --! Standard, using the configuration file mechanism</i>

      when 792 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The types in this assertion must both be either floating point or signed integer");

      when 793 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "This base type must have a defined range in the configuration file");
         --! If a predefined type is to be used in a base type assertion or in
         --! a derived type declaration, then it must appear in the configuration
         --! file and have a well-defined range.

      when 794 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Range of subtype exceeds range of base type");

      when 795 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "A base type assertion must be in the same declarative region as that of the full type definition");

      when 796 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "This type already has a base type: either it already has a " &
              "base type assertion, or is explicitly derived, or is a predefined type");
         --! A base type assertion can only be given exactly once. Explicitly derived
         --! scalar types and predefined types never need a base type assertion.

      when 797 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The base type in a floating point base type assertion must have a defined accuracy");

      when 798 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The accuracy of the base type in a base type assertion must be at " &
              "least that of the type which is the subject of the assertion");
      when 799 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Only a simple type can be the subject of a base type assertion ");

      when 800 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Modulus must be a positive power of 2");
         --! In SPARK, modular types must have a modulus which is a positive
         --! power of 2

      when 801 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Modular types may only be used in SPARK95");
         --! Ada83 (and hence SPARK83) does not include modular types

      when 803 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Unary arithmetic operators are not permitted for modular types");
         --! Unary arithmetic operators are of little value.  The &quot;abs&quot; and &quot;+&quot; operators
         --! have no effect for modular types, and so are not required.  The unary minus
         --! operator is a source of potential confusion, and so is not permitted in
         --! SPARK

      when 804 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Universal expression may not be implicitly converted to a modular " &
              "type here. " &
              "Left hand operand requires qualification to type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! A universal expression cannot be used as the left hand operand of a binary
         --! operator if the right hand operand is of a modular type.  Qualification of
         --! the
         --! left hand expression is required in this case.

      when 805 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Universal expression may not be implicitly converted to a modular " &
              "type here. " &
              "Right hand operand requires qualification to type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! A universal expression cannot be used as the right hand operand of a binary
         --! operator if the left hand operand is of a modular type.  Qualification of
         --! the
         --! right hand expression is required in this case.

      when 814 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Default_Bit_Order must be of type Bit_Order");
         --! The only possible type for the constant System.Default_Bit_Order
         --! is System.Bit_Order when it appears in the configuration file

      when 815 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The only allowed values of Default_Bit_Order are Low_Order_First and High_Order_First");
         --! System.Bit_Order is implicity declared in package System when a configuration
         --! file is given. This is an enumeration type with only two literals
         --! Low_Order_First and High_Order_First

      when 820 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Abstract types are not currently permitted in SPARK");
         --! Only non-abstract tagged types are currently supported.  It is hoped to
         --! lift this
         --! restriction in a future Examiner release.

      when 821 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "This type declaration must be a tagged record because it's private type is tagged");
         --! If a type is declared as &quot;tagged private&quot; then its full declaration must
         --! be a tagged
         --! record.

      when 822 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not a tagged type; only tagged types may be extended");
         --! In SPARK, &quot;new&quot; can only be used to declare a type extension; other
         --! derived types are not permitted.

      when 823 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "This type may not be extended in the same package in which it is declared");
         --! SPARK only permits types from another library package to be extended.
         --! This rule prevents
         --! overloading of inherited operations.

      when 824 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "This package already extends a type from package ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => ".  Only one type extension per package is permitted");
         --! SPARK only permits one type extension per package.  This rule prevents
         --! overloading of inherited operations.

      when 825 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " expected in order to complete earlier private extension");
         --! Since SPARK only permits one type extension per package it follows that
         --! the declaration
         --! &quot;new XXX with private&quot; in a package visible part must be paired with &quot;new
         --! XXX with record...&quot;
         --! in its private part.  The ancestor type XXX must be the same in both
         --! declarations.

      when 826 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Type extension is not permitted in SPARK 83");
         --! Type extension is an Ada 95 feature not included in Ada or SPARK 83.

      when 827 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The actual parameter associated with a tagged formal parameter in an " &
              "inherited operation must be an object not an expression");
         --! There are several reasons for this SPARK rule.  Firstly, Ada requires
         --! tagged parameters to be
         --! passed by reference and so an object must exist at least implicitly.
         --! Secondly, in order to
         --! perform flow analysis of inherited subprogram calls, the Examiner needs
         --! identify what subset of the
         --! information available at the point of call
         --! is passed to and from the called subprogram.  Since information can only
         --! flow through objects it
         --! follows that actual parameter must be an object.

      when 828 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Tagged types and tagged type extensions may only be declared in library-level package specifications");
         --! This SPARK rule facilitates the main uses of tagged types while greatly
         --! simplifying visibility rules.

      when 829 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Illegal re-declaration: this subprogram shares the same name as the inheritable root operation ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " but does not override it");
         --! To avoid overloading, SPARK prohibits more than one potentially visible
         --! subprogram having the
         --! same name.

      when 830 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "A private type may not be implemented as a tagged type or an extension of a tagged type");
         --! This rule means that a private type can only be implemented as a tagged
         --! type if the private
         --! type itself is tagged.

      when 831 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Extended tagged types may only be converted in the direction of their root type");
         --! This is an Ada rule: type conversions simply omit unused fields of the
         --! extended type.  It follows
         --! that conversions must be in the direction of the root type.

      when 832 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Only tagged objects, not expressions, may be converted");
         --! For flow analysis purposes the Examiner needs to know what subset of the
         --! information in
         --! the unconverted view is available in the converted view.  Since
         --! information can only flow
         --! through objects it follows that only objects can be converted.

      when 833 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Invalid record aggregate: type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has a private ancestor");
         --! If an extended type has a private ancestor then an extension aggregate
         --!  must be used rather
         --! than a normal aggregate.

      when 834 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Null records are only permitted if they are tagged");
         --! An empty record can have no use in a SPARK program others than as a
         --! root type from which
         --! other types can be derived and extended. For this reason, null records
         --! are only allowed
         --! if they are tagged.

      when 835 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not an extended tagged record type");
         --! An extension aggregate is only appropriate if the record type it is
         --! defining is
         --! an extended record.  A normal aggregate should be used for other record
         --! (and array)
         --! types.

      when 836 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "This expression does not represent a valid ancestor type of the aggregate ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! The expression before the reserved word &quot;with&quot; must be of an ancestor type
         --! of the overall aggregate
         --! type.  In SPARK, the ancestor expression may not be a subtype mark.

      when 837 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Invalid record aggregate: there is a private ancestor between the " &
              "type of this expression " &
              "and the type of the aggregate ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! The ancestor type can be an tagged type with a private extension;
         --! however, there must be no private
         --! extensions between the ancestor type and the type of the aggregate.

      when 838 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Incomplete aggregate: null record cannot be used here because fields in ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " require values");
         --! The aggregate form &quot;with null record&quot; can only be used if the type of
         --! the aggregate is a null record
         --! extension of the ancestor type.  If any fields are added between the
         --! ancestor type and the aggregate type then
         --! values need to be supplied for them so &quot;null record&quot; is inappropriate.

      when 839 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "This package already contains a root tagged type or tagged" &
              " type extension. " &
              "Only one such declaration per package is permitted");
         --! SPARK permits one root tagged type or one tagged type extension per
         --! package, but not both.  This rule prevents
         --! the declaration of illegal operations with more than one controlling
         --! parameter.

      when 840 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "A tagged or extended type may not appear here. " &
              "SPARK does not permit the declaration of primitive functions " &
              "with controlling results");
         --! A primitive function controlled by its return result would be almost
         --! unusable in SPARK because a data
         --! flow error would occur wherever it was used.

      when 841 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The return type in the declaration of this function contained an error. " &
              "It is not possible to check the validity of this return type");
         --! Issued when there is an error in the return type on a function's
         --! initial declaration.  In this situation
         --! we cannot be sure what return type is expected in the function's body.
         --! It would be misleading to simply
         --! report a type mismatch since the types might match perfectly and both
         --! be wrong.  Instead, the Examiner reports
         --! the above error and refuses to analyse the function body until its
         --! specification is corrected.

      when 842 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Pragma Atomic_Components is not permitted in SPARK when the Ravenscar profile is selected");

      when 843 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Pragma Volatile_Components is not permitted in SPARK when the Ravenscar profile is selected");

      when 844 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Missing or contradictory overriding_indicator for operation ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => ". This operation successfully overrides its parent operation");
         --! In SPARK2005, an operation which successfully overrides a parent operation
         --! must be specified as Overriding.
      when 845 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Subprogram ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " does not successfully override a parent operation");
         --! In SPARK2005, an overriding operation must successfully override an
         --! operation inherited from the parent.
      when 850 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "This construct may only be used when the Ravenscar profile is selected");
         --! Support for concurrent features of the SPARK language, including
         --! protected objects,
         --! tasking, etc. are only supported when the Ravenscar profile is selected.

      when 851 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The parameter to pragma Atomic must be a simple_name");
         --! The parameter to pragma Atomic must be a simple_name; and may not be
         --! passed using
         --! a named association

      when 852 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "pragma Atomic may only appear in the same immediate scope as the type to which it applies");
         --! This is an Ada rule (pragma Atomic takes a local name see LRM 13.1(1)).
         --! Note that this precludes the use of pragma Atomic on a predefined type

      when 853 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "pragma Atomic may only apply to a scalar base type, or to a non-tagged " &
              "record type with exactly 1 field that is a predefined scalar type");
         --! pragma Atomic may only be applied to base types that are scalar. (i.e.
         --! enumeration types, integer types, real types, modular types) or a non-tagged
         --! record type with a single field which is a predefined scalar type, such
         --! as Integer, Character or Boolean. As an additional special case, a
         --! record type with a single field of type System.Address is also allowed.

      when 854 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "pragma Atomic takes exactly one parameter");

      when 855 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The type of own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not consistent with its modifier");
         --! An own variable with a task modifier must be of a task type. A task own
         --! variable must have the task modifier. An own variable with a protected
         --! modifier must be a protected object, suspension object or pragma
         --! atomic type. A protected or suspension object own variable must
         --! have the protected modifier.

      when 858 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "A variable that appears in a protects property list may not appear in a refinement clause");
         --! A variable in a protects list is effectively protected and hence
         --! cannot be refined.

      when 859 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "A protected own variable may not appear in a refinement clause");
         --! Protected state cannot be refined or be used as refinement constituents

      when 860 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " appears in a protects list and hence must appear in the initializes clause");
         --! Protected state (including all refinement constituents) must be initialized.

      when 861 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Both abstract own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " and refinement constitutent ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " must have an Integrity property");
         --! If an abstract own variable has an Integrity property, then so must
         --! all its refinement constituents, and vice-versa.

      when 862 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Both abstract own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " and refinement constitutent ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " must have the same Integrity value");
         --! If both an abstract own variable and a refinement constituent have
         --! Integrity properties specified, then the value of the Integrity
         --! must be the same.

      when 863 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is protected and may not appear in an initializes clause");
         --! Protected own variables must always be initialized, and should not appear in
         --! initializes annotations.

      when 864 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Unexpected initialization specification - ");
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "all own variables of this package ");
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "are either implicitly initialized, or do not require initialization");
         --! An own variable initialization clause and that of its refinement
         --!  constituents must be
         --! consistent.

      when 865 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Field ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " is part of the ancestor part of this aggregate and does not require a value here");
         --! An extension aggregate must supply values for all fields that are part of
         --! the overall aggregate
         --! type but not those which are part of the ancestor part.

      when 866 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The expression in a delay_until statement must be of type Ada.Real_Time.Time");
         --! When the Ravenscar Profile is selected, the delay until statement may be
         --! used.
         --! The argument of this statement must be of type Ada.Real_Time.Time.

      when 867 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Subprogram ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " contains a delay statement but does not have a delay property");
         --! Any subprogram that may call delay until must have a delay property in a
         --! declare annotation.  Your subprogram is directly or indirectly making a
         --! call to delay until

      when 868 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Protected object ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " may only be declared immediately within a library package");
         --! This error message is issued if a type mark representing a protected type
         --! appears anywhere other than in
         --! a library level variable declaration or library-level own variable type
         --! announcement.

      when 869 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Protected type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " already contains an Entry declaration; only one Entry is permitted");
         --! The Ravenscar profile prohibits a protected type from declaring more than
         --! one entry.

      when 870 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Protected type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " does not have any operations, at least one operation must be declared");
         --! A protected type which provides no operations can never be used so SPARK
         --! requires the
         --! declaration of at least one.

      when 871 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "A type can only be explicitly derived from a predefined Integer or Floating Point type or" &
              " from a tagged record type");

      when 872 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " is not protected; only protected items may be globally accessed by protected operations");
         --! In order to avoid the possibility of shared data corruption,
         --! SPARK prohibits protected operations
         --! from accessing unprotected data items.

      when 873 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "This subprogram requires a global annotation which references the protected type name ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! In order to statically-detect certain bounded errors defined by the
         --! Ravenscar profile, SPARK
         --! requires every visible operation of protected type to globally
         --! reference the abstract state of
         --! the type.

      when 874 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Protected state ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " must be initialized at declaration");
         --! Because there is no guarantee that a concurrent thread that initializes a
         --! protected object will be executed before one that reads it, the only way
         --! we can be sure that a protected object is properly initialized is to do
         --! so at the point of declaration. You have either declared some protected
         --! state and not included an initialization or you have tried to initialize
         --! some protected state in package body elaboration.

      when 875 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Protected type expected; access discriminants may only refer to protected types in SPARK");
         --! Access discriminants have been allowed in SPARK solely to allow devices
         --! made up of co-operating
         --! Ravenscar-compliant units to be constructed.  For this reason only
         --! protected types may appear in
         --! access discriminants.

      when 876 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "This protected type or task declaration must include " &
              "either a pragma Priority or " &
              "pragma Interrupt_Priority");
         --! To allow the static detection of certain bounded errors defined by the
         --! Ravenscar profile, SPARK requires
         --! an explicitly-set priority for each protected type, task type or object
         --! of those types.  The System.Default_Priority
         --! may used explicitly provided package System has been defined in the
         --! configuration file.

      when 877 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Priority values require an argument which is an expression of type integer");

      when 878 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "This protected type declaration contains a pragma Attach_Handler " &
              "and must therefore also " &
              "include a pragma Interrupt_Priority");
         --! To allow the static detection of certain bounded errors defined by the
         --! Ravenscar profile, SPARK requires
         --! an explicitly-set priority for each protected type or object.
         --! The System.Default_Priority may used explicitly
         --! provided package System has been defined in the configuration file.

      when 879 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Unexpected pragma ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => ": this pragma may not appear here");
         --! pragma Interrupt_Priority must be the first item in
         --! a protected type declaration or task type declaration; pragma Priority
         --! must be the first item in
         --! a protected type declaration, task type declaration or the main program.

      when 880 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Pragma Priority or Interrupt_Priority expected here");
         --! Issued when a pragma other than Priority or Interrupt_Priority appears as
         --! the first item in
         --! a protected type or task type declaration.

      when 881 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The priority of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " must be in the range ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         --! See LRM D.1(17).

      when 882 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Integrity property requires an argument which is an expression of type Natural");

      when 883 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Pragma Interrupt_Handler may not be used; SPARK does not support" &
              " the dynamic attachment of interrupt handlers [LRM C3.1(9)]");
         --! Interrupt_Handler is of no use unless dynamic attachment of interrupt
         --! handlers is to be used.

      when 884 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Pragma Attach_Handler is only permitted immediately after the " &
              "corresponding protected procedure declaration in a protected type " &
              "declaration");

         --! Pragma Attach_Handler may only be used within a protected type
         --! declaration.  Furthermore, it must
         --! immediately follow a protected procedure declaration with the same name
         --! as the first argument to the pragma.

      when 885 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Pragma Attach_Handler may only be applied to a procedure with no parameters");
         --! See LRM C.3.1(5).

      when 887 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "A discriminant may only appear alone, not in an expression");
         --! Issued when a task or protected type priority is set using an expression
         --! involving a discriminant. The use
         --! of such an expression greatly complicates the static evaluation of the
         --! priority of task or protected subtypes
         --! thus preventing the static elimination of certain Ravenscar bounded errors.

      when 888 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Unexpected Delay, ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " already has a Delay property");
         --! A procedure may only have a maximum of one delay annotation.

      when 889 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " must have the suspendable property");
         --! The type used to declare this object must be a protected type with and
         --! entry or a suspension object type

      when 890 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The name ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " already appears in the suspends list");
         --! Items may not appear more than once in an a suspends list.

      when 891 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Task type or protected type required");
         --! Issued in a subtype declaration where the constraint is a discriminant
         --! constraint.  Only task and protected types
         --! may take a discriminant constraint as part of a subtype declaration.

      when 892 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Array type, task type or protected type required");
         --! Issued in a subtype declaration where the constraint is a either a
         --! discriminant constraint or an index
         --! constraint (these two forms cannot always be distinguished
         --!  syntactically).  Only task and protected types
         --! may take a discriminant constraint and only array types may
         --! take an index constraint as part of a
         --! subtype declaration.

      when 893 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Number of discriminant constraints differs from number of known discriminants of type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! Issued in a subtype declaration if too many or two few discriminant
         --! constraints are supplied.

      when 894 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Only variables of a protected type may be aliased");
         --! SPARK supports the keyword aliased in variable declarations only so
         --! that protected and task types can support
         --! access discriminants.  Since it has no other purpose it may not be
         --! used except in a protected object declaration.

      when 895 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Attribute Access may only be applied to variables which are declared as aliased, variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not aliased");
         --! This is a slightly annoying Ada issue.  Marking a variable as aliased
         --! prevents it being placed in a register
         --! which would make pointing at it hazardous; however, SPARK only permits
         --! 'Access on protected types which are
         --! limited and therefore always passed by reference anyway and immune from
         --! register optimization.   Requiring
         --! aliased on protected objects that will appear in discriminant
         --! constraints is therefore unwanted syntactic sugar
         --! only.

      when 896 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The task type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " does not have an associated body");
         --! Issued at the end of a package body if a task type declared in its
         --!  specification contains neither a body
         --! nor a body stub for it.

      when 897 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The protected type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " does not have an associated body");
         --! Issued at the end of a package body if a protected type declared in
         --! its specification contains neither a body
         --! nor a body stub for it.

      when 898 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not a protected or task type which requires a body");
         --! Issued if a body or body stub for a task or protected type is
         --! encountered and there is no matching specification.

      when 899 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "A body for type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has already been declared");
         --! Issued if a body or body stub for a task or protected type is
         --! encountered and an earlier body
         --! has already been encountered.

      when 901 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Suspension object ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " may only be declared immediately within a library package specification or body");
         --! Suspension objects must be declared at library level. They cannot
         --!  be used in protected type state or as local variables in subprograms.

      when 902 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Recursive use of typemark ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " in known descriminant");

      when 903 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Protected or suspension object types cannot be used to declare constants");
         --! Protected and suspension objects are used to ensure integrity of shared
         --! objects. If it is necessary to share constant data then these constructs
         --! should not be used.

      when 904 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Protected or suspension objects cannot be used as subprogram parameters");
         --! SPARK does not currently support this feature.

      when 905 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Protected or suspension objects cannot be returned from functions");
         --! SPARK does not currently support this feature.

      when 906 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Protected or suspension objects cannot be used in composite types");
         --! Protected and suspension objects cannot be used in record or array
         --! structures.

      when 907 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Delay until must be called from a task or unprotected procedure body");
         --! You are calling delay until from an invalid construct.
         --! Any construct that calls delay until must have a delay property in the
         --! declare annotation. This construct must be one of a task or procedure body
         --!

      when 908 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Blocking properties are not allowed in protected scope");
         --! Procedures in protected scope must not block and therefore blocking
         --! properties are prohibited

      when 909 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Object ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " cannot suspend");
         --! You are either applying the suspendable property to an own variable
         --! that cannot suspend or you have declared a variable (whose own variable
         --! has the suspendable property) which cannot suspend. Or you have used an
         --! item in a suspends list that does not have the suspendable property.
         --! An object can only suspend if it is a suspension object or a protected
         --! type with an entry.

      when 910 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Name ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " must appear in the suspends list property for the enclosing unit");
         --! Protected entry calls and calls to
         --! Ada.Synchronous_Task_Control.Suspend_Until_True may block the currently
         --! executing task. SPARK requires you announce this fact by placing the
         --! actual callee name in the suspends list for the enclosing unit.

      when 911 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The argument in pragma Priority for the main program must be an " &
              "integer literal " &
              "or a local constant of static integer value");
         --! If the main program priority is not an integer literal then you
         --! should declare a constant that has the required value in the
         --! declarative part of the main program prior to the position of the pragma.

      when 912 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "This call contains a delay property that is not propagated to the enclosing unit");
         --! The call being made has a declare annotation that contains a delay
         --! property. SPARK requires that this property is propagated up
         --! the call chain and hence must appear in a declare annotation
         --! for the enclosing unit.

      when 913 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "This call has a name in its suspends list which is not propagated to the enclosing unit");
         --! The call being made has a declare annotation that contains a suspends
         --! list. SPARK requires that the entire list is propagated up the call
         --! chain and hence must appear in a declare annotation for the enclosing unit.

      when 914 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The name ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " specified in the suspends list can never be called");
         --! You have specified the name of a protected or suspension object in
         --! the suspends list that can never be called by this procedure or task.

      when 915 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Procedure ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has a delay property but cannot delay");
         --! You have specified a delay property for this procedure but delay
         --! until can never be called from it.

      when 916 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " Protected object ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has a circular dependency in subprogram ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         --! The type of the protected object mentions the protected object name in
         --! the derives list for the given subprogram

      when 917 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Procedure ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " cannot be called from a protected action");
         --! The procedure being called may block and hence cannot be called from
         --! a protected action.

      when 918 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The delay property is not allowed for ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! The delay property may only be applied to a procedure

      when 919 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The priority property is not allowed for ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! The priority property can only be applied to protected own variables
         --! which are type announced. If the type has been declared it must be a
         --! protected type

      when 920 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The suspends property is not allowed for ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! The suspends property may only be applied to task type specifications
         --! and procedures

      when 921 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The identifier ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not recognised as a component of a property list");
         --! The property list can only specify the reserved word delay, suspends or
         --! priority.

      when 922 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " must have the priority property");
         --! In order to perform the ceiling priority checks the priority property must
         --! be given to all own variables of protected type.

      when 923 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The procedure ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " cannot be called from a function as it has a blocking side effect");
         --! Blocking is seen as a side effect and hence procedures that potentially
         --! block cannot be called from functions.

      when 924 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The suspendable property is not allowed for ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! Objects that suspend must be declared as own protected variables

      when 925 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The own variable or task ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " must have a type announcement");
         --! Own variables of protected type and own tasks must have a type announcement

      when 926 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Illegal declaration of task ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => ". Task objects must be declared at library level");
         --! Task objects must be declared in library level package specifications or
         --! bodies.

      when 927 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The own task annotation for this task is missing the name ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " in its suspends list");
         --! The task type declaration has name XXX in its list and this must appear
         --! in the own task annotation

      when 928 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Private elements are not allowed for protected type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! Protected type XXX has been used to declare a protected, moded own variable.
         --! Protected, moded own variables are refined onto a set of virtual elements with
         --! the same mode. As such private elements are not allowed.

      when 929 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Unexpected declare annotation. Procedure ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " should have the declare annotation on the specification");
         --! Declare annotations cannot appear on the procedure body if it appears
         --! on the procedure specification

      when 930 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Task ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " does not appear in the own task annotation for this package");
         --! A task has been declared that is not specified as an own task of the
         --! package.

      when 931 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Task ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " does not have a definition");
         --! A task name appears in the own task annotation for this package but
         --! is never declared

      when 932 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The priority for protected object ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " does not match that given in the own variable declaration");
         --! The priority given in the priority property must match that given in
         --! the protected type.

      when 933 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "A pragma Priority is required for the main program when Ravenscar Profile is enabled");
         --! When SPARK profile Ravenscar is selected, all tasks, protected objects
         --! and the main program must explicitly be assigned a priority.

      when 934 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Priority ceiling check failure: the priority of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is less than that of ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! The active priority of a task is the higher of its base priority
         --! and the ceiling priorities of all protected objects that it is
         --! executing. The active priority at the point of a call to a
         --! protected operation must not exceed the ceiling priority of the
         --! callee.

      when 935 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " must have the interrupt property");
         --! An own variable has been declared using a protected type with a pragma
         --! attach handler. Such objects are used in interrupt processing and must
         --! have the interrupt property specified in their own variable declaration

      when 936 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The interrupt property is not allowed for ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! The interrupt property can only be applied to protected own variables
         --! that are type announced. If the type is declared then it must be a
         --! protected type that contains an attach handler

      when 937 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The protects property is not allowed for ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! The protects property can only be applied to protected own variables
         --! that are type announced. If the type is declared then it must be a
         --! protected type.

      when 938 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The unprotected variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is shared by ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " and ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name3,
                      Scope => Err_Num.Scope);
         --! XXX is an unprotected variable that appears in the global list of the
         --! threads YYY and ZZZ. Unprotected variables cannot be shared between
         --! threads in SPARK. A thread is one of: the main program, a task, an
         --! interrupt handler.

      when 939 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The suspendable item ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is referenced by ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " and ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name3,
                      Scope => Err_Num.Scope);
         --! XXX is an own variable with the suspends property that appears in the
         --! suspends list of the threads YYY and ZZZ. SPARK prohibits this to
         --! prevent more than one thread being suspended on the same item at any
         --! one time. A thread is one of: the main program, a task, an interrupt
         --!  handler.

      when 940 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " is a protected own variable. Protected variables may not be used in proof contexts");
         --! The use of protected variables in pre and postconditions or other proof annotations is not (currently)
         --! supported.  Protected variables are volatile because they can be changed at any time by another program
         --! thread and this may invalidate some common proof techniques.  The prohibition of protected variables
         --! does not prevent proof of absence of run-time errors nor proof of protected operation bodies.  See the
         --! manual &quot;SPARK Proof Manual&quot; for more details.

      when 941 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The type of own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " must be local to this package");
         --! The type used to an announce an own variable with a protects property
         --! must be declared in the same package.

      when 942 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Only one instance of the type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is allowed");
         --! Type XXX has a protects property. This means there can be only one object
         --! in the package that has this type or any subtype of this type.

      when 943 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The name ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " cannot appear in a protects list");
         --! All items in a protects list must be unprotected own variables owned by
         --! this package

      when 944 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The name ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is already protected by ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         --! The name XXX appears in more than one protects list. The first time
         --! it appeared was for own variable YYY. XXX should appear in at most
         --! one protects list.

      when 945 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The property ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " must be given a static expression for its value");
         --! This property can only accept a static expression.

      when 946 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " must only ever be accessed from operations in protected type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         --! The own variable XXX is protected by the protected type YYY and hence
         --! must never be accessed from anywhere else.

      when 947 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " appears in a protects list for type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " but is not used in the body");
         --! The protected type YYY claims to protect XXX via a protects property.
         --! However, the variable XXX is not used by any operation in YYY.

      when 948 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The type of own variable or task ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " must be a base type");
         --! Own tasks and protected own variables of a protected type must be
         --! announced using the base type. The
         --! subsequent variable declaration may be a subtype of the base type

      when 949 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Unexpected partition annotation: a global annotation may only " &
              "appear here when " &
              "the Ravenscar profile is selected");
         --! When the sequential SPARK profile is selected, the global and derives
         --! annotation on the main program describes
         --! the entire program's behaviour.  No additional, partition annotation
         --! is required or permitted.  Note that an
         --! annotation must appear here if the Ravenscar profile is selected.

      when 950 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Partition annotation expected: a global and, optionally, a " &
              "derives annotation must " &
              "appear after 'main_program' when the Ravenscar profile is selected");
         --! When the Ravenscar profile is selected the global and derives annotation
         --! on the main program describes the
         --! behaviour of the environment task only, not the entire program.
         --! An additional annotation, called the
         --! partition annotation, is required to describe the entire program's
         --! behaviour; this annotation follows
         --! immediately after 'main_program;'

      when 951 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Inherited package ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => " contains tasks and/or interrupt handlers and must therefore appear in the preceding WITH clause");
         --! In order to ensure that a Ravenscar program is complete, SPARK requires
         --! that all 'active' packages
         --! inherited by the environment task also appear in a corresponding
         --! with clause.  This check ensures that
         --! any program entities described in the partition annotation are also
         --! linked into the program itself.

      when 952 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Subprogram ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is an interrupt handler and cannot be called");
         --! Interrupt handler operations cannot be called.

         --cfr1203 --error message only needed if we eliminate suspension objects from partition-level annos
         --cfr1203 when 985 =>
         --cfr1203    AppendName
         --cfr1203      (E_Str, Err_Num.Name1, Err_Num.Scope);
         --cfr1203    ELStrings.Append_String
         --cfr1203      (E_Str, " is a predefined suspension object which must not
         --cfr1203  appear in the partition global annotation");

      when 953 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Interrupt property error for own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => ". ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not an interrupt handler in type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name3,
                      Scope => Err_Num.Scope);
         --! The handler names in an interrupt property must match one in the
         --! protected type of the own variable.

      when 954 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Interrupt property error for own variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => ". Interrupt stream name ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name2,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is illegal");
         --! The stream name must be unprefixed and not already in use within the
         --! scope of the package.

      when 955 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " can only appear in the partition wide flow annotation");
         --! Interrupt stream variables are used only to enhance the partition
         --! wide flow annotation and must not be used elsewhere.

      when 956 =>
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " already appears in as an interrupt handler in the interrupt mappings");
         --! An interrupt handler can be mapped onto exactly one interrupt stream
         --! variable. An interrupt stream variable may be mapped onto many interrupt
         --!  handlers.

      when 957 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Consecutive updates of protected variable ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " are disallowed when they do not depend directly on its preceding value");
         --! A protected variable cannot be updated without direct reference to its
         --! preceding value more than once within a subprogram or task.
         --! Each update of a protected variable may have a wider effect than
         --! just the change of value of the protected variable. The overall
         --! change is considered to be the accumulation of all updates and
         --! reads of the protected variable and to preseve this information flow
         --! successive updates must directly depend on the preceding value of
         --! the variable

      when 958 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "A task may not import the unprotected state ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! A task may not import unprotected state unless it is mode in.
         --! This is because under the concurrent elaboration policy, the task cannot
         --! rely on the state being initialized before it is run.

      when 959 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Unprotected state ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is exported by a task and hence must not appear in an initializes clause");
         --! Own variable XXX is being accessed by a task. The order in which the task
         --! is run and the own variable initialized is non-deterministic under a
         --! concurrent elaboration policy. In this case SPARK forces the task to
         --! perform the initialization and as such the own variable must not appear
         --! in an initializes clause.

      when 960 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "The function Ada.Real_Time.Clock " &
              "can only be used directly (1) in an assignment or return statement or (2) " &
              "to initialize a library a level constant");
         --! <li>To avoid ordering effects, functions which globally access own
         --! variables which have modes (indicating that they are connected to
         --! the external environment) may only appear directly in assignment or
         --! return statements. They may not appear as actual parameters or in any
         --! other form of expression.
         --! <li>SPARK relaxes the illegal use of function calls in elaboration code in
         --! the case of the function Ada.Real_Time.Clock. However the function can only
         --! be used to directly initialize a constant value.

      when 961 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "This property value is of an incorrect format");
         --! Please check the user manual for valid property value formats.

      when 962 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Error(s) detected by VC Generator. See the .vcg file for more information");
         --! This message is echoed to the screen if an unrecoverable
         --! error occurs which
         --! makes the generation of VCs for the current subprogram impossible.
         --! Another message more precisely identifying the problem will be placed in
         --! the .vcg file.

      when 986 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "A protected function may not call a locally-declared protected procedure");
         --! See LRM 9.5.1 (2). A protected function has read access to the
         --! protected elements of the type whereas
         --! the called procedure has read-write access. There is no way in which
         --! an Ada compiler can determine whether
         --! the procedure will illegally update the protected state or not so the
         --! call is prohibited by the rules of Ada.
         --! (Of course, in SPARK, we know there is no function side effect but the
         --! rules of Ada must prevail nonetheless).

      when 987 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Task types and protected types may only be declared in package specifications");
         --! The Examiner performs certain important checks at the whole program level
         --! such as detection of illegal sharing of
         --! unprotected state and partition-level information flow analysis.
         --! These checks require visibility of task
         --! types and protected types (especially those containing interrupt
         --! handlers).  SPARK therefore requires these
         --! types to be declare in package specifications.  Subtypes and objects
         --! of task types, protected types and their
         --! subtypes may be declared in package bodies.

      when 988 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Illegal re-use of identifier ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "; this identifier is used in a directly visible protected type");
         --! SPARK does not allow the re-use of operation names which are already
         --! in use in a directly visible
         --! protected type.  The restriction is necessary to avoid overload resolution
         --! issues in the protected
         --! body.  For example, type PT in package P declares operation K.
         --! Package P also declares an operation K.
         --! From inside the body of PT, a call to K could refer to either of the
         --! two Ks since both are directly visible.

      when 989 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The last statement of a task body must be a plain loop with no exits");
         --! To prevent any possibility of a task terminating (which can lead to a
         --! bounded error), SPARK requires
         --! each task to end with a non-terminating loop.  The environment task (or
         --! &quot;main program&quot;) does not need
         --! to end in a plain loop provided the program closure includes at least
         --! one other task.  If there are
         --! no other tasks, then the environment task must be made non-terminating
         --! with a plain loop.

      when 990 =>
         E_Strings.Append_String
           (E_Str => E_Str,
            Str   => "Unexpected annotation, a task body may have only global and derives annotations");
         --! Issued if a pre, post or declare annotation is attached to a task body.

      when 991 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Unexpected task body, ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not the name of a task declared in this package specification");
         --! Issued if task body is encountered for which there is no preceding
         --! declaration.

      when 992 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "A body for task type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has already been declared");
         --! Issued if a duplicate body or body stub is encountered for a task.

      when 993 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "There is no protected type declaration for ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! Issued if a body is found for a protected types for which there is
         --! no preceding declaration.

      when 994 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Invalid guard, ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not a Boolean protected element of this protected type");
         --! The SPARK Ravenscar rules require a simple Boolean guard which must
         --! be one of the protected elements
         --! of the type declaring the entry.

      when 995 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Unexpected entry body, ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " is not the name of an entry declared in this protected type");
         --! Local entries are not permitted so a protected body can declare at
         --! most one entry body and that must have
         --! declared in the protected type specification.

      when 996 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The protected operation ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => ", declared in this type, does not have an associated body");
         --! Each exported protected operation must have a matching implementation
         --! in the associated protected body.

      when 997 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "A body for protected type ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => " has already been declared");
         --! Each protected type declaration must have exactly one matching
         --! protected body or body stub.

      when 998 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "There is no protected type declaration for ");
         Append_Name (E_Str => E_Str,
                      Name  => Err_Num.Name1,
                      Scope => Err_Num.Scope);
         --! Issued if a protected body or body stub is found and there is no
         --! matching declaration for it.

      when 999 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "This feature of Generics is not yet implemented");
         --! Generics are currently limited to instantiation of Unchecked_Conversion.

         -- The following semantic error numbers are unused.  When adding a new
         -- semantic error, pick one of these, add a case for it above and remove
         -- it from this case.
      when 0          |
        146 .. 147 |
        175        |
        177 .. 179 |
        183 .. 189 |
        192 .. 199 |
        202 .. 249 |
        256 .. 272 |
        274 .. 299 |
        344 .. 398 |
        404 .. 405 |
        424        |
        426 .. 499 |
        510 .. 549 |
        553        |
        556 .. 599 |
        612        |
        625 .. 629 |
        661        |
        663 .. 699 |
        726 .. 729 |
        731 .. 749 |
        758 .. 769 |
        802        |
        806 .. 813 |
        816 .. 819 |
        846 .. 849 |
        856 .. 857 |
        886        |
        900        |
        963 .. 985 =>
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "UNKNOWN ERROR NUMBER PASSED TO SemanticErr");
   end case;

   AppendReference (E_Str, Err_Num.Reference);
   Append_Explanation;
   E_Strings.Append_String (E_Str => E_Str,
                            Str   => ".");
end SemanticErr;
