<?php
/**
 * Forwards_Driver:: defines an API for implementing forwarding backends
 * for Forwards.
 *
 * $Horde: forwards/lib/Driver.php,v 1.29 2006/02/17 16:51:01 jan Exp $
 *
 * Copyright 2001-2006 Mike Cochrane <mike@graftonhall.co.nz>
 *
 * See the enclosed file LICENSE for license information (ASL). If you
 * did not receive this file, see http://www.horde.org/licenses/asl.php.
 *
 * @author  Mike Cochrane <mike@graftonhall.co.nz>
 * @since   Forwards 2.1
 * @package Forwards
 */
class Forwards_Driver {

    /**
     * Hash containing configuration data.
     *
     * @var array
     */
    var $_params;

    /**
     * The error string returned to the user if an error occurs.
     *
     * @var string
     */
    var $_error = '';

    /**
     * Constructs a new Forwards_Driver object.
     *
     * @param array $params  A hash containing connection parameters.
     */
    function Forwards_Driver($params = null)
    {
        if (is_null($params)) {
            $params = Horde::getDriverConfig('server', null);
        }
        $this->_params = $params;
    }

    /**
     * Gets the error message.
     *
     * @return string  The error message.
     */
    function getError()
    {
        return $this->_error;
    }

    /**
     * Begins forwarding of mail for a user.
     *
     * @param string $user        The username of the user.
     * @param string $realm       The realm of the user.
     * @param string $password    The password of the user.
     * @param string $target      The email address that mail should be
     *                            forwarded to.
     * @param boolean $keeplocal  Keep a copy of forwarded mail in the local
     *                            mailbox.
     *
     * @return boolean  True on success, false otherwise.
     */
    function enableForwarding($user, $realm = 'default', $password = '',
                              $target, $keeplocal = false)
    {
        return false;
    }

    /**
     * Stops forwarding of mail for a user.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return boolean  True on success, false otherwise.
     */
    function disableForwarding($user, $realm = 'default', $password = '')
    {
        return false;
    }

    /**
     * Retrieves current state of mail redirection for a user.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return boolean  True if forwarding is enabled, false otherwise.
     */
    function isEnabledForwarding($user, $realm = 'default', $password)
    {
        return false;
    }

    /**
     * Checks if user is keeping a local copy of forwarded mail.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return boolean  True if user is keeping a local copy of mail,
     *                  otherwise false.
     */
    function isKeepLocal($user, $realm = 'default', $password)
    {
        return false;
    }

    /**
     * Retrieves current target of mail redirection for a user.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return string  The current forwarding mail address, or false on error.
     */
    function currentTarget($user, $realm = 'default', $password)
    {
        return false;
    }

    /**
     * Parse an email address list and return it in a known standard form.
     * This will attempt to add the domain (realm) to unqualified addresses
     * if the realm is non-blank and not 'default'.
     *
     * @param string $user   The email address.
     * @param string $realm  The domain/realm to add if none is present.
     *
     * @return string  The email address(es), or false on error.
     */
    function _make_email_address($user, $realm)
    {
        $domain = ($realm != 'default') ? $realm : '';
        $email = '';

        require_once 'Mail/RFC822.php';
        $parser = &new Mail_RFC822();
        $parsed_email = $parser->parseAddressList($user, $domain, false);
        if (is_array($parsed_email) && count($parsed_email) > 0) {
            for ($i=0; $i < count($parsed_email); $i++) {
               $email .= !empty($email) ? ',' : '';
               if (is_object($parsed_email[$i])) {
                 $email .= $parsed_email[$i]->mailbox;
                 $email .= !empty($parsed_email[$i]->host)
                        ? '@' . $parsed_email[$i]->host
                        : '';
              } else {
                 $email .= $parsed_email[$i];
              }
            }
        } else {
            $this->_error = _("Can't parse your email address");
            $email = false;
        }

        return $email;
    }

    /**
     * Gets a concrete Forwards_Driver instance.
     *
     * @param string $driver  The type of concrete Forwards_Driver subclass to
     *                        return.  The code for the driver is dynamically
     *                        included.
     * @param array $params   A hash containing any additional configuration or
     *                        connection parameters a subclass might need.
     *
     * @return Forwards_Driver  The newly created concrete Forwards_Driver
     *                          instance, or false on error.
     */
    function &factory($driver = null, $params = null)
    {
        if (is_null($driver)) {
            $driver = $GLOBALS['conf']['server']['driver'];
        }

        $driver = basename($driver);

        include_once dirname(__FILE__) . '/Driver/' . $driver . '.php';
        $class = 'Forwards_Driver_' . $driver;
        if (class_exists($class)) {
            $forwards = &new $class($params);
        } else {
            $forwards = false;
        }

        return $forwards;
    }

    /**
     * Gets a reference to a concrete Forwards_Driver instance.
     *
     * It will only create a new instance if no Forwards_Driver instance with
     * the same parameters currently exists.
     *
     * This should be used if multiple storage sources are required.
     *
     * This method must be invoked as follows:<pre>
     *   $var = &Forwards_Driver::singleton();</pre>
     *
     * @param string $driver  The type of concrete Forwards_Driver subclass to
     *                        The code for the driver is dynamically included.
     * @param array $params   A hash containing any additional configuration or
     *                        connection parameters a a subclass might need.
     *
     * @return Forwards_Driver  The created concrete Forwards_Driver instance,
     *                          or false on error.
     */
    function &singleton($driver = null, $params = null)
    {
        static $instances;

        if (is_null($driver)) {
            $driver = $GLOBALS['conf']['server']['driver'];
        }

        if (is_null($params)) {
            $params = Horde::getDriverConfig('server', $driver);
        }

        if (!isset($instances)) {
            $instances = array();
        }

        $signature = serialize(array($driver, $params));
        if (!isset($instances[$signature])) {
            $instances[$signature] = &Forwards_Driver::factory($driver, $params);
        }

        return $instances[$signature];
    }

}
