/* SNAC

 * Copyright (C) 1999 the Free Software Foundation

 * Authors : Matias Mutchinick, Jan Struyf        

 *         

 * This program is free software; you can redistribute it and/or modify

 * it under the terms of the GNU General Public License as published by

 * the Free Software Foundation; either version 2 of the License, or

 * (at your option) any later version.

 *

 * This program is distributed in the hope that it will be useful,

 * but WITHOUT ANY WARRANTY; without even the implied warranty of

 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the

 * GNU General Public License for more details.

 *

 * You should have received a copy of the GNU General Public License

 * along with this program; if not, write to the Free Software

 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */







#ifndef __EVALSTRING_H__

#define __EVALSTRING_H__



#include <glib.h>



typedef struct __Funtable Funtable;

struct __Funtable {

	gchar      *name;

	gdouble   (*func)();

}; 



typedef struct __Optable Optable;

struct __Optable {

	gchar      *name;

	gint        prec;

	gint        id;

	gboolean    need_sep;

	gdouble   (*func)();

}; 



/*

 * keep_double_within_limits

 * @num   : a number

 * @error : pointer to an error state variable

 * Ensure that @num is within the limits of calculation

 * max = 1e300 -> oveflow, min = 1e-300 -> zero.

 */

gdouble

keep_double_within_limits(gdouble  num,

			  gint    *error);



/* str_s_f_cmp

 * @str       : a string

 * @ptr_str   : a string

 * @ptr_start : the start of a substring in @ptr_str

 * @ptr_end   : the end of a substring in @ptr_str

 * Compare the whole string @str with the substring in @ptr_str 

 * limited by @ptr_start and @ptr_end.

 * The return value follows the rules of (strcmp). 

 */

gint 

str_s_f_cmp(gchar      *str,

	    gchar      *ptr_str,

	    gint        ptr_start, 

	    gint        ptr_end);





/*

 * round_double_fix

 * @num    : a number

 * @digits : number of digits

 * round @num for fixed notation at @digit digits.

 */

gdouble 

round_double_fix(gdouble   num,

		 gint      digits);





/*

 * round_double_std

 * @num    : a number

 * @digits : number of digits

 * round @num the standard way at @digit digits.

 */

gdouble

round_double_std(gdouble  num,

		 gint     digits);



/*

 * round_double

 * @num    : a number

 * @format : the format for rounding the number

 * @digits : number of digits

 * rounds a number for some specific needs (number of digits and

 * formating).

 */

gdouble

round_double(gdouble    num,

	     gint       format,

	     gint       digits);





/*

 * rel_str_to_double

 * @expr      : string representation of a math expression (or number)

 * @num_start : where in @expr is the first character of the number

 * @num_end   : where in @expr is the end of the number

 * @error     : pointer to error state variable

 * Return the numeric result of a string reperesentation of a number.

 * Checks for errors, allowed syntax is ...

 * [-/+][n][.]nnnn[e[+/-]nn]

 */

gdouble 

rel_str_to_double(gchar     *expr,

		  gint       num_start, 

		  gint       num_end,

		  gint      *error);





/* 

 * get_function

 * @str   : string of a math expression

 * @start : the start of the function in the string

 * @end   : the end of the function in the string

 * @error : pointer to an error state variable

 * Return a pointer to a Funtable of the function requested in @str

 * between @start and @end.

 */

Funtable *

get_function(gchar     *str,

	     gint       start, 

	     gint       end,

	     gint      *error);





/*

 * eval_function

 */

gdouble

eval_function(Funtable     *ftable,

	      gdouble       arg,

	      gint         *error);





/*

 * operate

 * @num1  : a number

 * @num2  : a number

 * @oper  : character representation of an oper

 * @error : pointer to an error state indicator

 * return the result of the operation that @oper represents between 

 * @num1 and @num2. If an error ocurrs then set @error.

 */

gdouble 

operate(Optable      *otable,

        gdouble       num1, 

	gdouble       num2, 

	gint         *error);



/*

 * prec

 * @oper : character representation of an oper

 * return the precedence of the operandus.

 */

gint 

prec(gchar oper);





/*

 * pick_as_oper

 * @oper : character representation of an oper

 * return 1 if the character is an operandus.

 */

gint 

pick_as_oper(gchar oper);







/*

 * set_ag_error

 * @error  : type of the error

 * @start  : start of the argument tha causes the error in the string

 * @end    : end of the argument tha causes the error in the string

 * Set the error state for the calculator, where starts, and where ends.

 */

gint

set_ag_error(gint     error,

	     gint     start, 

	     gint     end);







/******************************************************************************/

/*                                                                            */

/* eval_expression                                                            */

/* @expr      : string representation of a mathematical expression            */

/* @str_start : the begining of the expression in the string                  */

/* @str_end   : the end of the expression in the string                       */

/*                                                                            */

/******************************************************************************/



gdouble 

eval_expression(gchar    *expr,

		gint      str_start,

		gint      str_end);





/*

 * eval_string

 * @expr   : character string representing a mathematical expression

 * @format : format (FIX, STD, ENG, SCI) to round for

 * @size   : number of digits to round at (1 ... 12)

 * @angle  : angle messure to use (RAD, DEG, GRAD)

 * Given a string reperesenting a math expression, tis function returns

 * a numeric value of the result, rounded specially for @format and @size

 * and in @angle messure

 */

gdouble 

eval_string(gchar      *expr,

	    gint        format,

	    gint        size,

	    gint        angle);





#endif /* __EVALSTRING_H__ */





