/*==================================================================
 * sfdofunc.c - Sound font undo/redo item functions
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/

#include <stdio.h>
#include <gtk/gtk.h>
#include "i18n.h"
#include "sfdofunc.h"
#include "sfont.h"
#include "sfundo.h"
#include "uif_sfont.h"
#include "uif_sftree.h"
#include "util.h"

static gint dofunc_noitem_restore (SFDoItem *item);
static gint dofunc_noitem_restate (SFDoItem *item, SFDoItem *newitem);
static gpointer dofunc_noitem_save_state (SFItemID itemid);

static gint dofunc_sfitem_restore (SFDoItem *item);
static gint dofunc_sfitem_restate (SFDoItem *refitem, SFDoItem *newitem);
static void dofunc_sfitem_free (SFDoItem *item);
static gpointer dofunc_sfitem_save_state (SFItemID itemid);

static gint dofunc_sfinfo_restore (SFDoItem *item);
static gint dofunc_sfinfo_restate (SFDoItem *item, SFDoItem *newitem);
static void dofunc_sfinfo_free (SFDoItem *item);
static gpointer dofunc_sfinfo_save_state (SFItemID sfitemid, guint8 id);

SFDoFuncInfo sfdofuncs[] = {
  { NULL, NULL, NULL},      /* DOFUNC_INVALID */
    { N_("Pre sound font item"), dofunc_noitem_restore,
      dofunc_noitem_restate, NULL },
    { N_("Sound font item"), dofunc_sfitem_restore,
      dofunc_sfitem_restate, dofunc_sfitem_free },
  { N_("Sound font info string"), dofunc_sfinfo_restore, dofunc_sfinfo_restate,
    dofunc_sfinfo_free },
};


/* ------------------------------------------------ */
/* created item save state (state of non-existence) */
/* ------------------------------------------------ */

void
dofunc_noitem_save (SFItemID itemid)
{
  gpointer st;

  SFUNDO_ENTER ();

  st = dofunc_noitem_save_state (itemid);
  sfdo_add (DOFUNC_NOITEM, st);
}

static gint
dofunc_noitem_restore (SFDoItem *item)
{
  SFItemID itemid;

  /* fetch item id and remove the item (the original state) */
  itemid = GPOINTER_TO_INT (item->state);
  uisf_sfitem_delete (itemid);

  return (OK);
}

/* save redo state (SF item itself) */
static gint
dofunc_noitem_restate (SFDoItem *item, SFDoItem *newitem)
{
  newitem->type = DOFUNC_SFITEM;
  newitem->state = dofunc_sfitem_save_state (GPOINTER_TO_INT (item->state));

  return (OK);
}

static gpointer
dofunc_noitem_save_state (SFItemID itemid)
{
  /* just save the item id of the node */
  return (GINT_TO_POINTER (itemid));
}

/* ------------------------------------- */
/* sound font item state (save the item) */
/* ------------------------------------- */

typedef struct _DoState_SFItem
{
  SFNodeType type;  /* item type: NODE_(PRESET, INST, SAMPLE, PZONE, IZONE) */
  UISFont *uisf;		/* the item's sound font */
  gint pos;  /* position within items of type (every type except Preset) */
  SFItemID pitemid;  /* ID of Preset or Inst for P/IZONEs, else SF item ID */
  gpointer data;		/* pointer to item data */
}DoState_SFItem;

void
dofunc_sfitem_save (SFItemID itemid)
{
  DoState_SFItem *st;

  SFUNDO_ENTER ();

  st = dofunc_sfitem_save_state (itemid);
  sfdo_add (DOFUNC_SFITEM, st);
}

static gint
dofunc_sfitem_restore (SFDoItem *item)
{
  DoState_SFItem *st;
  gpointer data;

  st = item->state;

  switch (st->type)
    {
    case NODE_PRESET:
      /* duplicate preset and make item IDs into inst ptrs */
      data = sfont_preset_dup ((SFPreset *)(st->data), SFDUP_RESTORE);
      break;
    case NODE_INST:
      /* duplicate inst in restore mode */
      data = sfont_inst_dup ((SFInst *)(st->data), SFDUP_RESTORE);
      break;
    case NODE_SAMPLE:
      /* dup sample in restore mode (retain item ID) and dorefcount++ */
      data = sfont_sample_dup ((SFSample *)(st->data), SFDUP_RESTORE, FALSE);
      break;
    case NODE_PZONE:
      /* dup zone in restore mode */
      data = sfont_zone_dup ((SFZone *)(st->data), SFDUP_RESTORE);
      break;
    case NODE_IZONE:
      /* dup zone in restore mode */
      data = sfont_zone_dup ((SFZone *)(st->data), SFDUP_RESTORE);
      break;
    default:  /* shouldn't happen */
      return (OK);
    }

  uisf_sfitem_add (st->pitemid, data, st->pos, st->type);

  return (OK);
}

static gint
dofunc_sfitem_restate (SFDoItem *refitem, SFDoItem *newitem)
{
  DoState_SFItem *st;

  st = refitem->state;

  newitem->type = DOFUNC_NOITEM;
  /* SFItemID is first item of SFPreset, SFInst, SFSample and SFZone structs */
  newitem->state = dofunc_noitem_save_state (*((SFItemID *)(st->data)));

  return (OK);
}

static void
dofunc_sfitem_free (SFDoItem *item)
{
  DoState_SFItem *st;

  st = item->state;

  switch (st->type)
    {
    case NODE_PRESET:		/* Destroy preset */
      sfont_preset_destroy ((SFPreset *) (st->data));
      break;
    case NODE_INST:		/* Destroy instrument */
      sfont_inst_destroy ((SFInst *) (st->data));
      break;
    case NODE_SAMPLE:		/* Destroy sample (dorefcount--) */
      sfont_sample_destroy ((SFSample *) (st->data), FALSE);
      break;
    case NODE_PZONE:
    case NODE_IZONE:
      sfont_zone_destroy ((SFZone *) (st->data));
      break;
    default:  /* shouldn't happen */
      break;
    }

  g_free (st);
}

/* saves a sound font item (Preset, Instrument or Sample). With Presets and
Instruments all zones are copied, but pointer references to instruments/samples
are changed to item ids */
static gpointer
dofunc_sfitem_save_state (SFItemID itemid)
{
  DoState_SFItem *st;
  UISFont *uisf;
  SFPreset *pset;
  SFInst *inst;
  GtkCTreeNode *node, *sfnode;
  SFTreeRef *ref;
  GSList *p;

  st = g_malloc (sizeof (DoState_SFItem));

  node = SFTREE_LOOKUP_ITEMID (itemid);

  g_return_val_if_fail (node != NULL, NULL);

  ref = SFTREE_NODE_REF (node);
  p = ref->dptr;

  st->type = ref->type;

  sfnode = sftree_find_parent_by_type (node, NODE_SFONT);
  uisf = SFTREE_SFNODE_UISF (sfnode);
  st->uisf = uisf;

  switch (ref->type)
    {
    case NODE_PRESET:
      /* duplicate preset and make inst ptrs item IDs */
      st->data = sfont_preset_dup ((SFPreset *)(p->data), SFDUP_ARCHIVE);
      st->pitemid = SFTREE_NODE_REF (sfnode)->itemid;
      break;
    case NODE_INST:
      /* duplicate inst in archive mode */
      st->data = sfont_inst_dup ((SFInst *)(p->data), SFDUP_ARCHIVE);
      st->pos = g_slist_index (uisf->sf->inst, p->data);
      st->pitemid = SFTREE_NODE_REF (sfnode)->itemid;
      break;
    case NODE_SAMPLE:
      /* dup sample in archive mode (retain item ID) and dorefcount++ */
      st->data = sfont_sample_dup ((SFSample *)(p->data),SFDUP_ARCHIVE, FALSE);
      st->pos = g_slist_index (uisf->sf->sample, p->data);
      st->pitemid = SFTREE_NODE_REF (sfnode)->itemid;
      break;
    case NODE_PZONE:
      /* get parent preset */
      node = GTK_CTREE_ROW (node)->parent;
      pset = (SFPreset *)(((GSList *)(SFTREE_NODE_REF (node)->dptr))->data);

      /* dup preset zone in archive mode */
      st->data = sfont_zone_dup ((SFZone *)(p->data), SFDUP_ARCHIVE);
      st->pos = g_slist_index (pset->zone, p->data);
      st->pitemid = pset->itemid;
      break;
    case NODE_IZONE:
      /* get parent instrument */
      node = GTK_CTREE_ROW (node)->parent;
      inst = (SFInst *)(((GSList *)(SFTREE_NODE_REF (node)->dptr))->data);

      /* dup instrument zone in archive mode */
      st->data = sfont_zone_dup ((SFZone *)(p->data), SFDUP_ARCHIVE);
      st->pos = g_slist_index (inst->zone, p->data);
      st->pitemid = inst->itemid;
      break;
    default:  /* shouldn't happen */
      break;
    }

  return (st);
}

typedef struct _DoState_SFInfo
{
  SFItemID sfitemid;
  guint8 id;
  gchar *str;
}DoState_SFInfo;

void
dofunc_sfinfo_save (SFItemID sfitemid, guint8 id)
{
  DoState_SFInfo *st;

  SFUNDO_ENTER ();

  st = dofunc_sfinfo_save_state (sfitemid, id);
  sfdo_add (DOFUNC_SFINFO, st);
}

static gint
dofunc_sfinfo_restore (SFDoItem *item)
{
  DoState_SFInfo *st;
  GtkCTreeNode *node;
  UISFont *uisf;

  st = item->state;
  node = SFTREE_LOOKUP_ITEMID (st->sfitemid);

  g_return_val_if_fail (node != NULL, FAIL);

  uisf = SFTREE_SFNODE_UISF (node);
  sfont_set_info (uisf->sf, st->id, st->str);

  if (st->id == INAM_ID)  /* if restored info is sfont name, update tree */
    sftree_update_sfont_node_label (uisf);

  return (OK);
}

static gint
dofunc_sfinfo_restate (SFDoItem *item, SFDoItem *newitem)
{
  DoState_SFInfo *st;

  st = item->state;
  newitem->type = item->type;  /* just copy the type */
  newitem->state = dofunc_sfinfo_save_state (st->sfitemid, st->id);

  return (OK);
}

static void
dofunc_sfinfo_free (SFDoItem *item)
{
  DoState_SFInfo *st;

  st = item->state;
  g_free (st->str);
  g_free (st);
}

static gpointer
dofunc_sfinfo_save_state (SFItemID sfitemid, guint8 id)
{
  DoState_SFInfo *st;
  GtkCTreeNode *node;
  SFData *sfont;

  st = g_malloc (sizeof (DoState_SFInfo));
  st->sfitemid = sfitemid;
  st->id = id;
  node = SFTREE_LOOKUP_ITEMID (sfitemid);

  g_return_val_if_fail (node != NULL, NULL);

  sfont = SFTREE_SFNODE_SFONT (node);
  st->str = g_strdup (sfont_get_info (sfont, id));

  return (st);
}

typedef struct _DState_SFGen
{
  gboolean set;			/* TRUE if generator is set (has a value) */
  SFItemID itemid;		/* item ID of generator's zone */
  SFGen gen;			/* the generator */
}
DState_SFGen;
