/*==================================================================
 * uif_pianospan.c - User interface piano and key/velocity span routines
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/
#include <stdio.h>
#include <gtk/gtk.h>
#include "uiface.h"
#include "uif_pianospan.h"
#include "uif_piano.h"
#include "uif_sfont.h"
#include "uif_sftree.h"
#include "sfont.h"
#include "widgets/piano.h"
#include "widgets/keyspan.h"
#include "widgets/ptrstrip.h"
#include "wavetable.h"
#include "smurfcfg.h"

static void pianospan_cb_span_changed (KeySpan * keyspan, GSList * zone);
static void pianospan_cb_span_state_changed (KeySpan * keyspan,
  GtkStateType state, GSList * p);
static gint pianospan_find_zone_compare (SFTreeRef * nref, GSList * pzone);
static void pianospan_cb_scrollbar_automatic (GtkAdjustment * adj,
  GtkWidget * scrollbar);
static void pianospan_cb_piano_vel_range_hadj (GtkAdjustment * padj,
  GtkAdjustment * radj);
static void pianospan_cb_ptrstrip_change (GtkWidget * ptrstrip, guint ptrndx);

gint pianospan_mode = PIANOSPAN_PIANO;

GtkWidget *pianospan_ptrstrip = NULL;	/* root key pointer strip widget */
GtkWidget *pianospan_velbar = NULL;	/* velocity bar widget */
GtkWidget *pianospan_velbar_box = NULL;	/* velbar's box to keep left justified */
GtkWidget *pianospan_view_box;	/* the box widget for piano/velocity */
GtkWidget *pianospan_keyrange_box;	/* Key range box (to pack the spans into) */
GPtrArray *pianospan_spans;	/* Ptr array of active spans */

guchar velbar_scolor[3] = { 0, 0, 0x40 };	/* velbar start color */
guchar velbar_ecolor[3] = { 0, 0, 0xFF };	/* velbar end color */

/* variables to disable entry into callbacks */
static gboolean pianospan_no_entry = FALSE;

static guchar *velbar_linebuf;	/* buffer for one line of velocity bar */


void
pianospan_set_mode (gint mode)
{
  if (mode == pianospan_mode)
    return;

  pianospan_clear ();

  if (mode == PIANOSPAN_PIANO)
    {
      gtk_widget_ref (pianospan_velbar_box);
      gtk_container_remove (GTK_CONTAINER (pianospan_view_box),
	pianospan_velbar_box);
      gtk_box_pack_start (GTK_BOX (pianospan_view_box), uipiano_widg,
	FALSE, FALSE, 0);
      gtk_widget_unref (uipiano_widg);
    }
  else
    {
      gtk_widget_ref (uipiano_widg);
      gtk_container_remove (GTK_CONTAINER (pianospan_view_box),
	uipiano_widg);
      gtk_box_pack_start (GTK_BOX (pianospan_view_box), pianospan_velbar_box,
	FALSE, FALSE, 0);
      gtk_widget_unref (pianospan_velbar_box);
    }

  pianospan_mode = mode;
}

/* range of a span changed */
static void
pianospan_cb_span_changed (KeySpan * keyspan, GSList * zone)
{
  SFGenAmount amt;

  amt.range.lo = keyspan->lokey;
  amt.range.hi = keyspan->hikey;

  if (pianospan_mode == PIANOSPAN_PIANO)
    sfont_gen_set (uisf_selected_uisfont->sf, (SFZone *) (zone->data),
      Gen_KeyRange, amt, FALSE);
  else
    sfont_gen_set (uisf_selected_uisfont->sf, (SFZone *) (zone->data),
      Gen_VelRange, amt, FALSE);

  if (uisf_selected_elem_type == NODE_PRESET)
    {
      wtbl_sfitem_changed (((SFPreset *)uisf_selected_elem)->itemid,
			   WTBL_ITEM_CHANGE);
    }
  else
    {
      wtbl_sfitem_changed (((SFInst *)uisf_selected_elem)->itemid,
			   WTBL_ITEM_CHANGE);
    }
}

/* active state of a span changed */
static void
pianospan_cb_span_state_changed (KeySpan * keyspan,
  GtkStateType state, GSList * p)
{
  GtkCTreeNode *node;
  KeySpan *span;
  gint i;
  /* 'state' callback parameter is probably the old state?? */

  if (GTK_WIDGET_STATE (GTK_WIDGET (keyspan)) != GTK_STATE_SELECTED
    || pianospan_no_entry)
    return;

  /* locate the corresponding zone for this span and activate it */
  if (uisf_selected_elem_type == NODE_PRESET)
    {
      node = gtk_ctree_find_by_row_data_custom (sftree_widg,
	uisf_selected_uisfont->nodes->preset, p,
	(GCompareFunc) pianospan_find_zone_compare);
    }
  else if (uisf_selected_elem_type == NODE_INST)
    {
      node = gtk_ctree_find_by_row_data_custom (sftree_widg,
	uisf_selected_uisfont->nodes->inst, p,
	(GCompareFunc) pianospan_find_zone_compare);
    }

  /* clear all other spans (except for this selected one) */
  i = pianospan_spans->len;
  while (i--)
    {
      span = g_ptr_array_index (pianospan_spans, i);
      if (span != keyspan)
	gtk_widget_set_state (GTK_WIDGET (span), GTK_STATE_NORMAL);
    }

  if (node)
    {
      sftree_clear_selection ();
      gtk_ctree_select (sftree_widg, node);
    }
}

/* find a particular zone in the sound font tree */
static gint
pianospan_find_zone_compare (SFTreeRef * nref, GSList * pzone)
{
  if ((nref->type == NODE_PZONE || nref->type == NODE_IZONE) &&
    nref->dptr == pzone) return (0);
  return (1);
}

/* update key/velocity spans for active element */
void
pianospan_update (void)
{
  GSList *zone = NULL;
  GSList *p, *p2;
  SFGen *gen;
  GtkWidget *keyspan;
  gint8 lo, hi;

  pianospan_clear ();

  if (uisf_selected_elem_type == NODE_PRESET)
    zone = ((SFPreset *) uisf_selected_elem)->zone;
  else if (uisf_selected_elem_type == NODE_INST)
    zone = ((SFInst *) uisf_selected_elem)->zone;

  if (!zone) return;		/* if there aren't any zones, return */

  p = zone;

  if (!((SFZone *) (p->data))->instsamp)	/* global zone? */
    p = g_slist_next (p);	/* yes: skip */

  while (p)
    {
      lo = 0;			/* default range is all keys */
      hi = 127;

      if ((p2 = ((SFZone *) (p->data))->gen))
	{			/* if gen.count > 0 */
	  gen = (SFGen *) (p2->data);
	  if (pianospan_mode == PIANOSPAN_PIANO)
	    {			/* piano mode? */
	      if (gen->id == Gen_KeyRange)
		{
		  lo = gen->amount.range.lo;
		  hi = gen->amount.range.hi;
		}
	    }
	  else
	    {			/* velocity mode */
	      if (gen->id == Gen_KeyRange)
		{		/* only KeyRange preceeds VelRange */
		  p2 = g_slist_next (p2);	/* advance to next gen */
		  if (p2)
		    gen = (SFGen *) (p2->data);
		  else
		    gen = NULL;
		}
	      if (gen && gen->id == Gen_VelRange)
		{		/* gen is velocity range? */
		  lo = gen->amount.range.lo;
		  hi = gen->amount.range.hi;
		}
	    }
	}

      keyspan = keyspan_new ();
      keyspan_set_span (KEYSPAN (keyspan), lo, hi);

      if (pianospan_mode == PIANOSPAN_PIANO)
	keyspan_set_mode (KEYSPAN (keyspan), KEYSPAN_KEYMODE);
      else
	keyspan_set_mode (KEYSPAN (keyspan), KEYSPAN_VELMODE);

      gtk_signal_connect (GTK_OBJECT (keyspan), "span_change",
	GTK_SIGNAL_FUNC (pianospan_cb_span_changed), (gpointer) p);
      gtk_signal_connect (GTK_OBJECT (keyspan), "state-changed",
	GTK_SIGNAL_FUNC (pianospan_cb_span_state_changed), (gpointer) p);
      gtk_widget_show (GTK_WIDGET (keyspan));
      gtk_box_pack_start (GTK_BOX (pianospan_keyrange_box), keyspan, FALSE,
	FALSE, 0);

      /* add keyspan ptr to keyspan ptr array */
      g_ptr_array_add (pianospan_spans, (gpointer) keyspan);

      if (p->data == uisf_selected_zone)
	{			/* this span is active? */
	  pianospan_no_entry = TRUE;	/* ?: yes, set its state to active */
	  gtk_widget_set_state (GTK_WIDGET (keyspan), GTK_STATE_SELECTED);
	  pianospan_no_entry = FALSE;
	}

      p = g_slist_next (p);
    }
}

void
pianospan_clear (void)
{				/* clear span window */
  gint i;

  i = pianospan_spans->len;
  while (i--)
    {
      gtk_widget_destroy (GTK_WIDGET (g_ptr_array_index (pianospan_spans,
	    0)));
      g_ptr_array_remove_index_fast (pianospan_spans, 0);
    }
}

void
pianospan_select_span (gint spannum)
{
  KeySpan *span;

  span = g_ptr_array_index (pianospan_spans, spannum);

  pianospan_no_entry = TRUE;
  gtk_widget_set_state (GTK_WIDGET (span), GTK_STATE_SELECTED);
  pianospan_no_entry = FALSE;
}

void
pianospan_clear_selection (void)
{				/* unselects all spans */
  KeySpan *span;
  gint i;

  i = pianospan_spans->len;
  while (i--)
    {
      span = g_ptr_array_index (pianospan_spans, i);
      gtk_widget_set_state (GTK_WIDGET (span), GTK_STATE_NORMAL);
    }
}

void
pianospan_clear_rootkey_ptrstrip (void)
{
  gtk_signal_handler_block_by_func (GTK_OBJECT (pianospan_ptrstrip),
			    (GtkSignalFunc) pianospan_cb_ptrstrip_change, NULL);

  ptrstrip_set_pointer (PTRSTRIP (pianospan_ptrstrip), 0, -1);

  gtk_signal_handler_unblock_by_func (GTK_OBJECT (pianospan_ptrstrip),
			    (GtkSignalFunc) pianospan_cb_ptrstrip_change, NULL);
}

void
pianospan_update_rootkey_ptrstrip (void)
{
  SFGenAmount *amt;
  gint xpos;

  if (uisf_selected_elem_type == NODE_INST && uisf_selected_zone)
    {				/* inst root key override */
      amt = sfont_gen_get (uisf_selected_zone, Gen_OverrideRootKey);
      if (amt)
	xpos = piano_key_to_xpos (amt->sword);
      else
	xpos = 0;
    }
  else if (uisf_selected_elem_type == NODE_SAMPLE)
    {				/* sample root key */
      xpos = piano_key_to_xpos (((SFSample *) uisf_selected_elem)->origpitch);
    }
  else return;			/* not izone or sample selection! */

  gtk_signal_handler_block_by_func (GTK_OBJECT (pianospan_ptrstrip),
			    (GtkSignalFunc) pianospan_cb_ptrstrip_change, NULL);

  ptrstrip_set_pointer (PTRSTRIP (pianospan_ptrstrip), 0, xpos);

  gtk_signal_handler_unblock_by_func (GTK_OBJECT (pianospan_ptrstrip),
			    (GtkSignalFunc) pianospan_cb_ptrstrip_change, NULL);
}

static void
pianospan_cb_scrollbar_automatic (GtkAdjustment * adj, GtkWidget * scrollbar)
{
  if (GTK_WIDGET_VISIBLE (scrollbar))
    {
      if ((adj->upper - adj->lower) <= adj->page_size)
	{
	  gtk_widget_hide (scrollbar);
	}
    }
  else
    {
      if ((adj->upper - adj->lower) > adj->page_size)
	{
	  gtk_widget_show (scrollbar);
	}
    }
}

static void
pianospan_cb_piano_vel_range_hadj (GtkAdjustment * padj, GtkAdjustment * radj)
{
  gtk_adjustment_set_value (radj, padj->value);
}

static void
pianospan_cb_ptrstrip_change (GtkWidget * ptrstrip, guint ptrndx)
{
  gint keynum;
  gint ptrxpos, newxpos;
  SFGenAmount amt, *getamt;
  GtkPSPtr *psptr;

  if (ptrndx != 0 || !(PTRSTRIP (ptrstrip)->pointers)) return;

  /* we want first pointer */
  psptr = (GtkPSPtr *) (PTRSTRIP (ptrstrip)->pointers->data);
  ptrxpos = psptr->xpos;

  /* get current value of root key */
  if (uisf_selected_elem_type == NODE_INST && uisf_selected_zone)
    {				/* -- inst root key override */

      /* ptr at very left hand side of the pointer bar disables rootkey
         override, which means the rootkey from the sample is used */
      if (ptrxpos > 3)
	{			/* NO, ptr is not far left */
	  keynum = piano_xpos_to_key (ptrxpos);
	  newxpos = piano_key_to_xpos (keynum);
	}
      else
	{			/* YES, ptr is at far left */
	  keynum = -1;
	  newxpos = 0;
	}

      getamt = sfont_gen_get (uisf_selected_zone, Gen_OverrideRootKey);

      /* check if new keynum is different then old */
      if ((!getamt && keynum == -1) || (getamt && getamt->sword == keynum))
	{
	  /* new keynum is the same as old, stop signal propagation */
	  gtk_signal_emit_stop_by_name (GTK_OBJECT (ptrstrip),
	    "pointer_change");
	}
      else
	{			/* keynum is different then old, so update it */
	  amt.sword = keynum;	/* set the override root key gen */
	  sfont_gen_set (uisf_selected_uisfont->sf, uisf_selected_zone,
	    Gen_OverrideRootKey, amt, FALSE);

	  /* notify wavetable of change */
	  if (uisf_selected_elem_type == NODE_PRESET)
	    {
	      wtbl_sfitem_changed (((SFPreset *)uisf_selected_elem)->itemid,
				   WTBL_ITEM_CHANGE);
	    }
	  else
	    {
	      wtbl_sfitem_changed (((SFInst *)uisf_selected_elem)->itemid,
				 WTBL_ITEM_CHANGE);
	    }
	}
    }
  else if (uisf_selected_elem_type == NODE_SAMPLE)
    {				/* -- sample root key */
      keynum = piano_xpos_to_key (ptrxpos);
      newxpos = piano_key_to_xpos (keynum);

      if (keynum == ((SFSample *) uisf_selected_elem)->origpitch)
	{
	  /* new keynum is the same as old, stop signal propagation */
	  gtk_signal_emit_stop_by_name (GTK_OBJECT (ptrstrip),
	    "pointer_change");
	}
      else
	{
	  ((SFSample *) uisf_selected_elem)->origpitch = keynum;

	  /* notify wavetable of change */
	  wtbl_sfitem_changed (((SFSample *)uisf_selected_elem)->itemid,
			       WTBL_ITEM_CHANGE);
	}
    } else return;		/* not instrument or sample selection! */

  psptr->xpos = newxpos;	/* lock the ptr to key centers */
}

void
pianospan_draw_velocity_bar (void)
{
  gint i, i2, lastlevel = 0;
  float levelinc, level = 0.0;
  gint rval, gval, bval;
  float rf, gf, bf, rinc, ginc, binc;

  rval = velbar_scolor[0];
  gval = velbar_scolor[1];
  bval = velbar_scolor[2];

  rf = (float) rval;
  gf = (float) gval;
  bf = (float) bval;

  rinc = (float) (velbar_ecolor[0] - rval + 1) / PIANO_DEFAULT_SIZEX;
  ginc = (float) (velbar_ecolor[1] - gval + 1) / PIANO_DEFAULT_SIZEX;
  binc = (float) (velbar_ecolor[2] - bval + 1) / PIANO_DEFAULT_SIZEX;

  /* 128 steps for 128 levels of velocity */
  levelinc = 128.0 / PIANO_DEFAULT_SIZEX;

  /* draw a single line of the velocity bar */
  for (i = 0, i2 = 0; i < PIANO_DEFAULT_SIZEX; i++, i2 += 3)
    {
      velbar_linebuf[i2] = rval;
      velbar_linebuf[i2 + 1] = gval;
      velbar_linebuf[i2 + 2] = bval;

      if (lastlevel != (int) level)
	{
	  lastlevel = (int) level;
	  rval = (int) rf;
	  gval = (int) gf;
	  bval = (int) bf;
	}

      level += levelinc;
      rf += rinc;
      gf += ginc;
      bf += binc;
    }

  /* duplicate the line for each row */
  for (i = 0; i < PIANO_DEFAULT_SIZEY; i++)
    gtk_preview_draw_row (GTK_PREVIEW (pianospan_velbar), velbar_linebuf,
      0, i, PIANO_DEFAULT_SIZEX);
}

GtkWidget *
pianospan_create (void)
{
  GtkWidget *box_main, *vbox, *box;
  GtkWidget *view;
  GtkWidget *hscrollbar, *vscrollbar;
  GtkAdjustment *hadj, *vadj;

  /* main outer box */
  box_main = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (box_main);

  /* everything cept the vertical scrollbar goes in here */
  vbox = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox);
  gtk_box_pack_start (GTK_BOX (box_main), vbox, TRUE, TRUE, 0);

  /* horizontal scrollbar controls both viewports */
  hscrollbar = gtk_hscrollbar_new (NULL);
  gtk_widget_show (hscrollbar);
  hadj = gtk_range_get_adjustment (GTK_RANGE (hscrollbar));
  gtk_signal_connect (GTK_OBJECT (hadj), "changed",
    GTK_SIGNAL_FUNC (pianospan_cb_scrollbar_automatic), hscrollbar);

  /* piano viewport */
  view = gtk_viewport_new (NULL, NULL);
  gtk_viewport_set_hadjustment (GTK_VIEWPORT (view), GTK_ADJUSTMENT (hadj));
  gtk_container_border_width (GTK_CONTAINER (view), 0);
  gtk_viewport_set_shadow_type (GTK_VIEWPORT (view), GTK_SHADOW_OUT);
  gtk_widget_show (GTK_WIDGET (view));
  gtk_box_pack_start (GTK_BOX (vbox), view, FALSE, FALSE, 0);

  pianospan_view_box = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (GTK_WIDGET (pianospan_view_box));
  gtk_container_add (GTK_CONTAINER (view), pianospan_view_box);

  pianospan_ptrstrip = ptrstrip_new ();
  gtk_widget_set_usize (pianospan_ptrstrip, PIANO_DEFAULT_SIZEX, -1);
  ptrstrip_new_pointer (PTRSTRIP (pianospan_ptrstrip), -1);
  gtk_signal_connect (GTK_OBJECT (pianospan_ptrstrip), "pointer_change",
    (GtkSignalFunc) pianospan_cb_ptrstrip_change, NULL);
  gtk_widget_show (pianospan_ptrstrip);
  gtk_box_pack_start (GTK_BOX (pianospan_view_box), pianospan_ptrstrip,
    FALSE, FALSE, 0);

  /* create the piano widget */
  uipiano_create ();
  gtk_widget_show (uipiano_widg);
  gtk_box_pack_start (GTK_BOX (pianospan_view_box), uipiano_widg,
    FALSE, FALSE, 0);

  /* create box to pack preview in so it stays left justified */
  pianospan_velbar_box = gtk_hbox_new (FALSE, 0);
  /* ref velocity bar box cuz we ain't packin' it now */
  gtk_widget_ref (pianospan_velbar_box);
  gtk_widget_show (pianospan_velbar_box);

  /* create the velocity widget */
  pianospan_velbar = gtk_preview_new (GTK_PREVIEW_COLOR);
  gtk_preview_size (GTK_PREVIEW (pianospan_velbar), PIANO_DEFAULT_SIZEX,
    PIANO_DEFAULT_SIZEY);
  gtk_preview_set_expand (GTK_PREVIEW (pianospan_velbar), FALSE);

  /* load velocity bar start and end colors from preferences */
  if (!smurfcfg_parse_hashcolor (smurfcfg_get_val (SMURFCFG_VELBAR_SCOLOR)->
      v_string, velbar_scolor))
    {
      velbar_scolor[0] = 0;
      velbar_scolor[1] = 0;
      velbar_scolor[2] = 0x40;
    }
  if (!smurfcfg_parse_hashcolor (smurfcfg_get_val (SMURFCFG_VELBAR_ECOLOR)->
      v_string, velbar_ecolor))
    {
      velbar_ecolor[0] = 0;
      velbar_ecolor[1] = 0;
      velbar_ecolor[2] = 0xFF;
    }

  /* allocate buffer space for one line of velocity bar */
  velbar_linebuf = g_new (guchar, PIANO_DEFAULT_SIZEX * 3);
  pianospan_draw_velocity_bar ();
  gtk_widget_show (pianospan_velbar);
  gtk_box_pack_start (GTK_BOX (pianospan_velbar_box), pianospan_velbar,
    FALSE, FALSE, 0);

  /* vertical scrollbar controls keyrange widget view only */
  vscrollbar = gtk_vscrollbar_new (NULL);
  gtk_widget_show (vscrollbar);
  vadj = GTK_RANGE (vscrollbar)->adjustment;
  gtk_signal_connect (GTK_OBJECT (vadj), "changed",
    GTK_SIGNAL_FUNC (pianospan_cb_scrollbar_automatic), vscrollbar);
  gtk_box_pack_start (GTK_BOX (box_main), vscrollbar, FALSE, FALSE, 0);

  /* keyrange viewport */
  view = gtk_viewport_new (NULL, vadj);
  gtk_widget_set_usize (view, KEYSPAN_DEFAULT_SIZEX, 0);

  gtk_signal_connect (GTK_OBJECT (hadj), "value_changed",
    GTK_SIGNAL_FUNC (pianospan_cb_piano_vel_range_hadj),
    (gpointer) gtk_viewport_get_hadjustment (GTK_VIEWPORT (view)));

  gtk_container_border_width (GTK_CONTAINER (view), 0);
  gtk_viewport_set_shadow_type (GTK_VIEWPORT (view), GTK_SHADOW_IN);
  gtk_widget_show (GTK_WIDGET (view));

  pianospan_keyrange_box = box = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (box);
  gtk_container_add (GTK_CONTAINER (view), box);

  gtk_box_pack_start (GTK_BOX (vbox), view, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hscrollbar, FALSE, FALSE, 0);

  pianospan_spans = g_ptr_array_new ();

  return (box_main);
}
