/***************************************************************************
    smb4kmounter.cpp  -  The core class that mounts the shares.
                             -------------------
    begin                : Die Jun 10 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qapplication.h>
#include <qdir.h>
#include <qtextstream.h>
#include <qregexp.h>
#include <qmap.h>
#include <qthread.h>

// KDE includes
#include <kapplication.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kdebug.h>

// system includes
#include <sys/statvfs.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <dirent.h>

#ifdef __FreeBSD__
#include <pwd.h>
#include <sys/param.h>
#include <sys/ucred.h>
#include <sys/mount.h>
#include <qfileinfo.h>
#endif

// Application specific includes
#include "smb4kmounter.h"
#include "smb4kglobal.h"
using namespace Smb4KGlobal;


class Smb4KMounterPrivate
{
  public:
    Smb4KMounterPrivate()
    {
      timerTicks = 0;
      data = NULL;
    }
    ~Smb4KMounterPrivate()
    {
      if ( data )
      {
        delete data;
      }
    }
    int timerTicks;
    Smb4KDataItem *data;
    class Thread : public QThread
    {
      public:
        Thread() : QThread(), m_mountpoint( QString::null ), m_broken( true ) {}
        ~Thread() {}

        void setMountpoint( const QString &mp ) { m_mountpoint = mp; }

        virtual void run()
        {
          if ( m_mountpoint.isEmpty() )
          {
            kdFatal() << "Smb4KMounterPrivate::Thread: No mountpoint specified" << endl;
          }

          DIR *dir = NULL;

          if ( (dir = opendir( m_mountpoint.local8Bit() )) != NULL )
          {
            m_broken = false;
          }
          else
          {
            m_broken = true;
          }

          closedir( dir );

          m_mountpoint = QString::null;
        }

        bool isBroken() { return m_broken; }

      private:
        QString m_mountpoint;
        bool m_broken;
    };

    Thread thread;
};

Smb4KMounterPrivate mp;




Smb4KMounter::Smb4KMounter( QObject *parent, const char *name ) : QObject( parent, name )
{
  m_proc = new KProcess( this, "MounterProcess" );
  m_proc->setUseShell( true );

  m_password_handler = new Smb4KPasswordHandler( this, "MounterPasswordHandler" );

  m_working = false;

  // Do some initial actions:
  // - Import all mounted SMB shares,
  // - Mount all shares that have to be mounted.
  m_queue.setAutoDelete( true );

  m_queue.enqueue( new QString( QString( "%1:" ).arg( Import ) ) );
  m_queue.enqueue( new QString( QString( "%1:" ).arg( MountRecent ) ) );

  connect( m_proc,  SIGNAL( processExited( KProcess * ) ),
           this,    SLOT( slotProcessExited( KProcess * ) ) );
  connect( m_proc,  SIGNAL( receivedStdout( KProcess *, char *, int ) ),
           this,    SLOT( slotReceivedStdout( KProcess *, char *, int ) ) );
  connect( m_proc,  SIGNAL( receivedStderr( KProcess *, char *, int ) ),
           this,    SLOT( slotReceivedStderr( KProcess *, char *, int ) ) );
  connect( timer(), SIGNAL( timeout() ),
           this,    SLOT( init() ) );
  connect( kapp,    SIGNAL( shutDown() ),
           this,    SLOT( slotShutdown() ) );
}


Smb4KMounter::~Smb4KMounter()
{
  abort();


  for ( QValueList<Smb4KShare *>::Iterator it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
  {
    delete *it;
  }

  m_mounted_shares.clear();
}


/***************************************************************************
   Aborts any action of the mounter.
***************************************************************************/


void Smb4KMounter::abort()
{
  config()->setGroup( "Super User Privileges" );
  bool run_suid = config()->readBoolEntry( "Run SUID", false );
  QString suid_program = config()->readEntry( "SUID Program", QString::null );

  m_queue.clear();

  if ( run_suid && !suid_program.isEmpty() )
  {
    if ( m_proc->isRunning() )
    {
      KProcess p;
      p.setUseShell( true );
      p << QString( "%1 smb4k_kill -15 %2" ).arg( suid_program ).arg( m_proc->pid() );
      p.start( KProcess::DontCare, KProcess::NoCommunication );
    }
  }
  else
  {
    if ( m_proc->isRunning() )
    {
      m_proc->kill();
    }
  }
}


/***************************************************************************
   Mounts recently used shares.
***************************************************************************/


void Smb4KMounter::mountRecent()
{
  config()->setGroup( "Mount Options" );
  QString mount_prefix = config()->readEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) );

  if ( config()->readBoolEntry( "Mount Recent", false ) )
  {
    if ( config()->hasGroup( "Recently Mounted Shares" ) )
    {
      config()->setGroup( "Recently Mounted Shares" );

      QString temp;

      for ( int index = 0; ; index++ )
      {
        temp = config()->readEntry( QString( "%1" ).arg( index ), QString::null );

        if ( !temp.isEmpty() )
        {
          Smb4KShare *share = findShareByName( temp );

          if ( !share || share->isForeign() )
          {
#ifndef __FreeBSD__
            mountShare( QString::null, temp.section( "/", 2, 2 ), QString::null, temp.section( "/", 3, 3 ) );
#else
            mountShare( QString::null, temp.section( "/", 2, 2 ).section( "@", 1, 1 ), QString::null, temp.section( "/", 3, 3 ) );
#endif
          }
          else
          {
            continue;
          }
        }
        else
        {
          break;
        }
      }

      m_working = false;
      emit state( MOUNTER_STOP );
    }
    else
    {
      m_working = false;
      emit state( MOUNTER_STOP );
    }
  }
  else
  {
    m_working = false;
    emit state( MOUNTER_STOP );
  }
}


/***************************************************************************
   Imports all shares, that are mounted externally.
***************************************************************************/

void Smb4KMounter::import()
{
  config()->setGroup( "Mount Options" );
  QString mount_prefix = config()->readEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) );

#ifndef __FreeBSD__

  QValueList<Smb4KShare *> shares;

  // /proc/mounts is read to import the externally mounted shares.
  if ( m_proc_mounts.name().isEmpty() )
  {
    m_proc_mounts.setName( "/proc/mounts" );
  }

  if ( !QFile::exists( m_proc_mounts.name() ) )
  {
    if ( !m_proc_error )
    {
      m_proc_error = true;
      showCoreError( ERROR_FILE_NOT_FOUND, QString( "%1" ).arg( m_proc_mounts.name() ) );
    }
  }
  else
  {
    if ( m_proc_mounts.open( IO_ReadOnly ) )
    {
      QTextStream ts( &m_proc_mounts );
      ts.setEncoding( QTextStream::Locale );

      while( !ts.atEnd() )
      {
        QString line = ts.readLine().stripWhiteSpace();

        if ( line.contains( "smbfs", false ) != 0 )
        {
          QString share_and_path = line.section( "smbfs", 0, 0 ).stripWhiteSpace();
          QString name = share_and_path.section( " ", 0, 0 ).stripWhiteSpace().replace( "\\040", "\040" );
          QString path = share_and_path.section( " ", 1, 1 ).stripWhiteSpace();

          if ( path.contains( "\\040" ) != 0 || path.contains( "\040" ) != 0 )
          {
            name.replace( "_", "\040" );
            path.replace( "\\040", "\040" );
          }

          int uid = line.section( "uid=", 1, 1 ).section( ",", 0, 0 ).stripWhiteSpace().toInt();
          int gid = line.section( "gid=", 1, 1 ).section( ",", 0, 0 ).stripWhiteSpace().toInt();

          // Test if share is broken
          bool broken = true;
          Smb4KShare *share = NULL;

          if ( ((share = findShareByName( name )) != NULL && !share->isBroken()) || share == NULL )
          {
            mp.thread.setMountpoint( path );
            mp.thread.start();
            mp.thread.wait( 1000 );
            mp.thread.terminate();
            mp.thread.wait();

            broken = mp.thread.isBroken();

            if ( broken )
            {
              showCoreError( ERROR_SHARE_WENT_OFFLINE, name );
            }
          }
          else
          {
            broken = true;
          }

          shares.append( new Smb4KShare( name, path, "smbfs", uid, gid, broken ) );

          continue;
        }
        else if ( line.contains( "cifs", false ) != 0 )
        {
          QString share_and_path = line.section( "cifs", 0, 0 ).stripWhiteSpace();
          QString name = share_and_path.section( " ", 0, 0 ).stripWhiteSpace().replace( "\\040", "\040" );
          QString path = share_and_path.section( " ", 1, 1 ).stripWhiteSpace();

          if ( path.contains( "\\040" ) != 0 || path.contains( "\040" ) != 0 )
          {
            name.replace( "_", "\040" );
            path.replace( "\\040", "\040" );
          }

          QString login = line.section( "username=", 1, 1 ).section( ",", 0, 0 ).stripWhiteSpace();

          // Test if share is broken
          bool broken = true;
          Smb4KShare *share = NULL;

          if ( ((share = findShareByName( name )) != NULL && !share->isBroken()) || share == NULL )
          {
            mp.thread.setMountpoint( path );
            mp.thread.start();
            mp.thread.wait( 1000 );
            mp.thread.terminate();
            mp.thread.wait();

            broken = mp.thread.isBroken();
          }
          else
          {
            broken = true;
          }

          bool foreign = true;

          if ( (!broken &&
               (QDir( path ).canonicalPath().startsWith( QDir( mount_prefix ).canonicalPath() ) ||
                QDir( path ).canonicalPath().startsWith( QDir::home().canonicalPath() ))) ||
               (broken && (path.startsWith( QDir::homeDirPath() ) || path.startsWith( mount_prefix ))) )
          {
            foreign = false;
          }

          shares.append( new Smb4KShare( name, path, "cifs", login, foreign, broken ) );

          continue;
        }
      }

      m_proc_mounts.close();
    }
    else
    {
      showCoreError( ERROR_READING_FILE, QString( "%1" ).arg( m_proc_mounts.name() ) );
    }
  }

#else

  struct statfs *buf;
  int count = getmntinfo( &buf, 0 );

  if ( count == 0 )
  {
    int err_code = errno;

    showCoreError( ERROR_IMPORTING_SHARES, strerror( err_code ) );

    m_working = false;
    return;
  }

  QValueList<Smb4KShare *> shares;

  for ( int i = 0; i < count; ++i )
  {
    if ( !strcmp( buf[i].f_fstypename, "smbfs" ) )
    {
      QString share_name( buf[i].f_mntfromname );
      QString path( buf[i].f_mntonname );
      QString fs( buf[i].f_fstypename );

      QFileInfo info( QString( buf[i].f_mntonname )+"/." );

      int uid = (int)info.ownerId();
      int gid = (int)info.groupId();

      // Test if share is broken
      bool broken = true;
      Smb4KShare *share = NULL;

      if ( ((share = findShareByName( share_name )) != NULL && !share->isBroken()) || share == NULL )
      {
        mp.thread.setMountpoint( path );
        mp.thread.start();
        mp.thread.wait( 1000 );
        mp.thread.terminate();
        mp.thread.wait();

        broken = mp.thread.isBroken();
      }
      else
      {
        broken = true;
      }

      shares.append( new Smb4KShare( share_name, path, fs, uid, gid, broken ) );

      continue;
    }
  }

  // Apparently, under FreeBSD we do not need to delete
  // the pointer (see manual page).

#endif

  // Check, if the entries are already in the list.
  // If yes, replace the entry in the temporary by the one from the
  // old list of shares. This way we won't get into trouble with
  // shares that contain spaces.
  if ( m_mounted_shares.count() != 0 )
  {
    for ( QValueList<Smb4KShare *>::Iterator it = shares.begin(); it != shares.end(); ++it )
    {
      Smb4KShare *s = findShareByPath( (*it)->path() );

      if ( s )
      {
        s->setBroken( (*it)->isBroken() );

        delete *it;
        *it = new Smb4KShare( *s );

        continue;
      }
      else
      {
        continue;
      }
    }
  }

  // Delete all entries of m_mounted_shares.
  for ( QValueList<Smb4KShare *>::Iterator it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
  {
    delete *it;
  }

  m_mounted_shares.clear();

  m_mounted_shares = shares;

  // Now get the information about disk usage etc.
  // If you change something here, be sure to check the
  // code in processMount(), too.
  for ( QValueList<Smb4KShare *>::Iterator it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
  {
    if ( !(*it)->isBroken() )
    {
      struct statvfs fs;

      if ( statvfs( (*it)->path().local8Bit(), &fs ) == -1 )
      {
        int err_code = errno;

        (*it)->setTotalDiskSpace( -1 );
        (*it)->setFreeDiskSpace( -1 );

        showCoreError( ERROR_GETTING_USAGE, strerror( err_code ) );
      }
      else
      {
        double kB_block = (double)(fs.f_bsize / 1024);
        double total = (double)(fs.f_blocks*kB_block);
        double free = (double)(fs.f_bfree*kB_block);

        (*it)->setTotalDiskSpace( total );
        (*it)->setFreeDiskSpace( free );
      }

      // You don't want to use ANY Qt function here, or Smb4K will
      // slow down dramatically.
    }
    else
    {
      (*it)->setTotalDiskSpace( -1 );
      (*it)->setFreeDiskSpace( -1 );
    }
  }

  emit updated();

  m_working = false;
}


/***************************************************************************
   Mounts a share. (Public part)
***************************************************************************/

void Smb4KMounter::mountShare( const QString &workgroup, const QString &host, const QString &ip, const QString &share )
{
  QString share_name = QString::null;

  if ( QString::compare( share, "homes" ) == 0 )
  {
    share_name = specifyUser( host );
  }
  else
  {
    share_name = share;
  }

  if ( !share_name.stripWhiteSpace().isEmpty() )
  {
    m_queue.enqueue( new QString( QString( "%1:%2:%3:%4:%5" ).arg( Mount ).arg( workgroup ).arg( host ).arg( ip ).arg( share_name ) ) );
  }
}



/***************************************************************************
   Mounts a share. (Private part)
***************************************************************************/

void Smb4KMounter::mount( const QString &workgroup, const QString &host, const QString &ip, const QString &share )
{
  mp.data = new Smb4KDataItem( workgroup, host, QString( "'%1'" ).arg( share.stripWhiteSpace() ), ip );

  config()->setGroup( "Super User Privileges" );
  bool run_suid = config()->readBoolEntry( "Run SUID", false );
  QString suid_program = config()->readEntry( "SUID Program", QString::null );

  config()->setGroup( "Samba" );
  mp.data->setFilesystem( config()->readEntry( "Mount Filesystem", "smbfs" ) );

  config()->setGroup( "Mount Options" );
  QString mount_prefix = config()->readPathEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) );
  bool lowercase = config()->readBoolEntry( "Force Lowercase", false );

  QString share_dir = share.stripWhiteSpace();

  // Test whether the share is already mounted.
  QString test_name = QString( "//%1/%2" ).arg( mp.data->host(), mp.data->share() );

  Smb4KShare *test_share;

  if ( (test_share = findShareByPath( QString( "%1%2/%3" ).arg( mount_prefix, mp.data->host(), share_dir ) )) != NULL )
  {
    if ( !test_share->isForeign() )
    {
      emit alreadyMountedShare( test_share->canonicalPath() );
      m_working = false;
      emit state( MOUNTER_STOP );
      return;
    }
  }
  else
  {
    if ( (test_share = findShareByName( QString( "//%1/%2" ).arg( mp.data->host(), mp.data->share() ) )) != NULL )
    {
      if ( !test_share->isForeign() )
      {
        emit alreadyMountedShare( test_share->canonicalPath() );
        m_working = false;
        emit state( MOUNTER_STOP );
        return;
      }
    }
  }

  delete test_share;

  // Create the mount point.
  QDir *dir = new QDir( mount_prefix );

  if ( !dir->exists() )
  {
    if ( !dir->mkdir( dir->canonicalPath() ) )
    {
      showCoreError( ERROR_MKDIR_FAILED, dir->path() );
      m_working = false;
      emit state( MOUNTER_STOP );
      return;
    }
  }

  QString host_dir = lowercase ? host.lower() : host;

  dir->setPath( dir->path()+"/"+host_dir );

  if ( !dir->exists() )
  {
    if ( !dir->mkdir( dir->canonicalPath() ) )
    {
      showCoreError( ERROR_MKDIR_FAILED, dir->path() );
      m_working = false;
      emit state( MOUNTER_STOP );
      return;
    }
  }

  share_dir = lowercase ? share_dir.lower() : share_dir;

  dir->setPath( dir->path()+"/"+share_dir );

  if ( !dir->exists() )
  {
    if ( !dir->mkdir( dir->canonicalPath() ) )
    {
      showCoreError( ERROR_MKDIR_FAILED, dir->path() );
      m_working = false;
      emit state( MOUNTER_STOP );
      return;
    }
  }

  mp.data->setPath( QDir::cleanDirPath( dir->path() ) );

  delete dir;

  // Mount the share.
  Smb4KAuthInfo *auth = m_password_handler->readAuth( workgroup, host, share );

  QString command;

#ifndef __FreeBSD__

  if ( !run_suid || suid_program.isEmpty() )
  {
    command = QString( "smb4k_mount --no-suid -t %1 " ).arg( mp.data->filesystem() );
  }
  else if ( run_suid && !suid_program.isEmpty() )
  {
    command = QString( "%1 smb4k_mount --suid -t %2 " ).arg( suid_program ).arg( mp.data->filesystem() );
  }

  command.append( QString( "//%1/%2 %3 " ).arg( KProcess::quote( host ) ).arg( KProcess::quote( share ) ).arg( KProcess::quote( mp.data->path() ) ) );

  command.append( "-o " );

  if ( !workgroup.isEmpty() )
  {
    if ( QString::compare( mp.data->filesystem(), "smbfs" ) == 0 )
    {
      command.append( QString( "workgroup=%1," ).arg( KProcess::quote( workgroup ) ) );
    }
    else
    {
      command.append( QString( "domain=%1," ).arg( KProcess::quote( workgroup ) ) );
    }
  }

  if ( !ip.isEmpty() )
  {
    command.append( QString( "ip=%1," ).arg( KProcess::quote( ip ) ) );
  }

  QString options = getMountOptions();

  if ( !options.stripWhiteSpace().isEmpty() )
  {
    command.append( options );
  }

  // We have to make sure, that if the user uses super or sudo,
  // the GID and the UID aren't the ones of root, except this
  // is explicitly wanted:
  if ( command.contains( "uid=", true ) == 0 && command.contains( "gid=", true ) == 0 && run_suid )
  {
    command.append( QString( "uid=%1,gid=%2," ).arg( QString( "%1" ).arg( (int)getuid() ), QString( "%1" ).arg( (int)getgid() ) ) );
  }

  // Setting the USER environment won't give you the right
  // user when using super/sudo.
  if ( !auth->user().isEmpty() )
  {
    command.append( QString( "username=%1" ).arg( KProcess::quote( auth->user() ) ) );
    mp.data->setCIFSLogin( auth->user() );
  }
  else
  {
    command.append( "guest" );
    mp.data->setCIFSLogin( "guest" );
  }

  m_proc->setEnvironment( "PASSWD", !auth->password().isEmpty() ? auth->password() : "" );

#else

  if ( !run_suid || suid_program.isEmpty() )
  {
    command = QString( "smb4k_mount --no-suid " );
  }
  else if ( run_suid && !suid_program.isEmpty() )
  {
    command = QString( "%1 smb4k_mount --suid " ).arg( suid_program );
  }

  QString options = getMountOptions();

  if ( !options.stripWhiteSpace().isEmpty() )
  {
    command.append( options );
  }

  if ( !mp.data->ip().isEmpty() )
  {
    command.append( " -I "+KProcess::quote( mp.data->ip() ) );
  }

  if ( !mp.data->workgroup().isEmpty() )
  {
    command.append( " -W "+KProcess::quote( mp.data->workgroup() ) );
  }

  command.append( " -N" );

  if ( command.contains( "-u", true ) == 0 && command.contains( "-g", true ) == 0 && run_suid )
  {
    command.append( QString( " -u %1 -g %2" ).arg( QString( "%1" ).arg( (int)getuid() ) ).arg( QString( "%1" ).arg( (int)getgid() ) ) );
  }

  if ( !auth->user().isEmpty() )
  {
    command.append( " //"+KProcess::quote( auth->user() )+"@"+KProcess::quote( host )+"/"+KProcess::quote( share )+" "+KProcess::quote( mp.data->path() ) );
  }
  else
  {
    command.append( " //guest@"+KProcess::quote( host )+"/"+KProcess::quote( share )+" "+KProcess::quote( mp.data->path() ) );
  }

#endif

  delete auth;

  // Start the mount process:
  *m_proc << command;

  startProcess( Mount );
}


/****************************************************************************
   Unmounts a share. (Public part)
****************************************************************************/

void Smb4KMounter::unmountShare( Smb4KShare *share, bool noMessage )
{
  // Do *not* change share->canonicalPath(). It is necessary for the
  // checks below to work.
  QString *input = new QString( QString( "%1:%2:%3" ).arg( Unmount ).arg( share->canonicalPath() ).arg( noMessage ) );
  m_queue.enqueue( input );
}


/***************************************************************************
   Unmounts a share. (Private part)
***************************************************************************/

void Smb4KMounter::unmount( const QString &mountpoint, bool noMessage )
{
  config()->setGroup( "Super User Privileges" );
  bool run_suid = config()->readBoolEntry( "Run SUID", false );
  QString suid_program = config()->readEntry( "SUID Program", QString::null );

  config()->setGroup( "Mount Options" );
  bool allow_unmount_foreign = config()->readBoolEntry( "Unmount Foreign", false );
  QString mount_prefix = config()->readEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) );

  mp.data = new Smb4KDataItem();

  if ( !mountpoint.isEmpty() )
  {
    QString command;
    QString path = mountpoint;
    mp.data->setPath( path.replace( '\044', "\044" ) );

    Smb4KShare *share = findShareByPath( mountpoint );

    if ( share && !share->isForeign() )
    {
      if ( !run_suid || suid_program.isEmpty() )
      {
        command = QString( "smb4k_umount --no-suid --%1 " ).arg( share->filesystem() );
      }
      else if ( run_suid && !suid_program.isEmpty() )
      {
        command = QString( "%1 smb4k_umount --suid --%2 " ).arg( suid_program ).arg( share->filesystem() );
      }
    }
    else if ( share && share->isForeign() )
    {
      if ( allow_unmount_foreign )
      {
        if ( !run_suid || suid_program.isEmpty() )
        {
          command = QString( "smb4k_umount --no-suid --%1 " ).arg( share->filesystem() );
        }
        else if ( run_suid && !suid_program.isEmpty() )
        {
          command = QString( "%1 smb4k_umount --suid --%2 " ).arg( suid_program ).arg( share->filesystem() );
        }
      }
      else
      {
        if ( !noMessage )
        {
          showCoreError( ERROR_UNMOUNTING_NOT_ALLOWED, QString::null );
        }

        m_working = false;
        emit state( MOUNTER_STOP );

        return;
      }
    }

    command.append( KProcess::quote( mp.data->path() ) );

    *m_proc << command;
    startProcess( Unmount );
  }
  else
  {
    showCoreError( ERROR_MOUNTPOINT_EMPTY, QString::null );
    m_working = false;
    emit state( MOUNTER_STOP );
  }
}


/****************************************************************************
   Unmounts a dead share. (Public part)
****************************************************************************/

void Smb4KMounter::forceUnmountShare( Smb4KShare *share )
{
  // Do *not* change share->canonicalPath(). It is necessary for the
  // checks below to work.
  QString *input = new QString( QString( "%1:%2" ).arg( ForceUnmount ).arg( share->canonicalPath() ) );
  m_queue.enqueue( input );
}


/***************************************************************************
   Unmounts a dead share. (Private part)
***************************************************************************/

void Smb4KMounter::forceUnmount( const QString &mountpoint )
{
  config()->setGroup( "Super User Privileges" );
  bool force_unmount = config()->readBoolEntry( "Force Unmount", false );
  QString suid_program = config()->readEntry( "SUID Program", QString::null );

  config()->setGroup( "Mount Options" );
  bool allow_unmount_foreign = config()->readBoolEntry( "Unmount Foreign", false );
  QString mount_prefix = config()->readEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) );

  QString kernel = kernelVersion();

  if ( !force_unmount )
  {
    showCoreError( ERROR_FEATURE_NOT_ENABLED, QString::null );
    m_working = false;
    emit state( MOUNTER_STOP );
    return;
  }

  // This will most likely never be the case, but we have to be prepared:
  if ( suid_program.isEmpty() )
  {
    showCoreError( ERROR_FORCE_UNMOUNT_SUID_MISSING, QString::null );
    m_working = false;
    emit state( MOUNTER_STOP );
    return;
  }

  // Check if the kernel supports the lazy unmount.
  int kern_maj = kernel.section( ".", 0, 0 ).toInt();
  int kern_min = kernel.section( ".", 1, 1 ).toInt();
  int kern_rel = kernel.section( ".", 2, 2 ).toInt();

  if ( kern_maj < 2 ||
       ( kern_maj <= 2 && kern_min < 4 ) ||
       ( kern_maj <= 2 && kern_min <= 4 && kern_rel < 11 ) )
  {
    showCoreError( ERROR_WRONG_KERNEL, kernel );
    m_working = false;
    emit state( MOUNTER_STOP );
    return;
  }

  mp.data = new Smb4KDataItem();

  if ( !mountpoint.isEmpty() )
  {
    QString command;
    QString path = mountpoint;
    mp.data->setPath( path.replace( '\044', "\044" ) );

    Smb4KShare *share = findShareByPath( mountpoint );

    bool execute = false;

    if ( share && !share->isForeign() )
    {
      if ( KMessageBox::questionYesNo( 0, i18n( "Do you really want to force the unmounting of this share?" ), QString::null, KStdGuiItem::yes(), KStdGuiItem::no(), "Dont Ask Forced", KMessageBox::Notify ) == KMessageBox::Yes )
      {
        command = QString( "%1 smb4k_umount --suid -l --%2 " ).arg( suid_program ).arg( share->filesystem() );
        execute = true;
      }
      else
      {
        m_working = false;
        emit state( MOUNTER_STOP );
        return;
      }
    }
    else if ( share && share->isForeign() )
    {
      if ( allow_unmount_foreign )
      {
        if ( KMessageBox::questionYesNo( 0, i18n( "Do you really want to force the unmounting of this share?" ), QString::null, KStdGuiItem::yes(), KStdGuiItem::no(), "Dont Ask Forced", KMessageBox::Notify ) == KMessageBox::Yes )
        {
          command = QString( "%1 smb4k_umount --suid -l --%2 " ).arg( suid_program ).arg( share->filesystem() );
          execute = true;
        }
        else
        {
          m_working = false;
          emit state( MOUNTER_STOP );
          return;
        }
      }
      else
      {
        showCoreError( ERROR_UNMOUNTING_NOT_ALLOWED, QString::null );
        m_working = false;
        emit state( MOUNTER_STOP );
        return;
      }
    }

    command.append( KProcess::quote( mp.data->path() ) );

    if ( execute )
    {
      *m_proc << command;
      startProcess( Unmount );
    }
  }
  else
  {
    showCoreError( ERROR_MOUNTPOINT_EMPTY, QString::null );
    m_working = false;
    emit state( MOUNTER_STOP );
  }
}


/***************************************************************************
   Unmounts all shares at once. (Public part)
***************************************************************************/

void Smb4KMounter::unmountAllShares()
{
  QString *input = new QString( QString( "%1" ).arg( UnmountAll ) );
  m_queue.enqueue( input );
}


/***************************************************************************
   Unmounts all shares at once.
***************************************************************************/

void Smb4KMounter::unmountAll()
{
  for ( QValueListIterator<Smb4KShare *> it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
  {
    unmountShare( *it, true );
  }

  m_working = false;
}


/***************************************************************************
   Starts any process.
***************************************************************************/

void Smb4KMounter::startProcess( int state )
{
  m_buffer = QString::null;
  m_state = state;

  if ( m_state != Import )
  {
    QApplication::setOverrideCursor( waitCursor );
  }

  m_proc->start( KProcess::NotifyOnExit, KProcess::AllOutput );
}


/***************************************************************************
   Ends any process. This functions tells the mounter what to do
   afterwards.
***************************************************************************/

void Smb4KMounter::endProcess()
{
  switch ( m_state )
  {
    case Mount:
      processMount();
      break;
    case Unmount:
      processUnmount();
      break;
    case ForceUnmount:
      processUnmount();
      break;
    default:
      break;
  }

  m_state = Idle;

  if ( mp.data )
  {
    delete mp.data;
    mp.data = NULL;
  }

  QApplication::restoreOverrideCursor();
  m_proc->clearArguments();

  m_working = false;
  emit state( MOUNTER_STOP );
}


/***************************************************************************
   Process mounts.
***************************************************************************/

void Smb4KMounter::processMount()
{
  Smb4KShare *share = NULL;

#ifndef __FreeBSD__

  if ( m_proc->normalExit() )
  {
    if ( m_buffer.contains( "failed", true ) == 0 &&
         m_buffer.contains( "ERR", true ) == 0 &&
         m_buffer.contains( "/bin/sh:" ) == 0 &&
         m_buffer.contains( "mount:", true ) == 0 &&
         m_buffer.contains( "smbmnt" ) == 0 &&
         m_buffer.contains( mp.data->path() ) == 0 &&
         m_buffer.contains( "mount error" ) == 0 )
    {
      QString share_string;

      if ( mp.data->share().contains( "'" ) == 2 )
      {
        share_string = mp.data->share().replace( QRegExp( "'" ), "" );
      }

      QString name = QString( "//%1/%2" ).arg( mp.data->host() ).arg( share_string );

      bool broken = true;

      mp.thread.setMountpoint( mp.data->path() );
      mp.thread.start();
      mp.thread.wait( 1000 );
      mp.thread.terminate();
      mp.thread.wait();

      broken = mp.thread.isBroken();

      if ( QString::compare( mp.data->filesystem(), "smbfs" ) == 0 )
      {
        share =  new Smb4KShare( name, mp.data->path(), mp.data->filesystem(), (int)getuid(), (int)getgid(), broken );
        m_mounted_shares.append( share );
      }
      else if ( QString::compare( mp.data->filesystem(), "cifs" ) == 0 )
      {
        share = new Smb4KShare( name, mp.data->path(), mp.data->filesystem(), mp.data->cifsLogin(), false, broken );
        m_mounted_shares.append( share );
      }

      // If you change something here, please check the code in import()
      // as well.
      if ( share && !share->isBroken() )
      {
        struct statvfs fs;

        if ( statvfs( share->path().local8Bit(), &fs ) == -1 )
        {
          int err_code = errno;

          share->setTotalDiskSpace( -1 );
          share->setFreeDiskSpace( -1 );

          showCoreError( ERROR_GETTING_USAGE, strerror( err_code ) );
        }
        else
        {
          double kB_block = (double)(fs.f_bsize / 1024);
          double total = (double)(fs.f_blocks*kB_block);
          double free = (double)(fs.f_bfree*kB_block);

          share->setTotalDiskSpace( total );
          share->setFreeDiskSpace( free );
        }

        // You don't want to use ANY Qt function here, or Smb4K will
        // slow down dramatically.
      }
      else
      {
        share->setTotalDiskSpace( -1 );
        share->setFreeDiskSpace( -1 );
      }

      emit mountedShare( mp.data->path() );
    }
    else
    {
      if ( m_buffer.contains( "ERRbadpw" ) != 0 || m_buffer.contains( "ERRnoaccess" ) != 0
           || m_buffer.contains( "mount error 13 = Permission denied" ) != 0 )
      {
        QString share_string;

        if ( mp.data->share().contains( "'" ) == 2 )
        {
          share_string = mp.data->share().replace( QRegExp( "'" ), "" );
        }

        int state = Smb4KPasswordHandler::None;

        if ( m_buffer.contains( "ERRbadpw" ) != 0 )
        {
          state = Smb4KPasswordHandler::BadPassword;
        }
        else if ( m_buffer.contains( "ERRnoaccess" ) != 0 )
        {
          state = Smb4KPasswordHandler::AccessDenied;
        }
        else if ( m_buffer.contains( "mount error 13 = Permission denied" ) != 0 )
        {
          state = Smb4KPasswordHandler::PermDenied;
        }

        // If the user supplied auth information, we will retry mounting.
        if ( m_password_handler->askpass( mp.data->workgroup(), mp.data->host(), share_string, state ) )
        {
          mountShare( mp.data->workgroup(), mp.data->host(), mp.data->ip(), share_string );
        }
      }
      else if ( m_buffer.contains( "ERRnosuchshare" ) != 0 && mp.data->share().contains( "_" ) != 0 )
      {
        QString share_string = mp.data->share().replace( "_", " " );
        mountShare( mp.data->workgroup(), mp.data->host(), mp.data->ip(), share_string );
      }
      else
      {
        showCoreError( ERROR_MOUNTING_SHARE, m_buffer );
      }
    }
  }

#else

  if ( m_proc->normalExit() )
  {
    if ( m_buffer.contains( "syserr =", true ) == 0 &&
         /* To make sure we catch all errors, also check for the following
            strings. Maybe we can remove them?? */
         m_buffer.contains( "Authentication error", true ) == 0 &&
         m_buffer.contains( "Connection refused", true ) == 0 &&
         m_buffer.contains( "Operation not permitted", true ) == 0 )
    {
      import();  // FIXME: *cough* What is this for???

      QString share_string;

      if ( mp.data->share().contains( "'" ) == 2 )
      {
        share_string = mp.data->share().replace( QRegExp( "'" ), "" );
      }

      Smb4KAuthInfo *auth = m_password_handler->readAuth( mp.data->workgroup(), mp.data->host(), share_string );

      QString name = QString( "//%1@%2/%3" ).arg( auth->user().upper(), mp.data->host().upper(), share_string.upper() );

      delete auth;

      bool broken = true;

      mp.thread.setMountpoint( mp.data->path() );
      mp.thread.start();
      mp.thread.wait( 1000 );
      mp.thread.terminate();
      mp.thread.wait();

      broken = mp.thread.isBroken();

      share = new Smb4KShare( name, mp.data->path(), mp.data->filesystem(), (int)getuid(), (int)getgid(), broken );
      m_mounted_shares.append( share );

      // If you change something here, please check the code in import()
      // as well.
      if ( share && !share->isBroken() )
      {
        struct statvfs fs;

        if ( statvfs( share->path().local8Bit(), &fs ) == -1 )
        {
          int err_code = errno;

          share->setTotalDiskSpace( -1 );
          share->setFreeDiskSpace( -1 );

          showCoreError( ERROR_GETTING_USAGE, strerror( err_code ) );
        }
        else
        {
          double kB_block = (double)(fs.f_bsize / 1024);
          double total = (double)(fs.f_blocks*kB_block);
          double free = (double)(fs.f_bfree*kB_block);

          share->setTotalDiskSpace( total );
          share->setFreeDiskSpace( free );
        }

        // You don't want to use ANY Qt function here, or Smb4K will
        // slow down dramatically.
      }
      else
      {
        share->setTotalDiskSpace( -1 );
        share->setFreeDiskSpace( -1 );
      }

      emit mountedShare( mp.data->path() );
    }
    else
    {
      if ( m_buffer.contains( "Authentication error" ) != 0 )
      {
        QString share_string;

        if ( mp.data->share().contains( "'" ) == 2 )
        {
          share_string = mp.data->share().replace( QRegExp( "'" ), "" );
        }

        // If the user supplied auth information, we will retry mounting.
        if ( m_password_handler->askpass( mp.data->workgroup(), mp.data->host(), share_string, Smb4KPasswordHandler::AuthError ) )
        {
          mountShare( mp.data->workgroup(), mp.data->host(), mp.data->ip() , share_string );
        }
      }
      else
      {
        showCoreError( ERROR_MOUNTING_SHARE, m_buffer );
      }
    }
  }

#endif

  emit updated();
}


/***************************************************************************
   Process unmounts.
***************************************************************************/

void Smb4KMounter::processUnmount()
{
  config()->setGroup( "Mount Options" );
  QString mount_prefix = config()->readEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) );

  if ( m_proc->normalExit() )
  {
    // FIXME: This is a potential problem. What if the unmount
    // process returns some debugging info like the mount process
    // under SuSE? Need more info.
    if ( m_buffer.isEmpty() )
    {
      // Since the mount point is unique, the mount point is greped
      // and used to determine the share that has to be removed from
      // m_mountedShares.
      Smb4KShare *share = findShareByPath( mp.data->path() );

      if ( share->canonicalPath().startsWith( QDir( mount_prefix ).canonicalPath() ) )
      {
        QDir *d = new QDir( share->canonicalPath() );

        if ( d->rmdir( d->canonicalPath(), true ) )
        {
          d->cdUp();
          d->rmdir( d->canonicalPath(), true );
        }

        delete d;
      }

      m_mounted_shares.remove(share);
    }
    else
    {
      showCoreError( ERROR_UNMOUNTING_SHARE, m_buffer );
    }
  }

  emit updated();
}


/***************************************************************************
   Check if a share is already mounted
***************************************************************************/

bool Smb4KMounter::isMounted( const QString &name )
{
  return findShareByName( name ) != NULL ? true : false;
}


/***************************************************************************
   Find a share in the list with its path
***************************************************************************/

Smb4KShare* Smb4KMounter::findShareByPath(const QString &path)
{
  if ( path.isEmpty() )
  {
    return NULL;
  }

  QValueListIterator<Smb4KShare *> it;

  for ( it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
  {
    if( QString::compare( (*it)->path().upper(), path.upper() ) == 0 ||
        QString::compare( (*it)->canonicalPath().upper(), path.upper() ) == 0 )
    {
      break;
    }
  }

  return it == m_mounted_shares.end() ? NULL : *it;
}


/***************************************************************************
   Find a share in the list with its name
***************************************************************************/

Smb4KShare* Smb4KMounter::findShareByName(const QString &name)
{
  QString n = name;

  QValueListIterator<Smb4KShare *> it;

  for ( it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
  {
    if( QString::compare( (*it)->name().upper(), name.upper() ) == 0 )
    {
      break;
    }
    else if ( QString::compare( (*it)->name().upper(), n.replace( " ", "_" ).upper() ) == 0 )
    {
      break;
    }
    else
    {
      continue;
    }
  }

  return it == m_mounted_shares.end() ? NULL : *it;
}


/***************************************************************************
   Returns a list of mount points that belong to broken shares
***************************************************************************/

const QValueList<Smb4KShare *> Smb4KMounter::getBrokenShares()
{
  QValueList<Smb4KShare *> broken_shares;

  for ( QValueListIterator<Smb4KShare *> it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
  {
    if ( (*it)->isBroken() )
    {
      broken_shares.append( *it );

      continue;
    }
    else
    {
      continue;
    }
  }

  return broken_shares;
}


/***************************************************************************
   Returns the advanced Samba options as chosen by the user.
***************************************************************************/

const QString Smb4KMounter::getMountOptions()
{
  config()->setGroup( "Samba" );
  QString options;

  QMap<QString,QString> smb_map = globalSMBOptions();

  QString domain = config()->readEntry( "Domain", smb_map["workgroup"] );

#ifndef __FreeBSD__

  QString filesystem = config()->readEntry( "Mount Filesystem", "smbfs" );

  if ( QString::compare( filesystem, "smbfs" ) == 0 )
  {
    if ( config()->readBoolEntry( "Use Kerberos", false ) )
    {
      options.append( "krb," );
    }

    if ( !config()->readEntry( "NetBIOS Name", QString::null ).isEmpty() )
    {
      options.append( QString( "netbiosname='%1'," ).arg( config()->readEntry( "NetBIOS Name", QString::null ) ) );
    }

    if ( !config()->readEntry( "Socket Options", QString::null ).isEmpty() )
    {
      options.append( QString( "sockopt=\"%1\"," ).arg( config()->readEntry( "Socket Options", QString::null ) ) );
    }

    if ( !config()->readEntry( "NetBIOS Scope", QString::null ).isEmpty() )
    {
      options.append( QString( "scope=\"%1\"," ).arg( config()->readEntry( "NetBIOS Scope", QString::null ) ) );
    }

    if ( !config()->readEntry( "Mount Codepage", QString::null ).isEmpty() )
    {
      options.append( QString( "codepage=%1," ).arg( config()->readEntry( "Mount Codepage", QString::null ) ) );
    }

    if ( !config()->readEntry( "Mount Cache", QString::null ).isEmpty() )
    {
      options.append( QString( "ttl=%1," ).arg( config()->readEntry( "Mount Cache", QString::null ) ) );
    }

    if ( config()->readBoolEntry( "Mount Unicode", false ) )
    {
      options.append( "unicode," );
    }

    if ( config()->readBoolEntry( "Mount LFS", false ) )
    {
      options.append( "lfs," );
    }

    if ( !config()->readEntry( "Mount FMASK", QString::null ).isEmpty() )
    {
      options.append( QString( "fmask=%1," ).arg( config()->readEntry( "Mount FMASK", QString::null ) ) );
    }

    if ( !config()->readEntry( "Mount DMASK", QString::null ).isEmpty() )
    {
      options.append( QString( "dmask=%1," ).arg( config()->readEntry( "Mount DMASK", QString::null ) ) );
    }
  }
  else
  {
    // The file_mode and the dir_mode option _have_ to take the file mask
    // and the directory mask in octal, respectively!

    if ( !config()->readEntry( "Mount FMASK", QString::null ).isEmpty() )
    {
      options.append( QString( "file_mode=%1," ).arg( config()->readEntry( "Mount FMASK", QString::null ) ) );
    }

    if ( !config()->readEntry( "Mount DMASK", QString::null ).isEmpty() )
    {
      options.append( QString( "dir_mode=%1," ).arg( config()->readEntry( "Mount DMASK", QString::null ) ) );
    }

    if ( config()->readBoolEntry( "Mount RSize", false ) )
    {
      options.append( "rsize," );
    }

    if ( config()->readBoolEntry( "Mount WSize", false ) )
    {
      options.append( "wsize," );
    }
  }

  if ( !config()->readEntry( "Mount Charset", QString::null ).isEmpty() )
  {
    options.append( QString( "iocharset=%1," ).arg( config()->readEntry( "Mount Charset", QString::null ) ) );
  }

  if ( !config()->readEntry( "Mount UID", QString::null ).isEmpty() )
  {
    options.append( QString( "uid=%1," ).arg( config()->readEntry( "Mount UID", QString::null ) ) );
  }

  if ( !config()->readEntry( "Mount GID", QString::null ).isEmpty() )
  {
    options.append( QString( "gid=%1," ).arg( config()->readEntry( "Mount GID", QString::null ) ) );
  }

  options.append( QString( "port=%1," ).arg( config()->readNumEntry( "Port", 139 ) ) );

  if ( config()->readBoolEntry( "Mount ReadWrite", true ) )
  {
    options.append( "rw," );
  }
  else
  {
    options.append( "ro," );
  }

#else

  if ( !config()->readEntry( "Mount UID", QString::null ).isEmpty() )
  {
    options.append( QString( "-u %1" ).arg( config()->readEntry( "Mount UID", QString::null ) ) );
  }

  if ( !config()->readEntry( "Mount GID", QString::null ).isEmpty() )
  {
    options.append( QString( " -g %1" ).arg( config()->readEntry( "Mount GID", QString::null ) ) );
  }

  if ( !config()->readEntry( "Mount FMASK", QString::null ).isEmpty() )
  {
    options.append( QString( " -f %1" ).arg( config()->readEntry( "Mount FMASK", QString::null ) ) );
  }

  if ( !config()->readEntry( "Mount DMASK", QString::null ).isEmpty() )
  {
    options.append( QString( " -d %1" ).arg( config()->readEntry( "Mount DMASK", QString::null ) ) );
  }

  if ( !config()->readEntry( "Mount Charset", QString::null ).isEmpty() && !config()->readEntry( "Mount Codepage", QString::null ).isEmpty() )
  {
    options.append( QString( " -E %1:%2" ).arg( config()->readEntry( "Mount Charset", QString::null ) ).arg( config()->readEntry( "Mount Codepage", QString::null ) ) );
  }

#endif

  return options;
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


void Smb4KMounter::slotProcessExited( KProcess * )
{
  endProcess();
}


void Smb4KMounter::slotReceivedStdout( KProcess *, char *buf, int len )
{
  m_buffer.append( QString::fromLocal8Bit( buf, len ) );
}


void Smb4KMounter::slotReceivedStderr( KProcess *, char *buf, int len )
{
  m_buffer.append( QString::fromLocal8Bit( buf, len ) );
}


void Smb4KMounter::init()
{
  if ( !m_working && !m_queue.isEmpty() )
  {
    // Tell the mounter, that it is busy.
    m_working = true;

    QString *item = m_queue.dequeue();
    int todo = item->section( ":", 0, 0 ).toInt();

    switch ( todo )
    {
      case MountRecent:
        mountRecent();
        break;
      case Import:
        import();
        break;
      case Mount:
        emit state( MOUNTER_MOUNTING );
        mount( item->section( ":", 1, 1 ), item->section( ":", 2, 2 ), item->section( ":", 3, 3 ), item->section( ":", 4, 4 ) );
        break;
      case Unmount:
        emit state( MOUNTER_UNMOUNTING );
        unmount( item->section( ":", 1, 1 ), (bool)item->section( ":", 2, 2 ).toInt() );
        break;
      case ForceUnmount:
        emit state( MOUNTER_UNMOUNTING );
        forceUnmount( item->section( ":", 1, 1 ) );
        break;
      case UnmountAll:
        unmountAll();
        break;
      default:
        break;
    }

    delete item;
  }

  mp.timerTicks++;

  config()->setGroup( "Mount Options" );
  int interval = config()->readNumEntry( "Check Interval", 2500 );

  if ( mp.timerTicks * timerInterval() >= interval /* msec */ && (!m_working || m_queue.isEmpty()) )
  {
    m_queue.enqueue( new QString( QString( "%1:" ).arg( Import ) ) );
    mp.timerTicks = 0;
  }
}


void Smb4KMounter::slotShutdown()
{
  config()->setGroup( "Mount Options" );
  bool remount = config()->readBoolEntry( "Mount Recent", false );
  bool unmount_on_exit = config()->readBoolEntry( "Unmount All", false );
  QString mount_prefix = config()->readEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) );

  config()->deleteGroup( "Recently Mounted Shares" );

  if ( remount )
  {
    config()->setGroup( "Recently Mounted Shares" );

    int i = 0;

    for ( QValueListIterator<Smb4KShare *> it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
    {
      if ( (*it)->uid() == (int)getuid() || (*it)->canonicalPath().startsWith( QDir( mount_prefix ).canonicalPath() ) || (*it)->canonicalPath().startsWith( QDir::home().canonicalPath() ) )
      {
        config()->writeEntry( QString( "%1" ).arg( i++ ), (*it)->name() );
      }
      else
      {
        continue;
      }
    }
  }

  QDir dir;

  dir.cd( mount_prefix );

  QStringList dirs = dir.entryList( QDir::Dirs, QDir::DefaultSort );

  QValueList<Smb4KShare *> broken_shares = getBrokenShares();

  for ( QStringList::ConstIterator it = dirs.begin(); it != dirs.end(); ++it )
  {
    if ( QString::compare( *it, "." ) != 0 && QString::compare( *it, ".." ) != 0 )
    {
      bool broken = false;

      for ( QValueListIterator<Smb4KShare *> bs = broken_shares.begin(); bs != broken_shares.end(); ++bs )
      {
        if ( (*bs)->path().startsWith( mount_prefix+*it ) || (*bs)->canonicalPath().startsWith( mount_prefix+*it ) )
        {
          broken = true;

          break;
        }
        else
        {
          continue;
        }
      }

      if ( !broken )
      {
        dir.cd( *it );

        QStringList subdirs = dir.entryList( QDir::Dirs, QDir::DefaultSort );

        for ( QStringList::ConstIterator i = subdirs.begin(); i != subdirs.end(); ++i )
        {
          if ( QString::compare( *i, "." ) != 0 && QString::compare( *i, ".." ) != 0 )
          {
            dir.rmdir( *i );
          }
        }

        dir.cdUp();
        dir.rmdir( *it );
      }
    }
  }

  broken_shares.clear();

  if ( unmount_on_exit )
  {
    config()->setGroup( "Super User Privileges" );
    bool run_suid = config()->readBoolEntry( "Run SUID", false );
    QString suid_program = config()->readEntry( "SUID Program", QString::null );

    KProcess p;
    p.setUseShell( true );
    p.detach();

    QString command;

    for ( QValueListIterator<Smb4KShare *> it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
    {
      if ( !(*it)->isForeign() )
      {
        if ( run_suid )
        {
          command.append( QString( "%1 smb4k_umount --suid --%2 " ).arg( suid_program ).arg( (*it)->filesystem() ) ).append( KProcess::quote( (*it)->path().replace( '\044', "\044" ) ) ).append( " ; " );
        }
        else
        {
          command.append( QString( "smb4k_umount --no-suid --%1 " ).arg( (*it)->filesystem() ) ).append( KProcess::quote( (*it)->path().replace( '\044', "\044" ) ) ).append( " ; " );
        }

#ifndef __FreeBSD__
        command.append( "rmdir --ignore-fail-on-non-empty " ).append( KProcess::quote( (*it)->canonicalPath().replace( '\044', "\044" ) ) ).append( " ; " );
        command.append( "rmdir --ignore-fail-on-non-empty " ).append( KProcess::quote( (*it)->canonicalPath().section( "/", 0, -2 ) ) ).append( " ; " );
#else
        command.append( "rmdir " ).append( KProcess::quote( (*it)->canonicalPath().replace( '\044', "\044" ) ) ).append( " ; " );
        command.append( "rmdir " ).append( KProcess::quote( (*it)->canonicalPath().section( "/", 0, -2 ) ) ).append( " ; " );
#endif
      }
      else
      {
        continue;
      }
    }

    command.truncate( command.length() - 2 );

    p << command;
    p.start( KProcess::DontCare, KProcess::NoCommunication );
  }
}


#include "smb4kmounter.moc"
