package SliMP3::Buttons::Home;

# SliMP3 Server Copyright (C) 2001 Sean Adams, Slim Devices Inc.
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License,
# version 2.

use strict;
use File::Spec::Functions qw(:ALL);
use File::Spec::Functions qw(updir);

use SliMP3::Buttons::BrowseID3;
use SliMP3::Buttons::Common;
use SliMP3::Buttons::Playlist;
use SliMP3::Buttons::Search;
use SliMP3::Buttons::Settings;
use SliMP3::Buttons::Synchronize;
use SliMP3::Strings qw (string);

# button functions for top-level home directory

my @homeChoices;

my %functions = (
	'add' => sub  {
		my $client = shift;
	
		if ($homeChoices[$client->homeSelection] eq 'MUSIC') {
			# add the whole of the music folder to the playlist!
			SliMP3::Buttons::Block::block($client, string('ADDING_TO_PLAYLIST'), string('MUSIC'));
			SliMP3::Command::execute($client, ['playlist', 'add', SliMP3::Prefs::get('mp3dir')], \&SliMP3::Buttons::Block::unblock, [$client]);
		} elsif($homeChoices[$client->homeSelection] eq 'NOW_PLAYING') {
			SliMP3::Animation::showBriefly($client, string('CLEARING_PLAYLIST'), '');
			SliMP3::Command::execute($client, ['playlist', 'clear']);
		} else {
			(getFunctions())->{'right'}($client);
		}
	},
	'play' => sub  {
		my $client = shift;
	
		if ($homeChoices[$client->homeSelection] eq 'MUSIC') {
			# play the whole of the music folder!
			if (SliMP3::Playlist::shuffle($client)) {
				SliMP3::Buttons::Block::block($client, string('PLAYING_RANDOMLY_FROM'), string('MUSIC'));
			} else {
				SliMP3::Buttons::Block::block($client, string('NOW_PLAYING_FROM'), string('MUSIC'));
			}
			SliMP3::Command::execute($client, ['playlist', 'load', SliMP3::Prefs::get('mp3dir')], \&SliMP3::Buttons::Block::unblock, [$client]);
		} elsif($homeChoices[$client->homeSelection] eq 'NOW_PLAYING') {
			SliMP3::Command::execute($client, ['play']);
			#The address of the %functions hash changes from compile time to run time
			#so it is necessary to get a reference to it from a function outside of the hash
			(getFunctions())->{'right'}($client);
		}  elsif (($homeChoices[$client->homeSelection] eq 'BROWSE_BY_GENRE') ||
				  ($homeChoices[$client->homeSelection] eq 'BROWSE_BY_ARTIST') ||
				  ($homeChoices[$client->homeSelection] eq 'BROWSE_BY_ALBUM')) {
			if (SliMP3::Playlist::shuffle($client)) {
				SliMP3::Buttons::Block::block($client, string('PLAYING_RANDOMLY'), string('EVERYTHING'));
			} else {
				SliMP3::Buttons::Block::block($client, string('NOW_PLAYING'), string('EVERYTHING'));
			}
			SliMP3::Command::execute($client, ["playlist", "loadalbum", "*", "*", "*"], \&SliMP3::Buttons::Block::unblock, [$client]);
		} else {
			(getFunctions())->{'right'}($client);
		}
	},
	'up' => sub  {
		my $client = shift;
		my $newposition = SliMP3::Buttons::Common::scroll($client, -1, ($#homeChoices + 1), $client->homeSelection);
		$client->homeSelection($newposition);
		SliMP3::Display::update($client);
	},
	'down' => sub  {
		my $client = shift;
		my $newposition = SliMP3::Buttons::Common::scroll($client, +1, ($#homeChoices + 1), $client->homeSelection);
		$client->homeSelection($newposition);
		SliMP3::Display::update($client);
	},
	'left' => sub  {
		my $client = shift;
		# doesn't do anything, we're already at the top level
		SliMP3::Animation::bumpLeft($client);
	},
	'right' => sub  {
		my $client = shift;
		my @oldlines = SliMP3::Display::curLines($client);
		# navigate to the current selected top level item:
		if ($homeChoices[$client->homeSelection] eq 'NOW_PLAYING') {
			# reset to the top level of the music
			SliMP3::Buttons::Common::pushModeLeft($client, 'playlist');
		} elsif ($homeChoices[$client->homeSelection] eq 'BROWSE_BY_GENRE') {
			SliMP3::Buttons::Common::pushModeLeft($client, 'browseid3',{});
		} elsif ($homeChoices[$client->homeSelection] eq 'BROWSE_BY_ARTIST') {
			SliMP3::Buttons::Common::pushModeLeft($client, 'browseid3',{'genre'=>'*'});
		} elsif ($homeChoices[$client->homeSelection] eq 'BROWSE_BY_ALBUM') {
			SliMP3::Buttons::Common::pushModeLeft($client, 'browseid3', {'genre'=>'*', 'artist'=>'*'});
		} elsif ($homeChoices[$client->homeSelection] eq 'SETTINGS') {
			SliMP3::Buttons::Common::pushModeLeft($client, 'settings');
		} elsif ($homeChoices[$client->homeSelection] eq 'PLUGINS') {
			SliMP3::Buttons::Common::pushModeLeft($client, 'plugins');
		} elsif ($homeChoices[$client->homeSelection] eq 'BROWSE_MUSIC_FOLDER') {
			# reset to the top level of the music
			SliMP3::Buttons::Common::pushMode($client, 'browse');
			SliMP3::Buttons::Browse::loadDir($client, '', 'right', \@oldlines);
		} elsif ($homeChoices[$client->homeSelection] eq 'SAVED_PLAYLISTS') {
			SliMP3::Buttons::Common::pushMode($client, 'browse');
			SliMP3::Buttons::Browse::loadDir($client, '__playlists', 'right', \@oldlines);
		} elsif ($homeChoices[$client->homeSelection] eq 'SEARCH_FOR_ARTISTS') {
			SliMP3::Buttons::SearchFor::searchFor($client, 'ARTISTS');
			SliMP3::Buttons::Common::pushModeLeft($client, 'searchfor');
		} elsif ($homeChoices[$client->homeSelection] eq 'SEARCH_FOR_ALBUMS') {
			SliMP3::Buttons::SearchFor::searchFor($client, 'ALBUMS');
			SliMP3::Buttons::Common::pushModeLeft($client, 'searchfor');
		} elsif ($homeChoices[$client->homeSelection] eq 'SEARCH_FOR_SONGS') {
			SliMP3::Buttons::SearchFor::searchFor($client, 'SONGS');
			SliMP3::Buttons::Common::pushModeLeft($client, 'searchfor');
		} elsif ($homeChoices[$client->homeSelection] eq 'SEARCH') {
			SliMP3::Buttons::Common::pushModeLeft($client, 'search');
		}
	},
	'numberScroll' => sub  {
		my $client = shift;
		my $button = shift;
		my $digit = shift;
		my $newpos;
		$client->homeSelection(SliMP3::Buttons::Common::numberScroll($client, $digit, \@homeChoices, 0));
		SliMP3::Display::update($client);
	}
);

sub getFunctions {
	return \%functions;
}

sub setMode {
	my $client = shift;

	$client->lines(\&lines);
	updateMenu($client);
 	if (!defined($client->homeSelection) || $client->homeSelection < 0 || $client->homeSelection >= scalar(@homeChoices)) {
 		$client->homeSelection(0);
 	}
 }
 
 my @menuOptions = ('NOW_PLAYING',
 					'BROWSE_BY_GENRE',
 					'BROWSE_BY_ARTIST',
 					'BROWSE_BY_ALBUM',
 					'BROWSE_MUSIC_FOLDER',
 					'SEARCH_FOR_ARTISTS',
 					'SEARCH_FOR_ALBUMS',
 					'SEARCH_FOR_SONGS',
 					'SEARCH',
 					'SAVED_PLAYLISTS',
 					'PLUGINS',
 					'SETTINGS',
 					);
 
sub menuOptions {
	my %menuChoices = ();
	$menuChoices{""} = "";
	foreach my $menuOption (@menuOptions) {
		if ($menuOption eq 'BROWSE_MUSIC_FOLDER' && SliMP3::iTunes::useiTunesLibrary()) {
			next;
		}
		if ($menuOption eq 'SAVED_PLAYLISTS' && !SliMP3::Prefs::get('playlistdir')) {
			next;
		}
		$menuChoices{$menuOption} = string($menuOption);
	}
	return %menuChoices;
}

sub unusedMenuOptions {
	my %menuChoices = menuOptions();
	delete $menuChoices{""};
	foreach my $usedOption (@homeChoices) {
		delete $menuChoices{$usedOption};
	}
	return sort(keys(%menuChoices));
}

sub updateMenu {
	my $client = shift;
	@homeChoices = ();
	
	foreach my $menuItem (SliMP3::Prefs::getArray('menuItem')) {
		if ($menuItem eq 'BROWSE_MUSIC_FOLDER' && SliMP3::iTunes::useiTunesLibrary()) {
			next;
		}
		if ($menuItem eq 'SAVED_PLAYLISTS' && !SliMP3::Prefs::get('playlistdir') && !SliMP3::iTunes::useiTunesLibrary()) {
			next;
		}
		push @homeChoices, $menuItem;
	}
	if (!scalar @homeChoices) {
		push @homeChoices, 'NOW_PLAYING';
	}
}
 
sub jump {
	my $client = shift;
	my $item = shift;
	my $pos = 0;

	if (defined($item)) {
		foreach my $i (@homeChoices) {
			last if ($i eq $item);
			$pos++;
		}
	}

	if ($pos > scalar @homeChoices) {
		$pos = 0;
	}

	$client->homeSelection($pos);
}
#
# figure out the lines to be put up to display the directory
#
sub lines {
	my $client = shift;
	my ($line1, $line2);

	$line1 = string('HOME');
	$line2 = string($homeChoices[$client->homeSelection]);

	return ($line1, $line2, undef, SliMP3::VFD::symbol('rightarrow'));
}

1;

__END__
