/*
    This file is part of the sly ident daemon (slidentd).  slidentd 
    was written by Sean Hunter <sean@uncarved.com> as a minimal 
    RFC1413 (ident) daemon.

    slidentd is copyright (c) 2001 Uncarved Systems Ltd.

    slidentd is free software; you can redistribute it and/or modify
    it under the terms of version 2 of the GNU General Public License 
    as published by the Free Software Foundation.

    slidentd is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    in the file COPYING along with slidentd; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "slid_get_input.h"
#include "slid_read.h"
#include "slid_log.h"
#include <fmt.h>
#include <byte.h>
#include <scan.h>
#include <buffer.h>
#include <stralloc.h>
#include <stdlib.h>

#ifndef NDEBUG
#define SLID_INPUT_DEBUG_MESSAGES
#endif

#if SLIDENTD_LOG_BADINPUT != 0
static void
slid_badinput_log(stralloc * buf, int len)
{
	buffer *logbuf = slid_get_log_buffer();
	stralloc *remote_ip = slid_get_remote_ip();
	stralloc *remote_port = slid_get_remote_port();

	slid_put_datestamp(logbuf);
	buffer_puts(logbuf, "Input: ");
	buffer_put(logbuf, buf->s, buf->len);
	buffer_puts(logbuf, " Received from ");
	buffer_puts(logbuf, remote_ip->s);
	buffer_puts(logbuf, ":");
	buffer_puts(logbuf, remote_port->s);
	buffer_puts(logbuf, ". Buffer length ");
	buffer_putlong(logbuf, (long) (SLIDENTD_MAX_INPUT_BUFFLEN));
	buffer_puts(logbuf, " - ");
	buffer_putlong(logbuf, (long) len);
	buffer_putsflush(logbuf, " bytes parsed before error\n");
}
#else
#define slid_badinput_log(x,y)
#endif

/* get and parse a line of input, returning the port numbers that are in the request */
void
slid_get_input(int *server_portnum, int *client_portnum)
{
	stralloc line_buf = { 0 };
	char *read_buf;
	unsigned long dummy;
	buffer *logbuf = slid_get_log_buffer();

	int comma_pos = -1;
	int pos;

	read_buf = alloca(SLIDENTD_MAX_INPUT_BUFFLEN + 1);
	byte_zero(read_buf, SLIDENTD_MAX_INPUT_BUFFLEN + 1);

	if (slid_read(STDIN_FILENO, read_buf, SLIDENTD_MAX_INPUT_BUFFLEN) < 0) {
		slid_put_datestamp(logbuf);
		buffer_puts(logbuf, "Error reading client request: ");
		buffer_puts(logbuf, strerror(errno));
		buffer_putsflush(logbuf, "\n");
	}

	stralloc_copyb(&line_buf, read_buf, SLIDENTD_MAX_INPUT_BUFFLEN + 1);
	stralloc_0(&line_buf);

	/* Skip initial whitespace */
	pos = scan_whitenskip(line_buf.s, line_buf.len);

	/* The next thing should be the server port number */
	pos += scan_ulong(line_buf.s + pos, &dummy);
	if (dummy != (unsigned long) (int) dummy || dummy == 0UL
	    || dummy > (unsigned long) SLIDENTD_MAX_PORTNUM) {
		slid_badinput_log(&line_buf, pos);
		slid_die("Unable to extract a valid server port\n");
	}
	*server_portnum = (int) dummy;

	/* ...possibly followed by whitespace... */
	pos += scan_whitenskip(line_buf.s + pos, line_buf.len - pos);

	/* ...followed by a comma... */
	comma_pos = scan_noncharsetnskip(line_buf.s, ",", line_buf.len);
	if (comma_pos == line_buf.len) {
		stralloc *remote_ip = slid_get_remote_ip();
		stralloc *remote_port = slid_get_remote_port();

		slid_badinput_log(&line_buf, pos);
		slid_put_datestamp(logbuf);
		buffer_puts(logbuf,
			    "Invalid format - no valid comma in request string from ");
		buffer_puts(logbuf, remote_ip->s);
		buffer_puts(logbuf, ":");
		buffer_puts(logbuf, remote_port->s);
		buffer_putsflush(logbuf, "\n");
		slid_dies();
	}
	pos = comma_pos + 1;

	/* ...possibly followed by whitespace... */
	pos += scan_whitenskip(line_buf.s + pos, line_buf.len - pos);

	/* ...after that should be the client port number */
	pos += scan_ulong(line_buf.s + pos, &dummy);
	if (!pos || dummy != (unsigned long) (int) dummy || dummy == 0UL
	    || dummy > (unsigned long) SLIDENTD_MAX_PORTNUM) {
		slid_badinput_log(&line_buf, pos);
		slid_die("Unable to extract a valid client port\n");
	}
	*client_portnum = (int) dummy;
#if 0
	/* ...after that should be nothing else other than more whitespace.. */
	pos += scan_charsetnskip(line_buf.s + pos, "\r\n\t \0");
	if ((unsigned) pos != line_buf.len) {
		slid_badinput_log(&line_buf, pos);
		slid_die("Additional non-whitespace characters in input\n");
	}
#endif

#ifdef SLID_INPUT_DEBUG_MESSAGES
	{

		slid_put_datestamp(logbuf);
		buffer_puts(logbuf, "Input: ");
		buffer_put(logbuf, line_buf.s, line_buf.len);
		buffer_puts(logbuf, " (");
		buffer_putint(logbuf, line_buf.len);
		buffer_puts(logbuf, " bytes read) got [");
		buffer_putint(logbuf, *server_portnum);
		buffer_puts(logbuf, "] - [");
		buffer_putint(logbuf, *client_portnum);
		buffer_putsflush(logbuf, "]\n");
	}
#endif
	stralloc_free(&line_buf);
}
